/*
 * $Id: cmd_set.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <string.h>

#define C_GFLAG (1 << C_LFLG_SHFT)
#define C_TFLAG (2 << C_LFLG_SHFT)

/* Forward declaration
 */
void set_usage(command_t *);

static char *_debug_components;
static uint64_t _trace_threshold = 0;

/* 
 * set_cmd() -- Command for ...
 */
int
set_cmd(command_t *cmd)
{

	if(cmd->nargs==0){
		if(cmd->flags & C_GFLAG){
			if(set_klib_dbg_components(_debug_components))
				kl_error("Couldn't set debug components.\n");
		}
		if(cmd->flags & C_TFLAG){
			if(kl_set_trace_threshold(_trace_threshold)){
				kl_error("Invalid trace thrshold.\n");
			}
		}
		if (cmd->flags == 0) {
			/* display settings */
			fprintf(cmd->ofp, "    Debug components: %s\n"
				"     Trace threshold: %u\n"
				" Iteration threshold: %"FMT64"u\n",
				get_klib_dbg_components(),
				kl_get_trace_threshold(),
				def_iter_threshold);
		}
	} else {
		fprintf(KL_ERRORFP, "Invalid commandline.\n");
		set_usage(cmd);
		return(1);
	}
	return(0);
}

#define _SET_USAGE \
"\n        [-g components]"\
"\n        [-i iteration_threshold]"\
"\n        [-T trace_threshold]"\
"\n        [-w outfile]"

#define _SET_HELP \
"Set or display lcrash settings.\n"\
"\nOPTIONS:"\
"\n -g components"\
"\n       Set debug components. I.e. components for which trace information"\
"\n       will be displayed. Supported debug components are:"\
"\n                 'alloc'    -    liballoc"\
"\n                 'bfd'      -    bfd handling"\
"\n                 'btree'    -    btree implementation"\
"\n                 'compress' -    (un)compressing routines for the dump"\
"\n                 'init'     -    klib initialization stuff"\
"\n                 'memmap'   -    memory mapping"\
"\n                 'module'   -    kernel module handling"\
"\n                 'signal'   -    signal handling"\
"\n                 'stabs'    -    stabs types handling"\
"\n                 'symbol'   -    symbol information handling"\
"\n                 'type'     -    type information handling"\
"\n       'none' stands for no component. 'all' means all components."\
"\n       To specify more than one component, the arguments must be"\
"\n       separated by commas (e.g. \"-g symbol,bfd\"). If 'all' is"\
"\n       specified together with other debug components, this means to"\
"\n       turn on all debug components and afterwards to turn off"\
"\n       (\"negate\") the other ones."\
"\n -i iteration_threshold"\
"\n       By default certain loops are interrupted after 10'000 iterations"\
"\n       to avoid endless loops while following invalid pointers. Using"\
"\n       this option you can change the threshold for the current command."\
"\n       A value '0' means infinite iteration threshold, i.e. no"\
"\n       interruption of the loop is caused by reaching any threshold."\
"\n -T trace_threshold"\
"\n       Set global trace threshold for lcrash/libklib debugging."\
"\n       Default is 0. Possible values are 0-4. Trace messages are shown"\
"\n       only for certain debug components -- see \"-g\" option."\
"\nWithout any options, 'set' displays current settings for all configurable"\
"\nvalues."

/*
 * set_usage() -- Print the usage string for the 'module' command.
 */
void
set_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _SET_USAGE);
}


/*
 * set_help() -- Print the help information for the 'module' command.
 */
void
set_help(command_t *cmd)
{
	CMD_HELP(cmd, _SET_USAGE, _SET_HELP);
}

/*
 * set_parse() -- Parse the command line arguments for 'module'.
 */
int
set_parse(command_t *cmd)
{
	option_t *op;

	_debug_components = NULL;

	if (set_cmd_flags(cmd, (C_WRITE), "g:i:T:")) {
		return(1);
	}
	op = cmd->options;

	while (op) {
		switch(op->op_char) {
			case 'g':
				cmd->flags |= C_GFLAG;
				_debug_components = op->op_arg;
				break;
			case 'i':
				cmd->flags |= C_ITER;
				kl_get_value(op->op_arg, 0, 0,
					     &def_iter_threshold);
				if(KL_ERROR){
					fprintf(KL_ERRORFP, "Invalid value for "
						"def_iter_threshold.\n");
					return(1);
				}
				break;
			case 'T':
				cmd->flags |= C_TFLAG;
				kl_get_value(op->op_arg, 0, 0,
					     &_trace_threshold);
				if(KL_ERROR){
					fprintf(KL_ERRORFP, "Invalid value for "
						"trace_threshold.\n");
					return(1);
				}
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * set_complete() -- Complete arguments of 'module' command.
 */
char *
set_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	set_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
