/*
 * $Id: dis.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <stdarg.h> /* for va_start() etc. */

/*
 * declaration of static functions
 */
static void list_instructions(FILE*);
static void dump_instr(kaddr_t, uint64_t, int, FILE*);

/* keep dump arch dependend dis stuff */
dis_info_t dis_info;

/*
 * init disassembler according to dump architecture
 */
int
dis_init(FILE *ofp, int dumparch)
{
	memset(&dis_info, 0, sizeof(dis_info_t));

	/* set defaults functions */
	LIST_INSTRUCTIONS = list_instructions;
	DUMP_INSTR = dump_instr;
	get_func_name = kl_lkup_symname;

	switch(dumparch){
#ifdef DUMP_ARCH_ALPHA
	case KL_ARCH_ALPHA:
		if(dis_init_alpha(ofp, dumparch)){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_ARM
	case KL_ARCH_ARM:
		if(dis_init_arm(ofp, dumparch)){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_I386
	case KL_ARCH_I386:
		if(dis_init_i386(ofp, dumparch)){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_IA64
	case KL_ARCH_IA64:
		if(dis_init_ia64(ofp, dumparch)){
			return(1);
		}
		break;
#endif
#ifdef DUMP_ARCH_PPC64
	case KL_ARCH_PPC64:
		if(dis_init_ppc64(ofp, dumparch)){
			return(1);
		}
		get_func_name = get_func_name_ppc64;
		break;
#endif
#if defined(DUMP_ARCH_S390) || defined(DUMP_ARCH_S390X)
	case KL_ARCH_S390:
	case KL_ARCH_S390X:
		if(dis_init_s390(ofp, dumparch)){
			return(1);
		}
		break;
#endif
	default:
		/* XXX set error code */
		return(1);
	}
	return(0);
}

/*
 * dis_fprintf()
 */
int
dis_fprintf(PTR ofp, const char *fmt, ...)
{
        char buffer[256];
        va_list ap;

        va_start(ap, fmt);
        vsprintf(buffer, fmt, ap);
        va_end(ap);

        i_fprintf((FILE *)ofp, "%s", buffer);
        return(0);
}

/*
 * getidmem()
 */
int
getidmem(bfd_vma addr, bfd_byte *buf, unsigned int length,
	 struct disassemble_info *dip)
{
	/* Fill the provided buffer with bytes from
	 * memory, starting at address 'addr' for 'length bytes.
	 */
	GET_BLOCK(addr, length, buf);
	if (KL_ERROR) {
		return(1);
	}
	return(0);
}

/*
 * dis_printintaddr()
 */
void
dis_printintaddr(bfd_vma addr, struct disassemble_info *dip, int flag)
{
	int offset = 0;
	syment_t *sp;

	if ((sp = kl_lkup_symaddr(addr))) {
		offset = addr - sp->s_addr;
	}

	/* Print out address
	 */
	dip->fprintf_func(dip->stream, "0x%"FMTPTR"x", (kaddr_t)addr);

	/* Print out symbol name 
	 */
	if (sp) {
		if (offset) {
			dip->fprintf_func(dip->stream, " <%s+%d>", 
				sp->s_name, offset);
		} else {
			dip->fprintf_func(dip->stream, " <%s>", sp->s_name);
		}
	} 

	/* Line things up properly for current function
	 */
	if (flag) {
		if (offset == 0) {
			dip->fprintf_func(dip->stream, ":       ");
		} else if (offset < 10) {
			dip->fprintf_func(dip->stream, ":     ");
		} else if (offset < 100) {
			dip->fprintf_func(dip->stream, ":    ");
		} else if (offset < 1000) {
			dip->fprintf_func(dip->stream, ":   ");
		} else if (offset < 10000) {
			dip->fprintf_func(dip->stream, ":  ");
		} else {
			dip->fprintf_func(dip->stream, ": ");
		}
	}
}

/*
 * dis_printaddr()
 */
void
dis_printaddr(bfd_vma addr, struct disassemble_info *dip)
{
	dis_printintaddr(addr, dip, 0);
}

/*
 * dis_getsym()
 */
int
dis_getsym(bfd_vma addr, struct disassemble_info *dip)
{
        return 0;
}

/*
 * set_dis_ofp()
 */
void
set_dis_ofp(FILE *ofp)
{
	/* Because the disassemble makes use of callbacks to display
	 * the disassembly output, we need to have a way to make sure
	 * the output is going where we want it to go. Since the disinfo
	 * struct is hidden from view, we have to have a function to do 
	 * this for us.
	 */
	if (ofp) {
		DISINFO.stream = ofp;	
	}
}

/*
 * list_instructions() -- stab routine which is set as default in dis_info.
 */
static void
list_instructions(FILE *ofp)
{
	fprintf(ofp, "This functionality is not provided for this dump "
		"architecture.\n");
	return;
}

/*
 * dump_instr() -- Just a stab routine which is set as default in dis_info.
 */
static void
dump_instr(kaddr_t addr, uint64_t count, int flags, FILE *ofp)
{
	fprintf(ofp, "This functionality is not provided for this dump "
		"architecture.\n");
	return;
}
