/*
 * $Id: lc_trace.h,v 1.1 2004/12/21 23:26:19 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#ifndef __LC_TRACE_H
#define __LC_TRACE_H

/* Stack frame updating macro */
#define UPDATE_FRAME(FUNCNAME, PC, RA, SP, FP, ASP, SRCNAME, LINE_NO, SIZE) \
        curframe->funcname = FUNCNAME; \
        curframe->pc = PC; \
        curframe->sp = SP; \
        curframe->ra = RA; \
        curframe->fp = FP; \
        curframe->asp = ASP; \
        curframe->srcfile = SRCNAME; \
        curframe->line_no = LINE_NO; \
        curframe->frame_size = SIZE; \
        curframe->ptr = curstkidx; \
        kl_enqueue((element_t **)&trace->frame, (element_t *)curframe); \
        trace->nframes++; \

/* internal structure to keep information about one stack frame */
typedef struct sframe_rec {
	struct sframe_rec      *next;
	struct sframe_rec      *prev;
	int			flag;
	int			level;
	char		       *funcname;
	char		       *srcfile;
	int			line_no;
	kaddr_t			pc;
	kaddr_t			ra;
	kaddr_t			sp;
	kaddr_t			fp;
	kaddr_t			bsp; /* for IPF */
	kaddr_t			pfs; /* for IPF */
	kaddr_t			cfm; /* for IPF */
	uaddr_t	                asp;
	int			frame_size;
	int			ptr;
        int                     msize; /* for IPF */
        int                     bsize; /* for IPF */
	uint64_t		error;
} sframe_t;

#define FRMFLG_MODNAME          0x1     /* Could not find funcname */

/* stack segment structure */
struct stack_s {
	int			type;
	uint32_t		size;
	kaddr_t			addr;
	void	               *ptr;
};

/* Stack types */
#define S_USERSTACK	0
#define S_KERNELSTACK	1
#define S_ASYNCSTACK    2

#define MAX_STACK_SEGMENTS 2
/* stack trace header */
typedef struct trace_rec {
	int			flags;
	kaddr_t			task;
	void			*tsp;
	struct stack_s		stack[MAX_STACK_SEGMENTS]; /* to be changed */
	int			stackcnt;
	sframe_t	       *frame;
	int			nframes;
} trace_t;

/* function pointers used for dump arch dependend tracing routines */
typedef int(*setup_trace_rec_t) (kaddr_t, kaddr_t, int, trace_t*);
typedef	void(*print_trace_t) (trace_t*, int, FILE*);
typedef	int(*task_trace_t) (kaddr_t, int, FILE*);
typedef	int(*print_traces_t) (kaddr_t, int, int, FILE*);
typedef	int(*find_trace_t) (kaddr_t, kaddr_t, kaddr_t, kaddr_t, trace_t*, int);
typedef	int(*dumptask_trace_t) (kaddr_t, int, FILE*);
typedef int(*do_list_t) (kaddr_t, int, FILE*);


/* internal structure to store dump arch specific trace stuff */
typedef struct trace_info_s{
	setup_trace_rec_t setup_trace_rec;
	print_trace_t     print_trace;
	task_trace_t      task_trace;
	print_traces_t    print_traces;
	find_trace_t      find_trace;
	dumptask_trace_t  dumptask_trace;
	do_list_t         do_list;
	int               stack_segments;
	kaddr_t           stack_size;
} trace_info_t;

/* place to keep dump arch dependend stuff for stack tracing */
extern trace_info_t trace_info;

#define	SETUP_TRACE_REC trace_info.setup_trace_rec
#define	PRINT_TRACE     trace_info.print_trace
#define	TASK_TRACE      trace_info.task_trace
#define	PRINT_TRACES    trace_info.print_traces
#define	FIND_TRACE      trace_info.find_trace
#define	DUMPTASK_TRACE  trace_info.dumptask_trace
#define DO_LIST         trace_info.do_list
#define STACK_SEGMENTS  trace_info.stack_segments
#define STACK_SIZE      trace_info.stack_size

/* include dump architecture dependend header */
#ifdef DUMP_ARCH_ALPHA
# include <lc_trace_alpha.h>
#endif
#ifdef DUMP_ARCH_ARM
# include <lc_trace_arm.h>
#endif
#ifdef DUMP_ARCH_IA64
# include <lc_trace_ia64.h>
#endif
#ifdef DUMP_ARCH_I386
# include <lc_trace_i386.h>
#endif
#ifdef DUMP_ARCH_PPC64
# include <lc_trace_ppc64.h>
#endif
#ifdef DUMP_ARCH_S390
# include <lc_trace_s390.h>
#endif
#ifdef DUMP_ARCH_S390X
# include <lc_trace_s390x.h>
#endif
#ifdef DUMP_ARCH_X86_64
# include <lc_trace_x86_64.h>
#endif

/*
 * function declarations
 */

/* allocate a stack frame structure */
sframe_t* alloc_sframe(trace_t*, int);

/* free stack frame structure */
void free_sframes(trace_t*);

/* allocate trace record */
trace_t*  alloc_trace_rec(int);

/* free trace record */
void free_trace_rec(trace_t*);

/* clean trace record */
void clean_trace_rec(trace_t*);

/* print banner for stack trace output */
void trace_banner(FILE*);

/* dump contents of stack frame */
void dump_stack_frame(trace_t*, sframe_t*, FILE*);

/* return offset of pc in from beginning of function */
int  pc_offset(kaddr_t);

/* init trace routines stuff -> calls arch dependent init routines */
int trace_init(int);

#endif /* __LC_TRACE_H */
