/*
 * $Id: kl_hwconfig.h,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libconfig.
 * A library which provides a framework for managing system hardware
 * and software configuration information.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#ifndef __KL_HWCONFIG_H
#define __KL_HWCONFIG_H

/* Control structure for hardware config data
 */
typedef struct hwconfig_s {
	int                    	 flags;           
	time_t                 	 time;       /* Time config was generated */
	uint64_t               	 sys_id;     /* system ID (can be zero) */
	int                    	 sys_type;   /* type of system */
	int                    	 hwcp_count;
	struct hwcomponent_s	*hwcp_root;
	string_table_t        	*st;         /* Pointer to String table */
} hwconfig_t;

/* Flag values -- must coexist with K_TEMP/K_PERM 
 */
#define LIVE_FLAG       0x04      /* config generated from live system */
#define STRINGTAB_FLAG  0x08      

/* Data structure that contains basic hardware inventory information
 */
typedef struct invent_data_s {
	int			inv_class;
	int                     inv_type;
	int                 	inv_controller;
	int                 	inv_unit;
	int                     inv_state;
	int                     inv_vhndl;
} invent_data_t;

#define HWAFLG(hcp) (((hcp)->flags & K_PERM) ? K_PERM : K_TEMP)

/* Every discrete part of a system is considered a component. The 
 * hwcomponent_s structure contains generic information about a 
 * single hardware component. Component specific data can be attached
 * via the c_data field. In addition, any architecture specific data
 * can be attached via the c_arch field. 
 *
 * Category and type values are to be defined by the application, as
 * are likely to be very architecture and platform specific.
 */
typedef struct hwcomponent_s {
	/* The order of the first six (6) fields MUST NOT be changed 
	 * as they map to fields in a htnode_s struct.
	 */
	struct hwcomponent_s *next;             /* Must be first */
	struct hwcomponent_s *prev;             /* Must be second */
	struct hwcomponent_s *parent;           /* component's parent */
	struct hwcomponent_s *cmplist;          /* list of sub-components */
	int                   seq;              /* sequence number */
	int                   level;            /* depth in hwconfig tree */
	uint64_t              sys_id;
	int                   flags;
	int                   category;         /* component category */
	int                   type;             /* component category type */
	invent_data_t         invent;           /* hw inventory info */
	char                 *location;         /* if board, slot name */
	char                 *name;             /* component name */
	char                 *serial_number;    /* component serial # */
	char                 *part_number;      /* component part # */
	char                 *revision;         /* component revision # */
	int                   enabled;          /* -1 if not relevant */
	void                 *type_data;        /* type specific data */
	void                 *arch;             /* arch specific data */
	void                 *pvt;              /* application data */
	hwconfig_t           *hwconfig;         /* hwconfig control struct */
} hwcomponent_t;

#define c_inv_class             invent.inv_class
#define c_inv_type              invent.inv_type
#define c_inv_controller        invent.inv_controller
#define c_inv_unit              invent.inv_unit
#define c_inv_state             invent.inv_state
#define c_inv_vhndl             invent.inv_vhndl

/* Major Hardware Component Categories -- Generic values to 
 * be used with all architectures and platforms.
 */
#define HW_SYSTEM               1000  /* The root of the hwconfig tree */
#define HW_PARTITION            1001
#define HW_RACK                 1002  
#define HW_MODULE               1003  
#define HW_POWER_SUPPLY         1004
#define HW_SYSBOARD             1005  /* motherboard, backplane, etc. */
#define HW_BOARD                1006  /* including daugter boards, etc. */
#define HW_CPU                  1007
#define HW_MEMORY               1008
#define HW_BUS                  1009
#define HW_IO                   1010
#define HW_PCI    		1011
#define HW_SCSI    		1012
#define HW_UNKNOWN            0xffff

/* HW_RACK Types
 */
#define RACK_UNKNOWN          0xffff

/*
 * HW_MODULE
 */
#define MODULE_UNKNOWN        0xffff

/* HW_SYSBOARD Types
 */
#define SBD_MOTHERBOARD         1001
#define SBD_BACKPLANE           1002
#define SBD_MIDPLANE            1003
#define SBD_FRONTPLANE          1004
#define SBD_UNKNOWN           0xffff

/* HW_BOARD Types 
 */
#define BD_NONE                    0
#define BD_PROCESSOR            1001  /* (System board w/ CPU, memory, etc. */
#define BD_VIDIO                1002  
#define BD_GRAPHICS             1003  
#define BD_NETWORK              1004  
#define BD_SCSI                 1005  
#define BD_IO                   1006  
#define BD_UNKNOWN            0xffff

/* HW_POWER_SUPPLY Types
 */
#define PWR_POWER_SUPPLY        1000 /* Generic Power Supply */
#define PWR_UNKNOWN           0xffff 

/* HW_CPU Types 
 */
#define CPU_MIPS                1000 /* Generic MIPS CPU type */
#define CPU_MIPS_R2000          1001 
#define CPU_MIPS_R2000A         1002 
#define CPU_MIPS_R3000          1003 
#define CPU_MIPS_R3000A         1004 
#define CPU_MIPS_R4000          1005 
#define CPU_MIPS_R4400          1006 
#define CPU_MIPS_R4600          1007 
#define CPU_MIPS_R4650          1008 
#define CPU_MIPS_R4700          1009 
#define CPU_MIPS_R5000          1010 
#define CPU_MIPS_R6000          1011 
#define CPU_MIPS_R6000A         1012 
#define CPU_MIPS_R8000          1013 
#define CPU_MIPS_R10000         1014 
#define CPU_MIPS_R12000         1015 
#define CPU_MIPS_R14000         1016 
/* Additional MIPS CPU types here */
#define CPU_INTEL               2000 /* Generic INTEL CPU type */
#define CPU_INTEL_I386		2001
#define CPU_INTEL_IA64		2002
/* Additional Intel CPU types here */

/* Additional CPU types here starting with 3000 */

#define CPU_UNKNOWN           0xffff

/* HW_MEMORY Types
 */
#define MEM_MAIN_MEMORY         1001
#define MEM_MEMORY_BANK         1002
#define MEM_UNKNOWN           0xffff 

/* HW_BUS Types
 */
#define BUS_SCSI                1001
#define BUS_PCI                 1002
#define BUS_UNKNOWN           0xffff

/* HW_IO Types
 */
#define IO_UNKNOWN            0xffff

/* DEVICE Types (for HW_PCI, HW_SCSI, etc.)
 */
#define DEV_FLOPPY	        1001
#define DEV_HD	        	1002
#define DEV_TAPE	        1003
#define DEV_CD		        1004
#define DEV_DVD	        	1005
#define DEV_UNKNOWN	      0xffff


/* Flag to indicate that hwcomponent one should be inserted after 
 * hwcomponent two. 
 */
#define INSERT_AFTER               1

/* Function prototypes
 */
hwcomponent_t *alloc_hwcomponent(
		hwconfig_t *  	/* hwconfig_s pointer */,
		int		/* category */,
		int		/* type */);

hwcomponent_t *dup_hwcomponent(
                hwcomponent_t * /* hwcomponent to dup */);

hwcomponent_t *next_hwcmp(
                hwcomponent_t * /* current hwcomponent_s pointer */);

hwcomponent_t *prev_hwcmp(
                hwcomponent_t * /* current hwcomponent_s pointer */);

void hwcmp_insert_next(
                hwcomponent_t * /* current hwcomponent_s pointer */,
                hwcomponent_t * /* next hwcomponent_s pointer */);

void hwcmp_add_child(
                hwcomponent_t * /* current hwcomponent_s pointer */,
                hwcomponent_t * /* pointer to child to add */);

void hwcmp_add_peer(
                hwcomponent_t * /* current hwcomponent_s pointer */,
                hwcomponent_t * /* pointer to peer to add */);

void replace_hwcomponent(
                hwcomponent_t * /* old hwcomponent_s pointer */,
                hwcomponent_t * /* new hwcomponent_s pointer */);

void insert_hwcomponent(
                hwcomponent_t * /* hwcomponent_s pointer */, 
                hwcomponent_t * /* hwcomponent_s pointer */, 
                int             /* flag */);

void unlink_hwcomponent(
                hwcomponent_t * /* hwcomponent_s pointer */);

void free_next_hwcmp(
                hwcomponent_t * /* hwcomponent_s pointer */); 

void free_hwcomponents(
                hwcomponent_t * /* hwcomponent_s pointer */);

hwcomponent_t *hw_find_location(
                hwcomponent_t * /* pointer to hwcomponent_s list */, 
                hwcomponent_t * /* hwconfig_s pointer */);

hwcomponent_t *hw_find_insert_point(
                hwcomponent_t * /* pointer to hwcomponent_s list */, 
                hwcomponent_t * /* hwconfig_s pointer */, 
                int *           /* pointer to location flag */);

int compare_hwcomponents(
                hwcomponent_t * /* first hwcomponent */, 
                hwcomponent_t * /* second hwcomponent */);

hwconfig_t *kl_alloc_hwconfig(
                int             /* flag */);

int kl_update_hwconfig(
                hwconfig_t *    /* hwconfig_s pointer */);

void kl_update_hwcomponent(
		hwcomponent_t *	/* pointer to hwcomponent */);

void kl_free_hwconfig(
                hwconfig_t *    /* hwconfig_s pointer */);

#endif /* __KL_HWCONFIG_H */
