#!/usr/bin/env python
#############################################################################
#
#  Linux Desktop Testing Project http://ldtp.freedesktop.org
# 
#  Author:
#     A. Nagappan <nagappan@gmail.com>
# 
#  Copyright 2004 - 2007 Novell, Inc.
# 
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Library General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
# 
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Library General Public License for more details.
# 
#  You should have received a copy of the GNU Library General Public
#  License along with this library; if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA 02110, USA.
#
#############################################################################

import re
import os
import sys
import time

# Let us not register our application under at-spi application list
os.environ ['GTK_MODULES'] = ''

try:
    import pyatspi as atspi, Accessibility
except ImportError:
    import atspi
    import Accessibility

from ldtplib.ldtplibutils import *

_ldtpDebug = os.getenv ('LDTP_DEBUG') # Enable debugging

class objInst:
    table = 0
    canvas = 0
    columnHeader = 0
    comboBox = 0
    pageTabList = 0
    pageTab = 0
    spinButton = 0
    button = 0
    tbutton = 0
    radioButton = 0
    checkBox = 0
    tree = 0
    treeTable = 0
    layeredPane = 0
    text = 0
    calView = 0
    panel = 0
    filler = 0
    menuBar = 0
    menu = 0
    separator = 0
    scrollBar = 0
    scrollPane = 0
    splitPane = 0
    slider = 0
    htmlContainer = 0
    progressBar = 0
    statusBar = 0
    toolBar = 0
    lst = 0
    label = 0
    icon = 0
    alert = 0
    dialog = 0
    unknown = 0
    embeddedComponent = 0

    def reset (self):
        self.table = 0
        self.canvas = 0
        self.columnHeader = 0
        self.comboBox = 0
        self.pageTabList = 0
        self.pageTab = 0
        self.spinButton = 0
        self.button = 0
        self.tbutton = 0
        self.radioButton = 0
        self.checkBox = 0
        self.tree = 0
        self.treeTable = 0
        self.layered_pane = 0
        self.text = 0
        self.calView = 0
        self.panel = 0
        self.filler = 0
        self.menuBar = 0
        self.menu = 0
        self.separator = 0
        self.scrollBar = 0
        self.scrollPane = 0
        self.splitPane = 0
        self.slider = 0
        self.htmlContainer = 0
        self.progressBar = 0
        self.statusBar = 0
        self.toolBar = 0
        self.label = 0
        self.lst = 0
        self.icon = 0
        self.alert = 0
        self.dialog = 0
        self.unknown = 0
        self.embeddedComponent = 0

class LdtpClientContext:
    def __init__ (self, localeLang, windowName, appHandle, appmap, request, response, guiHandle):
        self.localeLang = localeLang
        self.windowName = windowName
        self.appHandle  = appHandle
        self.appmap     = appmap
        self.request    = request
        self.response   = response
        self.guiHandle  = guiHandle

class LdtpGuiHandle:
    def __init__ (self, handle, classId):
        self.handle  = handle
        self.classId = classId

class Packet:
    def __init__ (self, packet, length):
        self.packet = packet
        self.length = length

class ParentPathInfo:
    def __init__ (self, childIndex, count):
        self.childIndex = childIndex
        self.count      = count

class Node:
    def __init__ (self, childIndex, node):
        self.childIndex = childIndex
        self.node = node

class LdtpRequest:
    def __init__ (self, requestType, command, application, requestId, context, component, actionName, argsList):
        self.requestType = requestType
        self.command     = command
        self.application = application
        self.requestId   = requestId
        self.context     = context
        self.component   = component
        self.actionName  = actionName
        self.argsList    = argsList

class LdtpResponse:
    def __init__ (self, status, data, length):
        self.status = status
        self.data   = data
        self.length = length

class SearchObjInfo:
    def __init__ (self, key, objIsWindow = False, pattern = None):
        self.key = key
        self.objIsWindow = objIsWindow
        self.pattern = pattern

class objInfo:
    def __init__ (self, prefix, objectType, instanceIndex, parentName):
        self.prefix     = prefix
        self.objType    = objectType
        self.instIndex  = instanceIndex
        self.parentName = parentName
        self.objectInst = objInst ()

class widget:
    def __init__ (self, widgetType, *args):
        self.widgetType = widgetType
        if len (args) == 5:
            self.label = args [0]
            self.parent = args [1]
            self.index = args [2]
            self.key   = args [3]
            self.hierarchy = args [4]
        else:
            self.label = ''
            self.parent = args [0]
            self.index = args [1]
            self.key   = args [2]
            self.hierarchy = args [3]

class window (widget):
    def __init__ (self, *args):
        self.children = {}
        widget.__init__ (self, *args)

class keyboard:
    def __init__ (self):
        pass
    charKeySynthVals = {38 : 'a', 56 : 'b', 54 : 'c', 40 : 'd', 26 : 'e', 41 : 'f', 42 : 'g', \
                        43 : 'h', 31 : 'i', 44 : 'j', 45 : 'k', 46 : 'l', 58 : 'm', 57 : 'n', \
                        32 : 'o', 33 : 'p', 24 : 'q', 27 : 'r', 39 : 's', 28 : 't', 30 : 'u', \
                        55 : 'v', 25 : 'w', 53 : 'x', 29 : 'y', 52 : 'z'} #A - Z
    digitKeySynthVals = {19 : '0', 10 : '1', 11 : '2', 12 : '3', 13 : '4', 14 : '5', 15 : '6', \
                         16 : '7', 17 : '8', 18 : '9'} #0 - 9
    symbolKeySynth = {20 : '-', 21 : '=', 34 : '[', 35 : ']', 47 : ';', 48 : '\'', 49 : '`', \
                      51 : '\\', 59 : ',', 60 : '.', 61 : '/', 10 : '!', 11 : '@', \
                      12 : '#', 13 : '$', 14 : '%', 15 : '^', 16 : '&', 17 : '*', 18 : '(', \
                      19 : ')', 20 : '_', 21 : '+', 34 : '{', 35 : '}', 47 : ':', 48 : '\"', \
                      49 : '~', 51 : '|', 59 : '<', 60 : '>', 61 : '?'}

    # PrtScrn: http://gentoo-wiki.com/TIP_Make_a_Screenshot_with_PrintScreen_Key
    # xmodmap -pke | grep -i print

    nonPrintKeySynth = {9 : '<escape>', 22 : '<backspace>',
                        37 : '<ctrl>', 115 : '<windowskey>',
                        23 : '<tab>', 36 : '<return>',
                        50 : '<shift>', 62 : '<shiftr>',
                        97 : '<home>', 103 : '<end>', 115 : '<window>',
                        64 : '<alt>', 113 : '<altr>',
                        98 : '<up>', 104 : '<down>', 102 : '<right>',
                        100 : '<left>', 65 : '<space>',
                        66 : '<caps>', 117 : '<menu>',
                        106 : '<insert>', 107 : '<delete>',
                        99 : '<pageup>', 105 : '<pagedown>',
                        77 : '<numlock>', 78 : '<scrolllock>',
                        67 : '<F1>', 68 : '<F2>', 69 : '<F3>', 70 : '<F4>',
                        71 : '<F5>', 72 : '<F6>', 73 : '<F7>', 74 : '<F8>',
                        75 : '<F9>', 76 : '<F10>', 95 : '<F11>', 96 : '<F12>',
                        111 : "prtscrn"}

class appmap:
    def __init__ (self, objectInst, registry):
        self.hashmap    = None
        self.hierarchy  = ''
        self.nodeIndex  = 0
        self.objTimeout = 0
        self.registry   = registry
        self.objectInst = objectInst
        self.lastNewContext      = None
        self.lastExistingContext = None
        self.applicationmap = window ('list_of_windows', None, None, None, None).children

    def getObjectInfo (self, accessible, parentName):
        role = accessible.getRole ()
        objectInfo = None
        if role == Accessibility.ROLE_PAGE_TAB:
            objectInfo = objInfo ('ptab', 'page_tab', self.objectInst.pageTab, parentName)
            self.objectInst.instIndex = self.objectInst.pageTab
            self.objectInst.pageTab += 1
        elif role == Accessibility.ROLE_PAGE_TAB_LIST:
            objectInfo = objInfo ('ptl', 'page_tab_list', self.objectInst.pageTabList, parentName)
            self.objectInst.instIndex = self.objectInst.pageTabList
            self.objectInst.pageTabList += 1
        elif role == Accessibility.ROLE_MENU:
            objectInfo = objInfo ('mnu', 'menu', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_CHECK_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'check_menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_RADIO_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'radio_menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_CHECK_BOX:
            objectInfo = objInfo ('chk', 'check_box', self.objectInst.checkBox, parentName)
            self.objectInst.instIndex = self.objectInst.checkBox
            self.objectInst.checkBox += 1
        elif role == Accessibility.ROLE_RADIO_BUTTON:
            objectInfo = objInfo ('rbtn', 'radio_button', self.objectInst.radioButton, parentName)
            self.objectInst.instIndex = self.objectInst.radioButton
            self.objectInst.radioButton += 1
        elif role == Accessibility.ROLE_SPIN_BUTTON:
            objectInfo = objInfo ('sbtn', 'spin_button', self.objectInst.spinButton, parentName)
            self.objectInst.instIndex = self.objectInst.spinButton
            self.objectInst.spinButton += 1
        elif role == Accessibility.ROLE_PUSH_BUTTON:
            objectInfo = objInfo ('btn', 'push_button', self.objectInst.button, parentName)
            self.objectInst.instIndex = self.objectInst.button
            self.objectInst.button += 1
        elif role == Accessibility.ROLE_TOGGLE_BUTTON:
            objectInfo = objInfo ('tbtn', 'toggle_button', self.objectInst.tbutton, parentName)
            self.objectInst.instIndex = self.objectInst.tbutton
            self.objectInst.tbutton += 1
        elif role == Accessibility.ROLE_TEXT:
            objectInfo = objInfo ('txt', 'text', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_PASSWORD_TEXT:
            objectInfo = objInfo ('txt', 'password_text', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_EDITBAR:
            objectInfo = objInfo ('txt', 'edit_bar', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_ENTRY:
            objectInfo = objInfo ('txt', 'entry', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_LIST:
            objectInfo = objInfo ('lst', 'list', self.objectInst.lst, parentName)
            self.objectInst.instIndex = self.objectInst.lst
            self.objectInst.lst += 1
        elif role == Accessibility.ROLE_TABLE:
            objectInfo = objInfo ('tbl', 'table', self.objectInst.table, parentName)
            self.objectInst.instIndex = self.objectInst.table
            self.objectInst.table += 1
        elif role == Accessibility.ROLE_TABLE_COLUMN_HEADER:
            objectInfo = objInfo ('tbl', 'table_column_header', self.objectInst.columnHeader, parentName)
            self.objectInst.instIndex = self.objectInst.columnHeader
            self.objectInst.columnHeader += 1
        elif role == Accessibility.ROLE_TREE:
            objectInfo = objInfo ('tree', 'tree', self.objectInst.tree, parentName)
            self.objectInst.instIndex = self.objectInst.tree
            self.objectInst.tree += 1
        elif role == Accessibility.ROLE_TREE_TABLE:
            objectInfo = objInfo ('ttbl', 'tree_table', self.objectInst.treeTable, parentName)
            self.objectInst.instIndex = self.objectInst.treeTable
            self.objectInst.treeTable += 1
        elif role == Accessibility.ROLE_TABLE_CELL:
            objectInfo = objInfo ('tblc', 'table_cell', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_FRAME:
            objectInfo = objInfo ('frm', 'frame', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_DIALOG:
            objectInfo = objInfo ('dlg', 'dialog', self.objectInst.dialog, parentName)
            self.objectInst.instIndex = self.objectInst.dialog
            self.objectInst.dialog += 1
        elif role == Accessibility.ROLE_WINDOW:
            objectInfo = objInfo ('dlg', 'dialog', self.objectInst.dialog, parentName)
            self.objectInst.instIndex = self.objectInst.dialog
            self.objectInst.dialog += 1
        elif role == Accessibility.ROLE_FONT_CHOOSER:
            objectInfo = objInfo ('dlg', 'font_chooser', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_FILE_CHOOSER:
            objectInfo = objInfo ('dlg', 'file_chooser', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_ALERT:
            objectInfo = objInfo ('dlg', 'alert', self.objectInst.alert, parentName)
            self.objectInst.instIndex = self.objectInst.alert
            self.objectInst.alert += 1
        elif role == Accessibility.ROLE_COMBO_BOX:
            objectInfo = objInfo ('cbo', 'combo_box', self.objectInst.comboBox, parentName)
            self.objectInst.instIndex = self.objectInst.comboBox
            self.objectInst.comboBox += 1
        elif role == Accessibility.ROLE_LAYERED_PANE:
            objectInfo = objInfo ('pane', 'layered_pane', self.objectInst.layeredPane, parentName)
            self.objectInst.instIndex = self.objectInst.layeredPane
            self.objectInst.layeredPane += 1
        elif role == Accessibility.ROLE_CALENDAR:
            objectInfo = objInfo ('calview', 'calendar_view', self.objectInst.calView, parentName)
            self.objectInst.instIndex = self.objectInst.calView
            self.objectInst.calView += 1
        elif role == Accessibility.ROLE_PANEL:
            objectInfo = objInfo ('pnl', 'panel', self.objectInst.panel, parentName)
            self.objectInst.instIndex = self.objectInst.panel
            self.objectInst.panel += 1
        elif role == Accessibility.ROLE_LABEL:
            objectInfo = objInfo ('lbl', 'label', self.objectInst.label, parentName)
            self.objectInst.instIndex = self.objectInst.label
            self.objectInst.label += 1
        elif role == Accessibility.ROLE_ICON:
            objectInfo = objInfo ('ico', 'icon', self.objectInst.icon, parentName)
            self.objectInst.instIndex = self.objectInst.icon
            self.objectInst.icon += 1
        elif role == Accessibility.ROLE_MENU_BAR:
            objectInfo = objInfo ('mbr', 'menu_bar', self.objectInst.menuBar, parentName)
            self.objectInst.instIndex = self.objectInst.menuBar
            self.objectInst.menuBar += 1
        elif role == Accessibility.ROLE_SCROLL_BAR:
            objectInfo = objInfo ('scbr', 'scroll_bar', self.objectInst.scrollBar, parentName)
            self.objectInst.instIndex = self.objectInst.scrollBar
            self.objectInst.scrollBar += 1
        elif role == Accessibility.ROLE_SCROLL_PANE:
            objectInfo = objInfo ('scpn', 'scroll_pane', self.objectInst.scrollPane, parentName)
            self.objectInst.instIndex = self.objectInst.scrollPane
            self.objectInst.scrollPane += 1
        elif role == Accessibility.ROLE_STATUS_BAR:
            objectInfo = objInfo ('stat', 'status_bar', self.objectInst.statusBar, parentName)
            self.objectInst.instIndex = self.objectInst.statusBar
            self.objectInst.statusBar += 1
        elif role == Accessibility.ROLE_SEPARATOR:
            objectInfo = objInfo ('spr', 'separator', self.objectInst.separator, parentName)
            self.objectInst.instIndex = self.objectInst.separator
            self.objectInst.separator += 1
        elif role == Accessibility.ROLE_FILLER:
            objectInfo = objInfo ('flr', 'filler', self.objectInst.filler, parentName)
            self.objectInst.instIndex = self.objectInst.filler
            self.objectInst.filler += 1
        elif role == Accessibility.ROLE_CANVAS:
            objectInfo = objInfo ('cnvs', 'canvas', self.objectInst.canvas, parentName)
            self.objectInst.instIndex = self.objectInst.canvas
            self.objectInst.canvas += 1
        elif role == Accessibility.ROLE_SLIDER:
            objectInfo = objInfo ('sldr', 'slider', self.objectInst.slider, parentName)
            self.objectInst.instIndex = self.objectInst.slider
            self.objectInst.slider += 1
        elif role == Accessibility.ROLE_SPLIT_PANE:
            objectInfo = objInfo ('splt', 'split_pane', self.objectInst.splitPane, parentName)
            self.objectInst.instIndex = self.objectInst.splitPane
            self.objectInst.splitPane += 1
        elif role == Accessibility.ROLE_HTML_CONTAINER:
            objectInfo = objInfo ('html', 'html_container', self.objectInst.htmlContainer, parentName)
            self.objectInst.instIndex = self.objectInst.htmlContainer
            self.objectInst.htmlContainer += 1
        elif role == Accessibility.ROLE_PROGRESS_BAR:
            objectInfo = objInfo ('pbr', 'progress_bar', self.objectInst.progressBar, parentName)
            self.objectInst.instIndex = self.objectInst.progressBar
            self.objectInst.progressBar += 1
        elif role == Accessibility.ROLE_TOOL_BAR:
            objectInfo = objInfo ('tbar', 'tool_bar', self.objectInst.toolBar, parentName)
            self.objectInst.instIndex = self.objectInst.toolBar
            self.objectInst.toolBar += 1
        elif role == Accessibility.ROLE_EMBEDDED:
            objectInfo = objInfo ('emb', 'embedded_component', self.objectInst.embeddedComponent, parentName)
            self.objectInst.instIndex = self.objectInst.embeddedComponent
            self.objectInst.embeddedComponent += 1
        elif role == Accessibility.ROLE_APPLICATION:
            objectInfo = objInfo ('app', 'application', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_EXTENDED:
            name = accessible.getRoleName ()
            if name == 'Calendar View':
                objectInfo = objInfo ('cal', 'calendar_view', self.objectInst.calView, parentName)
                self.objectInst.instIndex = self.objectInst.calView
                self.objectInst.calView += 1
            elif name == 'Calendar Event':
                objectInfo = objInfo ('cal', 'calendar_event', self.objectInst.calView, parentName)
                self.objectInst.instIndex = self.objectInst.calView
                self.objectInst.calView += 1
            else:
                if name is not None and name != '':
                    name = re.sub (' ', '_', name)
                # print 'extended', accessible.getRoleName (), accessible.getRole (), '*', accessible.name, '*'
                objectInfo = objInfo ('ukn', name, self.objectInst.unknown, parentName)
                self.objectInst.instIndex = self.objectInst.unknown
                self.objectInst.unknown += 1
        else:
            # print 'notlisted', accessible.getRoleName (), accessible.getRole (), '*', accessible.name, '*'
            roleName = accessible.getRoleName ()
            if roleName == None or roleName == '':
                name = 'unknown'
            else:
                name = re.sub (' ', '_', roleName)
            objectInfo = objInfo ('ukn', name, self.objectInst.unknown, parentName)
            self.objectInst.instIndex = self.objectInst.unknown
            self.objectInst.unknown += 1
        return objectInfo

    def get_non_conflicting_name (self, name):
        end_num = re.search ('\d*$', name)
        if end_num.start () == end_num.end (): # Numbers not postfixed yet
            num = 0
        else:
            num = int (name [end_num.start ():]) + 1
            name = name [:end_num.start ()]
        while name + str (num) in self.hashmap:
            num += 1
        return name + str (num)

    def doRemap (self, accessible, parentName = None, forceReScan = False, overloadedWindowName = False):
        if accessible is None:
            return
        role = accessible.getRole ()
        if role == Accessibility.ROLE_TABLE_CELL or \
               role == Accessibility.ROLE_SEPARATOR or \
               role == Accessibility.ROLE_LIST_ITEM:
            return
        name = accessible.name or 'unnamed'
        roleName = accessible.getRoleName ()
        description = accessible.description
        label = 'label'
        if name == 'unnamed':
            relationNames = self.getRelationName (accessible)
            if relationNames is not None and relationNames != '':
                name = relationNames
                label = 'label_by'
        #if _ldtpDebug and _ldtpDebug == '2':
        #    print '%s* %s "%s": %s' % ('\t', roleName, name, description)
        objectInfo = self.getObjectInfo (accessible, parentName)
        objHierarchyName = "%s#%d" % (objectInfo.prefix, objectInfo.instIndex)
        if objectInfo is not None and role != Accessibility.ROLE_APPLICATION:
            flag = False
            if role == Accessibility.ROLE_FRAME or role == Accessibility.ROLE_DIALOG or \
                   role == Accessibility.ROLE_ALERT or role == Accessibility.ROLE_FONT_CHOOSER or \
                   role == Accessibility.ROLE_FILE_CHOOSER or role == Accessibility.ROLE_WINDOW:
                flag = True
            if name != 'unnamed' and re.sub ("(?i) +", "", name) != '' and overloadedWindowName is False:
                strippedStr = escapeChars (name, False)
                ownName = objectInfo.prefix + strippedStr
                try:
                    ownName = ownName.encode ('utf-8')
                except UnicodeDecodeError:
                    ownName = unicode (ownName, 'utf-8').encode ('utf-8')
                if flag:
                    if self.applicationmap != {} and ownName in self.applicationmap and forceReScan == False:
                        if _ldtpDebug:
                            print 'has_key', ownName
                        # Since the window info is already in appmap, don't process further
                        return
                    curWindow = window (objectInfo.objType, name, parentName,
                                        accessible.getIndexInParent (),
                                        ownName, objHierarchyName)
                    if ownName in self.applicationmap and forceReScan:
                        del self.applicationmap [ownName]
                    self.applicationmap [ownName] = curWindow
                    self.hashmap = curWindow.children
                #else:
                #    if self.hashmap and ownName in self.hashmap:
                #        ownName = self.get_non_conflicting_name (ownName)
                #    if _ldtpDebug and self.hashmap:
                #        assert ownName not in self.hashmap, ('Name Conflict in Hashmap. ' + \
                #                                              ownName + 'already present\n' + foo)
                #    self.hashmap [ownName] = widget (objectInfo.objType, name, parentName, accessible.getIndexInParent ())
                if self.hashmap and ownName in self.hashmap:
                    ownName = self.get_non_conflicting_name (ownName)
                if _ldtpDebug and self.hashmap:
                    assert ownName not in self.hashmap, ('Name Conflict in Hashmap. ' + \
                                                          ownName + 'already present\n' + foo)
                self.hashmap [ownName] = widget (objectInfo.objType, name, parentName,
                                                 accessible.getIndexInParent (),
                                                 ownName, objHierarchyName)
                if _ldtpDebug and _ldtpDebug == '2':
                #     print u"%s={class=%s, %s=%s, child_index=%d, parent=%s}" % (ownName, objectInfo.objType,
                #                                                                 label, name,
                #                                                                 accessible.getIndexInParent (),
                #                                                                 parentName)
                    print "%s={class=%s, %s=%s, child_index=%d, parent=%s, key=%s, obj_index=%s}" % \
                        (ownName, objectInfo.objType, label, name, accessible.getIndexInParent (),
                         parentName, ownName, objHierarchyName)
                parentName = ownName
            elif overloadedWindowName is True:
                ownName = accessible.name
            else:
                ownName = "%s%d" % (objectInfo.prefix, objectInfo.instIndex)
                if flag:
                    if self.applicationmap != {} and ownName in self.applicationmap:
                        if _ldtpDebug:
                            print 'has_key', ownName
                        # Since the window info is already in appmap, don't process further
                        return
                    curWindow = window (objectInfo.objType, name, parentName,
                                        accessible.getIndexInParent (),
                                        ownName, objHierarchyName)
                    self.applicationmap [ownName] = curWindow
                    self.hashmap = curWindow.children
                else:
                    if self.hashmap and ownName in self.hashmap:
                        ownName = self.get_non_conflicting_name (ownName)
                        end_num = re.search ('\d*$', ownName)
                        if end_num.start () != end_num.end (): # Numbers not postfixed yet
                            objHierarchyName = "%s#%s" % (ownName [:end_num.start ()],
                                                          ownName [end_num.start ():])
                    self.hashmap [ownName] = widget (objectInfo.objType, parentName,
                                                     accessible.getIndexInParent (),
                                                     ownName, objHierarchyName)
                if _ldtpDebug and _ldtpDebug == '2':
                    print "%s={class=%s, child_index=%d, parent=%s, key=%s, obj_index=%s}" % \
                        (ownName, objectInfo.objType, accessible.getIndexInParent (),
                         parentName, ownName, objHierarchyName)
                parentName = ownName
        for i in range (accessible.childCount):
            child = accessible.getChildAtIndex (i)
            if child is None:
                continue
            childRole = child.getRole ()
            if childRole == Accessibility.ROLE_FRAME or childRole == Accessibility.ROLE_DIALOG or \
                   childRole == Accessibility.ROLE_ALERT or childRole == Accessibility.ROLE_FONT_CHOOSER or \
                   childRole == Accessibility.ROLE_FILE_CHOOSER:
                self.objectInst.reset ()
            # time.sleep (1.0 / pow (10, 2))
            try:
                self.doRemap (child, parentName)
            except:
                if _ldtpDebug:
                    if hasattr (traceback, 'format_exc'):
                        print traceback.format_exc ()
                    else:
                        print traceback.print_exc ()
            child.unref ()

    def push (self, head, value):
        if self.nodeIndex < 0:
            self.nodeIndex = 0
        head [self.nodeIndex] = value
        self.nodeIndex += 1

    def pop (self, head):
        value = head.pop (self.nodeIndex, -1)
        self.nodeIndex -= 1
        return value

    def tracePath2Parent (self, htContext, context, htComponent, cctxt):
        if htContext is None or htComponent is None or context is None or cctxt is None:
            return None
        head = None
        self.nodeIndex = 0
        self.push (head, htContext.index)
        parent = htContext.parent
        while 1:
            tmp = context
            if re.search (' ', context):
                tmp = self.escapeChar (context)
            if context is None or parent is None or \
                    context == parent or \
                    re.search (context, parent) or \
                    re.search (tmp, parent):
                break
            component = self.getProperty (htContext, parent)
            if component:
                prop = self.getProperty (component, "child_index")
                if prop is None:
                    return None
                self.push (head, prop)
                parent = self.getProperty (component, "parent")
                if _ldtpDebug:
                    print "Parent name: %s - Context name: %s - child index: %d" % \
                        (parent, context, prop)
            else:
                break
        return head

    def isRoleWindowType (self, _role):
        if _role == Accessibility.ROLE_FRAME or _role == Accessibility.ROLE_DIALOG or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_WINDOW:
            return True
        else:
            False

    def getRelationName (self, accessible):
        relations = []
        relationSet = accessible.getRelationSet ()
        for relation in relationSet:
            try:
                relations.append (relation._narrow (Accessibility.Relation))
            except:
                pass
        for relation in relations:
            _relationType = relation.getRelationType ()
            return relation.getTarget (0).name

    def getWinNameAppmapFormat (self, _windowName, _role):
        _windowType = None
        if _role == Accessibility.ROLE_FRAME:
            _windowType = 'frm'
        elif _role == Accessibility.ROLE_DIALOG or \
               _role == Accessibility.ROLE_ALERT or \
               _role == Accessibility.ROLE_WINDOW or \
               _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_FILE_CHOOSER:
            _windowType = 'dlg'
        else:
            if _ldtpDebug:
                print 'Not a window type'
            return None
        if _windowName and _windowName != '':
            return _windowType + escapeChars (_windowName, False)
        return _windowType + '0' # FIXME: Index has to be increased, instead of fixed 0

    def getParentNameAppmapFormat (self, accessible):
        # Provide the parent's accessible handle
        # It will be freed over here
        if accessible is None:
            return None
        _parentInfo = self.getObjectInfo (accessible, None)
        _parentName = _parentInfo.prefix + escapeChars (accessible.name)
        accessible.unref ()
        return _parentName

    def getObjNameAppmapFormat (self, accessible):
        if accessible is None:
            return None
        _objInfo = self.getObjectInfo (accessible, \
                                       self.getParentNameAppmapFormat (accessible.parent))
        if accessible.name and accessible.name != '' and _objInfo:
            _objName = _objInfo.prefix + escapeChars (accessible.name)
            return _objName
        return None

    def getObjRelNameAppmapFormat (self, accessible):
        if accessible is None:
            return None
        _objInfo = self.getObjectInfo (accessible, \
                                       self.getParentNameAppmapFormat (accessible.parent))
        _relationName = self.getRelationName (accessible)
        if _relationName and _relationName != '' and _objInfo:
            _objName = _objInfo.prefix + escapeChars (_relationName)
            return _objName
        return None

    def getParentHierarchyWithLabel (self, accessible, parentName = None):
        # Get parent hierarchy having label
        if accessible is None:
            return
        objectInfo = self.getObjectInfo (accessible, None)
        if accessible.name is not None and accessible.name != '':
            _role = accessible.getRole ()
            _name = ''
            _windowName = ''
            if self.isRoleWindowType (_role):
                _windowName = _name = self.getWinNameAppmapFormat (accessible.name, _role)
                self.hierarchy = '%(window)s|%(objtype)s:%(index)d:%(name)s;%(hierarchy)s' % \
                                {'window' : _windowName, 'objtype' : objectInfo.objType,
                                'index' : accessible.getIndexInParent (), \
                                'name' : _name, 'hierarchy' : self.hierarchy}
                return
#            else:
#                _name = self.getObjNameAppmapFormat (accessible)
#                _windowHandle = self.getWindowHandle (accessible)
#                if _windowHandle is not None:
#                    _windowName = self.getWinNameAppmapFormat (_windowHandle.name, _windowHandle.getRole ())
#                    _windowHandle.unref ()
        if objectInfo.objType != 'unknown' and accessible.getIndexInParent () != -1:
            if self.hierarchy == '':
                self.hierarchy = '%(objtype)s:%(index)d' % \
                                 {'objtype' : objectInfo.objType, 'index' : accessible.getIndexInParent ()}
            else:
                self.hierarchy = '%(objtype)s:%(index)d;%(hierarchy)s' % \
                                 {'objtype' : objectInfo.objType, 'index' : accessible.getIndexInParent (), 'hierarchy' : self.hierarchy}
        else:
            if _ldtpDebug:
                print 'Unknown objectInfo'
        _parent = accessible.parent
        if _parent is None:
            return
        self.getParentHierarchyWithLabel (_parent)
        _parent.unref ()
        return

    def getParentOfRole (self, accessible, parentRole):
        # Get parent of the given accessible handle based on the role
        # NOTE: parentRole should not be of any window type
        if accessible is None:
            return None
        if parentRole is None:
            return None
        _role = accessible.getRole ()
        if _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_DIALOG or _role == Accessibility.ROLE_FRAME or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_WINDOW:
            return None
        if _role == parentRole:
             # Has to be freed in the calling function,
             # assuming that the called function's accessible
             # handle is not equal to the new parent handle
            return accessible
        _parent = accessible.parent
        _handle = self.getParentOfRole (_parent, parentRole)
        if _handle != _parent:
            _parent.unref ()
        return _handle

    def getChildOfRole (self, accessible, childRole):
        if accessible is None:
            return None
        if childRole is None:
            return None
        _role = accessible.getRole ()
        if _role == childRole:
             # Has to be freed in the calling function,
             # assuming that the called function's accessible
             # handle is not equal to the new parent handle
            return accessible
        for i in range (accessible.childCount):
            child = accessible.getChildAtIndex (i)
            if child is None:
                continue
            tmpChild = self.getChildOfRole (child, childRole)
            if tmpChild:
                if tmpChild != child:
                    child.unref ()
                return tmpChild
            child.unref ()
        return None

    def getChildWindowHandle (self, accessible, name):
        flag = False
        if accessible is None:
            return None, None, flag
        if _ldtpDebug:
            print 'childCount', accessible.childCount
        for i in range (0, accessible.childCount):
            try:
                _app = accessible.getChildAtIndex (i)
            except:
                if _ldtpDebug:
                    if hasattr (traceback, 'format_exc'):
                        print traceback.format_exc ()
                    else:
                        print traceback.print_exc ()
                continue
            if _app is None:
                continue
            _role = _app.getRole ()
            windowName = self.getWinNameAppmapFormat (_app.name, _role)
            if _ldtpDebug:
                print 'Role: ', _app.getRoleName (), name
                if _app.name != None:
                    print '_app.name', _app.name, windowName
            if _app.name is not None and _app.name != '' and \
                (re.search (name, _app.name) != None or \
                re.search (name, windowName) != None):
                return _app, windowName
            _app.unref ()
        return None, None, flag

    def getAccessibleWindowHandle (self, accessible, context):
        if accessible:
            return self.getChildWindowHandle (accessible, context)

    def getWindowHandle (self, accessible):
        if accessible is None:
            return None
        _role = accessible.getRole ()
        if _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_DIALOG or _role == Accessibility.ROLE_FRAME or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_WINDOW:
                return accessible
        _parent = None
        _handle = None
        try:
            _parent = accessible.parent
            _handle = self.getWindowHandle (_parent)
        except:
            pass
        if _handle != _parent and _parent is not None:
            _parent.unref ()
        return _handle

    def getWindowList (self):
        windowList = []
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            return windowList
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                if _ldtpDebug:
                    print traceback.print_exc ()
                continue
            if _app is None:
                continue
            if _app.name is not None and _app.name != "":
                windowList.append (_app.name)
            _app.unref ()
        return windowList

    def updateWindowAppmapHandle (cctxt):
        flag   = False
        child  = None
        parent = None
        context    = None
        windowName = None
        parentName = None
        newHashTable = None

        context = cctxt.req.context

        if context is None:
            return LdtpErrorCode.ARGUMENT_NONE
        child, windowName = self.getWindowHandle (cctxt.appHandle, context)
        if child is None:
            return LdtpErrorCode.WIN_NOT_OPEN
        cctxt.windowName = windowName
        parent = child.parent
        if flag is False and parent:
            # FIXME
            return

    def getAccessibleContextHandle (self, appHandle, context):
        if appHandle is None or context is None:
            return None
        contextPattern = re.compile (context)
        if self.lastExistingContext:
            self.lastExistingContextPattern = re.compile (self.lastExistingContext)
        childCount = appHandle.childCount
        if _ldtpDebug and childCount and appHandle.name:
            print 'getAccessibleContextHandle: %s %ld' % (appHandle.name, childCount)
        tmpTable = []
        windowIndex = 0
        windowTmpIndex = 1
        for i in range (appHandle.childCount):
            child = appHandle.getChildAtIndex (i)
            if child is None:
                continue
            childRole = child.getRole ()
            childName = child.name
            windowName = None
            if childName:
                windowName = self.getWinNameAppmapFormat (childName, childRole)
            if windowName is None or re.search ("0", windowName):
                windowName = self.getWinNameAppmapFormat ("%d" % windowIndex, childRole)
                windowIndex += 1
            if windowName:
                if windowName in tmpTable is False:
                    tmpTable.append (windowName)
                else:
                    if childName:
                        windowName = self.getWinNameAppmapFormat ("%s%d" % (childName, windowTmpIndex),
                                                                  childRole)
                        windowTmpIndex += 1
                    if windowName and windowName in tmpTable is False:
                        tmpTable.append (windowName)
            if self.lastNewContext and self.lastExistingContext:
                if contextPattern and childName and self.lastExistingContextPattern and \
                        (re.search (self.lastExistingContextPattern, context) or \
                        (windowName and re.search (contextPattern, windowName))):
                    # FIXME: Localization related implementation
                    tmp = escapeChar (childName, ' ')
                    if self.lastNewContext is childName or \
                            self.lastNewContext is tmp or \
                            re.search (self.lastNewContext, childName):
                        return child
            # FIXME: Localization related implementation
            if _ldtpDebug and childName:
                print 'Context: %s Child name: %s' % (context, childName)
            if _ldtpDebug and windowName:
                print 'Window name: %s' % windowName
            if contextPattern and childName and \
                    (re.search (contextPattern, childName) or \
                         (windowName and re.search (contextPattern, windowName))):
                if _ldtpDebug:
                    print "Matched - Context: %s Child name: %s" % (context, childName)
                return child
            else:
                windowName = None
                child.unref ()

    def getAccessibleComponentHandle (self):
        return None

    def getComponentHandle (self, cctxt, windowProp, context, component):
        head      = None
        classType = None
        curWindow = None
        timeWait  = self.objTimeout
        curComponent    = None
        contextHandle   = None
        componentHandle = None
        currTime = prevTime = time.time ()

        if self.objTimeout is 0:
            timeWait = 5

        while currTime - prevTime < timeWait:
            self.updateWindowAppmapHandle (cctxt)

        return None

    def getAppHandle (self, appName):
        # App handle should be freed, if not None
        if appName == None or appName == '':
            return None
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            return None
        _app = ''
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                if _ldtpDebug:
                    if hasattr (traceback, 'format_exc'):
                        print traceback.format_exc ()
                    else:
                        print traceback.print_exc ()
                continue
            if _app is None:
                continue
            _name = ''
            if _app.name is not None and _app.name != "":
                if _ldtpDebug:
                    print 'DEBUG _app.name, _app.getRoleName', _app.name, _app.getRoleName ()
                _name = self.getWinNameAppmapFormat (_app.name, _app.getRole ())
                if _ldtpDebug:
                    print '_name', _name
            else:
                _app.unref ()
                continue
            if _ldtpDebug:
                print 'appName', appName, _name, _app.name
            if _name != None and re.search (appName, _name) != None or \
                re.search (appName, _app.name) != None:
                return _app
            else:
                _appHandle, windowName, flag = self.getChildWindowHandle (_app, appName)
                if _appHandle != None:
                    _app.unref ()
                    return _appHandle
            _app.unref ()
        return None

    def searchBasedOnKey (self, key, searchObjInfo):
        if key is None or searchObjInfo is None:
            return False
        #re.match ('(.*focus.*){1}', 'afocu8s123')
        #re.match ('(.*focus.*){1}', 'afocus123')
        #re.match ('(.*focus.*){1}', 'a focus123')
        if searchObjInfo.window is True:
            if key == searchObjInfo.key:
                return True
            if searchObjInfo.pattern is not None or searchObjInfo.pattern != '':
                if re.match (searchObjInfo.pattern, key) is not None:
                    return True
            tmpPattern = 'frm%s' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                return True
            tmpPattern = 'dlg%s' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                return True
            tmpPattern = 'frm%s*' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                return True
            tmpPattern = 'dlg%s*' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                return True
            return False
        if searchObjInfo.pattern is not None:
            return re.match (searchObjInfo.pattern, key)
        return False

    def searchBasedOnIndex (self, value, searchObjInfo):
        if value is None or searchObjInfo is None or searchObjInfo.key is None:
            return False
        flag = False
        if value.type is not None:
            if value.type == 'frame' or value.type == 'dialog' or \
               value.type == 'alert' or value.type == 'file_chooser' or \
               value.type == 'font_chooser':
                flag = True
        if value.label is None or value.label == '':
            return False
        if (searchObjInfo.window is False or flag is True) and \
               (value.hierarchy == searchObjInfo.key or \
                (searchObjInfo.pattern is not None and \
                 searchObjInfo.pattern != '' and \
                 re.search (searchObjInfo.pattern, value.hierarchy) is not None)):
            return True
        return False

    def searchBasedOnLabel (self, value, searchObjInfo):
        if value is None or searchObjInfo is None or searchObjInfo.key is None:
            return False
        flag = False
        if value.type is not None:
            if value.type == 'frame' or value.type == 'dialog' or \
               value.type == 'alert' or value.type == 'file_chooser' or \
               value.type == 'font_chooser':
                flag = True
        if value.label is None or value.label == '':
            return False
        if (searchObjInfo.window is False or flag is True) and \
               (value.label == searchObjInfo.key or \
                (searchObjInfo.pattern is not None and \
                 searchObjInfo.pattern != '' and \
                 re.search (searchObjInfo.pattern, value.label) is not None)):
            return True
        if re.search ('_', value.label) is None:
            return False
        if re.sub ('_', '', value.label) == searchObjInfo.key:
            return True
        return False

    def searchLabelName (self, key, value, unknLabelProperty):
        if key is None or value is None or unknLabelProperty is None:
            return False
        if value.index == unknLabelProperty.childIndex:
            if value.parent is not None and \
               unknLabelProperty.parentName is not None and \
               value.parent == unknLabelProperty.parentName:
                unknLabelProperty.objName = key
        return False

    def searchAfterStrippingSpace (self, key, data):
        if key is None or data is None:
            return False
        if key == data:
            return True
        if re.search (' ', data) is None:
            return False
        tmp = escapeChar (data, ' ')
        if key == tmp:
            return True
        if re.match ("(.*" + tmp + "){1}", key) or re.match (key, tmp):
            return True
        return False

    def getObjectDef (self, ht, context, isWindow):
        if ht is None or context is None or ht.children is None:
            return None
        if context in ht.children:
            return ht.children [context]
        obj = SearchObjInfo (context, isWindow, context)
        if re.search ('#', context):
            # Search based on index
            # ex: btn#1, mnu#1, etc
            for widget in ht.children.keys ():
                if self.searchBasedOnIndex (ht.children [widget], obj):
                    return widget
        for widget in ht.children.keys ():
            if self.searchBasedOnKey (widget, obj):
                return widget
        for widget in ht.children.keys ():
            if self.searchBasedOnLabel (ht.children [widget], obj):
                return widget
        for widget in ht.children.keys ():
            if self.searchAfterStrippingSpace ():
                return widget
        return None

    def getProperty (self, ht, property):
        if ht is None or ht.children is None:
            if _ldtpDebug:
                print getErrorMessage (LdtpErrorCode.APPMAP_NOT_INITIALIZED)
            return None
        if property is None:
            if _ldtpDebug:
                print getErrorMessage (LdtpErrorCode.ARGUMENT_NONE)
            return None
        if property in ht.children:
            return ht.children [property]
        return None
