#############################################################################
#
#  Linux Desktop Testing Project http://ldtp.freedesktop.org
# 
#  Author:
#     Nagappan Alagappan <nagappan@gmail.com>
# 
#  Copyright 2004 - 2007 Novell, Inc.
#  Copyright 2008 - 2009 Nagappan Alagappan
# 
#  This library is free software; you can redistribute it and/or
#  modify it under the terms of the GNU Library General Public
#  License as published by the Free Software Foundation; either
#  version 2 of the License, or (at your option) any later version.
# 
#  This library is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  Library General Public License for more details.
# 
#  You should have received a copy of the GNU Library General Public
#  License along with this library; if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA 02110, USA.
#
#############################################################################

import re
import os
import sys
import time
import fnmatch

# Let us not register our application under at-spi application list
os.environ ['GTK_MODULES'] = ''

try:
    import pyatspi as atspi, Accessibility
except ImportError:
    import atspi
    import Accessibility

from ldtplib.ldtplibutils import *

_ldtpDebug = os.getenv ('LDTP_DEBUG') # Enable debugging

class objInst:
    table = 0
    canvas = 0
    columnHeader = 0
    comboBox = 0
    pageTabList = 0
    pageTab = 0
    spinButton = 0
    button = 0
    tbutton = 0
    radioButton = 0
    checkBox = 0
    tree = 0
    treeTable = 0
    layeredPane = 0
    text = 0
    calView = 0
    panel = 0
    filler = 0
    menuBar = 0
    menu = 0
    separator = 0
    scrollBar = 0
    scrollPane = 0
    splitPane = 0
    slider = 0
    htmlContainer = 0
    progressBar = 0
    statusBar = 0
    toolBar = 0
    lst = 0
    label = 0
    icon = 0
    alert = 0
    dialog = 0
    unknown = 0
    embeddedComponent = 0

    def reset (self):
        self.table = 0
        self.canvas = 0
        self.columnHeader = 0
        self.comboBox = 0
        self.pageTabList = 0
        self.pageTab = 0
        self.spinButton = 0
        self.button = 0
        self.tbutton = 0
        self.radioButton = 0
        self.checkBox = 0
        self.tree = 0
        self.treeTable = 0
        self.layered_pane = 0
        self.text = 0
        self.calView = 0
        self.panel = 0
        self.filler = 0
        self.menuBar = 0
        self.menu = 0
        self.separator = 0
        self.scrollBar = 0
        self.scrollPane = 0
        self.splitPane = 0
        self.slider = 0
        self.htmlContainer = 0
        self.progressBar = 0
        self.statusBar = 0
        self.toolBar = 0
        self.label = 0
        self.lst = 0
        self.icon = 0
        self.alert = 0
        self.dialog = 0
        self.unknown = 0
        self.embeddedComponent = 0

class LdtpClientContext:
    def __init__ (self, windowName = None, appmap = None, appHandle = None,
                  request = None, response = None, guiHandle = None,
                  parentName = None, localeLang = 'en',
                  guiTimeOut = 30, objTimeOut = 5):
        self.windowName = windowName
        self.appmap     = appmap
        self.appHandle  = appHandle
        self.request    = request
        self.response   = response
        self.guiHandle  = guiHandle
        self.parentName = parentName
        self.localeLang = localeLang
        self.guiTimeOut = guiTimeOut
        self.objTimeOut = objTimeOut

class LdtpGuiHandle:
    def __init__ (self, handle = None, classId = 0):
        self.handle  = handle
        self.classId = classId

class ParentPathInfo:
    def __init__ (self, childIndex = None, count = None):
        self.childIndex = childIndex
        self.count      = count

class Node:
    def __init__ (self, childIndex = None, node = None):
        self.childIndex = childIndex
        self.node = node

class LdtpRequest:
    def __init__ (self, requestType = 'request', command = None, application = None,
                  requestId = None, context = None, component = None,
                  actionName = None, argsList = None):
        self.requestType = requestType
        self.command     = command
        self.application = application
        self.requestId   = requestId
        self.context     = context
        self.component   = component
        self.actionName  = actionName
        self.argsList    = argsList

class LdtpResponse:
    def __init__ (self, status, data, length):
        self.status = status
        self.data   = data
        self.length = length

class SearchObjInfo:
    def __init__ (self, key, objIsWindow = False, pattern = None, parent = None):
        self.key = key
        self.objIsWindow = objIsWindow
        self.pattern = pattern
        self.parent = parent

class objInfo:
    def __init__ (self, prefix, objectType, instanceIndex, parentName):
        self.prefix     = prefix
        self.objType    = objectType
        self.instIndex  = instanceIndex
        self.parentName = parentName
        self.objectInst = objInst ()

#class widget:
#    def __init__ (self, widgetType, *args):
#        self.widgetType = widgetType
#        if len (args) == 5:
#            self.label = args [0]
#            self.parent = args [1]
#            self.index = args [2]
#            self.key   = args [3]
#            self.hierarchy = args [4]
#        else:
#            self.label = ''
#            self.parent = args [0]
#            self.index = args [1]
#            self.key   = args [2]
#            self.hierarchy = args [3]

class widget:
    def __init__ (self, widgetType, *args):
        self.property = {}
        self.property ['class'] = widgetType
        if len (args) == 5:
            self.property ['label']     = args [0]
            self.property ['parent']    = args [1]
            self.property ['child_index']     = args [2]
            self.property ['key']       = args [3]
            self.property ['hierarchy'] = args [4]
        else:
            self.property ['label']     = ''
            self.property ['parent']    = args [0]
            self.property ['child_index']     = args [1]
            self.property ['key']       = args [2]
            self.property ['hierarchy'] = args [3]

#class window (widget):
#    def __init__ (self, *args):
#        self.children = {}
#        widget.__init__ (self, *args)

class window:
    def __init__ (self, *args):
        self.children = {}

class keyboard:
    def __init__ (self):
        pass
    charKeySynthVals = {38 : 'a', 56 : 'b', 54 : 'c', 40 : 'd', 26 : 'e', 41 : 'f', 42 : 'g', \
                        43 : 'h', 31 : 'i', 44 : 'j', 45 : 'k', 46 : 'l', 58 : 'm', 57 : 'n', \
                        32 : 'o', 33 : 'p', 24 : 'q', 27 : 'r', 39 : 's', 28 : 't', 30 : 'u', \
                        55 : 'v', 25 : 'w', 53 : 'x', 29 : 'y', 52 : 'z'} #A - Z
    digitKeySynthVals = {19 : '0', 10 : '1', 11 : '2', 12 : '3', 13 : '4', 14 : '5', 15 : '6', \
                         16 : '7', 17 : '8', 18 : '9'} #0 - 9
    symbolKeySynth = {20 : '-', 21 : '=', 34 : '[', 35 : ']', 47 : ';', 48 : '\'', 49 : '`', \
                      51 : '\\', 59 : ',', 60 : '.', 61 : '/', 10 : '!', 11 : '@', \
                      12 : '#', 13 : '$', 14 : '%', 15 : '^', 16 : '&', 17 : '*', 18 : '(', \
                      19 : ')', 20 : '_', 21 : '+', 34 : '{', 35 : '}', 47 : ':', 48 : '\"', \
                      49 : '~', 51 : '|', 59 : '<', 60 : '>', 61 : '?'}

    # PrtScrn: http://gentoo-wiki.com/TIP_Make_a_Screenshot_with_PrintScreen_Key
    # xmodmap -pke | grep -i print

    nonPrintKeySynth = {9 : '<escape>', 22 : '<backspace>',
                        37 : '<ctrl>', 115 : '<windowskey>',
                        23 : '<tab>', 36 : '<return>',
                        50 : '<shift>', 62 : '<shiftr>',
                        97 : '<home>', 103 : '<end>', 115 : '<window>',
                        64 : '<alt>', 113 : '<altr>',
                        98 : '<up>', 104 : '<down>', 102 : '<right>',
                        100 : '<left>', 65 : '<space>',
                        66 : '<caps>', 117 : '<menu>',
                        106 : '<insert>', 107 : '<delete>',
                        99 : '<pageup>', 105 : '<pagedown>',
                        77 : '<numlock>', 78 : '<scrolllock>',
                        67 : '<F1>', 68 : '<F2>', 69 : '<F3>', 70 : '<F4>',
                        71 : '<F5>', 72 : '<F6>', 73 : '<F7>', 74 : '<F8>',
                        75 : '<F9>', 76 : '<F10>', 95 : '<F11>', 96 : '<F12>',
                        111 : "prtscrn"}

class appmap:
    def __init__ (self, objectInst, registry):
        self.hashmap    = None
        self.hierarchy  = ''
        self.nodeIndex  = 0
        self.objTimeout = 0
        self.registry   = registry
        self.objectInst = objectInst
        self.remapEvent = threading.Lock ()
        self.currentWindowName  = None
        self.lastNewContext      = None
        self.lastExistingContext = None
        self.applicationmap = window ('list_of_windows', None, None, None, None).children

    def getObjectInfo (self, accessible, parentName):
        role = accessible.getRole ()
        objectInfo = None
        if role == Accessibility.ROLE_PAGE_TAB:
            objectInfo = objInfo ('ptab', 'page_tab', self.objectInst.pageTab, parentName)
            self.objectInst.instIndex = self.objectInst.pageTab
            self.objectInst.pageTab += 1
        elif role == Accessibility.ROLE_PAGE_TAB_LIST:
            objectInfo = objInfo ('ptl', 'page_tab_list', self.objectInst.pageTabList, parentName)
            self.objectInst.instIndex = self.objectInst.pageTabList
            self.objectInst.pageTabList += 1
        elif role == Accessibility.ROLE_MENU:
            objectInfo = objInfo ('mnu', 'menu', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_CHECK_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'check_menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_RADIO_MENU_ITEM:
            objectInfo = objInfo ('mnu', 'radio_menu_item', self.objectInst.menu, parentName)
            self.objectInst.instIndex = self.objectInst.menu
            self.objectInst.menu += 1
        elif role == Accessibility.ROLE_CHECK_BOX:
            objectInfo = objInfo ('chk', 'check_box', self.objectInst.checkBox, parentName)
            self.objectInst.instIndex = self.objectInst.checkBox
            self.objectInst.checkBox += 1
        elif role == Accessibility.ROLE_RADIO_BUTTON:
            objectInfo = objInfo ('rbtn', 'radio_button', self.objectInst.radioButton, parentName)
            self.objectInst.instIndex = self.objectInst.radioButton
            self.objectInst.radioButton += 1
        elif role == Accessibility.ROLE_SPIN_BUTTON:
            objectInfo = objInfo ('sbtn', 'spin_button', self.objectInst.spinButton, parentName)
            self.objectInst.instIndex = self.objectInst.spinButton
            self.objectInst.spinButton += 1
        elif role == Accessibility.ROLE_PUSH_BUTTON:
            objectInfo = objInfo ('btn', 'push_button', self.objectInst.button, parentName)
            self.objectInst.instIndex = self.objectInst.button
            self.objectInst.button += 1
        elif role == Accessibility.ROLE_TOGGLE_BUTTON:
            objectInfo = objInfo ('tbtn', 'toggle_button', self.objectInst.tbutton, parentName)
            self.objectInst.instIndex = self.objectInst.tbutton
            self.objectInst.tbutton += 1
        elif role == Accessibility.ROLE_TEXT:
            objectInfo = objInfo ('txt', 'text', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_PASSWORD_TEXT:
            objectInfo = objInfo ('txt', 'password_text', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_EDITBAR:
            objectInfo = objInfo ('txt', 'edit_bar', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_ENTRY:
            objectInfo = objInfo ('txt', 'entry', self.objectInst.text, parentName)
            self.objectInst.instIndex = self.objectInst.text
            self.objectInst.text += 1
        elif role == Accessibility.ROLE_LIST:
            objectInfo = objInfo ('lst', 'list', self.objectInst.lst, parentName)
            self.objectInst.instIndex = self.objectInst.lst
            self.objectInst.lst += 1
        elif role == Accessibility.ROLE_TABLE:
            objectInfo = objInfo ('tbl', 'table', self.objectInst.table, parentName)
            self.objectInst.instIndex = self.objectInst.table
            self.objectInst.table += 1
        elif role == Accessibility.ROLE_TABLE_COLUMN_HEADER:
            objectInfo = objInfo ('tbl', 'table_column_header', self.objectInst.columnHeader, parentName)
            self.objectInst.instIndex = self.objectInst.columnHeader
            self.objectInst.columnHeader += 1
        elif role == Accessibility.ROLE_TREE:
            objectInfo = objInfo ('tree', 'tree', self.objectInst.tree, parentName)
            self.objectInst.instIndex = self.objectInst.tree
            self.objectInst.tree += 1
        elif role == Accessibility.ROLE_TREE_TABLE:
            objectInfo = objInfo ('ttbl', 'tree_table', self.objectInst.treeTable, parentName)
            self.objectInst.instIndex = self.objectInst.treeTable
            self.objectInst.treeTable += 1
        elif role == Accessibility.ROLE_TABLE_CELL:
            objectInfo = objInfo ('tblc', 'table_cell', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_FRAME:
            objectInfo = objInfo ('frm', 'frame', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_DIALOG:
            objectInfo = objInfo ('dlg', 'dialog', self.objectInst.dialog, parentName)
            self.objectInst.instIndex = self.objectInst.dialog
            self.objectInst.dialog += 1
        elif role == Accessibility.ROLE_WINDOW:
            objectInfo = objInfo ('dlg', 'dialog', self.objectInst.dialog, parentName)
            self.objectInst.instIndex = self.objectInst.dialog
            self.objectInst.dialog += 1
        elif role == Accessibility.ROLE_FONT_CHOOSER:
            objectInfo = objInfo ('dlg', 'font_chooser', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_FILE_CHOOSER:
            objectInfo = objInfo ('dlg', 'file_chooser', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_ALERT:
            objectInfo = objInfo ('dlg', 'alert', self.objectInst.alert, parentName)
            self.objectInst.instIndex = self.objectInst.alert
            self.objectInst.alert += 1
        elif role == Accessibility.ROLE_COMBO_BOX:
            objectInfo = objInfo ('cbo', 'combo_box', self.objectInst.comboBox, parentName)
            self.objectInst.instIndex = self.objectInst.comboBox
            self.objectInst.comboBox += 1
        elif role == Accessibility.ROLE_LAYERED_PANE:
            objectInfo = objInfo ('pane', 'layered_pane', self.objectInst.layeredPane, parentName)
            self.objectInst.instIndex = self.objectInst.layeredPane
            self.objectInst.layeredPane += 1
        elif role == Accessibility.ROLE_CALENDAR:
            objectInfo = objInfo ('calview', 'calendar_view', self.objectInst.calView, parentName)
            self.objectInst.instIndex = self.objectInst.calView
            self.objectInst.calView += 1
        elif role == Accessibility.ROLE_PANEL:
            objectInfo = objInfo ('pnl', 'panel', self.objectInst.panel, parentName)
            self.objectInst.instIndex = self.objectInst.panel
            self.objectInst.panel += 1
        elif role == Accessibility.ROLE_LABEL:
            objectInfo = objInfo ('lbl', 'label', self.objectInst.label, parentName)
            self.objectInst.instIndex = self.objectInst.label
            self.objectInst.label += 1
        elif role == Accessibility.ROLE_ICON:
            objectInfo = objInfo ('ico', 'icon', self.objectInst.icon, parentName)
            self.objectInst.instIndex = self.objectInst.icon
            self.objectInst.icon += 1
        elif role == Accessibility.ROLE_MENU_BAR:
            objectInfo = objInfo ('mbr', 'menu_bar', self.objectInst.menuBar, parentName)
            self.objectInst.instIndex = self.objectInst.menuBar
            self.objectInst.menuBar += 1
        elif role == Accessibility.ROLE_SCROLL_BAR:
            objectInfo = objInfo ('scbr', 'scroll_bar', self.objectInst.scrollBar, parentName)
            self.objectInst.instIndex = self.objectInst.scrollBar
            self.objectInst.scrollBar += 1
        elif role == Accessibility.ROLE_SCROLL_PANE:
            objectInfo = objInfo ('scpn', 'scroll_pane', self.objectInst.scrollPane, parentName)
            self.objectInst.instIndex = self.objectInst.scrollPane
            self.objectInst.scrollPane += 1
        elif role == Accessibility.ROLE_STATUS_BAR:
            objectInfo = objInfo ('stat', 'status_bar', self.objectInst.statusBar, parentName)
            self.objectInst.instIndex = self.objectInst.statusBar
            self.objectInst.statusBar += 1
        elif role == Accessibility.ROLE_SEPARATOR:
            objectInfo = objInfo ('spr', 'separator', self.objectInst.separator, parentName)
            self.objectInst.instIndex = self.objectInst.separator
            self.objectInst.separator += 1
        elif role == Accessibility.ROLE_FILLER:
            objectInfo = objInfo ('flr', 'filler', self.objectInst.filler, parentName)
            self.objectInst.instIndex = self.objectInst.filler
            self.objectInst.filler += 1
        elif role == Accessibility.ROLE_CANVAS:
            objectInfo = objInfo ('cnvs', 'canvas', self.objectInst.canvas, parentName)
            self.objectInst.instIndex = self.objectInst.canvas
            self.objectInst.canvas += 1
        elif role == Accessibility.ROLE_SLIDER:
            objectInfo = objInfo ('sldr', 'slider', self.objectInst.slider, parentName)
            self.objectInst.instIndex = self.objectInst.slider
            self.objectInst.slider += 1
        elif role == Accessibility.ROLE_SPLIT_PANE:
            objectInfo = objInfo ('splt', 'split_pane', self.objectInst.splitPane, parentName)
            self.objectInst.instIndex = self.objectInst.splitPane
            self.objectInst.splitPane += 1
        elif role == Accessibility.ROLE_HTML_CONTAINER:
            objectInfo = objInfo ('html', 'html_container', self.objectInst.htmlContainer, parentName)
            self.objectInst.instIndex = self.objectInst.htmlContainer
            self.objectInst.htmlContainer += 1
        elif role == Accessibility.ROLE_PROGRESS_BAR:
            objectInfo = objInfo ('pbr', 'progress_bar', self.objectInst.progressBar, parentName)
            self.objectInst.instIndex = self.objectInst.progressBar
            self.objectInst.progressBar += 1
        elif role == Accessibility.ROLE_TOOL_BAR:
            objectInfo = objInfo ('tbar', 'tool_bar', self.objectInst.toolBar, parentName)
            self.objectInst.instIndex = self.objectInst.toolBar
            self.objectInst.toolBar += 1
        elif role == Accessibility.ROLE_EMBEDDED:
            objectInfo = objInfo ('emb', 'embedded_component', self.objectInst.embeddedComponent, parentName)
            self.objectInst.instIndex = self.objectInst.embeddedComponent
            self.objectInst.embeddedComponent += 1
        elif role == Accessibility.ROLE_APPLICATION:
            objectInfo = objInfo ('app', 'application', -1, parentName)
            self.objectInst.instIndex = -1
        elif role == Accessibility.ROLE_EXTENDED:
            name = accessible.getRoleName ()
            if re.search ('Calendar View', name, re.I):
                objectInfo = objInfo ('cal', 'calendar_view', self.objectInst.calView, parentName)
                self.objectInst.instIndex = self.objectInst.calView
                self.objectInst.calView += 1
            elif re.search ('Calendar Event', name, re.I):
                objectInfo = objInfo ('cal', 'calendar_event', self.objectInst.calView, parentName)
                self.objectInst.instIndex = self.objectInst.calView
                self.objectInst.calView += 1
            else:
                if name is not None and name != '':
                    name = re.sub (' ', '_', name)
                # print 'extended', accessible.getRoleName (), accessible.getRole (), '*', accessible.name, '*'
                objectInfo = objInfo ('unk', name, self.objectInst.unknown, parentName)
                self.objectInst.instIndex = self.objectInst.unknown
                self.objectInst.unknown += 1
        else:
            # print 'notlisted', accessible.getRoleName (), accessible.getRole (), '*', accessible.name, '*'
            roleName = accessible.getRoleName ()
            if roleName == None or roleName == '':
                name = 'unknown'
            else:
                name = re.sub (' ', '_', roleName)
            objectInfo = objInfo ('unk', name, self.objectInst.unknown, parentName)
            self.objectInst.instIndex = self.objectInst.unknown
            self.objectInst.unknown += 1
        return objectInfo

    def get_non_conflicting_name (self, name):
        end_num = re.search ('\d*$', name)
        if end_num.start () == end_num.end (): # Numbers not postfixed yet
            num = 0
        else:
            num = int (name [end_num.start ():]) + 1
            name = name [:end_num.start ()]
        while name + str (num) in self.hashmap:
            num += 1
        return name + str (num)

    def doRemap (self, accessible, parentName = None, forceReScan = False, overloadedWindowName = False):
        if accessible is None:
            return
        role = accessible.getRole ()
        if role == Accessibility.ROLE_TABLE_CELL or \
               role == Accessibility.ROLE_SEPARATOR or \
               role == Accessibility.ROLE_LIST_ITEM:
            return
        name = accessible.name or 'unnamed'
        roleName = accessible.getRoleName ()
        description = accessible.description
        if _ldtpDebug:
            print 'Role: %s - Description: %s' % (roleName, description)
        label = 'label'
        if name == 'unnamed':
            relationNames = self.getRelationName (accessible)
            if relationNames is not None and relationNames != '':
                name = relationNames
                label = 'label_by'
        #if _ldtpDebug and _ldtpDebug == '2':
        #    print '%s* %s "%s": %s' % ('\t', roleName, name, description)
        objectInfo = self.getObjectInfo (accessible, parentName)
        objHierarchyName = "%s#%d" % (objectInfo.prefix, objectInfo.instIndex)
        if objectInfo is not None and role != Accessibility.ROLE_APPLICATION:
            flag = False
            if role == Accessibility.ROLE_FRAME or role == Accessibility.ROLE_DIALOG or \
                   role == Accessibility.ROLE_ALERT or role == Accessibility.ROLE_FONT_CHOOSER or \
                   role == Accessibility.ROLE_FILE_CHOOSER or role == Accessibility.ROLE_WINDOW:
                flag = True
            if name != 'unnamed' and re.sub ("(?i) +", "", name) != '' and overloadedWindowName is False:
                strippedStr = escapeChars (name, False)
                ownName = objectInfo.prefix + strippedStr
                try:
                    ownName = ownName.encode ('utf-8')
                except UnicodeDecodeError:
                    ownName = unicode (ownName, 'utf-8').encode ('utf-8')
                if flag:
                    if self.applicationmap != {} and ownName in self.applicationmap \
                            and not forceReScan:
                        if _ldtpDebug:
                            print 'has_key', ownName
                        # Since the window info is already in appmap, don't process further
                        return
                    curWindow = window (objectInfo.objType, name, parentName,
                                        accessible.getIndexInParent (),
                                        ownName, objHierarchyName)
                    if ownName in self.applicationmap and forceReScan:
                        del self.applicationmap [ownName]
                    self.currentWindowName = ownName
                    self.applicationmap [ownName] = curWindow
                    self.hashmap = curWindow.children
                #else:
                #    if self.hashmap and ownName in self.hashmap:
                #        ownName = self.get_non_conflicting_name (ownName)
                #    if _ldtpDebug and self.hashmap:
                #        assert ownName not in self.hashmap, ('Name Conflict in Hashmap. ' + \
                #                                              ownName + 'already present\n' + foo)
                #    self.hashmap [ownName] = widget (objectInfo.objType, name, parentName, accessible.getIndexInParent ())
                if self.hashmap and ownName in self.hashmap:
                    ownName = self.get_non_conflicting_name (ownName)
                if _ldtpDebug and self.hashmap:
                    assert ownName not in self.hashmap, ('Name Conflict in Hashmap. ' + \
                                                          ownName + 'already present\n')
                self.hashmap [ownName] = widget (objectInfo.objType, name, parentName,
                                                 accessible.getIndexInParent (),
                                                 ownName, objHierarchyName)
                if _ldtpDebug and _ldtpDebug == '3':
                #     print u"%s={class=%s, %s=%s, child_index=%d, parent=%s}" % (ownName, objectInfo.objType,
                #                                                                 label, name,
                #                                                                 accessible.getIndexInParent (),
                #                                                                 parentName)
                    print "%s={class=%s, %s=%s, child_index=%d, parent=%s, key=%s, obj_index=%s}" % \
                        (ownName, objectInfo.objType, label, name, accessible.getIndexInParent (),
                         parentName, ownName, objHierarchyName)
                parentName = ownName
            elif overloadedWindowName is True:
                ownName = accessible.name
            else:
                ownName = "%s%d" % (objectInfo.prefix, objectInfo.instIndex)
                if flag:
                    if self.applicationmap != {} and ownName in self.applicationmap:
                        if _ldtpDebug:
                            print 'has_key', ownName
                        # Since the window info is already in appmap, don't process further
                        return
                    curWindow = window (objectInfo.objType, name, parentName,
                                        accessible.getIndexInParent (),
                                        ownName, objHierarchyName)
                    self.applicationmap [ownName] = curWindow
                    self.hashmap = curWindow.children
                else:
                    if self.hashmap and ownName in self.hashmap:
                        ownName = self.get_non_conflicting_name (ownName)
                        end_num = re.search ('\d*$', ownName)
                        if end_num.start () != end_num.end (): # Numbers not postfixed yet
                            objHierarchyName = "%s#%s" % (ownName [:end_num.start ()],
                                                          ownName [end_num.start ():])
                    self.hashmap [ownName] = widget (objectInfo.objType, parentName,
                                                     accessible.getIndexInParent (),
                                                     ownName, objHierarchyName)
                if _ldtpDebug and _ldtpDebug == '3':
                    print "%s={class=%s, child_index=%d, parent=%s, key=%s, obj_index=%s}" % \
                        (ownName, objectInfo.objType, accessible.getIndexInParent (),
                         parentName, ownName, objHierarchyName)
                parentName = ownName
        for i in range (accessible.childCount):
            child = accessible.getChildAtIndex (i)
            if child is None:
                continue
            childRole = child.getRole ()
            if childRole == Accessibility.ROLE_FRAME or childRole == Accessibility.ROLE_DIALOG or \
                   childRole == Accessibility.ROLE_ALERT or childRole == Accessibility.ROLE_FONT_CHOOSER or \
                   childRole == Accessibility.ROLE_FILE_CHOOSER:
                self.objectInst.reset ()
            # time.sleep (1.0 / pow (10, 2))
            try:
                self.doRemap (child, parentName)
                if self.currentWindowName:
                    self.applicationmap [self.currentWindowName].children = self.hashmap
            except:
                if _ldtpDebug:
                    if hasattr (traceback, 'format_exc'):
                        print traceback.format_exc ()
                    else:
                        print traceback.print_exc ()
            child.unref ()

    def push (self, head, value):
        if self.nodeIndex < 0:
            self.nodeIndex = 0
        head [self.nodeIndex] = value
        self.nodeIndex += 1

    def pop (self, head):
        if _ldtpDebug:
            print 'nodeIndex %d' % self.nodeIndex
        self.nodeIndex -= 1
        value = head.pop (self.nodeIndex, -1)
        return value

    def tracePath2Parent (self, htContext, context, htComponent, cctxt):
        if htContext is None or htComponent is None or context is None or cctxt is None:
            if _ldtpDebug:
                print htContext is None, htComponent is None, context is None, cctxt is None
            return None, None
        _head = {}
        self.nodeIndex = 0

        _prop = self.getProperty (htComponent, "child_index")
        if _prop is None:
            print 'htComponent', htComponent
            print traceback.print_stack ()
            if _ldtpDebug:
                print 'Unable to get property - %s' % lineno ()
            return None, None

        _classType = self.getProperty (htComponent, "class")
        if _ldtpDebug:
            print 'Class: %s' % _classType
        _childIndex = self.getProperty (htComponent, "child_index")
        if _ldtpDebug:
            print 'Child index: %d' % _childIndex
        self.push (_head, _childIndex)
        _parent = self.getProperty (htComponent, "parent")
        while 1:
            _tmp = context
            if re.search (' ', context):
                _tmp = escapeChar (context, ' ')
            if _ldtpDebug:
                print 'Tracepath2Parent %s %s' % (context, _parent)
                print _head, _classType, _parent
            if context is None or _parent is None or \
                    context == _parent or \
                    re.search (fnmatch.translate (context), _parent) or \
                    re.search (fnmatch.translate (_tmp), _parent):
                break
            _component = None
            if _parent in htContext.children:
                _component = htContext.children [_parent]
            if _component:
                _prop = self.getProperty (_component, "child_index")
                if _prop is None:
                    return None, None
                self.push (_head, _prop)
                _parent = self.getProperty (_component, "parent")
                if _ldtpDebug:
                    print "Parent name: %s - Context name: %s - child index: %d" % \
                        (_parent, context, _prop)
            else:
                break
        return _head, _classType

    def isRoleWindowType (self, _role):
        if _role == Accessibility.ROLE_FRAME or _role == Accessibility.ROLE_DIALOG or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_WINDOW:
            return True
        else:
            return False

    def getRelationName (self, accessible):
        relations = []
        relationSet = accessible.getRelationSet ()
        for relation in relationSet:
            try:
                relations.append (relation._narrow (Accessibility.Relation))
            except:
                pass
        for relation in relations:
            _relationType = relation.getRelationType ()
            if _ldtpDebug:
                print 'relationType', _relationType
            # FIXME: Not all relations navigated
            return relation.getTarget (0).name

    def getWinNameAppmapFormat (self, _windowName, _role):
        _windowType = None
        if _role == Accessibility.ROLE_FRAME:
            _windowType = 'frm'
        elif _role == Accessibility.ROLE_DIALOG or \
               _role == Accessibility.ROLE_ALERT or \
               _role == Accessibility.ROLE_WINDOW or \
               _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_FILE_CHOOSER:
            _windowType = 'dlg'
        else:
            if _ldtpDebug:
                print 'Not a window type'
            _windowType = ''
        if _windowName and _windowName != '':
            return '%s%s' % (_windowType, escapeChars (_windowName, False))
        return None

    def getParentNameAppmapFormat (self, accessible):
        # Provide the parent's accessible handle
        # It will be freed over here
        if accessible is None:
            return None
        _parentInfo = self.getObjectInfo (accessible, None)
        _parentName = _parentInfo.prefix + escapeChars (accessible.name)
        accessible.unref ()
        return _parentName

    def getObjNameAppmapFormat (self, accessible):
        if accessible is None:
            return None
        _objInfo = self.getObjectInfo (accessible, \
                                       self.getParentNameAppmapFormat (accessible.parent))
        if accessible.name and accessible.name != '' and _objInfo:
            _objName = _objInfo.prefix + escapeChars (accessible.name)
            return _objName
        return None

    def getObjRelNameAppmapFormat (self, accessible):
        if accessible is None:
            return None
        _objInfo = self.getObjectInfo (accessible, \
                                       self.getParentNameAppmapFormat (accessible.parent))
        _relationName = self.getRelationName (accessible)
        if _relationName and _relationName != '' and _objInfo:
            _objName = _objInfo.prefix + escapeChars (_relationName)
            return _objName
        return None

    def getParentHierarchyWithLabel (self, accessible, parentName = None):
        # Get parent hierarchy having label
        if accessible is None:
            return
        objectInfo = self.getObjectInfo (accessible, None)
        if accessible.name is not None and accessible.name != '':
            _role = accessible.getRole ()
            _name = ''
            _windowName = ''
            if self.isRoleWindowType (_role):
                _windowName = _name = self.getWinNameAppmapFormat (accessible.name, _role)
                self.hierarchy = '%(window)s|%(objtype)s:%(index)d:%(name)s;%(hierarchy)s' % \
                                {'window' : _windowName, 'objtype' : objectInfo.objType,
                                'index' : accessible.getIndexInParent (), \
                                'name' : _name, 'hierarchy' : self.hierarchy}
                return
#            else:
#                _name = self.getObjNameAppmapFormat (accessible)
#                _windowHandle = self.getWindowHandle (accessible)
#                if _windowHandle is not None:
#                    _windowName = self.getWinNameAppmapFormat (_windowHandle.name, _windowHandle.getRole ())
#                    _windowHandle.unref ()
        if objectInfo.objType != 'unknown' and accessible.getIndexInParent () != -1:
            if self.hierarchy == '':
                self.hierarchy = '%(objtype)s:%(index)d' % \
                                 {'objtype' : objectInfo.objType, 'index' : accessible.getIndexInParent ()}
            else:
                self.hierarchy = '%(objtype)s:%(index)d;%(hierarchy)s' % \
                                 {'objtype' : objectInfo.objType, 'index' : accessible.getIndexInParent (), 'hierarchy' : self.hierarchy}
        else:
            if _ldtpDebug:
                print 'Unknown objectInfo'
        _parent = accessible.parent
        if _parent is None:
            return
        self.getParentHierarchyWithLabel (_parent)
        return

    def getParentOfRole (self, accessible, parentRole):
        # Get parent of the given accessible handle based on the role
        # NOTE: parentRole should not be of any window type
        if accessible is None:
            return None
        if parentRole is None:
            return None
        _role = accessible.getRole ()
        if _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_DIALOG or _role == Accessibility.ROLE_FRAME or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_WINDOW:
            return None
        if _role == parentRole:
             # Has to be freed in the calling function,
             # assuming that the called function's accessible
             # handle is not equal to the new parent handle
            return accessible
        _parent = accessible.parent
        _handle = self.getParentOfRole (_parent, parentRole)
        # if _handle != _parent:
        #     _parent.unref ()
        return _handle

    def getChildOfRole (self, accessible, childRole):
        if accessible is None:
            return None
        if childRole is None:
            return None
        _role = accessible.getRole ()
        if _role == childRole:
             # Has to be freed in the calling function,
             # assuming that the called function's accessible
             # handle is not equal to the new parent handle
            return accessible
        for i in range (accessible.childCount):
            child = accessible.getChildAtIndex (i)
            if child is None:
                continue
            tmpChild = self.getChildOfRole (child, childRole)
            if tmpChild:
                if tmpChild != child:
                    child.unref ()
                return tmpChild
            child.unref ()
        return None

    def getChildWindowHandle (self, accessible, name):
        flag = False
        if accessible is None or name is None:
            return None, None, flag
        if _ldtpDebug:
            print 'childCount', accessible.childCount
        # Will be used when the window name is None, eg: dialog without title, dlg0
        j = 0
        # Will be used when the window name already exist
        k = 1
        # Used to check whether the window name already exist
        tmpWindowList = {}
        for i in range (0, accessible.childCount):
            try:
                _app = accessible.getChildAtIndex (i)
            except:
                if _ldtpDebug:
                    if hasattr (traceback, 'format_exc'):
                        print traceback.format_exc ()
                    else:
                        print traceback.print_exc ()
                continue
            if _app is None:
                continue
            _role = _app.getRole ()
            windowName = self.getWinNameAppmapFormat (_app.name, _role)
            if _ldtpDebug:
                print 'Role: ', _app.getRoleName (), name
                if _app.name and windowName:
                    print '_app.name', _app.name, windowName
            if windowName is None:
                # Create window title based on index and role
                windowName = self.getWinNameAppmapFormat ('%d' % j, _role)
                # Increment j value by 1 for next cycle
                j += 1
            if windowName:
                # Check whether the window name already exist in the tmpWindowList
                if tmpWindowList.has_key (windowName):
                    # Winodw name exist
                    # Append index to the name and try to get new name
                    windowName = self.getWinNameAppmapFormat ('%s%d' % (_app.name, k), _role)
                    # Increment k value by 1 for next cycle
                    k += 1
                    # Re-check whether the window name exist in tmpWindowList
                    if tmpWindowList.has_key (windowName) is False:
                        tmpWindowList [windowName] = windowName
                        flag = True
            # Check whether the expected window exist
            if _app.name and _app.name != '' and \
                (re.search (fnmatch.translate (name), _app.name) != None or \
                re.search (fnmatch.translate (name), windowName) != None):
                return _app, windowName, flag
            _app.unref ()
        return None, None, flag

    def getAccessibleWindowHandle (self, accessible, context):
        _handle = None
        _windowName = None
        flag = False
        if accessible:
            return self.getChildWindowHandle (accessible, context)
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            # Let us use the same format as getChildWindowHandle
            return _handle, _windowName, flag
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                continue
            if _app is None:
                continue
            if _app.name is not None:
                _handle, _windowName, flag = self.getChildWindowHandle (_app, context)
                if _handle:
                    return _handle, _windowName, flag
            _app.unref ()
        # Let us use the same format as getChildWindowHandle
        return _handle, _windowName, flag

    def getAccessibleAppHandle (self, appName, windowName = None):
        # Get top level application handle of given window name
        if appName is None:
            if _ldtpDebug:
                print 'appName argument None'
            return None, None

        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            if _ldtpDebug:
                print 'Unable to get desktop registry handle'
            return None, None
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                continue
            if _app is None:
                continue
            if _app.name is None:
                _app.unref ()
                continue

            # Incase of OO.o due to spaces, string compare gives different result.
            # Found because of extra spaces. Now stripped it ;)
            if re.search (' ', _app.name):
                _windowTitle = escapeChar (_app.name, ' ')
            else:
                _windowTitle = _app.name

            if _ldtpDebug:
                print 'Accessible application name: "%s" "%s"' % (_windowTitle, appName)

            if re.search (fnmatch.translate (appName), _windowTitle, re.I):
                # Top level application name matched
                if _ldtpDebug:
                    print 'Application name matched: %s - %s\t- Role: %s' % (_windowTitle,
                                                                             appName,
                                                                             _app.getRoleName ())
                if windowName is None:
                    _desktop.unref ()
                    if _ldtpDebug:
                        print 'Found application: %s %s' % (_app.name, _windowTitle)
                    return _app, _windowTitle
                if _ldtpDebug:
                    print 'Search for window name %s - child count %d' % (windowName,
                                                                          _app.childCount)
                for j in range (0, _app.childCount):
                    try:
                        _child = _app.getChildAtIndex (j)
                    except:
                        continue
                    if _child is None:
                        continue
                    if _child.name is None:
                        _child.unref ()
                        continue
                    if _ldtpDebug:
                        print 'Window name: %s Child %s' % (windowName,
                                                            _child.name)
                    # Just make sure window name also match apart from application name
                    # http://lists.freedesktop.org/archives/ldtp-dev/2008-September/000721.html
                    # Reported by Ara Pulido <ara@ubuntu.com>
                    if re.search (fnmatch.translate (windowName), _child.name, re.I):
                        _child.unref ()
                        _desktop.unref ()
                        if _ldtpDebug:
                            print 'Found application: %s %s' % (_app.name, _windowTitle)
                        return _app, _windowTitle
                    _child.unref ()
            _app.unref ()
        _desktop.unref ()
        if _ldtpDebug:
            print 'Unable to find application'
        return None, None

    def getWindowHandle (self, accessible):
        if accessible is None:
            return None
        _role = accessible.getRole ()
        if _role == Accessibility.ROLE_FILE_CHOOSER or _role == Accessibility.ROLE_FONT_CHOOSER or \
               _role == Accessibility.ROLE_DIALOG or _role == Accessibility.ROLE_FRAME or \
               _role == Accessibility.ROLE_ALERT or _role == Accessibility.ROLE_WINDOW:
                return accessible
        _parent = None
        _handle = None
        try:
            _parent = accessible.parent
            _handle = self.getWindowHandle (_parent)
        except:
            pass
        # if _handle != _parent and _parent is not None:
        #     _parent.unref ()
        return _handle

    def getAppList (self):
        appList = []
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            return appList
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                if _ldtpDebug:
                    print traceback.print_exc ()
                continue
            if _app is None:
                continue
            if _app.name is not None and _app.name != "":
                appList.append (_app.name)
            _app.unref ()
        return appList

    def getWindowList (self):
        windowList = []
        # Will be used when the window name is None, eg: dialog without title, dlg0
        k = 0
        # Will be used when the window name already exist
        l = 1
        # Used to check whether the window name already exist
        tmpWindowList = {}
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            return windowList
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                continue
            if _app is None:
                continue
            for j in range (0, _app.childCount):
                try:
                    _child = _app.getChildAtIndex (j)
                except:
                    if _ldtpDebug:
                        print traceback.print_exc ()
                    continue
                if _child is None:
                    continue
                _role = _child.getRole ()
                windowName = self.getWinNameAppmapFormat (_child.name, _role)
                if _ldtpDebug:
                    print 'Role: ', _child.getRoleName (), windowName
                    if _child.name and windowName:
                        print '_child.name', _child.name, windowName
                if windowName is None:
                    # Create window title based on index and role
                    windowName = self.getWinNameAppmapFormat ('%d' % k, _role)
                    # Increment k value by 1 for next cycle
                    k += 1
                if windowName:
                    # Check whether the window name already exist in the tmpWindowList
                    if tmpWindowList.has_key (windowName):
                        # Winodw name exist
                        # Append index to the name and try to get new name
                        windowName = self.getWinNameAppmapFormat ('%s%d' % (_child.name, l), _role)
                        # Increment l value by 1 for next cycle
                        l += 1
                        # Re-check whether the window name exist in tmpWindowList
                        if tmpWindowList.has_key (windowName) is False:
                            tmpWindowList [windowName] = windowName
                    # Append windo name to window list
                    windowList.append (windowName)
                _child.unref ()
            _app.unref ()
        return windowList

    def updateWindowAppmapHandle (self, cctxt):
        _child  = None
        _parent = None
        _context    = None
        _windowName = None
        _parentName = None
        _newHashTable = None

        _context = cctxt.request.context

        if _context is None:
            if _ldtpDebug:
                print 'Context argument None'
            return LdtpErrorCode.ARGUMENT_NONE, cctxt
        _child, _windowName, _flag = self.getAccessibleWindowHandle (cctxt.appHandle, _context)
        if _child is None:
            if _ldtpDebug:
                print 'Window not open'
            return LdtpErrorCode.WIN_NOT_OPEN, cctxt
        cctxt.windowName = _windowName
        _parent = _child.parent
        if _flag is False and _parent:
            _role = _parent.getRole ()
            if _ldtpDebug:
                print "Parent name %s %s" % (_parent.name, _parent.getRoleName ())
            if _role is not Accessibility.ROLE_APPLICATION:
                # Let us use child's role, if parent role is not application
                _role = _child.getRole ()
            _windowName = self.getWinNameAppmapFormat (_parent.name, _role)
            if _ldtpDebug and _windowName:
                print "Window name - Stripped: %s %s", _windowName, _parent.name
        if _parent:
            _parent.unref ()
        self.remapEvent.acquire ()
        _forceReScan = False
        if self.applicationmap != {} and _windowName in self.applicationmap:
            _forceReScan = True
            if _ldtpDebug:
                print 'Force rescan'
        self.doRemap (_child, _windowName, _forceReScan)
        self.remapEvent.release ()

        return LdtpErrorCode.SUCCESS, cctxt

    def getAccessibleContextHandle (self, appHandle, context):
        if appHandle is None or context is None:
            return None
        if _ldtpDebug:
            print 'Context: %s' % context
        contextPattern = re.compile (fnmatch.translate (context))
        if self.lastExistingContext:
            self.lastExistingContextPattern = re.compile (fnmatch.translate (self.lastExistingContext))
        else:
            self.lastExistingContextPattern = None
        childCount = appHandle.childCount
        if _ldtpDebug and childCount and appHandle.name:
            print 'getAccessibleContextHandle: %s %ld' % (appHandle.name, childCount)
        tmpTable = []
        windowIndex = 0
        windowTmpIndex = 1
        for i in range (appHandle.childCount):
            child = appHandle.getChildAtIndex (i)
            if child is None:
                continue
            childRole = child.getRole ()
            childName = child.name
            windowName = None
            if childName:
                windowName = self.getWinNameAppmapFormat (childName, childRole)
            if windowName is None or re.search ("0", windowName):
                windowName = self.getWinNameAppmapFormat ("%d" % windowIndex, childRole)
                windowIndex += 1
            if windowName:
                if windowName in tmpTable is False:
                    tmpTable.append (windowName)
                else:
                    if childName:
                        windowName = self.getWinNameAppmapFormat ("%s%d" % (childName, windowTmpIndex),
                                                                  childRole)
                        windowTmpIndex += 1
                    if windowName and windowName in tmpTable is False:
                        tmpTable.append (windowName)
            if self.lastNewContext and self.lastExistingContext:
                if (re.search (fnmatch.translate (self.lastExistingContext), context)) or \
                        (windowName and re.search (fnmatch.translate (context), windowName)):
                    # FIXME: Localization related implementation
                    tmp = escapeChar (childName, ' ')
                    if self.lastNewContext is childName or \
                            self.lastNewContext is tmp or \
                            re.search (fnmatch.translate (self.lastNewContext), childName):
                        return child
            # FIXME: Localization related implementation
            if _ldtpDebug:
                print 'Context: %s Child name: %s Window name: %s' % (context, childName, windowName)
            if (childName and re.search (fnmatch.translate (context), childName)) or \
                    (windowName and re.search (fnmatch.translate (context), windowName)):
                if _ldtpDebug:
                    print "Matched - Context: %s Child name: %s" % (context, childName)
                return child
            else:
                windowName = None
                child.unref ()

    def getAccessibleComponentHandle (self, head, contextHandle):
        if head is None or contextHandle is None:
            if _ldtpDebug:
                print 'Argument None'
            return None
        i = self.pop (head)
        if _ldtpDebug:
            print 'i = %d' % i
        _parent = contextHandle
        while i != -1:
            _child = _parent.getChildAtIndex (i)
            if _child is None:
                if _ldtpDebug:
                    print 'Parent: %s - index: %d' % (_parent.name, i)
                if _parent is not _child and _parent is not contextHandle:
                    _parent.unref ()
                return None
            if _ldtpDebug:
                print 'Name: %s Role: %s' % (_child.name, _child.getRoleName ())
            if _parent is not contextHandle:
                _parent.unref ()
            _parent = _child
            i = self.pop (head)
        return _parent

    def getComponentHandle (self, cctxt, windowProp, context, component):
        _head      = None
        _classType = None
        _curWindow = None
        _timeWait  = self.objTimeout
        _curComponent    = None
        _contextHandle   = None
        _componentHandle = None
        _currTime = _prevTime = time.time ()

        if not self.objTimeout:
            _timeWait = 5

        if _ldtpDebug:
            print 'Time: %d %d %d' % (_currTime, _prevTime, _timeWait)

        while _currTime - _prevTime < _timeWait:
            if _ldtpDebug:
                print 'Updating appmap - Component'
            errCode, cctxt = self.updateWindowAppmapHandle (cctxt)
            # Try with window property
            _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle, windowProp)
            if _contextHandle is None:
                # If unable to get, try with the provided context info
                _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle, context)

            if _contextHandle:
                if _ldtpDebug:
                    print 'Context handle name: %s' % _contextHandle.name
                _tmpContext = self.getWinNameAppmapFormat (_contextHandle.name,
                                                           _contextHandle.getRole ())
                if _tmpContext:
                    _curWindow = self.getObjectDef (cctxt.appmap.applicationmap,
                                                    _tmpContext, True, None)
                    _curComponent = self.getObjectDef (_curWindow, component,
                                                       False, cctxt.parentName)

                    if _ldtpDebug:
                        print dir (_curComponent), type (_curComponent), _curComponent, \
                            _curWindow, dir (_curWindow), type (_curWindow), _curWindow.children
                    _head, _classType = self.tracePath2Parent (_curWindow,
                                                               _tmpContext,
                                                               _curComponent,
                                                               cctxt)
                    if _head is None:
                        if _ldtpDebug:
                            print 'Head node is None'
                        return _componentHandle
                    _componentHandle = self.getAccessibleComponentHandle (_head, _contextHandle)
                    _contextHandle.unref ()
                    _head = None
            elif _ldtpDebug:
                print '_contextHandle is None - %s' % lineno ()

            if _componentHandle:
                _roleName = _componentHandle.getRoleName ()
                _str1 = _str2 = None
                if re.search (' ', _roleName):
                    _str1 = escapeChar (_roleName, ' ')
                else:
                    _str1 = _roleName
                if re.search (' ', _classType):
                    _str2 = escapeChar (_classType, ' ')
                elif re.search ('_', _classType):
                    _str2 = escapeChar (_classType, '_')
                else:
                    _str2 = _classType
                if _str1 and _str2:
                    if _ldtpDebug:
                        print 'str1: %s - str2: %s' % (_str1, _str2)
                    if re.search (fnmatch.translate (_str1), _str2, re.I) is None:
                        # Role types are not same, so we need to rescan
                        _componentHandle.unref ()
                        _componentHandle = None
                    _classType = None
            elif _ldtpDebug:
                print '_componentHandle is None - %s' % lineno ()

            if _componentHandle is None or \
                    _componentHandle.getRole () == Accessibility.ROLE_UNKNOWN:
                if _componentHandle:
                    _componentHandle.unref ()
                    _componentHandle = None
                # Sleep a second and then rescan
                time.sleep (1)
                _currTime = time.time ()
            else:
                # Break the loop
                break

        return _componentHandle

    def getAppHandle (self, appName):
        # App handle should be freed, if not None
        if appName == None or appName == '':
            return None
        _desktop = self.registry.getDesktop (0)
        if _desktop is None:
            return None
        _app = ''
        for i in range (0, _desktop.childCount):
            try:
                _app = _desktop.getChildAtIndex (i)
            except:
                continue
            if _app is None:
                continue
            _name = ''
            if _app.name is not None and _app.name != "":
                if _ldtpDebug:
                    print 'DEBUG _app.name, _app.getRoleName', _app.name, _app.getRoleName ()
                _name = self.getWinNameAppmapFormat (_app.name, _app.getRole ())
                if _ldtpDebug:
                    print '_name', _name
            else:
                _app.unref ()
                continue
            if _ldtpDebug:
                print 'appName', appName, _name, _app.name
            if _name != None and re.search (fnmatch.translate (appName), _name) != None or \
                re.search (fnmatch.translate (appName), _app.name) != None:
                return _app
            else:
                _appHandle, windowName, flag = self.getChildWindowHandle (_app, appName)
                if _appHandle != None:
                    _app.unref ()
                    return _appHandle
            _app.unref ()
        return None

    def guiExist (self, cctxt):
        _status, cctxt = self.getGuiHandle (cctxt)
        if cctxt and cctxt.guihandle and cctxt.guihandle.handle:
            cctxt.guihandle.handle.unref ()
        if _status == LdtpErrorCode.SUCCESS:
            cctxt.guihandle.handle.unref ()
            return LdtpErrorCode.SUCCESS
        else:
            return LdtpErrorCode.GUI_NOT_EXIST

    def getGuiHandle (self, cctxt):
        if cctxt is None or cctxt.request is None or cctxt.request.context is None:
            if _ldtpDebug:
                print 'Argument None - %s' % lineno ()
            return LdtpErrorCode.ARGUMENT_NONE, cctxt
        _context = cctxt.request.context
        _component = cctxt.request.component
        if _component is None or _component == "":
            _component = _context
        if cctxt.appmap.hashmap is None:
            errCode, cctxt = self.updateWindowAppmapHandle (cctxt)
            if errCode != LdtpErrorCode.SUCCESS:
                if _ldtpDebug:
                    print 'Unable to update context %s in appmap' % _context
                return errCode, cctxt
        if _ldtpDebug:
            print 'Context - Component: %s %s' % (_context, _component)

        _curWindow = self.getObjectDef (cctxt.appmap.applicationmap, _context, True, None)

        if _curWindow is None:
            if _ldtpDebug:
                print 'Unable to getObjectDef for context %s in appmap' % _context
            errCode, cctxt = self.updateWindowAppmapHandle (cctxt)
            if errCode != LdtpErrorCode.SUCCESS:
                if _ldtpDebug:
                    print 'Unable to update context %s in appmap' % _context
                return errCode, cctxt
            _curWindow = self.getObjectDef (cctxt.appmap.applicationmap, _context, True, None)

            if _curWindow is None:
                # From C implementation, using the following comments
                # If context is given without window type, then use dynamically created window name
                # Example: if context is Replace, instead of dlgReplace (in gedit app, replace dialog)
                # then we will try to search using our dynamic context dlgReplace
                _child, cctxt.windowName, flag = self.getAccessibleWindowHandle (cctxt.appHandle, _context)
                if _child is None:
                    return LdtpErrorCode.WIN_NOT_OPEN, cctxt
                _child.unref ()
                if cctxt.windowName:
                    _context = cctxt.request.context = cctxt.windowName
                    _curWindow = self.getObjectDef (cctxt.appmap.applicationmap, _context, True, None)
                if _curWindow is None:
                    return LdtpErrorCode.WIN_NAME_NOT_FOUND_IN_APPMAP, cctxt

        _curComponent = self.getObjectDef (_curWindow, _context, True, None)

        if _curComponent is None:
            return LdtpErrorCode.WIN_NAME_NOT_FOUND_IN_APPMAP, cctxt

        _appName = self.getProperty (_curComponent, "parent")

        if _ldtpDebug and _appName:
            print 'Application name: %s' % _appName

        _windowProp = self.getProperty (_curComponent, "label")

        if _ldtpDebug and _windowProp:
            print 'Window prop name: %s' % _windowProp

        if _context and _component and re.match (_context, _component, re.I):
            _curComponent = self.getObjectDef (_curWindow, _component, False, cctxt.parentName)
            if _curComponent is None:
                _startTime = time.time ()

                _tmpFlag = False
                while timeElapsed (_startTime) < cctxt.objTimeOut:
                    _tmpContext = _tmpComponent = None
                    if _curComponent is None:
                        _tmpContext = self.getContextDef (cctxt, _context, _component)
                        if _tmpContext:
                            _tmpComponent = self.getObjectDef (_tmpContext, _component, False, None)
                            if _tmpComponent:
                                _curWindow = _tmpContext
                                _curComponent = _tmpComponent
                                _tmpFlag = True
                                _tmpComponent = self.getObjectDef (_curWindow, _context, True, cctxt.parentName)
                                _appName = self.getProperty (_tmpComponent, 'parent')
                                if _ldtpDebug and _appName:
                                    print 'Application name: %s' % _appName
                                _windowProp = self.getProperty(_tmpComponent,'label')
                                if _ldtpDebug and _windowProp:
                                    print 'Window name: %s' % _windowProp
                    if _ldtpDebug:
                        print 'Updating appmap - context'
                    errCode, cctxt = self.updateWindowAppmapHandle (cctxt)
                    if errCode != LdtpErrorCode.SUCCESS:
                        return errCode, cctxt
                    _curWindow = self.getObjectDef (cctxt.appmap.applicationmap,
                                                    _context, True, None)
                    _curComponent = self.getObjectDef (_curWindow, _component,
                                                       False, cctxt.parentName)

                    if _curComponent is None:
                        time.sleep (1)
                    else:
                        _tmpFlag = True
                        _tmpComponent = self.getObjectDef (_curWindow, _context, True,
                                                           cctxt.parentName)
                        _appName = self.getProperty (_tmpComponent, 'parent')
                        if _ldtpDebug and _appName:
                            print 'Application name: %s' % _appName
                        _windowProp = self.getProperty(_tmpComponent,'label')
                        if _ldtpDebug and _windowProp:
                            print 'Window name: %s' % _windowProp
                        # Break from while loop
                        break
                if _tmpFlag is False:
                    return LdtpErrorCode.OBJ_NAME_NOT_FOUND_IN_APPMAP, cctxt

        if _ldtpDebug:
            print 'Appname: %s %s' % (_appName, cctxt.windowName)

        if cctxt.appHandle:
            cctxt.appHandle.unref ()

        cctxt.appHandle, cctxt.windowName = self.getAccessibleAppHandle (_appName,
                                                                         _windowProp)
        if cctxt.appHandle is None:
            if _ldtpDebug:
                print 'Application %s not running' % _appName
            return LdtpErrorCode.APP_NOT_RUNNING, cctxt

        if _ldtpDebug:
            print 'Window name: %s %s' % (cctxt.windowName, _windowProp)

        cctxt.windowName = _windowProp
        _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle,
                                                          _windowProp)

        if _contextHandle is None:
            _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle,
                                                              _context)

        if _contextHandle is None:
            cctxt.appHandle.unref ()
            if re.search (fnmatch.translate (_context), _component, re.I):
                return LdtpErrorCode.WIN_NOT_OPEN, cctxt
            else:
                return LdtpErrorCode.UNABLE_TO_GET_CONTEXT_HANDLE, cctxt

        if _ldtpDebug:
            print 'Context handle %s' % _contextHandle.name

        cctxt.guiHandle = LdtpGuiHandle (cctxt.appHandle)

        if _ldtpDebug:
            print 'Window: %s - Object: %s' % (_context, _component)

        # In case of waittillguiexist and waittillguinotexist,
        # both context and component members of the ClientContext 
        # structure points to same string and they don't need class_id.
        if _context and _component and re.search (fnmatch.translate (_context),
                                                  _component, re.I):
            return LdtpErrorCode.SUCCESS, cctxt

        _head, _classType = self.tracePath2Parent (_curWindow,
                                                   _context,
                                                   _curComponent,
                                                   cctxt)
        if _head is None:
            if _ldtpDebug:
                print 'Head node is None'
            return LdtpErrorCode.OBJ_INFO_MISMATCH, cctxt
        _componentHandle = self.getAccessibleComponentHandle (_head,
                                                              _contextHandle)
        _contextHandle.unref ()
        _head = None

        if _componentHandle:
            if _ldtpDebug:
                print 'Component name: %s' % _componentHandle.name
            _roleName = _componentHandle.getRoleName ()
            _str1 = _str2 = None
            if re.search (' ', _roleName):
                _str1 = escapeChar (_roleName, ' ')
            else:
                _str1 = _roleName
            if re.search (' ', _classType):
                _str2 = escapeChar (_classType, ' ')
            elif re.search ('_', _classType):
                _str2 = escapeChar (_classType, '_')
            else:
                _str2 = _classType
            if _str1 and _str2:
                if _ldtpDebug:
                    print 'str1: %s - str2: %s' % (_str1, _str2)
                if re.search (fnmatch.translate (_str1), _str2, re.I) is None:
                    # Role types are not same, so we need to rescan
                    _componentHandle.unref ()
                    _componentHandle = None
            _classType = None

        if _componentHandle is None or \
                _componentHandle.getRole () == Accessibility.ROLE_UNKNOWN:
            if _componentHandle:
                _componentHandle.unref ()
                _componentHandle = None
            if _windowProp:
                if _ldtpDebug:
                    print 'Window prop: %s' % _windowProp
                _componentHandle = self.getComponentHandle (cctxt, _windowProp, _context, _component)
                _curWindow = self.getObjectDef (cctxt.appmap.applicationmap, _context, True, None)
                _curComponent = self.getObjectDef (_curWindow, _component, False, cctxt.parentName)
                print '_componentHandle', _componentHandle, _windowProp, _context, _component
            elif _ldtpDebug:
                print '_windowProp is None'

        if _componentHandle is None:
            if _ldtpDebug and _component:
                print 'Unable to get object handle %s - %s' % (_component, lineno ())
            return LdtpErrorCode.GET_OBJ_HANDLE_FAILED, cctxt

        if self.isObjectMatching (_componentHandle, _curComponent) is False:
            if _ldtpDebug:
                print 'Object information: %s does not match with appmap entry' % _componentHandle.name
            _componentHandle.unref ()
            _componentHandle = None
            # Let us try rescan
            if _ldtpDebug:
                print 'Let us try rescan'
            _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle,
                                                              _windowProp)
            if _contextHandle is None:
                _contextHandle = self.getAccessibleContextHandle (cctxt.appHandle,
                                                                  _context)

            if _ldtpDebug and _contextHandle:
                print 'Context name: %s' % _contextHandle.name

            _tmpContext = self.getWinNameAppmapFormat (_contextHandle.name,
                                                       _contextHandle.getRole ())
            if _tmpContext:
                _componentHandle = self.getComponentHandle (cctxt, _windowProp, _tmpContext, _component)
                if _componentHandle is None:
                    if _ldtpDebug:
                        print 'Unable to get object handle %s - %s' % (_tmpContext, lineno ())
                    _contextHandle.unref ()
                    _componentHandle.unref ()
                    return LdtpErrorCode.GET_OBJ_HANDLE_FAILED, cctxt
            _curWindow = self.getObjectDef (cctxt.appmap.applicationmap, _tmpContext, True, None)
            _curComponent = self.getObjectDef (_curWindow, _component, False, cctxt.parentName)

            if self.isObjectMatching (_componentHandle, _curComponent) is False:
                if _ldtpDebug:
                    print 'Object information: %s does not match with appmap entry' % \
                        _componentHandle.name
                if _componentHandle:
                    _componentHandle.unref ()
                    _componentHandle = None
                if _contextHandle:
                    _contextHandle.unref ()
                    _contextHandle = None
                return LdtpErrorCode.OBJ_INFO_MISMATCH, cctxt

            # Custom class id is required as we handle cases for extended roles
            cctxt.guiHandle.classId = 0

            if _componentHandle:
                cctxt.guiHandle.classId = _componentHandle.getRole ()

            if cctxt.guiHandle.classId == Accessibility.ROLE_EXTENDED:
                _tmpName = _componentHandle.name
                if re.search ('calendar view', _tmpName, re.I):
                    cctxt.guiHandle.classId = ExtendedRole.CALENDAR_VIEW
                elif re.search ('calendar event', _tmpName, re.I):
                    cctxt.guiHandle.classId = ExtendedRole.CALENDAR_EVENT

            if _ldtpDebug:
                print 'Class id: %d %s' % (cctxt.guiHandle.classId,
                                           _componentHandle.getRoleName ())

            if cctxt.request.component:
                # Use the component name available as per the hash table instead of 
                # the user provided one
                cctxt.request.component = self.getProperty (_curComponent, "key")

        if cctxt.guiHandle.handle:
            cctxt.guiHandle.handle.unref ()
        cctxt.guiHandle.handle = _componentHandle
        return LdtpErrorCode.SUCCESS, cctxt

    def isObjectMatching (self, accessible, curComponent):
        if accessible is None or curComponent is None:
            return False
        _label = self.getProperty (curComponent, 'label')
        if _label:
            _objLabel = accessible.name
            _objDesc = accessible.description
            if _ldtpDebug:
                print 'Label: %s %s %s' % (_label, _objLabel,
                                           _objDesc)
            _regexp = None
            if re.search ('_', _label):
                _tmpLabel = escapeChar (_label, '_')
                if _ldtpDebug:
                    print 'Before: %s After: %s' % (_label, _tmpLabel)
                _regexp = re.search (_tmpLabel, _objLabel)
                _tmpLabel = None
            elif _objLabel and re.search ('...', _objLabel):
                # Ugly hack specific to VMware, as the team toggle button
                # which has label as ...
                _regexp = re.search (_label, _objLabel)
            elif _objDesc:
                _regexp = re.search (_label, _objDesc)
            if _regexp:
                if _ldtpDebug:
                    print 'Object matches'
                _regexp = None
                return True
            else:
                return False
        _label = self.getProperty (curComponent, 'label_by')
        if _label:
            _objLabel = self.getRelationName (accessible)
            if _ldtpDebug:
                print 'Label: %s %s' % (_label, _objLabel)
            _regexp = None
            if re.search ('_', _label):
                _tmpLabel = escapeChar (_label, '_')
                if _ldtpDebug:
                    print 'Before: %s After: %s' % (_label, _tmpLabel)
                _regexp = re.search (_tmpLabel, _objLabel)
                _tmpLabel = None
            if _regexp:
                if _ldtpDebug:
                    print 'Object matches'
                _regexp = None
                return True
            else:
                return False
        return True

    def searchBasedOnKey (self, key, searchObjInfo):
        if key is None or searchObjInfo is None:
            if _ldtpDebug:
                print 'One of the required argument is None - %s' % lineno ()
            return False
        #re.match ('(.*focus.*){1}', 'afocu8s123')
        #re.match ('(.*focus.*){1}', 'afocus123')
        #re.match ('(.*focus.*){1}', 'a focus123')
        if searchObjInfo.objIsWindow is True:
            if _ldtpDebug:
                print 'Object is window'
            if key == searchObjInfo.key:
                if _ldtpDebug:
                    print 'key matched'
                return True
            if searchObjInfo.pattern is not None or searchObjInfo.pattern != '':
                if re.match (searchObjInfo.pattern, key) is not None:
                    if _ldtpDebug:
                        print 'Search key based on pattern matched'
                    return True
            tmpPattern = 'frm%s' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                tmpPattern = None
                if _ldtpDebug:
                    print 'Search key based on frame matched'
                return True
            tmpPattern = 'dlg%s' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                tmpPattern = None
                if _ldtpDebug:
                    print 'Search key based on dialog matched'
                return True
            tmpPattern = 'frm%s*' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                tmpPattern = None
                if _ldtpDebug:
                    print 'Search key based on frame matched'
                return True
            tmpPattern = 'dlg%s*' % key
            if re.match (tmpPattern, searchObjInfo.key) is not None:
                tmpPattern = None
                if _ldtpDebug:
                    print 'Search key based on dialog matched'
                return True
            if _ldtpDebug:
                print 'Search failed'
            return False
        if searchObjInfo.pattern:
            return re.match (searchObjInfo.pattern, key)
        return False

    def searchBasedOnIndex (self, value, searchObjInfo):
        if _ldtpDebug:
            print 'Search Based On Index'
        if value is None or searchObjInfo is None or \
                searchObjInfo.key is None or hasattr (value, 'property') is False:
            if _ldtpDebug:
                print 'One of the required argument is None - %s' % lineno ()
            return False

        flag = False
        value = value.property

        if value ['class'] is not None:
            if value ['class'] == 'frame' or value ['class'] == 'dialog' or \
               value ['class'] == 'alert' or value ['class'] == 'file_chooser' or \
               value ['class'] == 'font_chooser':
                flag = True
        if value ['label'] is None or value ['label'] == '':
            return False
        if (searchObjInfo.objIsWindow is False or flag is True) and \
               (value ['hierarchy'] == searchObjInfo.key or \
                (searchObjInfo.pattern is not None and \
                 searchObjInfo.pattern != '' and \
                 re.search (fnmatch.translate (searchObjInfo.pattern),
                            value ['hierarchy']) is not None)):
            return True
        return False

    def searchBasedOnLabel (self, value, searchObjInfo):
        if value is None or searchObjInfo is None or \
                searchObjInfo.key is None or hasattr (value, 'property') is False:
            if _ldtpDebug:
                print 'One of the required argument is None - %s' % lineno ()
            return False

        flag = False
        value = value.property

        if value ['class'] is not None:
            if value ['class'] == 'frame' or value ['class'] == 'dialog' or \
               value ['class'] == 'alert' or value ['class'] == 'file_chooser' or \
               value ['class'] == 'font_chooser':
                flag = True
        if value ['label'] is None or value ['label'] == '':
            return False
        if (searchObjInfo.objIsWindow is False or flag is True) and \
               (value ['label'] == searchObjInfo.key or \
                (searchObjInfo.pattern is not None and \
                 searchObjInfo.pattern != '' and \
                 re.search (fnmatch.translate (searchObjInfo.pattern),
                            value ['label']) is not None)):
            if _ldtpDebug:
                print 'Search based on label matched'
            return True
        if re.search ('_', value ['label']) is None:
            return False
        if re.sub ('_', '', value ['label']) == searchObjInfo.key:
            return True
        return False

    def searchLabelName (self, key, value, unknLabelProperty):
        if _ldtpDebug:
            print 'Search Based On Label Name'
        if key is None or value is None or unknLabelProperty is None:
            if _ldtpDebug:
                print 'One of the required argument is None - %s' % lineno ()
            return False
        if value.index == unknLabelProperty.childIndex:
            if value.parent is not None and \
               unknLabelProperty.parentName is not None and \
               value.parent == unknLabelProperty.parentName:
                unknLabelProperty.objName = key
        return False

    def searchAfterStrippingSpace (self, key, data):
        if _ldtpDebug:
            print 'Search After Stripping Space'
        if key is None or data is None:
            if _ldtpDebug:
                print 'One of the required argument is None - %s' % lineno ()
            return False
        if re.search (' ', data.key) is None:
            return False
        tmp = escapeChar (data.key, ' ')
        if key == tmp:
            return True
        # if re.match ("(.*" + tmp + "){1}", key) or re.match (key, tmp):
        if re.match (fnmatch.translate (tmp), key) or \
                re.match (fnmatch.translate ('*' + tmp), key):
            return True
        return False

    def getObjectDef (self, ht, context, isWindow = False, parent = None):
        if ht is None or context is None:
            if _ldtpDebug:
                print 'One or more arguments to getObjectDef is None'
            return None

        if hasattr (ht, 'children'):
            ht = ht.children
        if context in ht:
            if _ldtpDebug:
                print 'Context %s found in ht using key search' % context
            return ht [context]

        obj = SearchObjInfo (context, isWindow, context)

        if re.search ('#', context):
            # Search based on index
            # ex: btn#1, mnu#1, etc
            for widget in ht.keys ():
                if self.searchBasedOnIndex (ht [widget], obj):
                    return widget

        for widget in ht.keys ():
            if self.searchBasedOnKey (widget, obj):
                return widget

        for widget in ht.keys ():
            if self.searchBasedOnLabel (ht [widget], obj):
                return widget

        for widget in ht.keys ():
            if self.searchAfterStrippingSpace (widget, obj):
                return widget

        return None

    def getProperty (self, ht, property):
        if ht is None:
            if _ldtpDebug:
                print LdtpErrorCode.APPMAP_NOT_INITIALIZED
            return None
        if property is None:
            if _ldtpDebug:
                print LdtpErrorCode.ARGUMENT_NONE
            return None
        if hasattr (ht, 'property') and property in ht.property:
            if _ldtpDebug:
                print 'Property: %s - %s' % (property, ht.property [property])
            return ht.property [property]
        if _ldtpDebug:
            print 'Unable to get the property %s - %s' % (property, lineno ())
        return None
