/*
 *  Copyright 1994-2020 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * EffecTV - Realtime Digital Video Effector
 * Copyright (C) 2001-2006 FUKUCHI Kentaro
 *
 * FireTV - clips incoming objects and burn them.
 * Copyright (C) 2001-2002 FUKUCHI Kentaro
 *
 * Fire routine is taken from Frank Jan Sorensen's demo program.
 */

#include "context.h"


uint32_t version = 0;
u_long options = BE_GFX|BEQ_IMAGE|BEQ_MUTE_CAM;
char desc[] = "FireTV plugin from the EffecTV project";
char dname[] = "TV fire";

enum LayerMode mode = LM_OVERLAY;

#define Decay 15
#define MAGIC_THRESHOLD 50

static Buffer8_t *buffer = NULL, *diff = NULL;

/*
 * fastrand - fast fake random number generator
 * Warning: The low-order bits of numbers generated by fastrand()
 *          are bad as random numbers. For example, fastrand()%4
 *          generates 1,2,3,0,1,2,3,0...
 *          You should use high-order bits.
 */
static unsigned int fastrand_val;

static unsigned int
fastrand()
{
  return (fastrand_val = fastrand_val * 1103515245 + 12345);
}

static void
fastsrand(unsigned int seed)
{
  fastrand_val = seed;
}


#ifdef PALETTE
static void
HSItoRGB(double h, double s, double i, int *r, int *g, int *b)
{
  double t, rv, gv, bv;

  rv = 1 + s * sin(h - 2 * M_PI / 3);
  gv = 1 + s * sin(h);
  bv = 1 + s * sin(h + 2 * M_PI / 3);
  t = 255.999 * i / 2;
  *r = trunc(rv * t);
  *g = trunc(gv * t);
  *b = trunc(bv * t);
}


static void
make_palette(Cmap8_t *palette)
{
  int i, r, g, b;

  for (i = 0; i < MaxColor; i++) {
    HSItoRGB(4.6 - 1.5 * i / MaxColor, (double)i / MaxColor, (double)i / MaxColor, &r, &g, &b);
    palette->colors[i].col.r = r;
    palette->colors[i].col.g = g;
    palette->colors[i].col.b = b;
  }

  for (i = MaxColor; i < 256; i++) {
    if (r < 255) {
      r++;
    }
    if (r < 255) {
      r++;
    }
    if (r < 255) {
      r++;
    }
    if (g < 255) {
      g++;
    }
    if (g < 255) {
      g++;
    }
    if (b < 255) {
      b++;
    }
    if (b < 255) {
      b++;
    }
    palette->colors[i].col.r = r;
    palette->colors[i].col.g = g;
    palette->colors[i].col.b = b;
  }
}
#endif


int8_t
create(Context_t *ctx)
{
  buffer = Buffer8_new();
  diff = Buffer8_new();
  fastsrand(time(NULL));

  return 1;
}


void
on_switch_on(Context_t *ctx)
{
  ctx->ref_taken[ctx->cam] = 0;
}


void
destroy(Context_t *ctx)
{
  Buffer8_delete(buffer);
  Buffer8_delete(diff);
}


void
run(Context_t *ctx)
{
  uint32_t i;
  int x, y;
  u_char v;
  Buffer8_t *src1;
  Buffer8_t *src2;
  Pixel_t *dst;

#ifdef PALETTE
  make_palette(ctx->cf->cur);
  ctx->cf->refresh = 1;
#endif

  pthread_mutex_lock(&ctx->cam_mtx[ctx->cam]);
  src1 = ctx->cam_save[ctx->cam][0];
#if 1
  src2 = ctx->cam_ref[ctx->cam];
  Buffer8_substract_y(src1, src2, MAGIC_THRESHOLD, diff);

  // for (i = 0; i < BUFFSIZE - WIDTH; i++) /* oliv3: why -WIDTH ? */
  for (i = 0; i < BUFFSIZE; i++) {
    buffer->buffer[i] |= diff->buffer[i];
  }
#endif
#if 0
  // for (i = 0; i < BUFFSIZE - WIDTH; i++) {
  for (i = 0; i < BUFFSIZE; i++) {
    // v = (src[i]>>16) & 0xff;
    v = src1->buffer[i];

    if (v > 150) {
      buffer->buffer[i] |= v;
    }
  }
#endif
#if 0
  // for (i = 0; i < BUFFSIZE - WIDTH; i++) {
  for (i = 0; i < BUFFSIZE; i++) {
    // v = src[i] & 0xff;
    v = src1->buffer[i];

    if (v < 60) {
      buffer->buffer[i] |= 0xff - v;
    }
  }
#endif
  pthread_mutex_unlock(&ctx->cam_mtx[ctx->cam]);

  for (x = 1; x < WIDTH - 1; x++) {
    i = WIDTH + x;
    for (y = 1; y < HEIGHT; y++) {
      v = buffer->buffer[i];

      if (v < Decay) {
        buffer->buffer[i - WIDTH] = 0;
      } else {
        buffer->buffer[i - WIDTH + fastrand() % 3 - 1] = v - (fastrand() & Decay);
      }
      i += WIDTH;
    }
  }

  dst = passive_buffer(ctx)->buffer;
  for (y = 0; y < HEIGHT; y++)
    // for (x = 1; x < WIDTH - 1; x++)
    for (x = 0; x < WIDTH; x++)
      // dest[y * WIDTH + x] = palette[buffer[y*video_width+x]];
    {
      dst[y * WIDTH + x] = buffer->buffer[y * WIDTH + x];
    }
}
