/***************************************************************************
 *   Copyright (C) 2007 by Miguel Chavez Gamboa                            *
 *   miguel.chavez.gamboa@gmail.com                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/
#include <KLocale>
#include <KMessageBox>
#include <KFileDialog>

#include <QByteArray>
#include <QRegExpValidator>

#include "producteditor.h"

ProductEditorUI::ProductEditorUI( QWidget *parent )
: QFrame( parent )
{
    setupUi( this );
}

ProductEditor::ProductEditor( QWidget *parent )
: KDialog( parent )
{
    ui = new ProductEditorUI( this );
    setMainWidget( ui );
    setCaption( i18n("Product Editor") );
    setButtons( KDialog::Ok|KDialog::Cancel );

    ui->btnChangeCode->setIcon(QIcon(DesktopIcon("edit-clear", 32)));

    //Set Validators for input boxes
    QRegExp regexpC("[0-9]{1,13}"); //(EAN-13 y EAN-8) .. y productos sin codigo de barras?
    QRegExpValidator * validatorEAN13 = new QRegExpValidator(regexpC, this);
    ui->editCode->setValidator(validatorEAN13);
    ui->editUtility->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3,ui->editUtility));
    ui->editTax->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3,ui->editTax));
    ui->editExtraTaxes->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3,ui->editExtraTaxes));
    ui->editCost->setValidator(new QDoubleValidator(0.00, 999999999999.99, 3, ui->editCost));
    ui->editStockQty->setValidator(new QDoubleValidator(0.00,999999999999.99, 3, ui->editStockQty));
    ui->editPoints->setValidator(new QIntValidator(0,999999999, ui->editPoints));
    ui->editFinalPrice->setValidator(new QDoubleValidator(0.00,999999999999.99, 3, ui->editFinalPrice));

    connect( ui->btnPhoto          , SIGNAL( clicked() ), this, SLOT( changePhoto() ) );
    connect( ui->btnCalculatePrice , SIGNAL( clicked() ), this, SLOT( calculatePrice() ) );
    connect( ui->btnChangeCode,      SIGNAL( clicked() ), this, SLOT( changeCode() ) );
    connect( ui->editCode, SIGNAL(textEdited(const QString &)), SLOT(checkIfCodeExists()));

    status = statusNormal;
    modifyCode = false;
    ui->labelError->hide();

    QTimer::singleShot(750, this, SLOT(checkIfCodeExists()));
}

ProductEditor::~ProductEditor()
{
    delete ui;
}

void ProductEditor::setDb(QSqlDatabase database)
{
  db = database;
  if (!db.isOpen()) db.open();
  populateCategoriesCombo();
  populateMeasuresCombo();
}

void ProductEditor::populateCategoriesCombo()
{
  QSqlQuery query(db);
  if (query.exec("select catid,text from categories;")) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      QString text = query.value(fieldText).toString();
      ui->categoriesCombo->addItem(text);
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}

void ProductEditor::populateMeasuresCombo()
{
  QSqlQuery query(db);
  if (query.exec("select id,text from measures;")) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      QString text = query.value(fieldText).toString();
      ui->measuresCombo->addItem(text);
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}

int ProductEditor::getCategoryId()
{
  QSqlQuery query(db);
  int code=-1;
  QString currentText = ui->categoriesCombo->currentText();
  if (query.exec("select catid,text from categories;")) {
    while (query.next() && (code == -1)) {
      int fieldId   = query.record().indexOf("catid");
      int fieldText = query.record().indexOf("text");
      int id = query.value(fieldId).toInt();
      QString text = query.value(fieldText).toString();
      if (text == currentText) {
        code = id;
      }
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
  return code;
}


int ProductEditor::getMeasureId()
{
  QSqlQuery query(db);
  int code=-1;
  QString currentText = ui->measuresCombo->currentText();
  if (query.exec("select id,text from measures;")) {
    while (query.next() && (code == -1)) {
      int fieldId   = query.record().indexOf("id");
      int fieldText = query.record().indexOf("text");
      int id = query.value(fieldId).toInt();
      QString text = query.value(fieldText).toString();
      if (text == currentText) {
        code = id;
      }
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
  return code;
}


void ProductEditor::setCategory(QString str)
{
 int idx = ui->categoriesCombo->findText(str,Qt::MatchCaseSensitive);
 if (idx > -1) ui->categoriesCombo->setCurrentIndex(idx);
 else { 
  qDebug()<<"Str not found:"<<str;
  }
}

void ProductEditor::setCategory(int i)
{
 QString text = getCategoryStr(i);
 setCategory(text);
}

void ProductEditor::setMeasure(int i)
{
 QString text = getMeasureStr(i);
 setMeasure(text);
}

void ProductEditor::setMeasure(QString str)
{
int idx = ui->measuresCombo->findText(str,Qt::MatchCaseSensitive);
 if (idx > -1) ui->measuresCombo->setCurrentIndex(idx);
 else { 
  qDebug()<<"Str not found:"<<str;
  }
}

QString ProductEditor::getCategoryStr(int c)
{
 QString str;
 QSqlQuery query(db);
 QString qstr = QString("select text from categories where catid=%1;").arg(c);
 if (query.exec(qstr)) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      str = query.value(fieldText).toString();
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
 return str;
}

QString ProductEditor::getMeasureStr(int c)
{
 QString str;
 QSqlQuery query(db);
 QString qstr = QString("select text from measures where id=%1;").arg(c);
 if (query.exec(qstr)) {
    while (query.next()) {
      int fieldText = query.record().indexOf("text");
      str = query.value(fieldText).toString();
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
 return str;
}

void ProductEditor::changePhoto()
{
 QString fname = KFileDialog::getOpenFileName();
  if (!fname.isEmpty()) {
    QPixmap p = QPixmap(fname);
    setPhoto(p);
  }
}

void ProductEditor::calculatePrice()
{
 double finalPrice=0.0;
 if (ui->editCost->text().isEmpty()) {
   ui->editCost->setFocus();
 }
 else if (ui->editUtility->text().isEmpty()) {
   ui->editUtility->setFocus();
 }
 else if (ui->editTax->text().isEmpty()) {
   ui->editTax->setText("0.0");
   ui->editTax->setFocus();
   ui->editTax->selectAll();
 }
 else {
  if (ui->editExtraTaxes->text().isEmpty()) {
   ui->editExtraTaxes->setText("0.0");
   ui->editExtraTaxes->setFocus();
   ui->editExtraTaxes->selectAll();
  }
  //TODO: if TAXes are included in cost...
  double cost    = ui->editCost->text().toDouble();
  double utility = ui->editUtility->text().toDouble();
  double tax     = ui->editTax->text().toDouble();
  double tax2    = ui->editExtraTaxes->text().toDouble(); 
  //Utility is calculated before taxes... Taxes include utility... is it ok?
  utility = ((utility/100)*cost);
  double cu      = cost + utility;
  tax     = ((tax/100)*cu);
  tax2    = ((tax2/100)*cu);
  finalPrice = cost + utility + tax + tax2;
  
  ui->editFinalPrice->setText(QString::number(finalPrice));
  ui->editFinalPrice->selectAll();
  ui->editFinalPrice->setFocus();
  }
}

void ProductEditor::changeCode()
{
  //this enables the code editing... to prevent unwanted code changes...
  enableCode();
  ui->editCode->setFocus();
  ui->editCode->selectAll();
}

void ProductEditor::checkIfCodeExists()
{
  ui->labelError->hide();
  enableButtonOk( true );
  QString codeStr = ui->editCode->text();
  if (codeStr.isEmpty()) {
    codeStr="-1";
    enableButtonOk(false);
  }
  QSqlQuery query(db);
  if (query.exec(QString("select * from products where code=%1;").arg(codeStr) )) {
    if (query.size() == 1 ) {
      status = statusMod;
      if (!modifyCode) {
        while (query.next()) {
          int fieldName = query.record().indexOf("name");
          int fieldPrice= query.record().indexOf("price");
          int fieldStockQty = query.record().indexOf("stockqty");
          int fieldCost     = query.record().indexOf("cost");
          int fieldUnits    = query.record().indexOf("units");
          int fieldSoldUnits= query.record().indexOf("soldunits");
          int fieldDateLastS= query.record().indexOf("datelastsold");
          int fieldTaxp     = query.record().indexOf("taxpercentage");
          int fieldEtraTax  = query.record().indexOf("extrataxes");
          int fieldCategory = query.record().indexOf("category");
          int fieldPhoto    = query.record().indexOf("photo");
          int fieldPoints   = query.record().indexOf("points");

          double   price    = query.value(fieldPrice).toDouble();
          double   stockq   = query.value(fieldStockQty).toDouble();
          QString  name     = query.value(fieldName).toString();
          double   cost     = query.value(fieldCost).toDouble();
          int      units    = query.value(fieldUnits).toInt();
          double   tax1     = query.value(fieldTaxp).toDouble();
          double   tax2     = query.value(fieldEtraTax).toDouble();
          int      category = query.value(fieldCategory).toInt();
          QByteArray photoBA= query.value(fieldPhoto).toByteArray();
          qulonglong points = query.value(fieldPoints).toULongLong();
          QString  lastSold = query.value(fieldDateLastS).toString();
          //double   soldUnits= query.value(fieldSoldUnits).toDouble();

          //Prepopulate dialog...
          ui->editDesc->setText(name);
          ui->editStockQty->setText(QString::number(stockq));
          setCategory(category);
          setMeasure(units);
          ui->editCost->setText(QString::number(cost));
          ui->editTax->setText(QString::number(tax1));
          ui->editExtraTaxes->setText(QString::number(tax2));
          ui->editFinalPrice->setText(QString::number(price));
          ui->editPoints->setText(QString::number(points));
          if (!photoBA.isEmpty()) {
            QPixmap photo;
            photo.loadFromData(photoBA);
            setPhoto(photo);
          }
        }
      } else {
        ui->labelError->show();
        enableButtonOk( false );
      }
    } else {
      if (!modifyCode) {
        //clear all used edits
        ui->editDesc->clear();
        ui->editStockQty->clear();
        setCategory(1);
        setMeasure(1);
        ui->editCost->clear();
        ui->editTax->clear();
        ui->editExtraTaxes->clear();
        ui->editFinalPrice->clear();
        ui->editPoints->clear();
        ui->editUtility->clear();
        ui->editFinalPrice->clear();
        ui->labelPhoto->setText("No Photo");
      }
      qDebug()<< "no product found with code "<<codeStr<<" .query.size()=="<<query.size();
    }
  }
  else {
    qDebug()<<"ERROR: "<<query.lastError();
  }
}


void ProductEditor::setPhoto(QPixmap p)
{
  int max = 150;
  QPixmap newPix;
  if ((p.height() > max) || (p.width() > max) ) {
    if (p.height() == p.width()) {
      newPix = p.scaled(QSize(max, max));
    }
    else if (p.height() > p.width() ) {
      newPix = p.scaledToHeight(max);
    }
    else  {
      newPix = p.scaledToWidth(max);
    }
  } else newPix=p;
  ui->labelPhoto->setPixmap(newPix);
  pix=newPix;
}

void ProductEditor::slotButtonClicked(int button)
{
  if (button == KDialog::Ok) {
    if (status == statusNormal) QDialog::accept();
    else {
      qDebug()<< "Button = OK, status == statusMOD";
      done(statusMod); 
    }
  }
  else QDialog::reject();
}

#include "producteditor.moc"
