/*====================================================================*
 -  Copyright (C) 2001 Leptonica.  All rights reserved.
 -  This software is distributed in the hope that it will be
 -  useful, but with NO WARRANTY OF ANY KIND.
 -  No author or distributor accepts responsibility to anyone for the
 -  consequences of using this software, or for whether it serves any
 -  particular purpose or works at all, unless he or she says so in
 -  writing.  Everyone is granted permission to copy, modify and
 -  redistribute this source code, for commercial or non-commercial
 -  purposes, with the following restrictions: (1) the origin of this
 -  source code must not be misrepresented; (2) modified versions must
 -  be plainly marked as such; and (3) this notice may not be removed
 -  or altered from any source or modified source distribution.
 *====================================================================*/

#ifndef  LEPTONICA_DEWARP_H
#define  LEPTONICA_DEWARP_H

/*
 *  dewarp.h
 *
 *     Data structure to hold arrays and results for generating
 *     horizontal and vertical disparity arrays based on textlines.
 *     Each disparity array is two-dimensional.  The vertical disparity
 *     array gives a vertical displacement, relative to the lowest point
 *     in the textlines.  The horizontal disparty array gives a horizontal
 *     displacement, relative to the minimum values (for even pages)
 *     or maximum values (for odd pages) of the left and right ends of
 *     full textlines.  Horizontal alignment always involves translations
 *     away from the book gutter.
 *
 *     We have intentionally separated the process of building models
 *     from the rendering process that uses the models.  For any page,
 *     the building operation either creates an actual model (that is,
 *     a model with at least the vertical disparity being computed, and
 *     for which the 'success' flag is set) or fails to create a model.
 *     However, at rendering time, a page can have one of two different
 *     types of models.
 *     (1) A valid model is an actual model that meets the rendering
 *         requirements, including whether a full model (horizontal
 *         and vertical disparity) is required, and limits on model
 *         curvatures.  See dewarpaTestForValidModel() for details.
 *         Valid models are identified by dewarpaInsertRefModels(), which
 *         sets the 'valid' field.  Only valid models are used for rendering.
 *     (2) A reference model is used by a page that doesn't have
 *         a valid model, but has a nearby valid model of the same
 *         parity (even/odd page) that it can use.  The range to search
 *         for a valid model is given by the 'maxdist' field.
 *     A page may have neither a valid nor a reference model, in
 *     which case rendering simply copies the input image.
 *     By default, only the vertical disparity is required for
 *     an actual model ('fullmodel' = 0).  The 'maxdist' parameter is
 *     input when the dewarpa is created.  The other rendering parameters
 *     have default values given in dewarp.c.  All parameters used by
 *     rendering can be set (or reset) using accessors.
 *
 *     After dewarping, use of the vertical disparity will cause all
 *     points on each altered curve to have a y-value equal to the
 *     minimum.  Use of horizontal disparity will cause the left
 *     and right edges of the textlines to be vertically aligned if
 *     they had been typeset flush-left and flush-right, respectively.
 *
 *     The sampled disparity arrays are expanded to full resolution,
 *     using linear interpolation, and this is further expanded
 *     by slope continuation to the right and below if the image
 *     is larger than the full resolution disparity arrays.  Then
 *     the disparity correction can be applied to the input image.
 *     If the input pix are 2x reduced, the expansion from sampled
 *     to full res uses the product of (sampling) * (redfactor).
 *
 *     The most accurate results are produced at full resolution, and
 *     this is generally recommended.
 */

    /* Note on versioning of the serialization of this data structure:
     * The dewarping utility and the stored data can be expected to change.
     * In most situations, the serialized version is ephemeral -- it is
     * not needed after being used.  No functions will be provided to
     * convert between different versions. */
#define  DEWARP_VERSION_NUMBER      2

struct L_Dewarpa
{
    l_int32             nalloc;        /* size of dewarp ptr array           */
    l_int32             maxpage;       /* maximum page number in array       */
    struct L_Dewarp   **dewarp;        /* array of ptrs to page dewarp       */
    struct Numa        *namodels;      /* list of page numbers for pages     */
                                       /* with page models                   */
    struct Numa        *napages;       /* list of page numbers with either   */
                                       /* page models or ref page models     */
    l_int32             redfactor;     /* reduction factor of input: 1 or 2  */
    l_int32             sampling;      /* disparity arrays sampling factor   */
    l_int32             minlines;      /* min number of long lines required  */
    l_int32             maxdist;       /* max distance for getting ref pages */
    l_int32             min_medcurv;   /* minimum median abs line curvature, */
                                       /* in micro-units                     */
    l_int32             max_medcurv;   /* maximum median abs line curvature, */
                                       /* in micro-units                     */
    l_int32             max_leftcurv;  /* maximum left edge line curvature,  */
                                       /* in micro-units                     */
    l_int32             max_rightcurv; /* maximum right edge line curvature, */
                                       /* in micro-units                     */
    l_int32             fullmodel;     /* both disparity arrays required     */
    l_int32             modelsready;   /* invalid models have been removed   */
                                       /* and refs built against valid set   */
};
typedef struct L_Dewarpa L_DEWARPA;


struct L_Dewarp
{
    struct Pix     *pixs;         /* source pix, 1 bpp                       */
    struct Pix     *pixd;         /* dewarped pix; 1, 8 or 32 bpp            */
    struct FPix    *sampvdispar;  /* sampled vert disparity array            */
    struct FPix    *samphdispar;  /* sampled horiz disparity array           */
    struct FPix    *fullvdispar;  /* full vert disparity array               */
    struct FPix    *fullhdispar;  /* full horiz disparity array              */
    struct Numa    *naflats;      /* sorted flat location of each line       */
    struct Numa    *nacurves;     /* sorted curvature of each line           */
    l_int32         w;            /* width of source image                   */
    l_int32         h;            /* height of source image                  */
    l_int32         pageno;       /* page number; important for reuse        */
    l_int32         sampling;     /* sampling factor of disparity arrays     */
    l_int32         redfactor;    /* reduction factor of pixs: 1 or 2        */
    l_int32         minlines;     /* min number of long lines required       */
    l_int32         nlines;       /* number of long lines found              */
    l_int32         medcurv;      /* median abs curvature in micro-units     */
    l_int32         leftcurv;     /* left edge curvature in micro-units      */
    l_int32         rightcurv;    /* right edge curvature in micro-units     */
    l_int32         nx;           /* number of sampling pts in x direction   */
    l_int32         ny;           /* number of sampling pts in y direction   */
    l_int32         hasref;       /* 0 if normal; 1 if has a refpage         */
    l_int32         refpage;      /* page with disparity model to use here   */
    l_int32         success;      /* sets to 1 if model build succeeds       */
    l_int32         valid;        /* sets to 1 if model is valid             */
    l_int32         debug;        /* sets to 1 if debug output requested     */
};
typedef struct L_Dewarp L_DEWARP;

#endif  /* LEPTONICA_DEWARP_H */
