/***********************************************************************************

	Copyright (C) 2007-2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_DATA_HEADER
#define LIFEOGRAPH_DATA_HEADER


#include "helpers.hpp"	// i18n headers


namespace LIFEO
{
const char						DB_FILE_HEADER[] = "LIFEOGRAPHDB";
const int						DB_FILE_VERSION_INT( 56 );
const int						DB_FILE_VERSION_INT_MIN( 51 );
const std::string::size_type	PASSPHRASE_MIN_SIZE( 4 );
const unsigned int				YEAR_MAX = 2099;
const unsigned int				YEAR_MIN = 1900;

enum Result
{
	OK,
	ABORTED,
	SUCCESS,
	FAILURE,
	COULD_NOT_START,
	COULD_NOT_FINISH,
	WRONG_PASSWORD,
	APPEARENTLY_ENCRYTED_FILE,
	APPEARENTLY_PLAIN_FILE,
	INCOMPATIBLE_FILE,
	CORRUPT_FILE,
	EMPTY_DATABASE
};

enum DATE_COMPONENTS
{
	YEAR = 1,
	MONTH = 2,
	DAY = 4
};


typedef unsigned long Date;

inline
unsigned int
isolate_year( Date date )
{
	return ( ( date & 0xFFFFE00 ) >> 9 );
}
inline
unsigned int
isolate_month( Date date )
{
	return( ( date & 0x1E0 ) >> 5 );
}
inline
unsigned int
isolate_day( Date date )
{
	return( date & 0x1F );
}
inline
Date
combine_date( unsigned int year, unsigned int month, unsigned int day )
{
	return( ( year << 9 ) | ( month << 5 ) | day );
}

inline Date make_year( unsigned int y ) { return( y << 9 ); }
inline Date make_month( unsigned int m ) { return( m << 5 ); }

inline
bool
validate_day( Date date )
{
	return( Glib::Date::get_days_in_month(
					static_cast< Glib::Date::Month >( isolate_month( date ) ),
					isolate_year( date ) ) >= isolate_day( date ) );
}

struct Theme
{
	Glib::ustring				font;
	std::string					color_base;
	std::string					color_text;
	std::string					color_heading;
	std::string					color_subheading;
	std::string					color_highlight;
};


class Tag
{
	public:
		typedef std::list<Tag*> Tagpool;

								~Tag( void );
		unsigned int			decrease_usercount( unsigned int = 1 );
		const std::string&		get_name( void ) const;
		void					set_name( const std::string& );

		static Tag*				create( const std::string& );
		static void				delete_tag( Tag * );
		static Tag*				get_tag( unsigned int );
		static Tag*				get_tag( const std::string & );
		static const Tagpool&	get_pool( void );
		static void				clear_pool( void );

	protected:
								Tag( const std::string&, unsigned int = 0 );

		std::string				Name;
		unsigned int			UserCount;

	private:
		// all declared tags
		static Tagpool			Pool;

};

// tags of an entry
class Tagset : public Tag::Tagpool
{
	public:
								~Tagset( void );
		bool					add_tag( const std::string& );
		bool					add_tag( Tag* );
		bool					checkfor_member( const Tag* ) const;
		const Tag*				get_tag( unsigned int ) const;
		bool					remove_tag( const std::string& );
		void					remove_tag( Tag* );

	protected:

	private:

};

class Chapter
{
	public:
		typedef std::list< Chapter* > Set;
		//~ static Chapter*			create (const Glib::ustring&, const Glib::Date&);
		static Chapter*			create( const Glib::ustring& );
		const Glib::ustring&	get_name( void ) const;
		void					set_name( const Glib::ustring& );
		bool					is_initialized( void ) { return m_flag_initialized; }
		const Glib::Date&		get_date_glib( void ) const;
		Glib::ustring			get_date_str( void ) const;
		Date					get_date( void ) const;
		void					set_date( const Glib::Date& );

	protected:
								Chapter( void );
								Chapter( const Glib::ustring&, const Glib::Date& );
								Chapter( const Glib::ustring& );

		Glib::ustring			m_name;
		Glib::Date				m_date_begin;
		bool					m_flag_initialized;
};

class Chapterset : public Chapter::Set
{
	public:
		typedef std::list< Chapterset* > Set;
								Chapterset( const Glib::ustring& );
								//~ ~Chapterset (void);

		const Glib::ustring&	get_name( void ) const;
		void					set_name( const Glib::ustring& );
		Chapter*				add_chapter( const Glib::ustring& );
		void					remove_chapter( Chapter* );
		bool					rename_chapter( Chapter*, const Glib::ustring& );
		bool					set_chapter_date( Chapter*, const Glib::ustring& );

	protected:
		Glib::ustring			m_name;
};

class Entry
{
	public:
								Entry();
								Entry( Date, const Glib::ustring&  );

		const Glib::ustring&	get_text() const { return m_text; }
		void					set_text( const Glib::ustring &text )
		{ m_text = text; }

		const Glib::ustring		get_title () const
		{
			if( m_text.size() < 1 )
				// TRANSLATORS: title of an empty diary entry
				return( _("<empty entry>") );
			unsigned int l_pos = m_text.find( '\n', 0 );
			if( l_pos == std::string::npos )
				return m_text;
			else
				return m_text.substr( 0, l_pos );
		}
		Glib::Date				get_date_glib( void ) const;
		Date					get_date( void ) const
		{ return m_date; }
		const std::string		get_date_str( void ) const;
		std::string				get_date_str( LIFEO::DATE_COMPONENTS ) const;
		std::string				get_year_str( void ) const;
		std::string				get_yearmonth_str( void ) const;
		std::string				get_monthday_str( void ) const;
		std::string				get_day_str( void ) const;

		bool					get_filtered_out( void ) const
		{ return m_flag_filteredout; }
		void					set_filtered_out( bool filteredout = true )
		{ m_flag_filteredout = filteredout; }

		unsigned int			get_day( void ) const
		{ return isolate_day( m_date ); }
		unsigned int			get_month( void ) const
		{ return isolate_month( m_date ); }
		unsigned int			get_year( void ) const
		{ return isolate_year( m_date ); }
		unsigned int			get_yearmonth( void ) const
		{ return( m_date & 0xFFFFFE0 ); }

		bool					is_favored( void ) const { return m_option_favored; }
		void					make_favored( void ) { m_option_favored = true; }
		void					make_unfavored( void ) { m_option_favored = false; }

		bool					get_spellcheck( void ) const
		{ return m_option_spellcheck; }
		void					set_spellcheck( bool opt_spell )
		{ m_option_spellcheck = opt_spell; }


		void					set_tags( const Tagset& );
		const Tagset&			get_tags( void ) const;
		Tagset&					get_tags( void );

	protected:
		Date					m_date;
		Glib::ustring			m_text;
		Tagset					m_tags;

		bool					m_option_favored;
		bool					m_option_spellcheck;
		bool					m_flag_filteredout;

	private:

};


typedef std::vector< Entry* >::iterator Entryiter;


class Database
{
	public:
								Database( void );

		// DISK I/O
		LIFEO::Result			read_body( void );
		LIFEO::Result			read_header( void );
		LIFEO::Result			read_header37( void );

		LIFEO::Result			write( void );
		bool					write_backup( void );
		LIFEO::Result			write_copy( const std::string&, const std::string& );
		LIFEO::Result			write_txt( const std::string& );

		void					clear( void );

		int						get_size() const { return m_entries.size(); }

		void					set_path( const std::string& );
		const std::string&		get_path( void ) const;
		bool					is_path_set( void ) const;

		void					set_passphrase( const std::string& );
		std::string				get_passphrase( void ) const;
		bool					compare_passphrase( const std::string& ) const;
		bool					is_passphrase_set( void ) const;

		//~ Entry*					get_entry( unsigned int, unsigned int, unsigned int );
		Entry*					get_entry( Date );
		Entry*					get_entry_create( Date );
		const Entry*			get_entry( int ) const;
		Entry*					get_entry( int );
		Entry*					get_entry_today( void );
		Entry*					get_entry_first( void );
		Entry*					get_entry_before( const Entry * const );
		Entry*					get_entry_after( const Entry * const );
		bool					is_first( const Entry * const ) const;
		bool					is_last( const Entry * const ) const;
		unsigned int			get_entryliststyle( void ) const
		{ return m_option_entryliststyle; }
		void					set_entryliststyle( unsigned int style )
		{ m_option_entryliststyle = style; }

		bool					add_entry( const Entry *);
		Entry*					create_entry( Date, const Glib::ustring& = "" );
		// adds a new entry to today even if there is already one or more:
		Entry*					add_today( void );
		bool					delete_entry( Entry * );
		bool					delete_entry( int );
		int						list_entries( void );

		int						set_filter( const Glib::ustring& );
		void					remove_filter( void );

		const Chapterset::Set*	get_chaptersets( void ) const
		{
			return &m_chaptersets;
		}
		Chapterset*				add_chapterset( const Glib::ustring& );
		void					remove_chapterset( Chapterset* );
		bool					rename_chapterset( Chapterset*, const Glib::ustring& );
		Glib::ustring			find_uniquename_chapterset( const Glib::ustring& );

		bool					get_spellcheck( void )
		{ return m_option_spellcheck; }
		void					set_spellcheck( bool opt_spell )
		{ m_option_spellcheck = opt_spell; }
		bool					is_encrypted( void )
		{ return m_flag_encrypted; }

		bool					has_theme( void )
		{ return( m_theme != NULL ); }
		Theme*					get_theme( void )
		{ return m_theme; }
		void					set_theme( Theme* theme )
		{	if( m_theme )
				delete m_theme;
			m_theme = theme;
		}
		void					dismiss_theme( void )
		{
			if( m_theme )
			{
				delete m_theme;
				m_theme = NULL;
			}
		}

	protected:
		std::vector< Entry* >	m_entries;
		Chapterset::Set			m_chaptersets;
		std::string				m_path;
		std::string				m_passphrase;

		LIFEO::Result			parse_db_body_text( std::stringstream& );
		LIFEO::Result			parse_db_body_text37( std::stringstream& );
		bool					create_db_body_text( std::stringstream& );
		bool					create_db_header_text( std::stringstream&, bool );
		LIFEO::Result			read_plain( void );
		LIFEO::Result			read_encrypted( void );
		LIFEO::Result			write_plain( const std::string&, bool = false );
		LIFEO::Result			write_encrypted( const std::string& );

		std::ios::pos_type		BodyPosition;

	private:
		// options
		bool					m_flag_encrypted;
		bool					m_option_spellcheck;
		unsigned int			m_option_entryliststyle;
		Theme					*m_theme;
		bool					m_flag_old;

	friend class Lifeograph;
};

} // end of namespace LIFEO

#endif
