/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>
#include <gtkmm/scrolledwindow.h>

#include "helpers.hpp"
#include "dialogchaptereditor.hpp"


using namespace LIFEO;


DialogChaptereditor::DialogChaptereditor( Database *ptr2database )
	:	Gtk::Dialog( _("Edit Chapters") ), m_ptr2database( ptr2database ),
	m_flag_groups_changed( false )
{
	Gtk::Table				*table_lists;
	Gtk::HBox				*hbox_sets, *hbox_chapters;
	Gtk::Label				*label_sets, *label_chapters;
	Gtk::ScrolledWindow		*scrolledwindow_sets, *scrolledwindow_chapters;

	try
	{
		table_lists = Gtk::manage( new Gtk::Table );
		hbox_sets = Gtk::manage( new Gtk::HBox );
		hbox_chapters = Gtk::manage( new Gtk::HBox );
		label_sets = Gtk::manage( new Gtk::Label( _("Chapter Sets:") ) );
		label_chapters = Gtk::manage( new Gtk::Label( _("Chapters:") ) );
		m_treeview_groups = Gtk::manage( new Gtk::TreeView );
		m_treeview_chapters = Gtk::manage( new Gtk::TreeView );
		scrolledwindow_sets = Gtk::manage( new Gtk::ScrolledWindow );
		scrolledwindow_chapters = Gtk::manage( new Gtk::ScrolledWindow );
		m_button_group_add = Gtk::manage( new Gtk::Button( _("Add _Set"), true ) );
		m_button_group_remove = Gtk::manage( new Gtk::Button( _("Remove Set") ) );
		m_button_chapter_add = Gtk::manage( new Gtk::Button( _("Add C_hapter"), true ) );
		m_button_chapter_remove = Gtk::manage( new Gtk::Button( _("Remove Chapter") ) );

		m_cellrenderertext_groupname = Gtk::manage( new CellrendererTextEditable );
		m_cellrenderertext_chaptername = Gtk::manage( new CellrendererTextEditable );
		m_cellrenderertext_chapterdate = Gtk::manage( new CellrendererTextEditable );
		m_treecolumn_setname = Gtk::manage(	new Gtk::TreeViewColumn( _("Name"),
											*m_cellrenderertext_groupname ) );
		m_treecolumn_chaptername = Gtk::manage(	new Gtk::TreeViewColumn( _("Name"),
												*m_cellrenderertext_chaptername ) );
		m_treecolumn_chapterdate = Gtk::manage(	new Gtk::TreeViewColumn( _("Begin Date"),
												*m_cellrenderertext_chapterdate ) );

	}
	catch( std::exception &ex )
	{
		throw ex;
	}

	m_liststore_groups = Gtk::ListStore::create( m_colrec_sets );
	m_liststore_chapters = Gtk::ListStore::create( m_colrec_chapters );

	m_treeview_groups->set_model( m_liststore_groups );
	m_treeview_chapters->set_model( m_liststore_chapters );
	m_treeview_groups->append_column( *m_treecolumn_setname );
	m_treeview_chapters->append_column( *m_treecolumn_chaptername );
	m_treeview_chapters->append_column( *m_treecolumn_chapterdate );

	m_treeview_chapters->set_rules_hint( true );

	m_treecolumn_chaptername->set_expand( true );

	m_treecolumn_setname->set_cell_data_func( *m_cellrenderertext_groupname,
			sigc::mem_fun( *this,
					&DialogChaptereditor::handle_celldata_groupname ) );
	m_treecolumn_chaptername->set_cell_data_func( *m_cellrenderertext_chaptername,
			sigc::mem_fun( *this,
					&DialogChaptereditor::handle_celldata_chaptername ) );
	m_treecolumn_chapterdate->set_cell_data_func( *m_cellrenderertext_chapterdate,
			sigc::mem_fun( *this,
					&DialogChaptereditor::handle_celldata_chapterdate ) );

	m_cellrenderertext_groupname->set_fixed_height_from_font( 1 );
	m_cellrenderertext_chaptername->set_fixed_height_from_font( 1 );
	m_cellrenderertext_chapterdate->set_fixed_height_from_font( 1 );

#ifdef GLIBMM_PROPERTIES_ENABLED
	m_cellrenderertext_groupname->property_editable() = true;
	m_cellrenderertext_chaptername->property_editable() = true;
	m_cellrenderertext_chapterdate->property_editable() = true;
#else
	m_cellrenderertext_groupname->set_property( "editable", true );
	m_cellrenderertext_chaptername->set_property( "editable", true );
	m_cellrenderertext_chapterdate->set_property( "editable", true );
#endif

	label_sets->set_alignment( Gtk::ALIGN_LEFT );
	label_chapters->set_alignment( Gtk::ALIGN_LEFT );

	scrolledwindow_sets->set_policy( Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC );
	scrolledwindow_chapters->set_policy( Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC );

	scrolledwindow_sets->add( *m_treeview_groups );
	scrolledwindow_chapters->add( *m_treeview_chapters );

	hbox_sets->pack_start( *m_button_group_add );
	hbox_sets->pack_start( *m_button_group_remove );
	hbox_sets->set_spacing( 5 );
	hbox_sets->set_homogeneous( true );
	hbox_chapters->pack_start( *m_button_chapter_add );
	hbox_chapters->pack_start( *m_button_chapter_remove );
	hbox_chapters->set_spacing( 5 );
	hbox_chapters->set_homogeneous( true );

	table_lists->attach(	*label_sets, 0, 1, 0, 1,
							Gtk::FILL, Gtk::SHRINK );
	table_lists->attach(	*label_chapters, 1, 2, 0, 1,
							Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	table_lists->attach(	*scrolledwindow_sets, 0, 1, 1, 2,
							Gtk::FILL, Gtk::EXPAND | Gtk::FILL );
	table_lists->attach(	*scrolledwindow_chapters, 1, 2, 1, 2,
							Gtk::EXPAND | Gtk::FILL, Gtk::EXPAND | Gtk::FILL );
	table_lists->attach(	*hbox_sets, 0, 1, 2, 3,
							Gtk::FILL, Gtk::SHRINK );
	table_lists->attach(	*hbox_chapters, 1, 2, 2, 3,
							Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	table_lists->set_spacings( 5 );

	get_vbox()->pack_start( *table_lists );
	get_vbox()->set_spacing( 5 );

	add_button( Gtk::Stock::CLOSE, Gtk::RESPONSE_OK );

	set_default_size( 550, 300 );

	m_button_group_add->signal_clicked().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::add_group ) );
	m_button_group_remove->signal_clicked().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::remove_group ) );
	m_button_chapter_add->signal_clicked().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::add_chapter ) );
	m_button_chapter_remove->signal_clicked().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::remove_chapter ) );
	m_treeview_groups->get_selection()->signal_changed().connect( sigc::mem_fun(
			*this, &DialogChaptereditor::handle_groups_selection_changed ) );
	m_treeview_chapters->get_selection()->signal_changed().connect( sigc::mem_fun(
			*this, &DialogChaptereditor::handle_chapters_selection_changed ) );

	m_cellrenderertext_groupname->signal_editing_started().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_editing_groupname ) );
	m_cellrenderertext_groupname->signal_edited().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_edited_groupname ) );

	m_cellrenderertext_chaptername->signal_editing_started().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_editing_chaptername ) );
	m_cellrenderertext_chaptername->signal_edited().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_edited_chaptername ) );

	m_cellrenderertext_chapterdate->signal_editing_started().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_editing_chapterdate ) );
	m_cellrenderertext_chapterdate->signal_edited().connect(
			sigc::mem_fun( *this, &DialogChaptereditor::handle_edited_chapterdate ) );

	populate_groups();

	m_button_group_remove->set_sensitive( false );
	m_button_chapter_add->set_sensitive( false );
	m_button_chapter_remove->set_sensitive( false );

	show_all();
}

void
DialogChaptereditor::populate_groups( void )
{
	for (	Chapterset::Set::const_iterator iter =
					m_ptr2database->get_chaptersets()->begin();
			iter != m_ptr2database->get_chaptersets()->end();
			iter++ )
	{
		Gtk::TreeRow row;
		row = *( m_liststore_groups->append() );
		row[ m_colrec_sets.m_name ] = (*iter)->get_name();
		row[ m_colrec_sets.m_ptr2group ] = (*iter);
	}
}

void
DialogChaptereditor::populate_chapters( const Chapterset* chapterset )
{
	m_liststore_chapters->clear();
	for (	Chapterset::const_iterator iter = chapterset->begin();
			iter != chapterset->end();
			iter++ )
	{
		Gtk::TreeRow row;
		row = *( m_liststore_chapters->append() );
		row[ m_colrec_chapters.m_name ] = (*iter)->get_name();
		row[ m_colrec_chapters.m_date ] = (*iter)->is_initialized() ?
					(*iter)->get_date_str() : _(STRING::NOT_SET_ITALIC);
		row[ m_colrec_chapters.m_ptr2chapter ] = (*iter);
	}
}

void
DialogChaptereditor::handle_groups_selection_changed( void )
{
	if( m_treeview_groups->get_selection()->count_selected_rows() < 1 )
	{
		m_button_group_remove->set_sensitive( false );
		m_button_chapter_add->set_sensitive( false );
		m_liststore_chapters->clear();
	}
	else
	{
		m_button_group_remove->set_sensitive( true );
		m_button_chapter_add->set_sensitive( true );
		Gtk::TreeRow row = *m_treeview_groups->get_selection()->get_selected();
		m_ptr2group = row[ m_colrec_sets.m_ptr2group ];
		populate_chapters( m_ptr2group );
	}
}

void
DialogChaptereditor::handle_chapters_selection_changed( void )
{
	if ( m_treeview_chapters->get_selection()->count_selected_rows() < 1 )
	{
		m_button_chapter_remove->set_sensitive( false );
	}
	else
	{
		m_button_chapter_remove->set_sensitive( true );
	}
}

void
DialogChaptereditor::handle_editing_groupname(	Gtk::CellEditable* editable,
												const Glib::ustring& string_path )
{
	Gtk::Entry* entry = dynamic_cast<Gtk::Entry*>( editable );
	if ( entry )
	{
		Gtk::TreePath path( string_path );
		Gtk::TreeIter iter = m_liststore_groups->get_iter( path );
		if ( iter )
		{
			Gtk::TreeRow row = *iter;
			entry->set_text( row[ m_colrec_sets.m_name ] );
		}
	}
}

void
DialogChaptereditor::handle_edited_groupname(	const Glib::ustring& string_path,
												const Glib::ustring& text )
{
	Gtk::TreePath path( string_path );
	Gtk::TreeIter iter = m_liststore_groups->get_iter( path );
	//~ if ( iter )
	Gtk::TreeRow row = *iter;
	Chapterset *ptr2chapterset = row[ m_colrec_sets.m_ptr2group ];

	if( text.size() && m_ptr2database->rename_chapterset( ptr2chapterset, text ) )
	{
		row[ m_colrec_sets.m_name ] = text;
		m_flag_groups_changed = true;
	}
}

void
DialogChaptereditor::handle_celldata_groupname(	Gtk::CellRenderer*,
											const Gtk::TreeModel::iterator& iter )
{
	if( iter )
	{
		Gtk::TreeModel::Row row = *iter;
		Glib::ustring view_text = row[m_colrec_sets.m_name];

#ifdef GLIBMM_PROPERTIES_ENABLED
		m_cellrenderertext_groupname->property_text() = view_text;
#else
		m_cellrenderertext_groupname->set_property( "text", view_text );
#endif
	}
}

void
DialogChaptereditor::handle_editing_chaptername(	Gtk::CellEditable* editable,
													const Glib::ustring& string_path )
{
	Gtk::Entry* entry = dynamic_cast<Gtk::Entry*>( editable );
	if ( entry )
	{
		Gtk::TreePath path( string_path );
		Gtk::TreeIter iter = m_liststore_chapters->get_iter( path );
		if( iter )
		{
			Gtk::TreeRow row = *iter;
			entry->set_text( row[ m_colrec_chapters.m_name ] );
		}
	}
}

void
DialogChaptereditor::handle_edited_chaptername(	const Glib::ustring& string_path,
												const Glib::ustring& text )
{
	Gtk::TreePath path( string_path );
	Gtk::TreeIter iter = m_liststore_chapters->get_iter( path );
	//~ if ( iter )
	Gtk::TreeRow row = *iter;
	Chapter *ptr2chapter = row[ m_colrec_chapters.m_ptr2chapter ];

	if( text.size() && m_ptr2group->rename_chapter( ptr2chapter, text ) )
	{
		row[ m_colrec_chapters.m_name ] = text;
	}
}

void
DialogChaptereditor::handle_celldata_chaptername(	Gtk::CellRenderer*,
													const Gtk::TreeModel::iterator& iter )
{
	if ( iter )
	{
		Gtk::TreeModel::Row row = *iter;
		Glib::ustring view_text = row[m_colrec_chapters.m_name];

#ifdef GLIBMM_PROPERTIES_ENABLED
		m_cellrenderertext_chaptername->property_text() = view_text;
#else
		m_cellrenderertext_chaptername->set_property( "text", view_text );
#endif
	}
}

void
DialogChaptereditor::handle_editing_chapterdate(	Gtk::CellEditable* editable,
													const Glib::ustring& string_path )
{
	Gtk::Entry* entry = dynamic_cast<Gtk::Entry*>( editable );
	if( entry )
	{
		Gtk::TreePath path( string_path );
		Gtk::TreeIter iter = m_liststore_chapters->get_iter( path );
		if( iter )
		{
			entry->set_max_length( 10 );
			Gtk::TreeRow row = *iter;
			Chapter *ptr2chapter = row[ m_colrec_chapters.m_ptr2chapter ];
			if( ptr2chapter->is_initialized() )
				entry->set_text( row[ m_colrec_chapters.m_date ] );
			else
				entry->set_text( "" );
		}
	}
}

void
DialogChaptereditor::handle_edited_chapterdate(	const Glib::ustring& string_path,
												const Glib::ustring& text )
{
	Gtk::TreePath path( string_path );
	Gtk::TreeIter iter = m_liststore_chapters->get_iter( path );
	if( iter )
	{
		Gtk::TreeRow row = *iter;
		Chapter *ptr2chapter = row[ m_colrec_chapters.m_ptr2chapter ];

		if( text.size() && m_ptr2group->set_chapter_date( ptr2chapter, text ) )
		{
			row[ m_colrec_chapters.m_date ] = ptr2chapter->get_date_str();
			populate_chapters( m_ptr2group );
		}
	}
}

void
DialogChaptereditor::handle_celldata_chapterdate(	Gtk::CellRenderer*,
													const Gtk::TreeModel::iterator& iter )
{
	if( iter )
	{
		Gtk::TreeModel::Row row = *iter;
		Glib::ustring view_text = row[m_colrec_chapters.m_date];

#ifdef GLIBMM_PROPERTIES_ENABLED
		m_cellrenderertext_chapterdate->property_markup() = view_text;
#else
		m_cellrenderertext_chapterdate->set_property( "markup", view_text );
#endif
	}
}

void
DialogChaptereditor::add_group( void )
{
	Gtk::TreeIter iter = m_liststore_groups->append();
	Gtk::TreeRow row = *iter;
	row[ m_colrec_sets.m_name ] =
			m_ptr2database->find_uniquename_chapterset( _("Untitled set") );
	row[ m_colrec_sets.m_ptr2group ] =
			m_ptr2database->add_chapterset( row[ m_colrec_sets.m_name ] );

	m_treeview_groups->set_cursor(	m_liststore_groups->get_path( iter ),
									*m_treecolumn_setname,
									*m_cellrenderertext_groupname, true );

	m_flag_groups_changed = true;
}

void
DialogChaptereditor::remove_group( void )
{
	if( m_treeview_groups->get_selection()->count_selected_rows() > 0 )
	{
		Gtk::TreeRow row = *m_treeview_groups->get_selection()->get_selected();
		m_ptr2group = row[ m_colrec_sets.m_ptr2group ];
		m_ptr2database->remove_chapterset( m_ptr2group );
		m_liststore_groups->clear();
		populate_groups();

		m_flag_groups_changed = true;
	}
}

void
DialogChaptereditor::add_chapter( void )
{
	Gtk::TreeIter iter = m_liststore_chapters->append();
	Gtk::TreeRow row = *iter;
	row[ m_colrec_chapters.m_name ] = _("Untitled chapter");
	row[m_colrec_chapters.m_date] = _(STRING::NOT_SET_ITALIC);
	row[ m_colrec_chapters.m_ptr2chapter ] =
			m_ptr2group->add_chapter( row[ m_colrec_chapters.m_name ] );

	m_treeview_chapters->set_cursor(	m_liststore_chapters->get_path( iter ),
										*m_treecolumn_chaptername,
										*m_cellrenderertext_chaptername, true );
}

void
DialogChaptereditor::remove_chapter( void )
{
	if( m_treeview_chapters->get_selection()->count_selected_rows() > 0 )
	{
		Gtk::TreeRow row = *m_treeview_chapters->get_selection()->get_selected();
		m_ptr2group->remove_chapter( row[ m_colrec_chapters.m_ptr2chapter ] );
		populate_chapters( m_ptr2group );
	}
}
