/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cmath>

#include "view_entry.hpp"
#include "widget_textview.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Glib::RefPtr<Gtk::TextTag>      TextbufferDiary::m_tag_strikethrough;
Glib::RefPtr<Gtk::TextTag>      TextbufferDiary::m_tag_highlight;
Glib::RefPtr<Gtk::TextTag>      TextbufferDiary::m_tag_checkbox;

Gdk::Color				TextbufferDiary::m_theme_color_match( "green" );
Gdk::Color				TextbufferDiary::m_theme_color_markup( "gray" );
Gdk::Color				TextbufferDiary::m_theme_color_link( "navy" );
Gdk::Color				TextbufferDiary::m_theme_color_linkbroken( "red" );

Gtk::TextBuffer					*UndoEdit::m_ptr2buffer;

// LINK
Link::Link(	const Glib::RefPtr< Gtk::TextMark > &start,
			const Glib::RefPtr< Gtk::TextMark > &end )
	:	m_mark_start( start ), m_mark_end( end )
{

}

Link::~Link()
{
	// TODO: is this necessary?
	Glib::RefPtr< Gtk::TextBuffer > buffer = m_mark_start->get_buffer();
	if ( buffer )
	{
		buffer->delete_mark( m_mark_start );
		buffer->delete_mark( m_mark_end );
	}
}

// LINK TO ENTRY
// static variable:
LinkEntry::Signal_void_Date LinkEntry::m_signal_activated;

LinkID::LinkID( const Glib::RefPtr< Gtk::TextMark > &start,
				const Glib::RefPtr< Gtk::TextMark > &end,
				DEID id )
:	Link( start, end ), m_id( id )
{
}

void
LinkID::go( void )
{
	Diary::d->get_element( m_id )->show();
}

LinkEntry::LinkEntry( const Glib::RefPtr< Gtk::TextMark > &start,
					  const Glib::RefPtr< Gtk::TextMark > &end,
					  Date date )
:	Link( start, end ), m_date( date )
{
}

void
LinkEntry::go( void )
{
	m_signal_activated.emit( m_date );
}

// LINK TO URI
Gtk::TextBuffer *LinkUri::m_ptr2buffer;

LinkUri::LinkUri(	const Glib::RefPtr< Gtk::TextMark > &start,
					const Glib::RefPtr< Gtk::TextMark > &end,
					const std::string& url )
	:	Link( start, end ), m_url( url )
{
}

void
LinkUri::go( void )
{
	GError *err = NULL;
	gtk_show_uri (NULL, m_url.c_str(), GDK_CURRENT_TIME, &err);
}

// LINK CHECKBOX
const Glib::ustring		LinkCheck::boxes = "☐☑☒";

LinkCheck::LinkCheck( const Glib::RefPtr< Gtk::TextMark > &start,
					  const Glib::RefPtr< Gtk::TextMark > &end,
					  unsigned int state_index )
	:	Link( start, end ), m_state_index( state_index )
{
}

void
LinkCheck::go( void )
{
	PRINT_DEBUG( "LinkCheck::go()" );
	m_state_index = ( m_state_index + 1 ) % 3;

	// link should be preserved
	TextviewDiary::m_buffer->m_flag_ongoingoperation = true;
	TextviewDiary::m_buffer->insert(
			m_mark_start->get_iter(), boxes.substr( m_state_index, 1) );

	Gtk::TextIter iter_start = m_mark_start->get_iter();
	iter_start++;
	TextviewDiary::m_buffer->erase( iter_start, m_mark_end->get_iter() );

	iter_start = m_mark_start->get_iter();
	Gtk::TextIter iter_end = m_mark_end->get_iter();
	TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_checkbox, iter_start, iter_end );

	iter_start = m_mark_end->get_iter();
	iter_end.forward_line();
	switch( m_state_index )
	{
		case 1:
		    TextviewDiary::m_buffer->remove_tag(
		            TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
			TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_highlight, iter_start, iter_end );
			break;
		case 2:
            TextviewDiary::m_buffer->remove_tag(
                    TextbufferDiary::m_tag_highlight, iter_start, iter_end );
			TextviewDiary::m_buffer->apply_tag(
					TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
			break;
		case 0:
            TextviewDiary::m_buffer->remove_tag(
                    TextbufferDiary::m_tag_strikethrough, iter_start, iter_end );
            TextviewDiary::m_buffer->remove_tag(
                    TextbufferDiary::m_tag_highlight, iter_start, iter_end );
            break;
	}

	TextviewDiary::m_buffer->m_flag_ongoingoperation = false;
}

// TEXTBUFFERDIARY =================================================================================
TextbufferDiary::TextbufferDiary( void )
:	Gtk::TextBuffer(),
	m_flag_settextoperation( false ), m_flag_ongoingoperation( false ),
	m_flag_parsing( false ),
	m_ptr2textview( NULL ), m_ptr2spellobj( NULL )
{
	LinkUri::m_ptr2buffer = this;
	UndoEdit::m_ptr2buffer = this;
	Glib::RefPtr< TagTable > tag_table = get_tag_table();

	m_tag_heading = Tag::create( "heading" );
	m_tag_heading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_heading->property_scale() = 1.5;
	tag_table->add( m_tag_heading );

	m_tag_subheading = Tag::create( "subheading" );
	m_tag_subheading->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_subheading->property_scale() = 1.2;
	tag_table->add( m_tag_subheading );

	m_tag_match = Tag::create( "match" );
	tag_table->add( m_tag_match );

	m_tag_markup = Tag::create( "markup" );
	m_tag_markup->property_scale() = 0.5;
	tag_table->add( m_tag_markup );

	m_tag_markup_link = Tag::create( "markup.link" );
	tag_table->add( m_tag_markup_link );

	m_tag_hidden = Tag::create( "hidden" );
	m_tag_hidden->property_invisible() = true;
	tag_table->add( m_tag_hidden );

	m_tag_bold = Tag::create( "bold" );
	m_tag_bold->property_weight() = Pango::WEIGHT_BOLD;
	tag_table->add( m_tag_bold );

	m_tag_italic = Tag::create( "italic" );
	m_tag_italic->property_style() = Pango::STYLE_ITALIC;
	tag_table->add( m_tag_italic );

	m_tag_strikethrough = Tag::create( "strikethrough" );
	m_tag_strikethrough->property_strikethrough() = true;
	tag_table->add( m_tag_strikethrough );

	m_tag_highlight = Tag::create( "highlight" );
	tag_table->add( m_tag_highlight );

    m_tag_comment = Tag::create( "comment" );
    m_tag_comment->property_scale() = 0.8;
    m_tag_comment->property_rise() = 5000;
    tag_table->add( m_tag_comment );

    m_tag_region = Tag::create( "region" );
    tag_table->add( m_tag_region );

	m_tag_link = Tag::create( "link" );
	m_tag_link->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link );

	m_tag_link_broken = Tag::create( "link.broken" );
	m_tag_link_broken->property_underline() = Pango::UNDERLINE_SINGLE;
	tag_table->add( m_tag_link_broken );

	// this is just for keeping the boundaries:
	m_tag_link_hidden = Tag::create( "link.hidden" );
	tag_table->add( m_tag_link_hidden );

	m_tag_checkbox = Tag::create( "checkbox" );
	m_tag_checkbox->property_weight() = Pango::WEIGHT_BOLD;
	m_tag_checkbox->property_scale() = 1.2;
	tag_table->add( m_tag_checkbox );
}

void
TextbufferDiary::handle_login( void )
{
	// SPELL CHECKING
	set_spellcheck( Diary::d->get_spellcheck() );
}

void
TextbufferDiary::handle_logout( void )
{
	// reset search str
	EntryParser::set_search_str( "" );

	m_flag_settextoperation = true;

	set_spellcheck( false );

	m_flag_settextoperation = false;
}

void
TextbufferDiary::set_search_str( const Glib::ustring &str )
{
    EntryParser::set_search_str( str );
    reparse();
}

bool
TextbufferDiary::select_searchstr_previous( void )
{
    if( m_search_str.size() <= 0 )
        return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_start.is_start() )
		return false;

	iter_bound_end = iter_bound_start;
	
	if( ! iter_bound_start.backward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_start.backward_to_tag_toggle( m_tag_match );
	iter_bound_start.backward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_end.backward_search(	m_search_str,
											Gtk::TextSearchFlags( 0 ),
											iter_start,
											iter_end,
											iter_bound_start ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

bool
TextbufferDiary::select_searchstr_next( void )
{
    if( m_search_str.size() <= 0 )
        return false;

	Gtk::TextIter iter_start, iter_end, iter_bound_start, iter_bound_end;
	get_selection_bounds( iter_bound_start, iter_bound_end );

	if( iter_bound_end.is_end() )
		return false;

	iter_bound_start = iter_bound_end;

	if( ! iter_bound_end.forward_to_tag_toggle( m_tag_match ) )
		return false;
	iter_bound_end.forward_to_tag_toggle( m_tag_match );
	if( iter_bound_end.has_tag( m_tag_match ) )
		iter_bound_end.forward_to_tag_toggle( m_tag_match );

	if( ! iter_bound_start.forward_search(	m_search_str,
													Gtk::TextSearchFlags( 0 ),
													iter_start,
													iter_end,
													iter_bound_end ) )
		return false;

	select_range( iter_start, iter_end );
	m_ptr2textview->scroll_to( iter_start );
	return true;
}

void
TextbufferDiary::parse( unsigned int start, unsigned int end )
{
	m_flag_parsing = true;

	// BEGINNING & END
	Gtk::TextIter   iter_start( get_iter_at_offset( start ) );
	Gtk::TextIter	const iter_end( get_iter_at_offset( end ) );
	// remove all tags:
	remove_all_tags( iter_start, iter_end );
	clear_links( start, end );
	EntryParser::parse( start, end );

	m_flag_parsing = false;
}

gunichar
TextbufferDiary::get_char_at( int i )
{
    return get_iter_at_offset( i ).get_char();
}

// PARSING APPLIERS ================================================================================
void
TextbufferDiary::apply_heading( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( 0 ) );
    Gtk::TextIter iter_end( iter_start );
    if( ! iter_start.ends_line() )
        iter_end.forward_to_line_end();
	apply_tag( m_tag_heading, iter_start, iter_end );

	if( ! m_flag_settextoperation )
	{
	    // TODO: can be improved:
		m_ptr2entry->calculate_title( get_text() );
		Lifeobase::base->handle_entry_title_changed( m_ptr2entry );
	}
}

void
TextbufferDiary::apply_subheading( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_current( iter_start );
    iter_current.forward_to_line_end();
    apply_tag( m_tag_subheading, iter_start, iter_current );
}

void
TextbufferDiary::apply_bold( void )
{
	apply_markup( m_tag_bold );
}

void
TextbufferDiary::apply_italic( void )
{
	apply_markup( m_tag_italic );
}

void
TextbufferDiary::apply_strikethrough( void )
{
	apply_markup( m_tag_strikethrough );
}

void
TextbufferDiary::apply_highlight( void )
{
	apply_markup( m_tag_highlight );
}

void
TextbufferDiary::apply_comment( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current ) );
    iter_end++;
    apply_tag( m_tag_comment, iter_start, iter_end );
}

void
TextbufferDiary::apply_ignore( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    apply_tag( m_tag_region, iter_start, iter_end );
}

void
TextbufferDiary::apply_hidden_link_tags( Gtk::TextIter &iter_end,
                                         const Glib::RefPtr< Tag > &tag_link )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );

	apply_tag( m_tag_markup_link, iter_start, iter_tab );
	apply_tag( m_tag_markup_link, iter_current, iter_end );

	apply_tag( tag_link, iter_tab, iter_current );
	apply_tag( m_tag_link_hidden, iter_start, iter_end );

	// hide link markup if cursor is not in it:
	Gtk::TextIter iter_insert = get_iter_at_mark( get_insert() );
	if( iter_insert.get_offset() < (int) pos_start ||
		iter_insert.get_offset() > (int) pos_current )
	{
		apply_tag( m_tag_hidden, iter_start, iter_tab );
		apply_tag( m_tag_hidden, iter_current, iter_end );
		parser_open_tag_begin = std::string::npos;
	}
	else
	{
		parser_open_tag_begin = pos_start;
		parser_open_tag_end = iter_end.get_offset();
	}
}

void
TextbufferDiary::apply_link( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );

	if( m_flag_hidden_link )
	{
        Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );
        Gtk::TextIter iter_url_start( get_iter_at_offset( pos_start + 1 ) );
        Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );
        m_list_links.push_back( new LinkUri( create_mark( iter_tab ),
                                             create_mark( iter_current ),
                                             get_slice( iter_url_start, iter_tab ) ) );

		apply_hidden_link_tags( iter_end, m_tag_link );
	}
	else
	{
		PRINT_DEBUG( "url: " + word_last );
		m_list_links.push_back( new LinkUri( create_mark( iter_start ),
											 create_mark( iter_current ),
											 word_last ) );

		apply_tag( m_tag_link, iter_start, iter_current );
	}
}

void
TextbufferDiary::apply_link_id( void )
{
	DiaryElement *element( Diary::d->get_element( id_last ) );

    if( element != NULL )
    {
        if( element->get_type() == DiaryElement::IT_ENTRY )
        {
            Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );
            Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
            Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );

            m_list_links.push_back( new LinkID( create_mark( iter_tab ),
                                                create_mark( iter_current ),
                                                id_last ) );

            apply_hidden_link_tags( iter_end, m_tag_link );
            return;
        }
    }
    // indicate dead links
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current + 1 ) );
    apply_tag( m_tag_link_broken, iter_start, iter_end );
}

void
TextbufferDiary::apply_link_date( void )
{
    // FIXME: date should be ready before arriving here
	date_last.set_day( int_last );

	if( date_last.is_valid() )
	{
		LinkStatus status( LS_OK );
		Entry *ptr2entry( Diary::d->get_entry( date_last.m_date + 1 ) ); // + 1 fixes order
		if( ptr2entry == NULL )
			status = LS_ENTRY_UNAVAILABLE;
		else
		if( date_last.get_pure() == m_ptr2entry->get_date().get_pure() )
			status = Diary::d->get_day_has_multiple_entries( date_last ) ? LS_OK : LS_CYCLIC;

        Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
        Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
        Gtk::TextIter iter_tab( get_iter_at_offset( pos_tab ) );

        if( status < LS_INVALID )
        {
            Gtk::TextIter iter_end( iter_current );
            iter_end++;
            // if hidden link:
            if( char_current == '>' )
            {
                m_list_links.push_back( new LinkEntry( create_mark( iter_tab ),
                                                       create_mark( iter_current ),
                                                       date_last ) );
                apply_hidden_link_tags( iter_end,
                        status == LS_OK ? m_tag_link : m_tag_link_broken );
			}
			else
			{
				m_list_links.push_back(
						new LinkEntry( create_mark( iter_start ),
                                       create_mark( iter_end ),
                                       date_last ) );

				apply_tag( status == LS_OK ? m_tag_link :
								m_tag_link_broken, iter_start, iter_end );
			}
		}
	}
}

void
TextbufferDiary::apply_check_unf( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current - 1 ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current ) );
    m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                           create_mark( iter_end ),
                                           0 ) );
	apply_tag( m_tag_checkbox, iter_start, iter_end );
}

void
TextbufferDiary::apply_check_fin( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current -1 ) );
    Gtk::TextIter iter_box( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                           create_mark( iter_box ),
                                           1 ) );
    apply_tag( m_tag_checkbox, iter_start, iter_box );
    apply_tag( m_tag_highlight, iter_box, iter_end );
}

void
TextbufferDiary::apply_check_ccl( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_current - 1 ) );
    Gtk::TextIter iter_box( get_iter_at_offset( pos_current ) );
    Gtk::TextIter iter_end( iter_start );
    iter_end.forward_to_line_end();
    m_list_links.push_back( new LinkCheck( create_mark( iter_start ),
                                           create_mark( iter_box ),
                                           2 ) );
    apply_tag( m_tag_checkbox, iter_start, iter_box );
    apply_tag( m_tag_strikethrough, iter_box, iter_end );
}

void
TextbufferDiary::apply_match( void )
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_search ) );
    Gtk::TextIter iter_end( get_iter_at_offset( pos_current+1 ) );
    apply_tag( m_tag_match, iter_start, iter_end );
}

//#define CALL_MEMBER_FN(object,ptrToMember)  ((object).*(ptrToMember))

inline void
TextbufferDiary::apply_markup( const Glib::RefPtr< Tag > &tag)
{
    Gtk::TextIter iter_start( get_iter_at_offset( pos_start ) );
    Gtk::TextIter iter_current( get_iter_at_offset( pos_current ) );
	Gtk::TextIter iter2( iter_start );
	iter2++;
	apply_tag( m_tag_markup, iter_start, iter2 );
	apply_tag( m_tag_hidden, iter_start, iter2 );
	apply_tag( tag, iter2, iter_current );
	iter2 = iter_current;
	iter2++;
	apply_tag( m_tag_markup, iter_current, iter2 );
	apply_tag( m_tag_hidden, iter_current, iter2 );
}

void
TextbufferDiary::set_richtext( Entry *entry )
{
	m_flag_settextoperation = true;
	clear_links();
	parser_open_tag_begin = std::string::npos;
	parser_open_tag_end = std::string::npos;
	parser_offset_insert_prev = std::string::npos;
	UndoManager::m->clear();
	UndoManager::m->freeze();
	m_ptr2entry = entry;
	set_theme( entry->get_theme_is_set() ?
			entry->get_theme() : Diary::d->get_default_theme() );
	Gtk::TextBuffer::set_text( entry->get_text() );
	place_cursor( begin() );
	UndoManager::m->thaw();
	m_flag_settextoperation = false;
}

void
TextbufferDiary::reparse( void )
{
	place_cursor( begin() );
	m_flag_settextoperation = true;
	parse( 0, get_char_count() );
	m_flag_settextoperation = false;
}

void
TextbufferDiary::on_insert( const Gtk::TextIter& iterator,
							const Glib::ustring& text,
							int bytes )
{
	const Glib::ustring::size_type offset_itr = iterator.get_offset();

	// UNDO
	if( ! UndoManager::m->is_freezed() )
	{
		UndoInsert * undo_insert = new UndoInsert( offset_itr, text );
		UndoManager::m->add_action( undo_insert );
	}
	else
	{
		Gtk::TextIter iter_scroll( iterator );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_insert( iterator, text, text.bytes() );

	if( m_flag_ongoingoperation )
		return;

	// PARSING LIMITS
	pos_start = 0;
	pos_end = get_char_count();
	if( m_flag_settextoperation == false )
	{
		if( offset_itr > 0 )
		{
			pos_start = get_text().rfind( '\n', offset_itr - 1 );
			if( pos_start == std::string::npos )
				pos_start = 0;
		}

		if( offset_itr < pos_end )
		{
			pos_end = get_text().find( '\n', offset_itr + 1 );
			if( pos_end == std::string::npos )
				pos_end = get_char_count();
		}
	}

	parse( pos_start, pos_end );
}

void
TextbufferDiary::process_space( void )
{
	Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter_end.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == std::string::npos )
		return;
	++pos_list;	// get rid of the new line char

	Gtk::TextIter iter_start = get_iter_at_offset( pos_list );
	Glib::ustring line = get_text( iter_start, iter_end );
	char char_lf = '\t';
	unsigned int size = pos_cursor - pos_list;

	for( unsigned int i = 0; i < size; i++ )
	{
		switch( line[ i ] )
		{
			case '\t':
				if( char_lf == '\t' )
				    char_lf = 'A';	// any list char like [ or *
				else
				if( char_lf != 'A' )	// multiple tabs are possible (indentation)
					return;
				pos_list++;	// indentation level
				break;
			case '[':
				if( char_lf != 'A' )
					return;
				char_lf = ']';
				break;
			case ']':
				if( char_lf != ']' || i != ( size - 1 ) )
					return;
				//char_lf = 'C';	// checkbox
				PRINT_DEBUG( "time to insert a checkbox" );
				iter_start = get_iter_at_offset( pos_list );	// get rid of the tab chars
				m_flag_ongoingoperation = true;
				erase( iter_start, iter_end );
				m_flag_ongoingoperation = false;
				iter_start = get_iter_at_offset( pos_list );	// refresh the iterator
				insert( iter_start, "☐" );
				break;
			case '*':
				if( char_lf != 'A' || i != ( size - 1 ) )
					return;
				iter_start = get_iter_at_offset( pos_list );	// get rid of the tab chars
				m_flag_ongoingoperation = true;
				erase( iter_start, iter_end );
				m_flag_ongoingoperation = false;
				iter_start = get_iter_at_offset( pos_list );	// refresh the iterator
				insert( iter_start, "•" );
				break;
			default:
				return;
		}
	}
}

bool
TextbufferDiary::process_new_line( void )
{
	Gtk::TextIter iter_end = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter_end.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list != std::string::npos &&
		pos_list > 0 &&					// previous line should not be the first
		pos_cursor - pos_list > 3 )		// previous line should be at least 3 chars long
	{
		++pos_list;	// get rid of the new line char
		Gtk::TextIter iter_start = get_iter_at_offset( pos_list );
		Glib::ustring line = get_text( iter_start, iter_end );

		if( line[ 0 ] == '\t' )
		{
			Glib::ustring text( "\n\t" );
			int value = 0;
			char char_lf = '*';

			for( unsigned int i = 1; i < pos_cursor - pos_list; i++ )
			{
				switch( line[ i ] )
				{
					// BULLETED LIST
					case L'•':
						if( char_lf != '*' )
							return false;
						char_lf = ' ';
						text += "• ";
						break;
					// CHECK LIST
					case L'☐':
					case L'☑':
					case L'☒':
						if( char_lf != '*' )
							return false;
						char_lf = ' ';
						text += "☐ ";
						break;
					// NUMBERED LIST
					case '0': case '1': case '2': case '3': case '4':
					case '5': case '6': case '7': case '8': case '9':
						if( char_lf != '*' && char_lf != '1' )
							return false;
						char_lf = '1';
						value *= 10;
						value += line[ i ] - '0';
						break;
					case '-':
					case '.':
					case ')':
						if( char_lf != '1' )
							return false;
						char_lf = ' ';
						text += Glib::ustring::compose( "%1%2 ", ++value, line.substr( i, 1 ) );
						break;
					case '\t':
						if( char_lf != '*' )
							return false;
						text += '\t';
						break;
					case ' ':
						if( char_lf != ' ' )
							return false;
						if( i == pos_cursor - pos_list - 1 )
						{
							erase( iter_start, iter_end );
							iter_start = get_iter_at_offset( pos_list );
							insert( iter_start, "\n" );
							return true;
						}
						else
						{
							insert( iter_end, text );
							return true;
						}
						break;
					default:
						return false;
				}
			}
		}
	}
	return false;
}

void
TextbufferDiary::on_erase(	const Gtk::TextIter& iter_start,
							const Gtk::TextIter& iter_end )
{
	if( ! UndoManager::m->is_freezed() )
	{
		UndoErase * undo_erase = new UndoErase(
				iter_start.get_offset(), get_slice( iter_start, iter_end ) );
		UndoManager::m->add_action( undo_erase );
	}
	else
	{
		Gtk::TextIter iter_scroll( iter_start );	// to remove constness
		m_ptr2textview->scroll_to( iter_scroll );
	}

	Gtk::TextBuffer::on_erase( iter_start, iter_end );

	// set_text() calls on_erase too:
	if( m_flag_settextoperation == false && m_flag_ongoingoperation == false )
	{
		pos_start = iter_start.get_offset();
		pos_end = iter_end.get_offset();
        parser_open_tag_begin = parser_open_tag_end = std::string::npos;

		if( pos_start > 0 )
		{
			pos_start = get_text().rfind( '\n', pos_start - 1 );
			if( pos_start == std::string::npos )
				pos_start = 0;
		}

		pos_end = get_text().find( '\n', pos_end );
		if( pos_end == std::string::npos )
			pos_end = get_char_count();

		parse( pos_start, pos_end );
	}
}

void
TextbufferDiary::on_apply_tag( const Glib::RefPtr< TextBuffer::Tag >& tag,
							   const Gtk::TextIter& iter_begin,
							   const Gtk::TextIter& iter_end )
{
	// do not check spelling of links:
	if( iter_begin.has_tag( m_tag_link ) && tag->property_name() == "gtkspell-misspelled" )
		return;
	else
		Gtk::TextBuffer::on_apply_tag( tag, iter_begin, iter_end );
}

void
TextbufferDiary::on_remove_tag(	const Glib::RefPtr< TextBuffer::Tag >& tag,
								const Gtk::TextIter& iter_begin,
								const Gtk::TextIter& iter_end )
{
	// do not remove gtkspell tags while parsing:
	if( m_flag_parsing && tag->property_name() == "gtkspell-misspelled" )
		return;
	else
		Gtk::TextBuffer::on_remove_tag( tag, iter_begin, iter_end );
}

void
TextbufferDiary::on_mark_set( const TextBuffer::iterator &iter,
							  const Glib::RefPtr< TextBuffer::Mark > &mark )
{
	Gtk::TextBuffer::on_mark_set( iter, mark );

	if( mark == get_insert() )
	{
		PRINT_DEBUG( "on_mark_set()" );
		bool flag_within_open_tag( false );
		if( parser_open_tag_begin != std::string::npos )
		{
			if( iter.get_offset() >= int( parser_open_tag_begin ) &&
				iter.get_offset() <= int( parser_open_tag_end ) )
				flag_within_open_tag = true;
			else
			{
				if( parser_open_tag_is_link )
				{
					PRINT_DEBUG( "OPEN TAG IS LINK" );
					parse( parser_open_tag_begin, parser_open_tag_end );
					parser_open_tag_begin = parser_open_tag_end = std::string::npos;
				}
				else
				{
					apply_tag( m_tag_hidden,
							   get_iter_at_offset( parser_open_tag_begin ),
							   get_iter_at_offset( parser_open_tag_end ) );
					parser_open_tag_begin = parser_open_tag_end = std::string::npos;
				}
			}
		}
		if( ! flag_within_open_tag )	// not within an already open tag
		{
			PRINT_DEBUG( "NOT WITHIN OPEN TAG" );
			if( iter.has_tag( m_tag_link_hidden ) )
			{
				PRINT_DEBUG( "WITHIN HIDDEN LINK" );
				Gtk::TextIter begin( iter );
				Gtk::TextIter end( iter );
				if( ! iter.begins_tag( m_tag_link_hidden ) )
				{
					PRINT_DEBUG( "BEGINS TAG" );
					begin.backward_to_tag_toggle( m_tag_link_hidden );
				}
				end.forward_to_tag_toggle( m_tag_link_hidden );
				parser_open_tag_begin = begin.get_offset();
				parser_open_tag_end = end.get_offset();
				parser_open_tag_is_link = true;
				parse( parser_open_tag_begin, parser_open_tag_end );
			}
			// STARTING MARKUP
			else
			if( iter.begins_tag( m_tag_markup ) )
			{
				parser_open_tag_begin = iter.get_offset();
				parser_open_tag_end = parser_open_tag_begin + 1;
				parser_open_tag_is_link = false;
				Gtk::TextIter begin( get_iter_at_offset( parser_open_tag_begin ) );
				Gtk::TextIter end( get_iter_at_offset( parser_open_tag_end ) );
				PRINT_DEBUG( "BEGINS MARKUP |" + get_slice(begin, end) + "|" );

				remove_tag( m_tag_hidden, begin, end );

				if( parser_offset_insert_prev != parser_open_tag_begin - 1 )
				{
					move_mark( get_selection_bound(), end );
					move_mark( mark, end );
				}
			}
			// FINISHING MARKUP
			else
			if( iter.ends_tag( m_tag_bold ) || iter.ends_tag( m_tag_italic ) ||
				iter.ends_tag( m_tag_highlight ) || iter.ends_tag( m_tag_strikethrough ) )
			{
				parser_open_tag_begin = iter.get_offset();
				parser_open_tag_end = parser_open_tag_begin + 1;
				parser_open_tag_is_link = false;
				Gtk::TextIter begin( get_iter_at_offset( parser_open_tag_begin ) );
				Gtk::TextIter end( get_iter_at_offset( parser_open_tag_end ) );
				PRINT_DEBUG( "ENDS MARKUP |" + get_slice(begin, end) + "|" );

				remove_tag( m_tag_hidden, begin, end );

				if( parser_offset_insert_prev == parser_open_tag_begin + 2 )
				{
					move_mark( get_selection_bound(), end );
					move_mark( mark, end );
				}
			}
		}

		parser_offset_insert_prev = iter.get_offset();
	}
}

Link*
TextbufferDiary::get_link( int offset ) const
{
	for ( ListLinks::const_iterator iter = m_list_links.begin();
		  iter != m_list_links.end();
		  ++iter )
	{
		if(	offset >= ( *iter )->m_mark_start->get_iter().get_offset() &&
			offset <= ( *iter )->m_mark_end->get_iter().get_offset() )
			return( *iter );
	}

	return NULL;
}

Link*
TextbufferDiary::get_link( const Gtk::TextIter &iter ) const
{
	for( ListLinks::const_iterator link = m_list_links.begin();
		  link != m_list_links.end();
		  ++link )
	{
		if( iter.in_range( ( *link )->m_mark_start->get_iter(),
						   ( *link )->m_mark_end->get_iter() ) )
			return( *link );
	}
	return NULL;
}

void
TextbufferDiary::clear_links( int pos_start, int pos_end )
{
	ListLinks::iterator iter_tmp;
	for( ListLinks::iterator iter = m_list_links.begin(); iter != m_list_links.end(); )
		if(	pos_start <= ( *iter )->m_mark_start->get_iter().get_offset() &&
			pos_end >= ( *iter )->m_mark_end->get_iter().get_offset() )
		{
			iter_tmp = iter;
			++iter;
			delete( *iter_tmp );
			m_list_links.erase( iter_tmp );
		}
		else
			++iter;
}

void
TextbufferDiary::clear_links( void )
{
	for( ListLinks::iterator iter = m_list_links.begin();
		 iter != m_list_links.end();
		 ++iter )
	{
		delete( *iter );
	}
	m_list_links.clear();
}

void
TextbufferDiary::handle_menu( Gtk::Menu *menu )
{
	Gtk::CheckMenuItem *menuitem_spell = Gtk::manage(
			new Gtk::CheckMenuItem( _( "Check Spelling" ) ) );

	Gtk::SeparatorMenuItem *separator = Gtk::manage( new Gtk::SeparatorMenuItem );

	menuitem_spell->set_active( Diary::d->get_spellcheck() );

	menuitem_spell->signal_toggled().connect(
			sigc::bind(	sigc::mem_fun( *this, &TextbufferDiary::set_spellcheck ),
						! Diary::d->get_spellcheck() ) );

	menuitem_spell->show();
	separator->show();

	menu->prepend( *separator );
	menu->prepend( *menuitem_spell );
}

void
TextbufferDiary::toggle_format(	Glib::RefPtr< Tag > tag, const Glib::ustring &markup )
{
	Gtk::TextIter iter_start, iter_end;
	if( get_has_selection() )
	{
        int pos_start( -2 ), pos_end( -1 );
        bool properly_separated( false );

        get_selection_bounds( iter_start, iter_end );
        iter_end--;

        if( ! iter_start.is_start() )
        {
            --iter_start;   // also evaluate the previous character
        }
        else
        {
            properly_separated = true;
            pos_start = -2;
        }


        for( ; ; ++iter_start )
        {
            if( iter_start.has_tag( tag ) )
                return;
            switch( iter_start.get_char() )
            {
                // do nothing if selection spreads over more than one line:
                case '\n':
                    if( pos_start > -2 )
                        return;
                    /* else no break */
				case ' ':
				case '\t':
				    if( pos_start == -2 )
				    {
				        properly_separated = true;
				        pos_start = -1;
				    }
					break;
//				case '*':
//				case '_':
//				case '#':
//				case '=':
//					if( iter_start.get_char() == markup[ 0 ] )
//						break;
					/* else no break */
				default:
                    if( pos_start == -2 )
                        pos_start = -1;
                    else
					if( pos_start == -1 )
						pos_start = iter_start.get_offset();
					pos_end = iter_start.get_offset();
					break;
			}
			if( iter_start == iter_end )
				break;
		}
		// add markup chars to the beginning and end:
		if( pos_start >= 0 )
		{
            if( properly_separated )
            {
                insert( get_iter_at_offset( pos_start ), markup );
                pos_end += 2;
            }
            else
            {
                insert( get_iter_at_offset( pos_start ), " " + markup );
                pos_end += 3;
            }

			insert( get_iter_at_offset( pos_end ), markup );
			place_cursor( get_iter_at_offset( pos_end ) );
		}
	}
	else	// no selection case
	{
		Glib::RefPtr< Gtk::TextMark > mark = get_insert();
		iter_start = mark->get_iter();
		if( Glib::Unicode::isspace( iter_start.get_char() ) ||
			iter_start.has_tag( TextbufferDiary::m_tag_markup ) )
		{
			if( ! iter_start.starts_line() )
				iter_start--;
			else
				return;
		}
		if( iter_start.has_tag( tag ) )
		{
			m_flag_ongoingoperation = true;

			// find start and end points of formatting:
			if( iter_start.starts_word() )
				iter_start++;	// necessary when cursor is between a space char
								// and non-space char
			iter_start.backward_to_tag_toggle( tag );
			backspace( iter_start );

			iter_end = mark->get_iter();
			if( iter_end.has_tag( tag ) )
				iter_end.forward_to_tag_toggle( TextbufferDiary::m_tag_markup );

			m_flag_ongoingoperation = false;

			backspace( ++iter_end );
		}
		else
		if( iter_start.get_tags().empty() )		// nested tags are not supported atm
		{
			// find word boundaries:
			if( !( iter_start.starts_word() || iter_start.starts_line() ) )
				iter_start.backward_word_start();
			insert( iter_start, markup );

			iter_end = mark->get_iter();
			if( !( iter_end.ends_word() || iter_end.ends_line() ) )
			{
				iter_end.forward_word_end();
				insert( iter_end, markup );
			}
			else
			{
				int offset = iter_end.get_offset();
				insert( iter_end, markup );
				place_cursor( get_iter_at_offset( offset ) );

			}
		}
	}
}

void
TextbufferDiary::toggle_bold( void )
{
	toggle_format( m_tag_bold, "*" );
}

void
TextbufferDiary::toggle_italic( void )
{
	toggle_format( m_tag_italic, "_" );
}

void
TextbufferDiary::toggle_strikethrough( void )
{
	toggle_format( m_tag_strikethrough, "=" );
}

void
TextbufferDiary::toggle_highlight( void )
{
	toggle_format( m_tag_highlight, "#" );
}

void
TextbufferDiary::show_comments( bool visible )
{
    m_tag_comment->property_invisible() = ! visible;
}

void
TextbufferDiary::handle_indent( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == std::string::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	insert( iter, "\t" );
}

void
TextbufferDiary::handle_unindent( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == std::string::npos || ( pos_cursor - pos_list ) < 2 )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	if( iter.get_char() != '\t' )
		return;
	Gtk::TextIter iter_end = get_iter_at_offset( pos_list + 1 );
	erase( iter, iter_end );
}

void
TextbufferDiary::add_bullet( void )
{
	Gtk::TextIter iter = get_iter_at_mark( get_insert() );
	if( iter.get_line() < 1 )	// no list in the first line
		return;
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == std::string::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	Gtk::TextIter iter_erase_begin( iter );
	char char_lf( 't' );	// tab
	while( true )
	{
		switch( iter.get_char() )
		{
			case L'•':	// remove bullet
				if( char_lf == 'b' )
					char_lf = 's';	// space
				else
					return;
				break;
			case ' ':
				if( char_lf == 's' )
					erase( iter_erase_begin, ++iter );
				return;
			case L'☐':
			case L'☑':
			case L'☒':
				// TODO: convert to bullet
				return;
			case '\t':
				char_lf = 'b';	// bullet
				iter_erase_begin = iter;
				break;
			case 0:	// end
			default:
				if( char_lf != 's' )
					insert( iter, "\t• " );
				return;
		}
		++iter;
	}
}

void
TextbufferDiary::add_checkbox( void )
{
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.get_line() < 1 )	// no list in the first line
		return;
	const Glib::ustring::size_type	pos_cursor = iter.get_offset();
	Glib::ustring::size_type		pos_list = get_text().rfind( '\n', pos_cursor - 1 );
	if( pos_list == std::string::npos )
		return;
	++pos_list;	// get rid of the new line char

	iter = get_iter_at_offset( pos_list );
	Gtk::TextIter iter_erase_begin( iter );
	char char_lf( 't' );	// tab
	while( true )
	{
		switch( iter.get_char() )
		{
			case L'☐':
			case L'☑':
			case L'☒':	// remove checkbox
				if( char_lf == 'c' )
					char_lf = 's';	// space
				else
					return;
				break;
			case ' ':
				if( char_lf == 's' )
					erase( iter_erase_begin, ++iter );
				return;
			case L'•':
				// TODO: convert to checkbox
				return;
			case '\t':
				char_lf = 'c';	// checkbox
				iter_erase_begin = iter;
				break;
			case 0:	// end
			default:
				if( char_lf != 's' )
					insert( iter, "\t☐ " );
				return;
		}
		++iter;
	}
}

void
TextbufferDiary::add_empty_line_above( void )
{
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.backward_line() )
		iter.forward_line();
	insert( iter, "\n" );
}

void
TextbufferDiary::remove_empty_line_above( void )
{
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.backward_line() )
		iter.forward_line();

	if( iter.get_line() < 1 )
		return;

	Gtk::TextIter iter_begin( --iter );
	iter_begin--;
	if( iter_begin.get_char() == '\n' )
		erase( iter_begin, iter );
}

void
TextbufferDiary::insert_link( DiaryElement *element )
{
	// TODO: implement a custom insert function that adds spaces where necessary
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	if( iter.get_tags().size() > 0 )
		return;
	if( iter.get_offset() > 0 )
		iter--;
	char c( iter.get_char() );
	if( c != ' ' && c != '\n' && c != '\t' )
		insert( get_iter_at_mark( get_insert() ), " ");
	insert( get_iter_at_mark( get_insert() ),
				Glib::ustring::compose( "<deid:%1\t%2>", element->get_id(), element->get_name() ) );
}

void
TextbufferDiary::insert_time_stamp( void )
{
	// TODO: implement a custom insert function that adds spaces where necessary
	Gtk::TextIter iter( get_iter_at_mark( get_insert() ) );
	//if( iter.get_tags().size() > 0 )
		//return;
	if( iter.get_offset() > 0 )
		iter--;
	char c( iter.get_char() );
	if( c != ' ' && c != '\n' && c != '\t' )
		insert( get_iter_at_mark( get_insert() ), " ");

	time_t date( time( NULL ) );
	insert( get_iter_at_mark( get_insert() ), Date::format_string( &date ) );
}

void
TextbufferDiary::set_spellcheck( bool option_spell )
{
	if( option_spell )
	{
		if( ! m_ptr2spellobj )
			m_ptr2spellobj = gtkspell_new_attach( m_ptr2textview->gobj(), NULL, NULL );
	}
	else
	if( m_ptr2spellobj )
	{
		gtkspell_detach( m_ptr2spellobj );
		m_ptr2spellobj = NULL;
	}

	Diary::d->set_spellcheck( option_spell );
}

void
TextbufferDiary::set_theme( const Theme *theme )
{
	m_ptr2textview->modify_font( theme->font );
	m_ptr2textview->modify_base( Gtk::STATE_NORMAL, theme->color_base );
	m_ptr2textview->modify_text( Gtk::STATE_NORMAL, theme->color_text );
	m_tag_heading->property_foreground_gdk() = theme->color_heading;
	m_tag_subheading->property_foreground_gdk() = theme->color_subheading;
	m_tag_highlight->property_background_gdk() = theme->color_highlight;

	Gdk::Color color_mid( midtone( theme->color_base, theme->color_text ) );

    m_tag_comment->property_foreground_gdk() = color_mid;
    m_tag_region->property_paragraph_background_gdk() = midtone(
            theme->color_base, theme->color_text, 0.9 );
	m_tag_match->property_background_gdk() = contrast(
			theme->color_base, m_theme_color_match );
	m_tag_markup->property_foreground_gdk() = color_mid;
	m_tag_markup_link->property_foreground_gdk() = color_mid;
	m_tag_link->property_foreground_gdk() = contrast(
			theme->color_base, m_theme_color_link );
	m_tag_link_broken->property_foreground_gdk() = contrast(
			theme->color_base, m_theme_color_linkbroken );
}

// TEXTVIEW ========================================================================================
// STATIC MEMBERS
TextbufferDiary			*TextviewDiary::m_buffer;

TextviewDiary::TextviewDiary( BaseObjectType *cobject, const Glib::RefPtr<Gtk::Builder>& )
:	Gtk::TextView( cobject ),
	m_cursor_hand( Gdk::HAND2 ), m_cursor_xterm( Gdk::XTERM ),
	m_ptr2cursor_last( &m_cursor_xterm ), m_link_hovered( NULL )
{
	set_buffer( static_cast< Glib::RefPtr< TextbufferDiary > >( m_buffer ) );
	m_buffer->set_textview( this );
	set_wrap_mode( Gtk::WRAP_WORD );
	set_left_margin( 10 );

	signal_populate_popup().connect(
			sigc::mem_fun( *m_buffer, &TextbufferDiary::handle_menu ) );
}

inline void
TextviewDiary::update_link( void )
{
	Gtk::TextIter		iter;
	const Gdk::Cursor	*ptr2cursor = &m_cursor_xterm;
	int					pointer_x, pointer_y;
	int					trailing, buffer_x, buffer_y;
    Gdk::ModifierType	modifiers;

    Gtk::Widget::get_window()->get_pointer( pointer_x, pointer_y, modifiers );
	window_to_buffer_coords( Gtk::TEXT_WINDOW_WIDGET,
							 pointer_x, pointer_y,
							 buffer_x, buffer_y );
	get_iter_at_position( iter, trailing, buffer_x, buffer_y );

	// FIX ITER IF NEEDED:
	if( iter.has_tag( m_buffer->m_tag_hidden ) )
		iter.forward_to_tag_toggle( m_buffer->m_tag_hidden );
	Gtk::TextIter iter2( iter );
	iter2++;
	if( ( iter2.ends_tag( m_buffer->m_tag_link ) ||
		  iter2.ends_tag( m_buffer->m_tag_link_broken ) ) && trailing > 0 )
		iter++;

	m_link_hovered = m_buffer->get_link( iter );

	if( m_link_hovered != NULL )
	{
		if( !( modifiers & Gdk::CONTROL_MASK ) )
			ptr2cursor = &m_cursor_hand;
	}

	if( ptr2cursor != m_ptr2cursor_last )
	{
		m_ptr2cursor_last = ptr2cursor;
		get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( *ptr2cursor );
	}
}

void
TextviewDiary::set_richtext( Entry *entry )
{
    m_link_hovered = NULL;
    m_buffer->set_richtext( entry );
}

bool
TextviewDiary::on_motion_notify_event( GdkEventMotion *event )
{
	update_link();
	return Gtk::TextView::on_motion_notify_event( event );
}

bool
TextviewDiary::on_button_press_event( GdkEventButton *event )
{
	if( m_link_hovered != NULL )
		if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK )
			return true;

	return Gtk::TextView::on_button_press_event( event );
}

bool
TextviewDiary::on_button_release_event( GdkEventButton *event )
{
	if( m_link_hovered != NULL )
		if( ( event->state & Gdk::CONTROL_MASK ) != Gdk::CONTROL_MASK )
		{
			m_link_hovered->go();
			return true;
		}

	return Gtk::TextView::on_button_release_event( event );
}

bool
TextviewDiary::on_key_press_event( GdkEventKey *event )
{
	if( ( event->state & Gdk::CONTROL_MASK ) == Gdk::CONTROL_MASK )
	{
		switch( event->keyval )
		{
			case GDK_b:
			case GDK_B:
				m_buffer->toggle_bold();
				break;
			case GDK_i:
			case GDK_I:
				m_buffer->toggle_italic();
				break;
			case GDK_s:
			case GDK_S:
				m_buffer->toggle_strikethrough();
				break;
			case GDK_h:
			case GDK_H:
				m_buffer->toggle_highlight();
				break;
			case GDK_t:
			case GDK_T:
				Lifeobase::base->focus_tag();
				break;
		}
	}
	else
	if( ( event->state & Gdk::MOD1_MASK )/* == Gdk::MOD1_MASK */)
	{
		PRINT_DEBUG( "mod1" );
		switch( event->keyval )
		{
            case GDK_c:
            case GDK_C:
                Lifeobase::base->toggle_comments();
                break;
			case GDK_i:
			case GDK_I:
				m_buffer->handle_indent();
				break;
			case GDK_u:
			case GDK_U:
				m_buffer->handle_unindent();
				break;
			case GDK_l:
			case GDK_L:
				if( event->state & Gdk::SHIFT_MASK )
					m_buffer->remove_empty_line_above();
				else
					m_buffer->add_empty_line_above();
				break;
			case GDK_t:
			case GDK_T:
				m_buffer->insert_time_stamp();
				break;
		}
	}
	else
	if( event->state == 0 )
	{
		switch( event->keyval )
		{
			case GDK_space:
				m_buffer->process_space();
				break;
			case GDK_Return:
				if( m_buffer->process_new_line() )
					return true;
				break;
			case GDK_Control_L:
			case GDK_Control_R:
				if( m_link_hovered )
					update_link();
				break;
		}
	}

	return Gtk::TextView::on_key_press_event( event );
}

//void
//TextviewDiary::on_style_changed( const Glib::RefPtr< Gtk::Style > &style_prev )
//{
//	Gtk::TextView::on_style_changed( style_prev );
////	TextbufferDiary = get_pango_context()->get_font_description();
//	TextbufferDiary::m_theme_font = get_style()->get_font();
//}

bool
TextviewDiary::on_key_release_event( GdkEventKey *event )
{
	if( event->keyval == GDK_Control_L || event->keyval == GDK_Control_R )
		if( m_link_hovered )
			update_link();
		
	return Gtk::TextView::on_key_release_event( event );
}
