/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeograph.hpp"
#include "app_window.hpp"
#include "diary.hpp"
#include "views.hpp"
#include "panel_extra.hpp"


using namespace LIFEO;


// TAG =============================================================================================
TagView::TagView()
: m_menu_themes( NULL )
{
    try
    {
        Lifeograph::builder->get_widget( "Bx_tag", m_vbox );
        Lifeograph::builder->get_widget( "G_tag_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "B_tag_dismiss", m_button_dismiss );
        Lifeograph::builder->get_widget( "frame_tag_edit", m_frame_edit );
        Lifeograph::builder->get_widget( "TB_tag_filter", m_togglebutton_filter );
        Lifeograph::builder->get_widget( "combobox_tag_category", m_combobox_category );
        Lifeograph::builder->get_widget_derived( "drawingarea_tag_chart", m_widgetchart );
        Lifeograph::builder->get_widget( "textview_tag_theme", m_textview_theme );
        Lifeograph::builder->get_widget( "menubutton_tag_themes", m_button_themes );
        Lifeograph::builder->get_widget( "button_tag_theme_reset", m_button_theme_reset );

        m_liststore_categories = Gtk::ListStore::create( colrec_category );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the tag view" );
    }

    Tag::shower = this;

    m_combobox_category->set_model( m_liststore_categories );
    m_combobox_category->pack_start( colrec_category.name );
    m_combobox_category->set_row_separator_func(
            sigc::mem_fun( this, &TagView::is_row_separator ) );

    // SIGNALS
    m_togglebutton_filter->signal_toggled().connect(
            sigc::mem_fun( this, &TagView::handle_filter_toggled ) );
    m_button_dismiss->signal_clicked().connect(
            sigc::mem_fun( this, &TagView::dismiss_tag ) );

    m_combobox_category->signal_changed().connect(
            sigc::mem_fun( this, &TagView::handle_category_changed ) );

    m_button_theme_reset->signal_clicked().connect(
            sigc::mem_fun( this, &TagView::reset_theme ) );

    // THEME TEXT TAGS
    Glib::RefPtr< Gtk::TextBuffer::TagTable > tag_table =
            m_textview_theme->get_buffer()->get_tag_table();

    m_tag_action = Gtk::TextTag::create();
    tag_table->add( m_tag_action );

    m_tag_heading = Gtk::TextTag::create();
    m_tag_heading->property_weight() = Pango::WEIGHT_BOLD;
    m_tag_heading->property_scale() = 1.5;
    tag_table->add( m_tag_heading );

    m_tag_subheading = Gtk::TextTag::create();
    m_tag_subheading->property_weight() = Pango::WEIGHT_BOLD;
    m_tag_subheading->property_scale() = 1.2;
    tag_table->add( m_tag_subheading );

    m_tag_normal = Gtk::TextTag::create();
    m_tag_normal->property_underline() = Pango::UNDERLINE_ERROR;
    tag_table->add( m_tag_normal );

    m_tag_background = Gtk::TextTag::create();
    m_tag_background->property_weight() = Pango::WEIGHT_BOLD;
    m_tag_background->property_justification() = Gtk::JUSTIFY_RIGHT;
    tag_table->add( m_tag_background );

    m_tag_match = Gtk::TextTag::create();
    tag_table->add( m_tag_match );

    m_tag_highlight = Gtk::TextTag::create();
    tag_table->add( m_tag_highlight );

    m_tag_comment = Gtk::TextTag::create();
    m_tag_comment->property_scale() = 0.8;
    m_tag_comment->property_rise() = 5000;
    tag_table->add( m_tag_comment );

    m_tag_link = Gtk::TextTag::create();
    m_tag_link->property_underline() = Pango::UNDERLINE_SINGLE;
    tag_table->add( m_tag_link );

    m_tag_link_broken = Gtk::TextTag::create();
    m_tag_link_broken->property_underline() = Pango::UNDERLINE_SINGLE;
    tag_table->add( m_tag_link_broken );

    m_tag_region = Gtk::TextTag::create();
    tag_table->add( m_tag_region );

    m_tag_font = Gtk::TextTag::create();
    tag_table->add( m_tag_font );

    m_textview_theme->signal_motion_notify_event().connect_notify(
            sigc::mem_fun( this, &TagView::handle_theme_motion_notify_event ) );
    m_textview_theme->signal_button_release_event().connect_notify(
            sigc::mem_fun( this, &TagView::handle_theme_button_release_event ) );
}

bool
TagView::check_title_applicable( const Glib::ustring& name ) const
{
    Tag *tag = Diary::d->get_tags()->get_tag( name );
    return( !( tag || name.empty() ) );
}

bool
TagView::apply_title( const Glib::ustring& name )
{
    // a caution against calls by an entry activation event:
    if( ! check_title_applicable( name ) )
        return false;

    Diary::d->get_tags()->rename( m_ptr2elem, name );
    AppWindow::p->panel_extra->refresh_elem( m_ptr2elem );

    return true;
}

void
TagView::handle_login()
{
    update_combobox_categories();
    m_combobox_category->set_sensitive( ! Diary::d->is_read_only() );
}

void
TagView::handle_filter_toggled()
{
    if( Lifeograph::m_internaloperation )
        return;

    const Tag *tag( m_togglebutton_filter->get_active() ? m_ptr2elem : NULL );
    Diary::d->set_filter_tag( tag );
    update_button_filter_tag();
    AppWindow::p->panel_extra->set_filtered_tag( tag );
    AppWindow::p->panel_diary->update_entry_list();
}

void
TagView::handle_category_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    Gtk::TreeIter iter = m_combobox_category->get_active();
    if( iter )
    {
        Gtk::TreeRow row = *iter;
        if( row[ colrec_category.type ]  == TCT_NEW ) // create new
        {
            CategoryTags *category = Diary::d->create_tag_ctg();
            m_ptr2elem->set_category( category );
            update_combobox_categories();
            AppWindow::p->panel_extra->populate();
            category->show();
            AppWindow::p->panel_main->start_title_edit();
        }
        else	// assign a category or reset
        {
            m_ptr2elem->set_category( row[ colrec_category.ptr ] );
            AppWindow::p->panel_extra->populate();
        }
    }
}

void
TagView::update_combobox_categories()
{
    m_liststore_categories->clear();

    Gtk::TreeRow row = * m_liststore_categories->append();

    row[ colrec_category.name ] = _( "None" );
    row[ colrec_category.ptr ] = NULL;
    row[ colrec_category.type ] = TCT_NONE;

    // separator 1:
    row = * m_liststore_categories->append();
    row[ colrec_category.type ] = TCT_SEPARATOR;
    row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;

    for( PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.begin();
         iter != Diary::d->m_tag_categories.end();
         ++iter )
    {
        row = * m_liststore_categories->append();

        row[ colrec_category.name ] = iter->first;
        row[ colrec_category.ptr ] = iter->second;
        row[ colrec_category.type ] = TCT_CATEGORY;
    }

    // separator 2:
    if( Diary::d->m_tag_categories.size() > 0 )
    {
        row = * m_liststore_categories->append();
        row[ colrec_category.type ] = TCT_SEPARATOR;
        row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
    }

    row = * m_liststore_categories->append();

    row[ colrec_category.name ] = _( "Create New..." );
    // a dummy value to differentiate it from "None":
    row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
    row[ colrec_category.type ] = TCT_NEW;
}

void
TagView::update_button_filter_tag()
{
    Lifeograph::m_internaloperation++;
    m_togglebutton_filter->set_active( Diary::d->get_filter_tag() == m_ptr2elem );
    Lifeograph::m_internaloperation--;
}

void
TagView::update_theme_view()
{
    Theme *theme = m_ptr2elem->get_theme();

    m_textview_theme->override_font( theme->font );
    m_textview_theme->override_background_color( theme->color_base, Gtk::STATE_FLAG_NORMAL );
    m_textview_theme->override_color( theme->color_text, Gtk::STATE_FLAG_NORMAL );

    m_textview_theme->override_background_color( theme->color_heading, Gtk::STATE_FLAG_SELECTED );
    m_textview_theme->override_color( theme->color_base, Gtk::STATE_FLAG_SELECTED );

    Gdk::RGBA color_mid( midtone( theme->color_base, theme->color_text ) );

    m_tag_heading->property_foreground_rgba() = theme->color_heading;
    m_tag_subheading->property_foreground_rgba() = theme->color_subheading;
    m_tag_highlight->property_background_rgba() = theme->color_highlight;
    m_tag_background->property_foreground_rgba() = theme->color_base;
    m_tag_background->property_background_rgba() = color_mid;

    m_tag_comment->property_foreground_rgba() = color_mid;
    m_tag_region->property_paragraph_background_rgba() = midtone(
            theme->color_base, theme->color_text, 0.9 );
    m_tag_match->property_foreground_rgba() = theme->color_base;
    m_tag_match->property_background_rgba() = contrast2(
            theme->color_base, Theme::s_color_match1, Theme::s_color_match2 );
    m_tag_link->property_foreground_rgba() = contrast2(
            theme->color_base, Theme::s_color_link1, Theme::s_color_link2 );
    m_tag_link_broken->property_foreground_rgba() = contrast2(
            theme->color_base, Theme::s_color_broken1, Theme::s_color_broken2 );

    // THEME VIEW CONTENTS
    m_textview_theme->get_buffer()->set_text( "" );

    add_action_theme_text( _( "Heading" ), m_tag_heading );

    add_plain_theme_text( " \n " );

    add_action_theme_text( _( "Subheading" ), m_tag_subheading );

    add_plain_theme_text( " \n" );

    add_action_theme_text( _( "Normal text" ), m_tag_normal );

    add_plain_theme_text( "  " );

    add_action_theme_text( _( "highlighted text" ), m_tag_highlight );

    add_plain_theme_text( "  " );

    add_tagged_theme_text( Glib::ustring::format( "[[", _( "comment" ), "]]" ), m_tag_comment );

    add_plain_theme_text( "  " );

    add_tagged_theme_text( _( "link" ), m_tag_link );

    add_plain_theme_text( "  " );

    add_tagged_theme_text( _( "broken link" ), m_tag_link_broken );

    add_plain_theme_text( "  " );

    add_tagged_theme_text( _( "match" ), m_tag_match );

    add_plain_theme_text( " \n" );

    add_action_theme_text(
            Glib::ustring::format(
                    _( "Font" ), ": ", m_ptr2elem->get_theme()->font.to_string() ),
                    m_tag_font );

    add_plain_theme_text( " \n" );

    add_action_theme_text( _( "Background" ), m_tag_background );

    add_plain_theme_text( " \n" );

    add_tagged_theme_text(
            Glib::ustring::format(
                    ".\t", _( "Click on the parts to edit" ), "." ), m_tag_region );
}

void
TagView::update_theme_menu()
{
    if( m_menu_themes )
    {
        m_button_themes->unset_popup();
        //delete m_menu_themes; seems not necessary after above...
    }

    m_menu_themes = Gtk::manage( new Gtk::Menu );

    for( PoolTags::const_iterator iter( Diary::d->get_tags()->begin() );
         iter != Diary::d->get_tags()->end();
         ++iter )
    {
        Tag* tag( iter->second );

        if( ! tag->get_has_own_theme() )
            continue;
        if( tag == m_ptr2elem )
            continue;

        Gtk::MenuItem *mi = Gtk::manage( new Gtk::MenuItem( tag->get_name() ) );
        mi->show();
        m_menu_themes->append( *mi );
        mi->signal_activate().connect(
                sigc::bind( sigc::mem_fun( this, &TagView::copy_theme_from ), tag ) );
    }

    m_button_themes->set_popup( *m_menu_themes );
    m_button_theme_reset->set_sensitive( m_ptr2elem->get_has_own_theme() );
}

void
TagView::reset_theme()
{
    m_ptr2elem->reset_theme();
    m_button_theme_reset->set_sensitive( false );
    update_theme_view();
    AppWindow::p->panel_main->set_icon( m_ptr2elem->get_icon32() );
    AppWindow::p->panel_extra->refresh_elem( m_ptr2elem );
}

void
TagView::copy_theme_from( const Tag* tag )
{
    m_ptr2elem->create_own_theme_duplicating( tag->get_theme() );
    m_button_theme_reset->set_sensitive( true );
    update_theme_view();
    AppWindow::p->panel_main->set_icon( m_ptr2elem->get_icon32() );
    AppWindow::p->panel_extra->refresh_elem( m_ptr2elem );
}

void
TagView::show( Tag& tag )
{
    // do nothing if it is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &tag ) )
        return;

    m_ptr2elem = &tag;

    AppWindow::p->panel_main->show( this );

    Lifeograph::m_internaloperation++;

    // BODY
    update_button_filter_tag();

    for( Gtk::TreeIter iter = m_liststore_categories->children().begin();
         iter != m_liststore_categories->children().end();
         ++iter )
    {
        if( ( *iter )[ colrec_category.ptr ] == tag.get_category() )
            m_combobox_category->set_active( iter );
    }

    Lifeograph::m_internaloperation--;

    m_frame_edit->set_visible( tag.get_type() == DiaryElement::ET_TAG );

    update_theme_view();
    update_theme_menu();

    // CHART DATA
    ChartPoints *points( new ChartPoints );
    for( Tag::Set::reverse_iterator iter = tag.rbegin(); iter != tag.rend(); ++iter )
    {
        Date date( ( *iter )->get_date() );
        if( ! date.is_ordinal() )
            points->increase( date.get_yearmonth() );
    }
    points->finish();
    m_widgetchart->set_points( points );
}

Gtk::Widget*
TagView::get_popover()
{
    return( ( Diary::d->is_read_only() || m_ptr2elem->get_type() == DiaryElement::ET_UNTAGGED ) ?
            NULL : m_grid_popover );
}

void
TagView::dismiss_tag()
{
    AppWindow::p->panel_main->hide_popover();

    Gtk::CheckButton check( _( "Also dismiss the entries associated with this tag" ) );
    if( ! AppWindow::p->confirm_dismiss_element( m_ptr2elem, &check ) )
        return;

    Diary::d->show();

    AppWindow::p->panel_main->remove_element_from_history( m_ptr2elem );

    // clear filter if necessary:
    if( Diary::d->get_filter_tag() == m_ptr2elem )
    {
        Diary::d->set_filter_tag( NULL );
    }

    Diary::d->dismiss_tag( m_ptr2elem, check.get_active() );
    AppWindow::p->panel_diary->update_entry_list();
    AppWindow::p->panel_extra->populate();
}

bool
TagView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel >& model,
                             const Gtk::TreeIter iter )
{
    Gtk::TreeRow row = * iter;
    return ( row[ colrec_category.type ] == TCT_SEPARATOR );
}

void
TagView::handle_theme_motion_notify_event( GdkEventMotion* event )
{
    Gtk::TextIter       iter;
    int                 trailing, buffer_x, buffer_y;
    static const Glib::RefPtr< Gdk::Cursor > cursor_hand( Gdk::Cursor::create( Gdk::HAND2 ) );
    static const Glib::RefPtr< Gdk::Cursor > cursor_xterm( Gdk::Cursor::create( Gdk::XTERM ) );

    m_textview_theme->window_to_buffer_coords( Gtk::TEXT_WINDOW_WIDGET,
                                               event->x, event->y, buffer_x, buffer_y );
    m_textview_theme->get_iter_at_position( iter, trailing, buffer_x, buffer_y );

    if( iter.has_tag( m_tag_action ) )
        m_textview_theme->get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( cursor_hand );
    else
        m_textview_theme->get_window( Gtk::TEXT_WINDOW_TEXT )->set_cursor( cursor_xterm );
}

void
TagView::handle_theme_button_release_event( GdkEventButton* event )
{
    Gtk::TextIter       iter;
    int                 trailing, buffer_x, buffer_y;
    m_textview_theme->window_to_buffer_coords( Gtk::TEXT_WINDOW_WIDGET,
                                               event->x, event->y, buffer_x, buffer_y );
    m_textview_theme->get_iter_at_position( iter, trailing, buffer_x, buffer_y );

    if( iter.has_tag( m_tag_action ) )
    {
        if( !m_ptr2elem->get_has_own_theme() )
            m_ptr2elem->create_own_theme_duplicating( ThemeSystem::get() );
        Theme* theme = m_ptr2elem->get_theme();

        if( iter.has_tag( m_tag_heading ) )
            launch_color_dialog( theme->color_heading );
        else if( iter.has_tag( m_tag_subheading ) )
            launch_color_dialog( theme->color_subheading );
        else if( iter.has_tag( m_tag_normal ) )
            launch_color_dialog( theme->color_text );
        else if( iter.has_tag( m_tag_highlight ) )
            launch_color_dialog( theme->color_highlight );
        else if( iter.has_tag( m_tag_background ) )
            launch_color_dialog( theme->color_base );
        else if( iter.has_tag( m_tag_font ) )
            launch_font_dialog();

        m_button_theme_reset->set_sensitive( true );
        AppWindow::p->panel_main->set_icon( m_ptr2elem->get_icon32() );
        AppWindow::p->panel_extra->refresh_elem( m_ptr2elem );
    }
}

void
TagView::add_action_theme_text( const Glib::ustring& text,
                                const Glib::RefPtr< Gtk::TextTag >& tag )
{
    std::vector< Glib::RefPtr< Gtk::TextTag > > tags;
    tags.push_back( m_tag_action );
    tags.push_back( tag );

    m_textview_theme->get_buffer()->insert_with_tags(
            m_textview_theme->get_buffer()->end(), text, tags );
}
void
TagView::add_tagged_theme_text( const Glib::ustring& text,
                                const Glib::RefPtr< Gtk::TextTag >& tag )
{
    m_textview_theme->get_buffer()->insert_with_tag(
            m_textview_theme->get_buffer()->end(), text, tag );
}
void
TagView::add_plain_theme_text( const Glib::ustring& text )
{
    m_textview_theme->get_buffer()->insert( m_textview_theme->get_buffer()->end(), text );
}

void
TagView::launch_color_dialog( Gdk::RGBA& color )
{
    Gtk::ColorChooserDialog *dlg = new Gtk::ColorChooserDialog;
    dlg->set_transient_for( * AppWindow::p );
    dlg->set_use_alpha( false );
    dlg->set_rgba( color );

    if( dlg->run() == Gtk::RESPONSE_OK )
    {
        color = dlg->get_rgba();
        update_theme_view();
    }

    delete dlg;
}

void
TagView::launch_font_dialog()
{
    Gtk::FontChooserDialog *dlg = new Gtk::FontChooserDialog;
    dlg->set_transient_for( * AppWindow::p );
    dlg->set_font_desc( m_ptr2elem->get_theme()->font );

    if( dlg->run() == Gtk::RESPONSE_OK )
    {
        m_ptr2elem->get_theme()->font = dlg->get_font_desc();
        update_theme_view();
    }

    delete dlg;
}

// TAG CATEGORY ====================================================================================
CategoryTagsView::CategoryTagsView()
{
    CategoryTags::shower = this;

    try
    {
        Lifeograph::builder->get_widget( "Bx_categorytags", m_vbox );
        Lifeograph::builder->get_widget( "G_tagctg_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "B_tagctg_dismiss", m_button_dismiss );
        Lifeograph::builder->get_widget_derived( "DA_ctgtag_chart", m_widgetchart );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the tag category view" );
    }

    m_button_dismiss->signal_clicked().connect(
            sigc::mem_fun( this, &CategoryTagsView::dismiss_category ) );
}

void
CategoryTagsView::show( CategoryTags& category )
{
    // do nothing if it is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &category ) )
        return;

    m_ptr2elem = &category;

    AppWindow::p->panel_main->show( this );

    // CHART DATA
    ChartPoints* points( new ChartPoints );
    for( CategoryTags::iterator iter_tag = category.begin();
         iter_tag != category.end();
         iter_tag++ )
    {
        Tag* tag( *iter_tag );

        for( Tag::Set::iterator iter = tag->begin(); iter != tag->end(); ++iter )
        {
            Date date( ( *iter )->get_date() );
            if( ! date.is_ordinal() )
                points->increase( date.get_yearmonth() );
        }
    }
    points->finish();
    m_widgetchart->set_points( points );
}

Gtk::Widget*
CategoryTagsView::get_popover()
{
    return( Diary::d->is_read_only() ? NULL : m_grid_popover );
}

bool
CategoryTagsView::check_title_applicable( const Glib::ustring& name ) const
{
    PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.find( name );
    return( iter == Diary::d->m_tag_categories.end() && name.empty() == false );
}

bool
CategoryTagsView::apply_title( const Glib::ustring& name )
{
    if( ! check_title_applicable( name ) )
        return false;

    Diary::d->m_tag_categories.rename_category( m_ptr2elem, name );
    AppWindow::p->panel_extra->populate();
    AppWindow::p->m_tag_view->update_combobox_categories();

    return true;
}

void
CategoryTagsView::dismiss_category()
{
    AppWindow::p->panel_main->hide_popover();

    Gtk::CheckButton check( _( "Also dismiss the tags in the category" ) );
    if( ! AppWindow::p->confirm_dismiss_element( m_ptr2elem, &check ) )
        return;

    Diary::d->show();

    AppWindow::p->panel_main->remove_element_from_history( m_ptr2elem );
    Diary::d->dismiss_tag_ctg( m_ptr2elem, check.get_active() );
    AppWindow::p->panel_extra->populate();
    AppWindow::p->m_tag_view->update_combobox_categories();
}

// CHAPTER =========================================================================================
ChapterView::ChapterView()
{
    Chapter::shower = this;

    try
    {
        Lifeograph::builder->get_widget( "Bx_chapter", m_vbox );
        Lifeograph::builder->get_widget( "G_chapter_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "I_chapter_not_todo", m_I_not_todo );
        Lifeograph::builder->get_widget( "RB_chapter_not_todo", m_button_not_todo );
        Lifeograph::builder->get_widget( "RB_chapter_todo", m_button_todo );
        Lifeograph::builder->get_widget( "RB_chapter_progressed", m_button_progressed );
        Lifeograph::builder->get_widget( "RB_chapter_done", m_button_done );
        Lifeograph::builder->get_widget( "RB_chapter_canceled", m_button_canceled );
        Lifeograph::builder->get_widget( "B_chapter_dismiss", m_button_dismiss );

//		Lifeograph::builder->get_widget( "label_chapter_period", m_label_period );
//		Lifeograph::builder->get_widget( "hbox_chapter_period", m_hbox_period );
        Lifeograph::builder->get_widget( "entry_chapter_begins", m_entry_begins );
        Lifeograph::builder->get_widget( "button_chapter_date_apply", m_button_date_apply );
        Lifeograph::builder->get_widget( "label_chapter_ends", m_label_ends );

        Lifeograph::builder->get_widget( "frame_chapter_edit", m_frame_edit );
        Lifeograph::builder->get_widget( "frame_chapter_actions", m_frame_actions );
        Lifeograph::builder->get_widget( "button_chapter_new_entry", m_button_new_entry );
    }
    catch( ... )
    {
        throw HELPERS::Error( "Failed to create the chapter view" );
    }

    // SIGNALS
    m_button_not_todo->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &ChapterView::set_todo_status ), ES::NOT_TODO ) );
    m_button_todo->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &ChapterView::set_todo_status ), ES::TODO ) );
    m_button_progressed->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &ChapterView::set_todo_status ), ES::PROGRESSED ) );
    m_button_done->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &ChapterView::set_todo_status ), ES::DONE ) );
    m_button_canceled->signal_toggled().connect(
            sigc::bind( sigc::mem_fun( this, &ChapterView::set_todo_status ), ES::CANCELED ) );

    m_button_dismiss->signal_clicked().connect(
            sigc::mem_fun( this, &ChapterView::dismiss_chapter ) );

    m_entry_begins->signal_changed().connect(
            sigc::mem_fun( *this, &ChapterView::handle_date_changed ) );
    m_entry_begins->signal_activate().connect(
            sigc::mem_fun( *m_button_date_apply, &Gtk::Button::clicked ) );
    m_button_date_apply->signal_clicked().connect(
            sigc::mem_fun( *this, &ChapterView::handle_date_applied ) );

    m_button_new_entry->signal_clicked().connect(
            sigc::mem_fun( *this, &ChapterView::add_new_entry ) );
}

void
ChapterView::dismiss_chapter()
{
    AppWindow::p->panel_main->hide_popover();

    Gtk::CheckButton check( _( "Also dismiss the entries within" ) );
    if( ! AppWindow::p->confirm_dismiss_element( m_ptr2elem, &check ) )
        return;

    Diary::d->show();

    AppWindow::p->panel_main->remove_element_from_history( m_ptr2elem );
    Diary::d->dismiss_chapter( m_ptr2elem, check.get_active() );
    AppWindow::p->panel_diary->update_entry_list();
}

bool
ChapterView::check_title_applicable( const Glib::ustring& str ) const
{
    return( ! str.empty() && str != m_ptr2elem->m_name );
}

bool
ChapterView::apply_title( const Glib::ustring& str )
{
    // a caution against calls by an entry activation event:
    if( ! check_title_applicable( str ) )
        return false;
    m_ptr2elem->set_name( str );
    Gtk::TreeRow row = * AppWindow::p->get_element_row( m_ptr2elem );
    row[ ListData::colrec->info ] = m_ptr2elem->get_list_str();
    return true;
}

void
ChapterView::handle_date_changed()
{
    Date date( m_entry_begins->get_text() );

    if( date.is_set() && ! date.is_ordinal() )
        m_button_date_apply->set_sensitive(
                ! Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) );
    else
        m_button_date_apply->set_sensitive( false );
}

void
ChapterView::handle_date_applied()
{
    // a caution against calls by an entry activation event:
    if( ! m_button_date_apply->is_sensitive() )
        return;

    Date date( m_entry_begins->get_text() );

    if( date.is_set() &&
        Diary::d->get_current_chapter_ctg()->set_chapter_date( m_ptr2elem, date.m_date ) )
    {
        m_button_date_apply->set_sensitive( false );
        Diary::d->update_entries_in_chapters();
        AppWindow::p->panel_diary->update_entry_list();
        update_labels();
    }
}

void
ChapterView::update_labels()
{
    if( ! m_ptr2elem->m_date_begin.is_ordinal() )
    {
        Lifeograph::m_internaloperation++;

        m_entry_begins->set_text( m_ptr2elem->m_date_begin.format_string() );

        CategoryChapters::iterator iter( Diary::d->get_current_chapter_ctg()->find(
                                                m_ptr2elem->get_date_t() ) );
        if( m_ptr2elem->m_time_span == 0 )
            m_label_ends->set_text( _( "Unlimited" ) );
        else
        {
            --iter;
            m_label_ends->set_text( iter->second->m_date_begin.format_string() );
        }

        Lifeograph::m_internaloperation--;
    }
}

void
ChapterView::add_new_entry()
{
    Diary::d->create_entry( m_ptr2elem->get_free_order().m_date );

    AppWindow::p->panel_diary->update_entry_list();
    AppWindow::p->panel_diary->update_calendar();
}

void
ChapterView::handle_login()
{
    bool editable( ! Diary::d->is_read_only() );
    m_entry_begins->set_sensitive( editable );
    m_button_date_apply->set_visible( editable );
    m_frame_actions->set_visible( editable );
}

void
ChapterView::show( Chapter& chapter )
{
    // do nothing if entry is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &chapter ) )
        return;

    m_ptr2elem = &chapter;

    AppWindow::p->panel_main->show( this );

    // BODY
    update_labels();

    bool flag_ordinal( chapter.m_date_begin.is_ordinal() );
    m_frame_edit->set_visible( ! flag_ordinal );
    if( ! Diary::d->is_read_only() )
        m_frame_actions->set_visible( flag_ordinal );

    m_I_not_todo->set( chapter.get_date().is_ordinal() ?
            Lifeograph::icons->chapter_32 : Lifeograph::icons->month_32 );

    switch( m_ptr2elem->get_todo_status() )
    {
        case ES::NOT_TODO:
            m_button_not_todo->set_active();
            break;
        case ES::TODO:
            m_button_todo->set_active();
            break;
        case ES::PROGRESSED:
            m_button_progressed->set_active();
            break;
        case ES::DONE:
            m_button_done->set_active();
            break;
        case ES::CANCELED:
            m_button_canceled->set_active();
            break;
    }
}

Gtk::Widget*
ChapterView::get_popover()
{
    return( Diary::d->is_read_only() ? NULL : m_grid_popover );
}

void
ChapterView::set_todo_status( ElemStatus status )
{
    if( Lifeograph::m_internaloperation )
        return;

    m_ptr2elem->set_todo_status( status );
    AppWindow::p->panel_main->set_icon( m_ptr2elem->get_icon32() );
    Gtk::TreeRow row = AppWindow::p->get_element_row( m_ptr2elem );
    row[ ListData::colrec->icon ] = m_ptr2elem->get_icon();
    row[ ListData::colrec->info ] = m_ptr2elem->get_list_str();
}

// FILTER ==========================================================================================
FilterView::ColrecEntries*  FilterView::colrec_entries = new ColrecEntries;

FilterView::FilterView()
{
    Filter::shower = this;

    try
    {
        Lifeograph::builder->get_widget( "Bx_filter", m_vbox );
        Lifeograph::builder->get_widget( "TB_filter_not_todo", m_tglbtn_todo_not );
        Lifeograph::builder->get_widget( "TB_filter_todo", m_tglbtn_todo_open );
        Lifeograph::builder->get_widget( "TB_filter_progressed", m_tglbtn_todo_progressed );
        Lifeograph::builder->get_widget( "TB_filter_done", m_tglbtn_todo_done );
        Lifeograph::builder->get_widget( "TB_filter_canceled", m_tglbtn_todo_canceled );
        Lifeograph::builder->get_widget( "TB_filter_not_favored", m_tglbtn_not_favored );
        Lifeograph::builder->get_widget( "TB_filter_favored", m_tglbtn_favored );
        Lifeograph::builder->get_widget( "TB_filter_not_trashed", m_tglbtn_not_trashed );
        Lifeograph::builder->get_widget( "TB_filter_trashed", m_tglbtn_trashed );

        Lifeograph::builder->get_widget( "L_filter_tag", m_label_tag );
        Lifeograph::builder->get_widget( "B_filter_clear_tag", m_button_clear_tag );

        Lifeograph::builder->get_widget_derived( "E_filter_date_start", m_entry_date_begin );
        Lifeograph::builder->get_widget_derived( "E_filter_date_end", m_entry_date_end );

        Lifeograph::builder->get_widget( "TV_filter_entries", m_treeview_entries );
        Lifeograph::builder->get_widget( "B_filter_remove_entry", m_button_remove_entry );
        Lifeograph::builder->get_widget( "B_filter_rm_all_entries", m_button_rm_all_entries );

        Lifeograph::builder->get_widget( "G_filter_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "B_filter_revert", m_button_revert_to_default );
        Lifeograph::builder->get_widget( "B_filter_make_default", m_button_make_default );

        m_liststore_entries = Gtk::ListStore::create( *colrec_entries );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the filter view" );
    }

    m_button_revert_to_default->signal_clicked().connect(
            sigc::mem_fun( this, &FilterView::revert_to_default ) );
    m_button_make_default->signal_clicked().connect(
            sigc::mem_fun( this, &FilterView::make_default ) );

    m_tglbtn_todo_not->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_todo_changed ) );
    m_tglbtn_todo_open->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_todo_changed ) );
    m_tglbtn_todo_progressed->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_todo_changed ) );
    m_tglbtn_todo_done->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_todo_changed ) );
    m_tglbtn_todo_canceled->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_todo_changed ) );

    m_tglbtn_trashed->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_trash_changed ) );
    m_tglbtn_not_trashed->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_trash_changed ) );

    m_tglbtn_favored->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_favored_changed ) );
    m_tglbtn_not_favored->signal_toggled().connect(
            sigc::mem_fun( this, &FilterView::handle_favored_changed ) );

    m_label_tag->drag_dest_set( Lifeograph::p->drag_targets_tag );
    m_label_tag->signal_drag_drop().connect(
            sigc::mem_fun( this, &FilterView::handle_tag_dropped ) );
    m_button_clear_tag->signal_clicked().connect(
            sigc::mem_fun( this, &FilterView::clear_tag ) );

    m_entry_date_begin->signal_changed().connect(
            sigc::mem_fun( this, &FilterView::handle_date_begin_changed ) );
    m_entry_date_end->signal_changed().connect(
            sigc::mem_fun( this, &FilterView::handle_date_end_changed ) );

    m_treeview_entries->set_model( m_liststore_entries );
    m_treeview_entries->append_column(
            * Gtk::manage( new Gtk::TreeView::Column( "", colrec_entries->name ) ) );
    m_treeview_entries->drag_dest_set( Lifeograph::p->drag_targets_entry );
    m_treeview_entries->signal_drag_drop().connect(
            sigc::mem_fun( this, &FilterView::handle_entry_dropped ) );
    m_treeview_entries->get_selection()->signal_changed().connect(
            sigc::mem_fun( this, &FilterView::handle_entry_selection_changed ) );

    m_button_remove_entry->signal_clicked().connect(
            sigc::mem_fun( this, &FilterView::remove_selected_entries ) );
    m_button_rm_all_entries->signal_clicked().connect(
            sigc::mem_fun( this, &FilterView::remove_all_entries ) );
}

void
FilterView::handle_login()
{
    m_button_make_default->set_visible( ! Diary::d->is_read_only() );
}

void
FilterView::show( Filter& filter )
{
    // do nothing is entry is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &filter ) )
        return;

    m_ptr2elem = &filter;

    AppWindow::p->panel_main->show( this );

    update();

/*
    if( ! Diary::d->is_read_only() )
    {
        bool flag_sensitive( ! theme.is_system() );
        m_menuitem_make_default->set_visible( &theme != Diary::d->get_default_theme() );
        m_menuseparator->set_visible( flag_sensitive );
        m_menuitem_dismiss->set_visible( flag_sensitive );
    }
    m_frame_font->set_visible( ! Diary::d->is_read_only() );
    m_frame_colors->set_visible( ! Diary::d->is_read_only() );*/
}

void
FilterView::update()
{
    ElemStatus fs( m_ptr2elem->get_status() );

    Lifeograph::m_internaloperation++;

    // STATUS
    m_tglbtn_favored->set_active( fs & ES::SHOW_FAVORED );
    m_tglbtn_not_favored->set_active( fs & ES::SHOW_NOT_FAVORED );
    m_tglbtn_trashed->set_active( fs & ES::SHOW_TRASHED );
    m_tglbtn_not_trashed->set_active( fs & ES::SHOW_NOT_TRASHED );
    m_tglbtn_todo_not->set_active( fs & ES::SHOW_NOT_TODO );
    m_tglbtn_todo_open->set_active( fs & ES::SHOW_TODO );
    m_tglbtn_todo_progressed->set_active( fs & ES::SHOW_PROGRESSED );
    m_tglbtn_todo_done->set_active( fs & ES::SHOW_DONE );
    m_tglbtn_todo_canceled->set_active( fs & ES::SHOW_CANCELED );

    // TAG
    if( m_ptr2elem->get_tag() )
        m_label_tag->set_text( m_ptr2elem->get_tag()->get_name() );
    else
        m_label_tag->set_text( _( STRING::DROP_TAG_TO_FILTER ) );

    m_button_clear_tag->set_visible( m_ptr2elem->get_tag() );

    // DATE INTERVAL
    if( m_ptr2elem->get_date_begin() > 0 )
        m_entry_date_begin->set_text( Date::format_string( m_ptr2elem->get_date_begin() ) );
    else
        m_entry_date_begin->set_text( "" );

    if( m_ptr2elem->get_date_end() != Date::DATE_MAX )
        m_entry_date_end->set_text( Date::format_string( m_ptr2elem->get_date_end() ) );
    else
        m_entry_date_end->set_text( "" );

    // INDIVIDUAL ENTRIES
    update_entries();

    Lifeograph::m_internaloperation--;
}

Glib::ustring
FilterView::get_info_str() const
{
    return _( "Determines the entries to be shown in the list" );
}

void
FilterView::revert_to_default()
{
    AppWindow::p->panel_main->hide_popover();

    m_ptr2elem->set( Diary::d->get_filter_default() );
    m_ptr2elem->set_status_outstanding();
    update();
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::make_default()
{
    AppWindow::p->panel_main->hide_popover();

    Diary::d->get_filter_default()->set( m_ptr2elem );
}

void
FilterView::handle_trash_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    bool trashed( m_tglbtn_trashed->get_active() );
    bool not_trashed( m_tglbtn_not_trashed->get_active() );

    if( !( trashed || not_trashed ) )
    {
        if( m_ptr2elem->get_status() & ES::SHOW_TRASHED )
            not_trashed = true;
        if( m_ptr2elem->get_status() & ES::SHOW_NOT_TRASHED )
            trashed = true;

        Lifeograph::m_internaloperation++;
        m_tglbtn_not_trashed->set_active( not_trashed );
        m_tglbtn_trashed->set_active( trashed );
        Lifeograph::m_internaloperation--;
    }

    m_ptr2elem->set_trash( trashed, not_trashed );
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::handle_favored_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    bool favored( m_tglbtn_favored->get_active() );
    bool not_favored( m_tglbtn_not_favored->get_active() );

    if( !( favored || not_favored ) )
    {
        if( m_ptr2elem->get_status() & ES::SHOW_FAVORED )
            not_favored = true;
        if( m_ptr2elem->get_status() & ES::SHOW_NOT_FAVORED )
            favored = true;

        Lifeograph::m_internaloperation++;
        m_tglbtn_favored->set_active( favored );
        m_tglbtn_not_favored->set_active( not_favored );
        Lifeograph::m_internaloperation--;
    }

    m_ptr2elem->set_favorites( favored, not_favored );
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::handle_todo_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    m_ptr2elem->set_todo( m_tglbtn_todo_not->get_active(),
                          m_tglbtn_todo_open->get_active(),
                          m_tglbtn_todo_progressed->get_active(),
                          m_tglbtn_todo_done->get_active(),
                          m_tglbtn_todo_canceled->get_active() );

    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::clear_tag()
{
    m_label_tag->set_text( _( STRING::DROP_TAG_TO_FILTER ) );
    m_button_clear_tag->hide();
    m_ptr2elem->set_tag( NULL );
    AppWindow::p->panel_extra->set_filtered_tag( NULL );
    AppWindow::p->panel_diary->update_entry_list();
}

bool
FilterView::handle_tag_dropped( const Glib::RefPtr< Gdk::DragContext >& context,
                                int x, int y, guint time )
{
    bool retval( true );

    if( Lifeograph::s_elem_dragged != NULL )
    {
        switch( Lifeograph::s_elem_dragged->get_type() )
        {
            case DiaryElement::ET_TAG:
            case DiaryElement::ET_UNTAGGED:
            {
                const Tag *tag( dynamic_cast< Tag* >( Lifeograph::s_elem_dragged ) );
                m_label_tag->set_text( Lifeograph::s_elem_dragged->get_name() );
                m_button_clear_tag->show();
                m_ptr2elem->set_tag( tag );
                AppWindow::p->panel_extra->set_filtered_tag( tag );
                AppWindow::p->panel_diary->update_entry_list();
            }
                break;
            default:
                retval = false;
                break;
        }
    }

    context->drag_finish( retval, false, time );

    return retval;
}

void
FilterView::handle_date_begin_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    m_entry_date_begin->unset_color();

    if( m_entry_date_begin->get_text_length() == 0 )
    {
        Diary::d->set_filter_date_begin( 0 );
    }
    else
    {
        Date date( m_entry_date_begin->get_text() );

        if( date.is_set() )
            Diary::d->set_filter_date_begin( date.m_date );
        else
            m_entry_date_begin->override_color( Gdk::RGBA( "Red" ) );
    }

    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::handle_date_end_changed()
{
    if( Lifeograph::m_internaloperation )
        return;

    m_entry_date_end->unset_color();

    if( m_entry_date_end->get_text_length() == 0 )
    {
        Diary::d->set_filter_date_end( Date::DATE_MAX );
    }
    else
    {
        Date date( m_entry_date_end->get_text() );

        if( date.is_set() )
            Diary::d->set_filter_date_end( date.m_date );
        else
            m_entry_date_end->override_color( Gdk::RGBA( "Red" ) );
    }

    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::update_entries()
{
    m_liststore_entries->clear();
    const EntrySet* es( m_ptr2elem->get_entries() );
    for( EntrySetIter iter = es->begin(); iter != es->end(); ++iter )
    {
        Gtk::TreeModel::Row row( *( m_liststore_entries->append() ) );
        row[ colrec_entries->name ] = ( *iter )->get_name();
        row[ colrec_entries->ptr ] = *iter;
    }

    m_button_rm_all_entries->set_sensitive( ! m_liststore_entries->children().empty( ) );
}

void
FilterView::remove_selected_entries()
{
    std::vector< Gtk::TreePath > paths(
            m_treeview_entries->get_selection()->get_selected_rows() );

    for( std::vector< Gtk::TreePath >::iterator iter = paths.begin();
         iter != paths.end();
         ++iter )
    {
        Gtk::TreeModel::Row row( * m_liststore_entries->get_iter( *iter ) );
        Entry* entry( row[ colrec_entries->ptr ] );
        m_ptr2elem->remove_entry( entry );
    }

    update_entries();
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

void
FilterView::remove_all_entries()
{
    m_ptr2elem->clear_entries();
    update_entries();
    AppWindow::p->panel_diary->update_calendar();
    AppWindow::p->panel_diary->update_entry_list();
}

bool
FilterView::handle_entry_dropped( const Glib::RefPtr< Gdk::DragContext >& context,
                                  int x, int y, guint time )
{
    bool retval( true );

    if( Lifeograph::s_elem_dragged != NULL )
    {
        switch( Lifeograph::s_elem_dragged->get_type() )
        {
            case DiaryElement::ET_ENTRY:
            {
                Entry* entry( dynamic_cast< Entry* >( Lifeograph::s_elem_dragged ) );
                m_ptr2elem->add_entry( entry );
                update_entries();
                AppWindow::p->panel_diary->update_calendar();
                AppWindow::p->panel_diary->update_entry_list();
                break;
            }
            case DiaryElement::ET_MULTIPLE_ENTRIES:
            {
                sigc::slot< void, Entry* > s = sigc::mem_fun( *m_ptr2elem, &Filter::add_entry );
                AppWindow::p->panel_diary->do_for_each_selected_entry( s );
                update_entries();
                AppWindow::p->panel_diary->update_calendar();
                AppWindow::p->panel_diary->update_entry_list();
                break;
            }
            default:
                retval = false;
                break;
        }
    }

    context->drag_finish( retval, false, time );

    return retval;
}

void
FilterView::handle_entry_selection_changed()
{
    m_button_remove_entry->set_sensitive(
            m_treeview_entries->get_selection()->count_selected_rows() > 0 );
}

// DIARY ===========================================================================================
DiaryView::DiaryView()
:	m_dialog_export( NULL )
{
    Diary::shower = this;
    m_ptr2elem = Diary::d;	// for now no other diary is shown in the view

    Gtk::Button*    button_export( NULL );

    try
    {
        Lifeograph::builder->get_widget( "Bx_diary", m_vbox );
        Lifeograph::builder->get_widget( "G_diary_popover", m_grid_popover );
        Lifeograph::builder->get_widget( "B_diary_import", m_button_import );
        Lifeograph::builder->get_widget( "B_diary_export", button_export );
        Lifeograph::builder->get_widget( "S_diary_encryption", m_separator_encryption );
        Lifeograph::builder->get_widget( "B_diary_encryption", m_button_encryption );

        Lifeograph::builder->get_widget( "label_path", m_label_path );
        Lifeograph::builder->get_widget( "combobox_diary_spellcheck", m_combobox_spellcheck );

        Lifeograph::builder->get_widget( "cmb_startup_type", m_combobox_startup_type );
        Lifeograph::builder->get_widget( "label_startup_elem", m_label_startup_elem );
        Lifeograph::builder->get_widget( "label_startup_elem_drop", m_label_startup_elem_drop );

        Lifeograph::builder->get_widget( "button_chapter_ctg_dismiss",
                m_button_chapter_ctg_dismiss );
        Lifeograph::builder->get_widget( "button_chapter_ctg_rename",
                m_button_chapter_ctg_rename );
        Lifeograph::builder->get_widget( "combo_chapter_ctg", m_combobox_chapter_ctg );

        Lifeograph::builder->get_widget_derived( "drawingarea_diary_chart", m_widgetchart );

        m_liststore_startup_type = Gtk::ListStore::create( colrec_startup_type );
        m_liststore_chapter_ctg = Gtk::ListStore::create( colrec_chapter_ctg );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create the diary view" );
    }

    // STARTUP
    m_combobox_startup_type->set_model( m_liststore_startup_type );
    m_combobox_startup_type->pack_start( colrec_startup_type.name );
    Gtk::TreeRow row( * m_liststore_startup_type->append() );
    row[ colrec_startup_type.name ] = _( "Show Most Current Entry" );
    row[ colrec_startup_type.type ] = HOME_CURRENT_ELEM;
    row = * m_liststore_startup_type->append();
    row[ colrec_startup_type.name ] = _( "Remember Last Item" );
    row[ colrec_startup_type.type ] = HOME_LAST_ELEM;
    row = * m_liststore_startup_type->append();
    row[ colrec_startup_type.name ] = _( "Always Show a Fixed Item" );
    row[ colrec_startup_type.type ] = DEID_DIARY;

    m_label_startup_elem_drop->drag_dest_set( Lifeograph::p->drag_targets_entry );

    // CHAPTER CATEGORIES
    m_combobox_chapter_ctg->set_model( m_liststore_chapter_ctg );
    m_combobox_chapter_ctg->set_entry_text_column( colrec_chapter_ctg.name );
    m_combobox_chapter_ctg->set_row_separator_func(
            sigc::mem_fun( this, &DiaryView::is_row_separator ) );

    // SIGNALS
    m_combobox_spellcheck->signal_changed().connect(
            sigc::mem_fun( this, &DiaryView::handle_language_changed ) );

    m_combobox_startup_type->signal_changed().connect(
            sigc::mem_fun( this, &DiaryView::handle_startup_type_changed ) );
    m_label_startup_elem_drop->signal_drag_motion().connect(
            sigc::mem_fun( this, &DiaryView::handle_startup_elem_drag_motion ), false );
    m_label_startup_elem_drop->signal_drag_drop().connect(
            sigc::mem_fun( this, &DiaryView::handle_startup_elem_dropped ) );
    m_label_startup_elem_drop->signal_activate_link().connect(
            sigc::mem_fun( this, &DiaryView::go_to_startup_elem ), false );

    m_combobox_chapter_ctg->signal_changed().connect(
            sigc::mem_fun( this, &DiaryView::handle_cur_chapter_ctg_changed ) );
    m_combobox_chapter_ctg->get_entry()->signal_activate().connect(
            sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
    m_button_chapter_ctg_rename->signal_clicked().connect(
            sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
    m_button_chapter_ctg_dismiss->signal_clicked().connect(
            sigc::mem_fun( this, &DiaryView::dismiss_cur_chapter_ctg ) );

    m_button_import->signal_clicked().connect(
            sigc::mem_fun( *this, &DiaryView::start_dialog_import ) );
    button_export->signal_clicked().connect(
            sigc::mem_fun( *this, &DiaryView::start_dialog_export ) );
    m_button_encryption->signal_clicked().connect(
            sigc::mem_fun( *this, &DiaryView::start_dialog_password ) );
}

DiaryView::~DiaryView()
{
    Diary::shower = NULL;
}

void
DiaryView::handle_login()
{
    Lifeograph::m_internaloperation++;

    // SPELL CHECKING
    m_combobox_spellcheck->remove_all();
    m_combobox_spellcheck->append( _( STRING::OFF ) );
    for( LanguageList::iterator iter = Lifeograph::s_lang_list.begin();
         iter != Lifeograph::s_lang_list.end(); ++iter )
    {
        m_combobox_spellcheck->append( *iter );
    }

    if( Diary::d->m_language.empty() )
        m_combobox_spellcheck->set_active_text( _( STRING::OFF ) );
    else
    {
        if( Lifeograph::s_lang_list.find( Diary::d->m_language ) ==
            Lifeograph::s_lang_list.end() )
            m_combobox_spellcheck->append( Diary::d->m_language );

        m_combobox_spellcheck->set_active_text( Diary::d->m_language );
    }

    update_combobox_chapter_ctg();
    if( Diary::d->m_startup_elem_id < HOME_FIXED_ELEM )
        m_combobox_startup_type->set_active( Diary::d->m_startup_elem_id - 1 );
    else
        m_combobox_startup_type->set_active( 2 );

    Lifeograph::m_internaloperation--;

    bool editable( ! Diary::d->is_read_only() );
    m_combobox_startup_type->set_sensitive( editable );
    m_combobox_spellcheck->set_sensitive( editable );
    m_combobox_chapter_ctg->set_sensitive( editable );

    m_button_import->set_visible( editable );
    m_separator_encryption->set_visible( editable );
    m_button_encryption->set_visible( editable );

    m_button_encryption->set_label( Diary::d->m_passphrase.empty() ? _( "Encrypt..." ) :
                                                _( STRING::CHANGE_PASSWORD ) );

    // BODY
    m_label_path->set_markup( Glib::ustring::compose( "<a href=\"file://%1\">%1</a>",
            Glib::path_get_dirname( Diary::d->m_path ) ) );
}

void
DiaryView::show( Diary& diary )
{
    // do nothing is entry is already the current element:
    if( AppWindow::p->panel_main->is_cur_elem( &diary ) )
        return;

    // not necessary for now as there is only one diary to show in the view:
    // m_ptr2elem = &diary;

    AppWindow::p->panel_main->show( this );

    // CHART DATA
    ChartPoints *points( new ChartPoints );
    for( EntryIter iter = diary.m_entries.begin(); iter != diary.m_entries.end(); ++iter )
    {
        Date date( iter->second->get_date() );
        if( ! date.is_ordinal() )
            points->increase( date.get_yearmonth() );
    }
    points->finish();
    m_widgetchart->set_points( points );
}

void
DiaryView::start_dialog_password()
{
    AppWindow::p->panel_main->hide_popover();

    if( DialogPassword::launch(
            m_ptr2elem->is_encrypted() ? DialogPassword::OT_CHANGE : DialogPassword::OT_ADD,
            m_ptr2elem ) == RESPONSE_GO )
    {
        AppWindow::p->panel_main->refresh_extra_info();
        m_button_encryption->set_label( _( STRING::CHANGE_PASSWORD ) );
    }
}

void
DiaryView::start_dialog_import()
{
    AppWindow::p->panel_main->hide_popover();

    Glib::RefPtr< Gtk::Builder > builder( Gtk::Builder::create_from_file( UIDIR "/import.ui" ) );

    DialogImport* dialog_import;
    builder->get_widget_derived( "D_import", dialog_import );

    if(  dialog_import->run() == RESPONSE_GO )
    {
        AppWindow::p->panel_diary->update_entry_list();
        AppWindow::p->panel_extra->populate();
        update_combobox_chapter_ctg();
    }

    dialog_import->hide();
    delete dialog_import;
}

void
DiaryView::start_dialog_export()
{
    AppWindow::p->panel_main->hide_popover();

    if( m_ptr2elem->is_encrypted() ?
            DialogPassword::launch( DialogPassword::OT_AUTHENTICATE,
                                    m_ptr2elem ) == RESPONSE_GO : true )
    {
        if( m_dialog_export == NULL )
            Lifeograph::builder->get_widget_derived( "D_export", m_dialog_export );

        m_dialog_export->run();
        m_dialog_export->hide();
    }
}

//void
//DiaryView::open_diary_folder()
//{
//	std::string uri = "file://" + Glib::path_get_dirname( m_path );
//	GError *err = NULL;
//	gtk_show_uri( NULL, uri.c_str(), GDK_CURRENT_TIME, &err);
//}

void
DiaryView::handle_language_changed()
{
    if( Lifeograph::m_internaloperation )
        return;
    std::string lang( m_combobox_spellcheck->get_active_text() );
    Diary::d->set_lang( lang == _( STRING::OFF ) ? "" : lang );
}

void
DiaryView::handle_startup_type_changed()
{
    if( ! Lifeograph::m_internaloperation )
    {
        Gtk::TreeRow row( * m_combobox_startup_type->get_active() );
        Diary::d->m_startup_elem_id = row[ colrec_startup_type.type ];
    }
    bool flag_show_fixed_item( Diary::d->m_startup_elem_id > HOME_FIXED_ELEM );
    m_label_startup_elem->set_visible( flag_show_fixed_item );
    m_label_startup_elem_drop->set_visible( flag_show_fixed_item );
    if( flag_show_fixed_item )
       update_startup_elem();
}

void
DiaryView::update_startup_elem()
{
    m_label_startup_elem_drop->set_markup(
            Glib::ustring::compose( "<a href=\"#\" title=\"%2\">%1</a>",
                                    Diary::d->get_startup_elem()->get_list_str(),
                                    _( "Drop items here to change" ) ) ); // tooltip
}

bool
DiaryView::handle_startup_elem_drag_motion( const Glib::RefPtr<Gdk::DragContext>& context,
                                            int x, int y, guint time )
{
    if( Lifeograph::s_flag_dragging )
        if( Lifeograph::s_elem_dragged )
            if( Lifeograph::s_elem_dragged->get_type() == DiaryElement::ET_DIARY ||
                Lifeograph::s_elem_dragged->get_type() == DiaryElement::ET_ENTRY )
            {
                context->drag_status( Gdk::ACTION_COPY, time );
                return true;
            }

    m_label_startup_elem_drop->drag_unhighlight();  // is called many times unnecessarily :(
    context->drag_refuse( time );
    return false;
}

bool
DiaryView::handle_startup_elem_dropped(
                const Glib::RefPtr< Gdk::DragContext >& context,
                int x, int y,
                guint time )
{
    bool retval( false );

    context->drag_finish( true, false, time );

    if( Lifeograph::s_flag_dragging )
    {
        if( Lifeograph::s_elem_dragged != NULL )
        {
            switch( Lifeograph::s_elem_dragged->get_type() )
            {
                case DiaryElement::ET_ENTRY:
                case DiaryElement::ET_DIARY:
                    Diary::d->m_startup_elem_id = Lifeograph::s_elem_dragged->get_id();
                    update_startup_elem();
                    retval = true;
                    break;
                default:
                    break;
            }
        }
    }

    return retval;
}

bool
DiaryView::go_to_startup_elem( const Glib::ustring& )
{
    DiaryElement *element( Diary::d->get_startup_elem() );
    if( element != Diary::d )
        element->show();

    return true;
}

void
DiaryView::update_combobox_chapter_ctg()
{
    m_liststore_chapter_ctg->clear();

    Gtk::TreeRow row;

    for( PoolCategoriesChapters::const_iterator iter =
                m_ptr2elem->m_chapter_categories.begin();
         iter != m_ptr2elem->m_chapter_categories.end();
         ++iter )
    {
        row = * m_liststore_chapter_ctg->append();

        CategoryChapters *category = iter->second;

        row[ colrec_chapter_ctg.name ] = category->get_name();
        row[ colrec_chapter_ctg.ptr ] = category;
        row[ colrec_chapter_ctg.type ] = CCCT_CATEGORY;

        // setting current item:
        if( m_ptr2elem->m_ptr2chapter_ctg_cur == category )
            m_combobox_chapter_ctg->set_active( row );
    }

    m_button_chapter_ctg_rename->set_visible( false );
    if( ! Diary::d->is_read_only() )
    {
        m_button_chapter_ctg_dismiss->set_visible( true );
        m_button_chapter_ctg_dismiss->set_sensitive(
                m_ptr2elem->m_chapter_categories.size() > 1 );
    }
    else
    {
        m_button_chapter_ctg_dismiss->set_visible( false );
    }

    // separator 2:
    if( m_ptr2elem->m_chapter_categories.size() > 0 )
    {
        row = * m_liststore_chapter_ctg->append();
        row[ colrec_chapter_ctg.type ] = CCCT_SEPARATOR;
    }

    row = * m_liststore_chapter_ctg->append();

    row[ colrec_chapter_ctg.name ] = _( "Create New..." );
    row[ colrec_chapter_ctg.ptr ] = NULL;
    row[ colrec_chapter_ctg.type ] = CCCT_NEW;
}

void
DiaryView::handle_cur_chapter_ctg_changed()
{
    if( Lifeograph::m_internaloperation ) return;

    Gtk::TreeIter iter = m_combobox_chapter_ctg->get_active();
    bool flag_renamed = ! ( iter ), flag_existing_ctg( false );

    if( flag_renamed )
    {
        Glib::ustring name = m_combobox_chapter_ctg->get_entry_text();
        bool flag_name_usable = m_ptr2elem->m_chapter_categories.count( name ) < 1;
        m_button_chapter_ctg_rename->set_sensitive( flag_name_usable );
    }
    else
    {
        Gtk::TreeRow row = *iter;
        switch( row[ colrec_chapter_ctg.type ] )
        {
            case CCCT_NEW: // create new
                m_ptr2elem->set_current_chapter_ctg( m_ptr2elem->create_chapter_ctg() );
                update_combobox_chapter_ctg();
                flag_existing_ctg = true;
                break;
            case CCCT_CATEGORY:
                flag_existing_ctg = true;
                m_ptr2elem->set_current_chapter_ctg( row[ colrec_chapter_ctg.ptr ] );
                AppWindow::p->panel_diary->update_entry_list();
                break;
            default:	// for separators
                break;
        }

        m_button_chapter_ctg_dismiss->set_sensitive( flag_existing_ctg );
    }

    if( ! Diary::d->is_read_only() )
    {
        m_button_chapter_ctg_rename->set_visible( flag_renamed );
        m_button_chapter_ctg_dismiss->set_visible( ! flag_renamed );
    }
}

void
DiaryView::rename_cur_chapter_ctg()
{
    Glib::ustring name = m_combobox_chapter_ctg->get_entry_text();
    if( name == m_ptr2elem->m_ptr2chapter_ctg_cur->get_name() )
        return;
    m_ptr2elem->rename_chapter_ctg( m_ptr2elem->m_ptr2chapter_ctg_cur, name );
    update_combobox_chapter_ctg();
}

void
DiaryView::dismiss_cur_chapter_ctg()
{
    if( ! AppWindow::p->confirm_dismiss_element( m_ptr2elem->m_ptr2chapter_ctg_cur ) )
        return;

    Diary::d->show();

    m_ptr2elem->dismiss_chapter_ctg( m_ptr2elem->m_ptr2chapter_ctg_cur );
    update_combobox_chapter_ctg();
}

bool
DiaryView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel >& model,
                             const Gtk::TreeIter iter )
{
    Gtk::TreeRow row = * iter;
    return ( row[ colrec_chapter_ctg.type ] == CCCT_SEPARATOR );
}
