/****************************************************************************
**
** $Id: cvscontrol.cpp,v 1.156 2004/05/03 14:52:34 hemer Exp $
**
** Copyright (C) 1999-2004 The LinCVS development team.
**    Tilo Riemer <riemer@lincvs.org>
**    Falk Brettschneider <gigafalk@yahoo.com>
**    Frank Hemer <frank@hemer.org>
**    Wim Delvaux <wim.delvaux@chello.be>
**    Jose Hernandez <joseh@tesco.net>
**    Helmut Koll <HelmutKoll@web.de>
**    Tom Mishima <tmishima@mail.at-m.or.jp>
**    Joerg Preiss <auba@auba.de>
**    Sven Trogisch <trogisch@iapp.de>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "config.h"

#include <qapplication.h>
#include <qtextcodec.h>
#include <qinputdialog.h>
#include <qfile.h>
#include <qcheckbox.h>
#include <qtextstream.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qstringlist.h>
#include <qregexp.h> 
#include <qprocess.h>
#include <assert.h>

#ifndef Q_WS_WIN
#include <unistd.h>
#else
#include <io.h>
#endif

#include "cvscontrol.h"
#include "dialogs.h"
#include "PixmapTimer.h"
#include "colortab.h"
#include "cvslistview.h"
#include "noncvslistview.h"
#include "cvsignorelistview.h"
#include "directory.h"
#include "DiffDialogImpl.h"
#include "LogDialogImpl.h"
#include "aboutdialogimpl.h"
#include "AnnotateDialogImpl.h"
#include "CommitDialogImpl.h"
#include "EditorsDialogImpl.h"
#include "HistoryDialogImpl.h"
#include "ProfilesDialogImpl.h"
#include "PropertiesDialogImpl.h"
#include "WatchersDialogImpl.h"
#include "AnnotateGrepLineDialogImpl.h"
#include "AnnotateGrepLinesDialogImpl.h"
#include "MergeDialogImpl.h"
#include "cvsconfig.h"
#include "login.h"
#include "LinCVSLog.h"
#include "cvstoolbar.h"
#include "TextEncoder.h"

#ifdef QT_NEED_PWDIALOGPATCH
#include "PasswordDialogImpl.h"
#endif

//----------------------------------------------------------------------------

const bool CvsControl::OVERRIDE = TRUE;

//----------------------------------------------------------------------------

CvsControl::CvsControl( QString startUpDir)
  : CMainWindow(0, "LinCVS main window", WType_TopLevel | WDestructiveClose)

// CvsControl::CvsControl(QString startUpDir)

{
   conf = 0;
   KILLED = false;
   createTmpDir();

   sshAgentIsRunning = false;
   sshAgentStarted = false;

   m_pInteractiveCmdThread = NULL;
   m_pCvsBuffer = new CvsBuffer();
   m_coImportLastAccessMethod = -1;
   m_coImportLastProfile = "";

   m_bExternalDiffIsRunning = false;
   m_curDiffFileName = "";      //used for external diff
   m_curTmpDiffFileName = "";   //used for external diff
   m_coImportVendorTag.clear();
   m_coImportReleaseTag.clear();
   
   m_curCommentFileName = "";
   
   readSettings();
   Debug::g_pLog = new CLinCVSLog();

   if (Debug::g_pLog) {
     QString msg = "\nLinCVS version: "+getVersion()+"\n";
#ifdef Q_WS_WIN
     msg += "for Windows\n";
#else
#ifdef Q_WS_MAC
     msg += "for Mac OS X\n";
#else
     msg += "for Unix\n";
#endif
#endif
     msg += "Compile Time: "+getCompileTime()+"\n";
     msg += "LogLevel: "+QString::number(Debug::g_logLevel)+"\n";
     Debug::g_pLog->log(Debug::LL_INFO, msg);
   }

   //we have set the application icon --> now we try to load icons from disk 
   loadPixmapsFromDisk(iconDir);

   globalListViewsEnabled = true;
   
   m_pCurCvsDir = NULL;
   m_pLastCvsCallDir = NULL;
   
   globalStopAction = false;
   m_chainedCommand = false;
   m_showWarning = true;
   m_interactiveCmdThreadIsRunning = false;
   
   /* initialize CommandInterface */
   initCommandInterface(this);

   if (bStartedInDir) {
       m_tmpStartUpDir = startUpDir;
   }

   /* initialize Monitoring */
   initMonitoring();
   if (DirWatch::b_isActive) {
     //No better way here since globals can't handle qt messages
     connect (getDirConnector(), SIGNAL(eventReceived()),this,SLOT(dirConnectorDataNotify()));
   }
}

//----------------------------------------------------------------------------

CvsControl::~CvsControl()
{
    releaseMonitoring();

    delete m_pCommitDlg;

    //remove any temporary file
    if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);
    if (m_curTmpDiffFileName.length()) QFile::remove(m_curTmpDiffFileName);
    
    //remove all tmp files from external diff
    QDir d(tmpDir);
    d.setFilter( QDir::Files | QDir::NoSymLinks );
    
    const QStringList list = d.entryList();
    unsigned int i;  
    for (i = 0; i < list.count(); i++) {
      if ( (WINVERSION && (list[i].find("fn") >= 0) ) || ( (!WINVERSION) && (list[i].find("tmp.") == 0)) ) {
	    QFile tmpFile(tmpDir + "/" + list[i]);
	    setPermission(tmpFile,READABLE | WRITEABLE);
	    tmpFile.remove();
      }
    }
    killSshAgent();

    delete Debug::g_pLog;
    Debug::g_pLog = 0;  //prevents logging after deleting log object
}

//----------------------------------------------------------------------------

void CvsControl::readSettings() {

    delete conf;
    conf = 0;   //paranoia

    QString tmpCvsRsh = getenv("CVS_RSH");
    QString tmpCvsPassPath = getenv("CVS_PASSFILE");
    QString tmpCvsWrappers = getenv("CVSWRAPPERS");
    if (tmpCvsPassPath.isEmpty()) tmpCvsPassPath = QDir::homeDirPath() + "/.cvspass";

    conf = new CvsConfig;

    expurgateProjectSettings();

    if (ExtApps::g_cvsRsh.path.isEmpty() && (!tmpCvsRsh.isEmpty())) {
	ExtApps::g_cvsRsh.path = tmpCvsRsh;   //use CVS_RSH
    }
    if (CVSPASSPATH.isEmpty() && (!tmpCvsPassPath.isEmpty())) {
	CVSPASSPATH = tmpCvsPassPath;
    }
    if (BINARYFILESPECLIST.isEmpty() && (!tmpCvsWrappers.isEmpty())) {
	//probably read file pointed to by CVSWRAPPERS, no docu available
// 	BINARYFILESPECLIST = tmpCvsWrappers;
    }

    openFileAppList.clear();
    QStringList::Iterator it;
    for (it = OPENFILELIST.begin(); it != OPENFILELIST.end(); it++) {
      FileAppItem item;
      int pos1 = (*it).find('\n');
      int pos2 = (*it).find('\n',pos1+1);
      int pos3 = (*it).find('\n',pos2+1);
      item.wildcard = (*it).mid(0,pos1);
      item.app = (*it).mid(pos1+1,pos2-pos1-1);
      item.params = (*it).mid(pos2+1,pos3-pos2-1);
      if ((*it).at( pos3+1) == '+') item.isRegExp = TRUE;
      else item.isRegExp = FALSE;
      openFileAppList.append(item);
    }
    viewFileAppList.clear();
    for (it = VIEWFILELIST.begin(); it != VIEWFILELIST.end(); it++) {
      FileAppItem item;
      int pos1 = (*it).find('\n');
      int pos2 = (*it).find('\n',pos1+1);
      int pos3 = (*it).find('\n',pos2+1);
      item.wildcard = (*it).mid(0,pos1);
      item.app = (*it).mid(pos1+1,pos2-pos1-1);
      item.params = (*it).mid(pos2+1,pos3-pos2-1);
      if ((*it).at( pos3+1) == '+') item.isRegExp = TRUE;
      else item.isRegExp = FALSE;
      viewFileAppList.append(item);
    }

}

//----------------------------------------------------------------------------

void CvsControl::writeSettings() {

  OPENFILELIST.clear();
  VIEWFILELIST.clear();
  FileAppList::Iterator it;
  for ( it = openFileAppList.begin(); it != openFileAppList.end(); ++it ) {
    OPENFILELIST.append((*it).wildcard+'\n'+(*it).app+'\n'+(*it).params+'\n'+((*it).isRegExp ? "+" : "-"));
  }
  for ( it = viewFileAppList.begin(); it != viewFileAppList.end(); ++it ) {
    VIEWFILELIST.append((*it).wildcard+'\n'+(*it).app+'\n'+(*it).params+'\n'+((*it).isRegExp ? "+" : "-"));
  }

  expurgateProjectSettings();

  writeSetup();
  writeCfg();
  delete conf;
  conf = 0;
}

//----------------------------------------------------------------------------

void CvsControl::expurgateProjectSettings() {
  unsigned int i;
  unsigned int j;
  QString workdir;
  QStringList projectNameList;
  QStringList topLevelProjectNameList;

  //remove unreferenced project settings
  projectSettings->getTopLevelProjects(topLevelProjectNameList);
  projectSettings->getProjects(projectNameList);
  QStringList tmpProjectNameList = projectNameList;
  QString joined;

  for (i=0; i<topLevelProjectNameList.count(); ++i) {//remove unreferenced entries from subProject list
    QStringList subProjects;
    if (projectSettings->getSubProjects(topLevelProjectNameList[i],subProjects)) {
      for (j=0; j<subProjects.count(); ++j) {
	QStringList::Iterator it = projectNameList.find(subProjects[j]);
	if (it == projectNameList.end()) {
	  projectSettings->removeSubProject(topLevelProjectNameList[i],subProjects[j]);
	  if (Debug::g_pLog) Debug::g_pLog->log(Debug::LL_INFO,"removing unreferenced sub-project from sub-project list: "+subProjects[j]);
	} else {
	  joined += "^e"+subProjects[j].replace(QRegExp("\\^"),"^0");
	}
      }
    }
    tmpProjectNameList.remove(tmpProjectNameList.find(topLevelProjectNameList[i]));
  }

  QStringList allSubProjectsList = QStringList::split("^e",joined);
  QStringList::Iterator it;
  for (it = allSubProjectsList.begin(); it != allSubProjectsList.end(); ++it) {
    (*it).replace(QRegExp("\\^0"),"^");
  }

  for (i=0; i<tmpProjectNameList.count(); ++i) {//remove unreferenced subProjects

    QStringList::Iterator it = allSubProjectsList.find(tmpProjectNameList[i]);
    if (it == allSubProjectsList.end()) {
      projectSettings->removeProject(tmpProjectNameList[i]);
      if (Debug::g_pLog) Debug::g_pLog->log(Debug::LL_INFO,"removing unreferenced sub-project: "+tmpProjectNameList[i]);
    }
  }
}

//----------------------------------------------------------------------------

void CvsControl::setSettings() {
  if (!DirWatch::b_useDirWatch) releaseMonitoring();
  Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA, "LogLevel set to: "+QString::number(Debug::g_logLevel));
}

//----------------------------------------------------------------------------

void CvsControl::startInSingleDir() {//startup option, only called by lincvs.cpp
  addProject(m_tmpStartUpDir,true);
}

//----------------------------------------------------------------------------

void CvsControl::initialDirScanning()//startup option, only called by lincvs.cpp
{
   QStringList projectNameList;
   projectSettings->getTopLevelProjects(projectNameList);

   QStringList removeList;
   unsigned int i;
   for(i = 0; i < projectNameList.count(); i++) {
     QString fullName;
     projectSettings->get(projectNameList[i],WORKDIR,fullName);
     if (addProject(fullName)) {
       
       applyProjectSettings(projectNameList[i]);
       
     } else {//globalStopAction!
       
       setStatusText( tr("Scanning aborted") );
       if (showYesNoQuestion( tr("Interrupted"),
			      tr("Remove project from workbench:")+"\n"+
			      fullName+"?" ) ) {

	 removeList.append(projectNameList[i]);
       }
     }
   }

   if (!removeList.isEmpty()) {
     for(i = 0; i < removeList.count(); ++i) {
       QString fullName;
       projectSettings->get(removeList[i],WORKDIR,fullName);
       Directory * dir = m_pWorkBenchTree->find(fullName);
       if (dir) delete dir;
       projectSettings->removeProject(removeList[i]);
     }
   }

   m_lastOpenedDir = QDir::homeDirPath();
   
   globalStopAction = false;

   initDone();
}

//----------------------------------------------------------------------------

void CvsControl::checkout( QString importLocalDirectory,
			   QString module,
			   QString checkoutAs,
			   QString checkoutAsDOption,
			   QString repository,
			   QString tmpCvsRsh,
			   QString tmpRevTagDate,
			   QString server,
			   QString user,
			   int sshAccess,
			   int mode,
			   int rshMode,
			   bool tmpRWPermission)
{

  m_coImportLocalDirectory = importLocalDirectory;
  QString workdir = importLocalDirectory;
  bool needcvsroot = false;
  QString cvsRoot;
  QString files = "";

  //check ssh access settings
  bUseSsh = false;
  bUseSshAgent = false;
  bUseSshAgentVars = false;
  switch( sshAccess) {
    case USESSH: {
      bUseSsh = true;
      break;
    }
    case USESSHAGENT: {
      bUseSshAgent = true;
      break;
    }
    case USESSHAGENTVARS: {
      bUseSshAgentVars = true;
      break;
    }
  }

  //modify application-wide lists
  adaptQStringList(module,&MODULEHISTORYLIST,HistorySize::g_profiles);
  adaptQStringList(checkoutAs,&CHECKOUTASLIST,HistorySize::g_profiles);
  adaptQStringList(m_coImportLocalDirectory,&WORKDIRHISTORYLIST,HistorySize::g_workdirs);
  adaptQStringList(user, &cvsUsers, HistorySize::g_profiles);
  adaptQStringList(server, &cvsServers, HistorySize::g_profiles);
  adaptQStringList(repository, &cvsRepositories, HistorySize::g_profiles);

  //set CVSROOT      
  if(mode == MODE_PSERVER) { //password server
    //set CVSROOT for pserver mode
    cvsRoot = ":pserver:";
    needcvsroot = true;
  } else if(mode == MODE_RSH) {//remote shell access
    needcvsroot = true;
    if(rshMode == RSH_EXT) {
      cvsRoot = ":ext:";
    } else {
      cvsRoot = ":server:";
    }
  }
  if (server.find(':') == -1) server += ":";
  if (needcvsroot) cvsRoot += user + "@" + server;
  cvsRoot +=  repository;
  
  if (tmpRWPermission) {
    files = "-r ";
  }
  files += CvsOptions::cmprStr();
  files += " co ";
  files += (CvsOptions::g_bPruneDirs ? "-P " : " ");
  
  if (!tmpRevTagDate.isNull()) {
    files += tmpRevTagDate + " ";
  }
  
  if (!checkoutAs.isEmpty()) {
    m_coModule =  checkoutAs;
//     int pos = -1;
//     if ( (pos = module.find("/"))>0) {// maybe adding a subdir to an aliased and allready checkedout module
//       if (QFileInfo(workdir+"/"+checkoutAs).exists()) {
// 	m_coModule = module.mid(++pos);
// 	m_coImportLocalDirectory = workdir+"/"+checkoutAs;
//       }
//     }
    files += "-d " + masqWs(checkoutAs) + " ";
    files += masqWs(module);
  } else if (!checkoutAsDOption.isEmpty()) {
    m_coModule =  checkoutAsDOption;
    files += masqWs(module);
  } else {
    m_coModule =  module;
    files += masqWs(module);
  }
  
  QString topModule = QString::null;

  int CMD;
  if (CvsOptions::g_bPruneDirs) CMD = CHECKOUT_CMD;
  else CMD = CHECKOUT_NO_PRUNE_CMD;

  callInteractiveCmd( topModule, workdir, cvsRoot,
                      files, CMD, tmpCvsRsh, false, this);
}

//----------------------------------------------------------------------------

void CvsControl::finishCheckOut(CvsBuffer* cvsBuffer)
{
   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);

   //check if cvs checkout succeeded
   bool checkoutFailed = true;
   unsigned int len = (*cvsBuffer).numLines();
   for ( unsigned int i = outputLineOffset;i<len;i++) {
     if ((*m_pCvsBuffer).textLine(i).startsWith("U ")) {
       checkoutFailed = false;
       break;
     }
   }
   if (checkoutFailed) return;

   //cvs dir is already in workbench?
   QString topProjectDir = "";
   QString subProjectDir = "";
   bool found = FALSE;
   Directory * myChild = NULL;

   QString importDir = m_coImportLocalDirectory + "/" + m_coModule;
   int pos = m_coModule.length();
   do {
     
     pos = m_coModule.findRev("/",pos-m_coModule.length()-1);
     importDir.truncate((unsigned int)(m_coImportLocalDirectory.length()+1+pos));
     QString module = m_coModule.mid(pos+1);

     if ( (myChild = m_pWorkBenchTree->find(importDir)) ) {
       
       if (myChild->isAnalyzed()) {

	 QString dirToAdd = importDir + "/" + module;
	 dirToAdd = dirToAdd.left(dirToAdd.find("/",myChild->fullName().length()+1));
	 Directory * unanalyzedDir = myChild->addDir(dirToAdd);
	 if (unanalyzedDir && myChild->isOpen()) {//will not be analyzed on open() because it is allready open
	   myChild->setOpen(TRUE);
	 }
	 if (myChild->isSelected()) {
	   myChild->activateItem(TRUE);//reread files
	 } else {
	   m_pWorkBenchTree->setSelected(myChild,TRUE);//reread files
	 }
       }

       showInfo( tr("Checkout Info"),
		 tr("This project subdir has been added.") );
       topProjectDir = myChild->topDir()->shortName();
       Directory * tmp = m_pWorkBenchTree->find(importDir + "/" + module);
       if (tmp) {
	 subProjectDir = tmp->topCvsDir()->relativeName();
       }
       found = TRUE;
     }

   } while ( !found && (pos > -1));


   if (!found) {

     myChild = (Directory*)(m_pWorkBenchTree->firstChild());
     while( myChild) {

       if (myChild->shortName() == m_coModule.left(m_coModule.find("/"))) {
	 showWarning( tr("Warning")+", "+tr("can't add to workbench"),
		      tr("A project with equal name is allready in workbench:")+"\n"+
		      myChild->shortName());
	 return;
       }
       myChild = (Directory*)(myChild->nextSibling());
     }
   }
   if ( !found && ((m_coModule == ".") || (m_coModule == "..")) ) {

     return;//-d option in CVSROOT/modules file

   }

   if (!found) {
     bool res = showYesNoQuestion( tr("Checkout Info"),
				   tr("Add this project to workbench:")+"\n"+
				   m_coModule.left(m_coModule.find("/"))+"?" );
     if (res) {

       Directory * item = new Directory( m_pWorkBenchTree,
					 NULL,
					 m_coImportLocalDirectory+"/"+m_coModule.left(m_coModule.find("/")), //add project, not importdir!
					 m_pCvsFileList,
					 m_pNonCvsFileList,
					 m_pCvsIgnoreFileList );

       item->connect(this,SLOT( slot_requestDirectorySuicide(const QVariant&)) );
       item->setAllToUpToDate();

       //write project settings
       projectSettings->set(item->shortName(),WORKDIR,item->fullName());

       topProjectDir = item->shortName();
       
       //select the dir
       m_pWorkBenchTree->setSelected(item,TRUE);

       //open the dir
       m_lastOpenedDir = item->fullName();
       m_lastOpenedDir.truncate(m_lastOpenedDir.findRev("/"));
       item->setOpen(true);

       if (globalStopAction) globalStopAction = FALSE;

     } else {
       return;
     }
   }

   if (!topProjectDir.isEmpty()) {
     int sshAccess = NOSSH;
     if (bUseSsh) sshAccess = USESSH;
     else if (bUseSshAgent) sshAccess = USESSHAGENT;
     else if (bUseSshAgentVars) sshAccess = USESSHAGENTVARS;
     if (!subProjectDir.isEmpty()) {
       projectSettings->set(subProjectDir,SSHACCESS,sshAccess);
     } else {
       projectSettings->set(topProjectDir,SSHACCESS,sshAccess);
     }
   }
}

//----------------------------------------------------------------------------


/**
 * Exports a CVS module into the repository.
 */
void CvsControl::import( QString importLocalDirectory,
			 QString module,
			 QString importLastComment,
			 QString user,
			 QString server,
			 QString repository,
			 QString tmpCvsRsh,
			 QString tagV,
			 QString tagR,
			 int mode,
			 int rshMode,
			 int sshAccess,
			 QStringList ignoreWildcards,
			 QStringList binaryWildcards)
{

  m_coImportLocalDirectory = importLocalDirectory;
  m_coImportLastComment = importLastComment;
  m_coModule = module;
  bool needcvsroot = false;
  QString cvsRoot;
  QString files;
  
  //modify application-wide lists
  adaptQStringList(module,&MODULEHISTORYLIST,HistorySize::g_profiles);
  adaptQStringList(importLocalDirectory,&WORKDIRHISTORYLIST,HistorySize::g_workdirs);
  adaptQStringList(tagV,&m_coImportVendorTag,20);
  adaptQStringList(tagR,&m_coImportReleaseTag,20);
  adaptQStringList(user, &cvsUsers, HistorySize::g_profiles);
  adaptQStringList(server, &cvsServers, HistorySize::g_profiles);
  adaptQStringList(repository, &cvsRepositories, HistorySize::g_profiles);
  
  //check ssh access settings
  bUseSsh = false;
  bUseSshAgent = false;
  bUseSshAgentVars = false;
  switch( sshAccess) {
    case USESSH: {
      bUseSsh = true;
      break;
    }
    case USESSHAGENT: {
      bUseSshAgent = true;
      break;
    }
    case USESSHAGENTVARS: {
      bUseSshAgentVars = true;
      break;
    }
  }
  
  //set CVSROOT      
  if(mode == MODE_PSERVER) { //password server
    //set CVSROOT for pserver mode
    cvsRoot = ":pserver:";
    needcvsroot = true;
  } else if(mode == MODE_RSH) {//remote shell access
    needcvsroot = true;
    if(rshMode == RSH_EXT) {
      cvsRoot = ":ext:";
    } else {
      cvsRoot = ":server:";
    }
  }
  if (server.find(':') == -1) server += ":";
  if (needcvsroot) cvsRoot += user + "@" + server;
  cvsRoot +=  repository;
  files = "-r " + CvsOptions::cmprStr();
  
  files += " import ";
  
  //ignore files:
  QString sIgnore = "-I ! ";
  if (!ignoreWildcards.empty()) {
    sIgnore += "-I \"";
    sIgnore += ignoreWildcards.join("\" -I \"");
    sIgnore += "\" ";
  }
  files += sIgnore;
  
  //binary import files:
  if (!binaryWildcards.empty()) {
    QString sBin = "-W \"";
    sBin += binaryWildcards.join(" -k 'b'\" -W \"");
    sBin += " -k 'b'\" ";
    files += sBin;
  }
  
  //add commitinfo
  files += "-m ";
  files += "\"" + I18n::g_pTextEncoder->encode(m_coImportLastComment) + "\" ";
  
  files += masqWs(m_coModule) + " " + tagV + " " + tagR;
  QString topModule = QString::null;
  callInteractiveCmd( topModule, m_coImportLocalDirectory, cvsRoot,  files, IMPORT_CMD, tmpCvsRsh);
}

//----------------------------------------------------------------------------

void CvsControl::importVendor( QString topModule, 
			       QString cvsRoot,
			       QString repository,
			       QString importLocalDirectory,
			       QString importLastComment,
			       QString tagV,
			       QString branchV,
			       QString tagR,
			       QStringList ignoreWildcards,
			       QStringList binaryWildcards)
{

  //save new template file
  if (m_coImportLastComment.stripWhiteSpace() != importLastComment.stripWhiteSpace()) {
    writeUserHstFile(importLastComment);
    m_coImportLastComment = importLastComment;
  }

  QString cmd = "import ";

  if (!branchV.isEmpty()) cmd += "-b " + branchV + " ";
  
  //ignore files:
  QString sIgnore = "-I ! ";
  if (!ignoreWildcards.empty()) {
    sIgnore += "-I \"";
    sIgnore += ignoreWildcards.join("\" -I \"");
    sIgnore += "\" ";
  }
  cmd += sIgnore;
  
  //binary import files:
  if (!binaryWildcards.empty()) {
    QString sBin = "-W \"";
    sBin += binaryWildcards.join(" -k 'b'\" -W \"");
    sBin += " -k 'b'\" ";
    cmd += sBin;
  }
  
  //add commitinfo
  cmd += "-m ";
  cmd += "\"" + importLastComment + "\" ";
  cmd += masqWs(repository) + " " + tagV + " " + tagR;

  callInteractiveCmd( topModule,
		      importLocalDirectory,
		      cvsRoot,
		      cmd,
		      IMPORT_CMD,
		      ExtApps::g_cvsRsh.path  //additional options of cvsRsh not supported yet
		      );
}

//----------------------------------------------------------------------------

bool CvsControl::addNewProject(const QString fileName, bool expand) {

  QStringList projectNameList;
  projectSettings->getProjects(projectNameList);
  unsigned int i;
  for(i=0; i < projectNameList.count(); i++) {
    QString fullName;
    projectSettings->get(projectNameList[i],WORKDIR,fullName);

    if ( fileName.mid(fileName.findRev("/")+1) == projectNameList[i]) {
      showWarning(tr("Warning")+", "+tr("adding aborted"),
		  tr("This project is already in workbench:\n")
		  +projectNameList[i]
		  +"\nat: "+fullName);
      setStatusText( tr("Adding aborted"), 4000 );
      return FALSE;
    }
  }

  setStatusText("");

  Directory * item = new Directory ( m_pWorkBenchTree, 
				     NULL,
				     fileName, 
				     m_pCvsFileList, 
				     m_pNonCvsFileList,
				     m_pCvsIgnoreFileList
				     );

  item->connect(this,SLOT( slot_requestDirectorySuicide(const QVariant&)) );

  projectSettings->set(item->shortName(),WORKDIR,item->fullName());

  m_lastOpenedDir = item->fullName();
  m_lastOpenedDir.truncate(m_lastOpenedDir.findRev("/"));

  if(expand) {
    item->setOpen(true);
  }
  
  if (globalStopAction) {
    setStatusText(tr("Adding aborted"));
    globalStopAction = false;
    return FALSE;
  } else {
    m_pWorkBenchTree->setSelected(item,TRUE);
    slot_checkStatusOfTree(item);
    return TRUE;
  }
}

//----------------------------------------------------------------------------

bool CvsControl::addProject(const char * fileName, bool expand)
{
   bool addNewProject = false;

   QString fn = fileName;
   if(fn.isEmpty()){//popup dialog
      addNewProject = true;
      expand = true;

      /* Using the Qt standard file dialog preserves the look and feel
       * of the application across platforms.
       */
      QString defaultDir = QString::null;
      if ( (m_pCurCvsDir = static_cast<Directory*>(m_pWorkBenchTree->currentItem())) ) {
	defaultDir = m_pCurCvsDir->fullName();
	defaultDir = defaultDir.left(defaultDir.findRev("/"));
      }

      blockGUI(false);
      fn = QFileDialog::getExistingDirectory (defaultDir, this, NULL,
					      tr("Add directory to workbench"),
					      true);
      blockGUI(true);

      // Remove trailing "/" from the path.
      fn.replace (QRegExp ("/+$"), "");
#ifdef Q_WS_WIN
      fn.replace (QRegExp ("\\\\"), "/");
#endif
   }
   
   if ( !fn.isEmpty() ) {

      if (addNewProject) {
	QStringList projectNameList;
	projectSettings->getProjects(projectNameList);
	unsigned int i;
	for(i=0; i < projectNameList.count(); i++) {
	  QString fullName;
	  projectSettings->get(projectNameList[i],WORKDIR,fullName);

	  if ( fn.mid(fn.findRev("/")+1) == projectNameList[i]) {
	    showWarning(tr("Warning")+", "+tr("adding aborted"),
			tr("This project is already in workbench:\n")
			+projectNameList[i]
			+"\nat: "+fullName);
	    setStatusText( tr("Adding aborted"), 4000 );
	    return FALSE;
	  }
	}
      }

      setStatusText("");

      Directory * item = new Directory ( m_pWorkBenchTree, 
					 NULL,
					 fn, 
					 m_pCvsFileList, 
					 m_pNonCvsFileList,
					 m_pCvsIgnoreFileList
					 );

      item->connect(this,SLOT( slot_requestDirectorySuicide(const QVariant&)) );

      if(addNewProject){
	projectSettings->set(item->shortName(),WORKDIR,item->fullName());
      }
      m_lastOpenedDir = item->fullName();
      m_lastOpenedDir.truncate(m_lastOpenedDir.findRev("/"));
      if(expand) {
	item->setOpen(true);
      }

      if (globalStopAction) {
	setStatusText(tr("Adding aborted"));
	globalStopAction = false;
	return FALSE;
      } else {
	if (!item->isDisabled()) {
	  m_pWorkBenchTree->setSelected(item,TRUE);
	  slot_checkStatusOfTree(item);
	}
	return TRUE;
      }
   }
   else setStatusText(tr("Adding aborted"));
   return FALSE;
}

//----------------------------------------------------------------------------

void CvsControl::slot_requestDirectorySuicide(const QVariant &name) {
  QString dirName = name.toString();
  m_DirsToKill.append(dirName);
  Timer::singleShot(0,this,SLOT(slot_execDirectorySuicide()) );
}

//----------------------------------------------------------------------------

void CvsControl::slot_execDirectorySuicide() {
  if (!m_DirsToKill.isEmpty()) {
    QString name = m_DirsToKill.first();
    m_DirsToKill.pop_front();
    Directory * dir = m_pWorkBenchTree->find(name);
    if (dir) {
      Directory * parentDir = dir->parent();
      if (parentDir) parentDir->removeChild(dir);
    }
  }
}

//----------------------------------------------------------------------------

void CvsControl::removeProjectSlot()
{
    Directory *dir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
    if(!dir) {
        showWarning(tr("Warning"), tr("There is no directory selected") );
        return;
    }

    if (dir != dir->topDir()) {
      showInfo( tr("Information"), 
		tr("This Directory is not a top level dir.") );
      return;
    }

    QString topModule = dir->shortName();
    projectSettings->removeProject(topModule);
    delete dir;
}

/*---------------------------------------------------------------------------*/
/*!
\fn			void CvsControl::disableProjectSlot(void)
\brief		Disable a dir at top level form workbench.
          Dir is still listed but not further updated and scanned until enabled
          again. Entry is inserted in file list in cfg file section DISABLED.


<BR><HR>*/
/*---------------------------------------------------------------------------*/
void CvsControl::disableProjectSlot(void) //only called from lincvs.cpp
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    bool blocked = isGuiBlocked();
    if (!blocked) blockGUI(true);
    QString topCvsModule = m_pCurCvsDir->topCvsDir()->relativeName();
    QString dirname = m_pCurCvsDir->fullName();
    QStringList disabled;
    projectSettings->get(topCvsModule,DISABLED,disabled);
    disabled.append(dirname);
    projectSettings->set(topCvsModule,DISABLED,disabled);
    m_pCurCvsDir->topCvsDir()->updateDisabledList(disabled);

    Directory *hlp = m_pCurCvsDir->parent();
    if (hlp) {
	hlp->reloadChild(dirname);
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->find( dirname);
	m_pWorkBenchTree->setSelected( m_pCurCvsDir, TRUE );
	slot_checkStatusOfTree(m_pCurCvsDir,OVERRIDE);
    }
    else {
	delete m_pCurCvsDir;
	addProject(dirname.latin1());
	applyProjectSettings(topCvsModule);
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->find( dirname);
	m_pWorkBenchTree->setSelected( m_pCurCvsDir, TRUE );
    }
    if (!blocked) blockGUI(false);
}


/*---------------------------------------------------------------------------*/
/*!
\fn			void CvsControl::enableProjectSlot()
\brief		Reread project after reenable.

<BR><HR>*/
/*---------------------------------------------------------------------------*/
void CvsControl::enableProjectSlot() //only called from lincvs.cpp
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    bool blocked = isGuiBlocked();
    if (!blocked) blockGUI(true); // takes some time on huge dirs
    QString topCvsModule = m_pCurCvsDir->topCvsDir()->relativeName();
    QString dirname = m_pCurCvsDir->fullName();
    QStringList disabled;
    projectSettings->get(topCvsModule,DISABLED,disabled);
    disabled.remove(dirname);
    projectSettings->set(topCvsModule,DISABLED,disabled);
    m_pCurCvsDir->topCvsDir()->updateDisabledList(disabled);

    Directory *hlp = (Directory *)m_pCurCvsDir->parent();
    if (hlp) {
	m_pCurCvsDir = hlp->reloadChild(dirname);
	if (m_pCurCvsDir == m_pCurCvsDir->topCvsDir()) applyProjectSettings(topCvsModule);
	if (!m_pCurCvsDir->isDisabled()) {
	  slot_checkStatusOfTree(m_pCurCvsDir);
	}
    }
    else {
	delete m_pCurCvsDir;
	addProject(dirname.latin1());// calls slot_checkStatusOfTree()
	m_pCurCvsDir = (Directory *)m_pWorkBenchTree->find( dirname);
	applyProjectSettings(topCvsModule);
    }
    m_pWorkBenchTree->setSelected(m_pCurCvsDir,TRUE);
    m_pCurCvsDir->activateItem(FALSE);
    if (!blocked) blockGUI(false);
}

/*---------------------------------------------------------------------------*/

void CvsControl::performRereadProject( QString moduleName, QString fullName) {
      setStatusText( "");
      addProject(fullName);
      projectSettings->set(moduleName,WORKDIR,fullName);
      setStatusText( tr("Ready"));
}

//----------------------------------------------------------------------------

void CvsControl::rereadProject() {

   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   bool blocked = isGuiBlocked();
   if (!blocked) blockGUI(true);
   m_pCurCvsDir = rereadProjectOfDir(m_pCurCvsDir);
   if (!blocked) blockGUI(false);
}

//----------------------------------------------------------------------------

// reopen tree, returns deepest opened dir in hierarchy or null, if not found in workbench
Directory * CvsControl::openTreeAt(const QString &topDirName,const QString &dirToOpen) {

  Directory * dir = NULL;
  if (ONTHEFLYSCANNING && (topDirName != dirToOpen)) {//can't find dir since it's not scanned jet!
    int from = topDirName.length();
    while (from >= 0) {
      dir = (Directory *)m_pWorkBenchTree->find(dirToOpen.left( (unsigned int)from));
      if (dir) {
	dir->setOpen(TRUE);//This will scan!
      }
      from = dirToOpen.find('/',from+1);
    }
  }
  dir = (Directory *)m_pWorkBenchTree->find( dirToOpen );
  if (dir) {
    if (ONTHEFLYSCANNING) dir->setOpen(TRUE);//This will scan!
    else dir->setAllOpen();
  }
  slot_checkStatusOfTree(m_pWorkBenchTree->find(topDirName));
  return dir;
}

//----------------------------------------------------------------------------

Directory * CvsControl::openTreeAndActivate( const QString &topDirName, QString dirToOpenAfterRescan) {
  // reopen tree
  int pos = dirToOpenAfterRescan.findRev("/");
  if (pos > (int)topDirName.length()) {
    openTreeAt(topDirName,dirToOpenAfterRescan.left(pos) );
  }
  //activate dir
  Directory * dir = NULL;
  dir = (Directory *)m_pWorkBenchTree->find( dirToOpenAfterRescan );
  if( dir )  {
    m_pWorkBenchTree->setSelected( dir, TRUE );
    dir->activateItem(FALSE);
    return dir;
  }
  return NULL;
}

//----------------------------------------------------------------------------

void CvsControl::applyProjectSettings(QString projectName) {
  if (projectName.isEmpty()) {//apply settings of all existing projects
    QStringList projectNameList;
    projectSettings->getTopLevelProjects(projectNameList);
    unsigned int i;
    for(i = 0; i < projectNameList.count(); ++i) {
      projectName = projectNameList[i];
      QStringList subProjectNameList;
      projectSettings->getSubProjects(projectName,subProjectNameList);

      Directory * dir;
      QString workdir;
      int autoupdate;
      bool autoUpdateInSubProject = FALSE;

      unsigned int j;
      for (j = 0; j < subProjectNameList.count(); ++j) {
	autoupdate = OFF;
	projectSettings->get(subProjectNameList[j],WORKDIR,workdir);
	projectSettings->get(subProjectNameList[j],PAUTOUPDATE,autoupdate);
	dir = (Directory *)m_pWorkBenchTree->find(workdir);
	if (dir) {
	  bool state = AUTOUPDATE && (autoupdate != OFF);
	  autoUpdateInSubProject |= state;
	  dir->setAutoUpdate(state);
	}
      }
      autoupdate = OFF;
      projectSettings->get(projectName,PAUTOUPDATE,autoupdate);
      projectSettings->get(projectName,WORKDIR,workdir);
      dir = (Directory *)m_pWorkBenchTree->find(workdir);
      if (dir) {
	dir->setAutoUpdate(AUTOUPDATE && ( (autoUpdateInSubProject) || (autoupdate != OFF) ) );
      }
    }
  } else {//apply only settings of 'projectName'
    QStringList subProjectNameList;
    projectSettings->getSubProjects(projectName,subProjectNameList);

    Directory * dir;
    QString workdir;
    int autoupdate;
    bool autoUpdateInSubProject = FALSE;

    unsigned int j;
    for (j = 0; j < subProjectNameList.count(); ++j) {
      autoupdate = OFF;
      projectSettings->get(subProjectNameList[j],WORKDIR,workdir);
      projectSettings->get(subProjectNameList[j],PAUTOUPDATE,autoupdate);
      dir = (Directory *)m_pWorkBenchTree->find(workdir);
      if (dir) {
	bool state = AUTOUPDATE && (autoupdate != OFF);
	autoUpdateInSubProject |= state;
	dir->setAutoUpdate(state);
      }
    }
    autoupdate = OFF;
    projectSettings->get(projectName,PAUTOUPDATE,autoupdate);
    projectSettings->get(projectName,WORKDIR,workdir);
    dir = (Directory *)m_pWorkBenchTree->find(workdir);
    if (dir) {
      dir->setAutoUpdate(AUTOUPDATE && ( (autoUpdateInSubProject) || (autoupdate != OFF) ) );
    }
  }
}

//----------------------------------------------------------------------------

void CvsControl::stopTimers(int type /* = TALL */) {
  if (type == TALL || type == TSTATUS) {
    m_timer.stop();
  }
  if (type == TALL || type == TUPDATE) {
    m_updateTimer.stop();
  }
}

//----------------------------------------------------------------------------

void CvsControl::startTimers(int type /* = TALL */) {
  if (type == TALL || type == TSTATUS) {
    if (Polling::checkStatusLevel>Polling::NONE && (!DirWatch::b_isActive) ) {
      if (!m_timer.isActive()) {
	m_timer.start(CHECKSTATUSINTERVALL);
      }
    }
  }
  if (type == TALL || type == TUPDATE) {
    if (!m_updateTimer.isActive() && AUTOUPDATE) {
      m_updateTimer.start(AUTOUPDATEINTERVALL*1000*60);
    }
  }
}

//----------------------------------------------------------------------------

/**
 * Use this Method to check project state and to start the state timer
 * It will stop and restart the timer 
 * appropriate according to the Settings
 */
void CvsControl::slot_checkStatus(bool override /*=FALSE*/) {

  checkInProgress(TRUE);

  stopTimers(TSTATUS);
  bool doBlock = !isGuiBlocked();
  if (doBlock) blockGUI(true,BLOCKNOCONTROLS | BLOCKNOCURSOR);
  Directory *dir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
  if ( dir && dir->isSelected() ) {

    if ( dir->isModified() ) {
      saveViewportPos(override);
      dir->activateItem(TRUE);
      recoverViewportPos();
    } else {

      QListViewItem *myChild = m_pCvsFileList->firstChild();
      while(myChild) {
	if(m_pCvsFileList->itemRect(myChild).isValid()) {
	  dir->checkAndShowStatus(static_cast<FileListViewItem*>(myChild));   
	}
	myChild = myChild->nextSibling();
      }
    }
    if (Polling::checkStatusLevel >= Polling::ALLDIRS) {
      dir =  dir->topDir();
      dir->recCheckForModifications(Polling::checkStatusLevel >= Polling::ALLFILES);
    }
  }

  if (Polling::checkStatusLevel >= Polling::ALLPROJECTS) {
    QListViewItem *myChild = m_pWorkBenchTree->firstChild();
    while(myChild) {
      if (myChild != dir) {
	((Directory *)myChild)->recCheckForModifications(Polling::checkStatusLevel >= Polling::ALLFILES);
      }
      myChild = myChild->nextSibling();
    }
  }

  checkInProgress(FALSE);

  startTimers(TSTATUS);
  m_timer.pause();
  if (doBlock) blockGUI(false,BLOCKNOCONTROLS | BLOCKNOCURSOR);
  
}

//----------------------------------------------------------------------------

void CvsControl::slot_checkStatusOfTree(Directory * dir, bool override /*=FALSE*/) {

  checkInProgress(TRUE);

  if ( dir && dir->isSelected() ) {

    if ( dir->isModified() ) {
      saveViewportPos(override);
      dir->activateItem(TRUE);
      recoverViewportPos();
    } else {

      QListViewItem *myChild = m_pCvsFileList->firstChild();
      while(myChild) {
	if(m_pCvsFileList->itemRect(myChild).isValid()) {
	  dir->checkAndShowStatus(static_cast<FileListViewItem*>(myChild));   
	}
	myChild = myChild->nextSibling();
      }
    }
  }

  dir->recCheckForModifications(TRUE);

  checkInProgress(FALSE);
}

//----------------------------------------------------------------------------

void CvsControl::dirConnectorDataNotify() {
  Directory *dir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
  //if(!dir) return; <-- not needed because we want to check dirs even if we have no dirs selected

  Directory * tmpDir = NULL;
  QString fileName;
  bool deleted;
  blockGUI(true,BLOCKNOCONTROLS | BLOCKNOCURSOR);
  while ( (tmpDir = checkMonitoredEvent(fileName,deleted))) {
//     qDebug("Notify Dir: "+tmpDir->fullName()+", file: "+fileName+", deleted: "+QString(deleted ? "true" : "false"));
    //tmpDir != NULL --> call only dir->activateItem if dir != NULL too
    if (dir == tmpDir) {
      saveViewportPos();
      dir->activateItem(TRUE);
      recoverViewportPos();
    } else {
      tmpDir->checkAndShowStatus(NULL,TRUE,TRUE);
    }
  }
  blockGUI(false,BLOCKNOCONTROLS | BLOCKNOCURSOR);
}

//----------------------------------------------------------------------------

bool CvsControl::slot_checkAutoUpdate() {

  if (!isGuiBlocked()) {
    stopTimers(TUPDATE);
    blockGUI(true);
  }

  static QString lastChecked = "";
  static QString lastSubChecked = "";
  static Directory* dir = NULL;
  static int lastCmd = -1;

  if (dir) {//parse last call and set state without refreshing the file pane
    switch(lastCmd) {
      case OFF: {
	break;
      }
      case QUERY: {
	dir->parseCvsCallResult(m_pCvsBuffer, QUERY_UPDATE_CMD);
	break;
      }
      case QUERYALL: {
	dir->parseCvsCallResult(m_pCvsBuffer, QUERY_UPDATE_ALL_CMD);
	break;
      }
      case STATUS: {
	dir->parseCvsCallResult(m_pCvsBuffer, STATUS_CMD);
	break;
      }
      case UPDATE: {
	dir->parseCvsCallResult(m_pCvsBuffer, UPDATE_DIR_CMD);
	break;
      }
      case UPDATEALL: {
	dir->parseCvsCallResult(m_pCvsBuffer, UPDATE_DIR_CMD);
	break;
      }
      default: break;
    }
  }

  QStringList projectNameList;
  projectSettings->getTopLevelProjects(projectNameList);

  int from = 0;
  if (!lastChecked.isEmpty()) {
    if (lastSubChecked.isEmpty()) {
      from = projectNameList.findIndex( lastChecked) + 1;
    } else {
      from = projectNameList.findIndex( lastChecked);
    }
  }

  for(unsigned int i = (unsigned int)from; i < projectNameList.count(); i++) {

    QString moduleName = projectNameList[i];
    QStringList subProjectNameList;
    projectSettings->getSubProjects(moduleName,subProjectNameList);

    if (subProjectNameList.isEmpty()) {

      if (autoUpdate(moduleName,&dir,lastCmd)) {
	lastChecked = moduleName;
	return true;//call successfull
      }

    } else {

      int subFrom = 0;
      if (!lastSubChecked.isEmpty()) {
	subFrom = subProjectNameList.findIndex( lastSubChecked) + 1;
      }
    
      for(unsigned int j = (unsigned int)subFrom; j < subProjectNameList.count(); j++) {
      
	QString subProject = subProjectNameList[j];
	if (autoUpdate(subProject,&dir,lastCmd)) {
	  lastChecked = moduleName;
	  lastSubChecked = subProject;
	  return true;//call successfull
	}
      }
      lastSubChecked = "";

      if (autoUpdate(moduleName,&dir,lastCmd)) {
	lastChecked = moduleName;
	return true;//call successfull
      }

    }
  }

  lastChecked = "";
  lastCmd = -1;
  dir = NULL;
  recoverViewportPos();

  blockGUI(false);
  startTimers(TUPDATE);

  return false;
}

//----------------------------------------------------------------------------

bool CvsControl::autoUpdate(QString topModule, Directory** callDir, int &lastCmd) {

  int autoupdate;
  QString workdir;
  QStringList disabled;
  Directory * dir = NULL;

  projectSettings->get(topModule,WORKDIR,workdir);
  projectSettings->get(topModule,DISABLED,disabled);

  if (disabled.find(workdir) != disabled.end()) return FALSE;
  dir = (Directory *)m_pWorkBenchTree->find(workdir);
  if (dir && !dir->isDisabled() ) {
    projectSettings->get(topModule,PAUTOUPDATE,autoupdate);
  } else return FALSE;
  *callDir = dir;

  switch(autoupdate) {
    case OFF: {
      break;
    }
    case QUERY: {

      lastCmd = QUERY;

      //run the cvs call
      QString command = "";
      if (!bRWPermission) command += "-r ";
      command += CvsOptions::cmprStr() + " -n update ";
      command += CvsOptions::g_bPruneDirs ? "-P" : "";
	
      QString files = "";

      int CMD;
      if (CvsOptions::g_bPruneDirs) CMD = AUTOUPDATE_DIR_CMD;
      else CMD = AUTOUPDATE_DIR_NO_PRUNE_CMD;
      callInteractiveCmd( topModule, workdir, command,
                          files, CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
      return true;

      break;
    }
    case QUERYALL: {

      lastCmd = QUERYALL;

      //run the cvs call
      QString command = "";
      if (!bRWPermission) command += "-r ";
      command += CvsOptions::cmprStr() + " -n update ";
      command += CvsOptions::g_bPruneDirs ? "-P" : "";
      command += " -d ";
	
      QString files = "";

      int CMD;
      if (CvsOptions::g_bPruneDirs) CMD = AUTOUPDATE_DIR_CMD;
      else CMD = AUTOUPDATE_DIR_NO_PRUNE_CMD;
      callInteractiveCmd( topModule, workdir, command,
                          files, CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
      return true;

      break;
    }
    case STATUS: {

      lastCmd = STATUS;

      //run the cvs call
      QString command = "status -v";
	
      QString files = "";

      callInteractiveCmd( topModule, workdir, command,
                          files, AUTOUPDATE_DIR_CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
      return true;

      break;
    }
    case UPDATE: {

      lastCmd = UPDATE;

      //run the cvs call
      QString command = "";
      if (!bRWPermission) command += "-r ";
      command += CvsOptions::cmprStr() + " update ";
      command += CvsOptions::g_bPruneDirs ? "-P" : "";
	
      QString files = "";

      int CMD;
      if (CvsOptions::g_bPruneDirs) CMD = AUTOUPDATE_DIR_CMD;
      else CMD = AUTOUPDATE_DIR_NO_PRUNE_CMD;
      callInteractiveCmd( topModule, workdir, command,
                          files, CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
      return true;

      break;
    }
    case UPDATEALL: {

      lastCmd = UPDATE;

      //run the cvs call
      QString command = "";
      if (!bRWPermission) command += "-r ";
      command += CvsOptions::cmprStr() + " update ";
      command += CvsOptions::g_bPruneDirs ? "-P" : "";
      command += " -d ";
	
      QString files = "";

      int CMD;
      if (CvsOptions::g_bPruneDirs) CMD = AUTOUPDATE_DIR_CMD;
      else CMD = AUTOUPDATE_DIR_NO_PRUNE_CMD;
      callInteractiveCmd( topModule, workdir, command,
                          files, CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
      return true;

      break;
    }
    default: break;
  }
  lastCmd = -1;
  return FALSE;
}

//----------------------------------------------------------------------------

void CvsControl::test() { /* -->please keep test() method for testing,
				    slot is enabled if .lincvsrc has <DEBUG = true> entrie<-- */

  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }
  recValidateCvsFilesStatus(false,true);
}

void CvsControl::slot_onlineHelp() {
  runExternal("\"" + ExtApps::g_remoteBrowser.path +"\" "
              + ExtApps::g_remoteBrowser.options + " "
              + tr("http://cvsbook.red-bean.com/cvsbook.html"));
}

void CvsControl::slot_debug() {

  QString debugLogFile;
  QFileDialog* fd = new QFileDialog( QDir::currentDirPath(), tr("Debug Log (*.log)"), this, "file dialog", TRUE );
  fd->setCaption(LC_APPNAME);
  fd->setMode( QFileDialog::AnyFile );
  fd->setSelection("debug.log");
  if ( fd->exec() == QDialog::Accepted) {
    debugLogFile = fd->selectedFile();
    if ( debugLogFile.isEmpty()) return;
  } else {
    return;
  }

  QFile file(debugLogFile);
  if (file.exists()) {
    if (!showYesNoQuestion( tr("Warning"), 
			    tr("The filename:") + " "
			    + debugLogFile + " "
			    + tr("already exists.")
			    + "\n" 
			    + tr("Overwrite it?")) )
      return;
    
    file.remove();
  }

  if (file.open(IO_WriteOnly)) {
    QTextStream s(&file);                        // serialize using f
    s << "----------start debug log------------\n";
    s << "Date: "+QDateTime::currentDateTime().toString(Qt::ISODate)+"\n";
    s << "LinCVS version: "+getVersion()+"\n";
#ifdef Q_WS_WIN
    s << "for Windows\n";
#else
#ifdef Q_WS_MAC
    s << "for Mac OS X\n";
#else
    s << "for Unix\n";
#endif
#endif
    s << "Compile Time: "+getCompileTime()+"\n";
    fillCmdDebugStream(s);
    s << "----------end debug log------------";
    file.close();
  } else {
    showWarning( tr("Error:"),
		 tr("No write access to: ") + debugLogFile);
  }
}

//----------------------------------------------------------------------------

void CvsControl::fillCmdDebugStream(QTextStream &s) {

  s << m_debugCvsCallString;
  s << "outputLineOffset: "+QString::number(outputLineOffset)+"\n";
  s << "CvsBuffer: ->"+m_pCvsBuffer->readAll()+"<-\n";
  s << "==================================================================\n";
  s << "CvsBuffer single line mode: \n";
  unsigned int len = (*m_pCvsBuffer).numLines();
  for ( unsigned int i = outputLineOffset;i<len;i++) {
      s << "->"+(*m_pCvsBuffer).textLine(i)+"<-\n";
  }
}

//----------------------------------------------------------------------------

// use this if visible cvs output doesn't match CVS/Repository i.e. merge with -j xxx -j xxx
void CvsControl::recValidateCvsFilesStatus(bool forceCacheReset /*= FALSE */, bool forceEntriesReset /*= FALSE */) {

    // iter down all dirs, check state of folders and set entries
    if(!m_pCurCvsDir) return;
    qApp->setOverrideCursor(waitCursor); // takes some time on huge dirs
    m_pCurCvsDir->validateCvsFilesStatus(TRUE,forceCacheReset,forceEntriesReset);
    qApp->restoreOverrideCursor();
}

//----------------------------------------------------------------------------

void CvsControl::statusDir() {

   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();
   QString cvsRoot = "status";
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, STATUS_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::updateProjectTagList() {

  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }
  Directory * topDir = m_pCurCvsDir->topCvsDir();
  QString dir = topDir->fullName();
  QString cvsRoot = "status -v";
  QString files = "";
  QString topModule = topDir->relativeName();
  callInteractiveCmd( topModule, dir, cvsRoot,
                      files, GET_TAG_INFO_PROJECT_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::readCurrentTagList() {

  m_currentTagList.clear();
  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }
  QString files = getNamesOfSelectedFilesWithoutSelCheck();
   
  QString dir = m_pCurCvsDir->fullName();
  QString cvsRoot = "status -v";
  QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
  callInteractiveCmd( topModule, dir, cvsRoot,
                      files, GET_TAG_INFO_FILES_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::updateAny() {
   // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         updateFile();
         return;
      }
      myChild = myChild->itemBelow();
   }
   
   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      updateDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::mergeAny() {
   // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         mergeFile();
         return;
      }
      myChild = myChild->itemBelow();
   }
   
   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      mergeDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::queryUpdateDir()
{
    updateDir(true, false);
}

//----------------------------------------------------------------------------

void CvsControl::updateDir()
{
    updateDir(false, false);
}

//----------------------------------------------------------------------------

void CvsControl::queryUpdateAllDir()
{
    updateDir(true,true);
}

//----------------------------------------------------------------------------

void CvsControl::updateAllDir()
{
    updateDir(false,true);
}

//----------------------------------------------------------------------------

void CvsControl::updateDir(bool query, bool dOption)
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() +
     (query ? " -n " : " ") + "update " + 
     (CvsOptions::g_bPruneDirs ? "-P " : "");
   command += dOption ? "-d " : "";

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   int CMD;
   if (query) {
     if (dOption) {
       CMD = QUERY_UPDATE_ALL_CMD;
     } else {
       CMD = QUERY_UPDATE_CMD;
     }
   } else {
     CMD = UPDATE_DIR_CMD;
   }

   callInteractiveCmd( topModule, dir, command, files, 
		       CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::updateVirtualDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() +
     " update -d " + 
     (CvsOptions::g_bPruneDirs ? "-P " : "");

   QString dir;
   QString files;
   m_newFileNameList.clear();
   if (m_pCurCvsDir->isVirtual()) {
     files = m_pCurCvsDir->fullName();
     m_pCurCvsDir = m_pCurCvsDir->topDir()->searchLastValidDirOfPath(m_pCurCvsDir->fullName());
     dir = m_pCurCvsDir->fullName();
     m_newFileNameList.append(files);
     files = files.mid(dir.length()+1);
     int pos = files.find("/");
     if (pos > -1) {
       files.truncate(pos);//update first virtual dir in hierarchy
     }
   } else {
     dir = m_pCurCvsDir->fullName();
     files = getNamesOfSelectedFilesWithoutSelCheck();
     m_newFileNameList = QStringList::split(" ",files);
     QStringList::Iterator iter;
     for (iter = m_newFileNameList.begin(); iter != m_newFileNameList.end(); iter++) {
       (*iter).replace( QRegExp("\""),"");
     }
   }
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   callInteractiveCmd( topModule, dir, command, files, 
		       UPDATE_VIRTUAL_DIR_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::replaceDir()
{
    replaceDir(false);
}

//----------------------------------------------------------------------------

void CvsControl::replaceAllDir()
{
    replaceDir(true);
}

//----------------------------------------------------------------------------

void CvsControl::replaceDir(bool dOption)
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() + " update -C " +
     (CvsOptions::g_bPruneDirs ? "-P " : "");
   command += dOption ? "-d " : "";

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   int CMD;
   CMD = UPDATE_DIR_CMD;

   callInteractiveCmd( topModule, dir, command,
                       files, CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::updateRevDir(QString cvsTag)
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = "";                                             
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() + " update " +
     (CvsOptions::g_bPruneDirs ? "-P " : "");
   command += CvsOptions::g_bBringOverNewDirs ? "-d " : "";
   command += cvsTag;

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   callInteractiveCmd( topModule, dir, command,
                       files, UPDATE_DIR_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::revertToRevDir(QString param /*=QString::null*/) {

  static QString topModule;
  static QString dir;
  static QString files;
  static QString module;
  static QString tag;
  static QString updateParam;
  static QString cvsRoot;
  static int state;

  QString command;
  if (!param.isNull()) {
    if (!(m_pCurCvsDir = getSelectedDir())) {
      return;
    }
    dir = m_pCurCvsDir->fullName();
    files = "";
    module = m_pCurCvsDir->repository();
    topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    cvsRoot = m_pCurCvsDir->getCvsRoot();
    updateParam = param;
    if (updateParam.startsWith(":")) state = 0;
    else state = 1;
    tag = "TMP-TAG-"+m_pCurCvsDir->userName()+"-"+QDateTime::currentDateTime().toString("yyyyMMddhhmmzzz")+"-";
    command = "tag "+tag+"CURRENT";
  } else {
    switch( state) {
      case 1: {
	command = "-d \""+cvsRoot+"\" rtag -D \""+updateParam.mid(1)+"\" "+tag+"TO \""+module+"\"";
	break;
      }
      case 2: {
	command = "";
	if (!bRWPermission) command += "-r ";
	command += CvsOptions::cmprStr() + " update " +
	  (CvsOptions::g_bPruneDirs ? "-P " : "");
	command += CvsOptions::g_bBringOverNewDirs ? "-d " : "";
	if (updateParam.startsWith(":")) command += "-j \""+tag+"CURRENT\" -j \"" + tag + "TO\"";
	else {
	  command += "-j \""+tag+"CURRENT\" -j \"" + updateParam + "\"";
	  ++state;
	}
	break;
      }
      case 3: {
	command = "-d \""+cvsRoot+"\" rtag -d \""+tag+"TO\" \""+module+"\"";
	break;
      }
      case 4: {
	command = "tag -d "+tag+"CURRENT";
	break;
      }
      case 5: {
	m_pCurCvsDir = m_pWorkBenchTree->findBestMatch(dir);
	if (m_pCurCvsDir) {
	  m_pCurCvsDir->recRemoveTmpEntries(FALSE);
	  m_pWorkBenchTree->setSelected(m_pCurCvsDir,true);
	  m_pCurCvsDir->activateItem(TRUE);
	  recoverViewportPos();
	  return;
	}
	break;
      }
      default: {
	qDebug("revertToRevDir: should never reach this line");
	return;
      }
    }
  }
  ++state;
  callInteractiveCmd( topModule, dir, command,
                      files, REVERT_DIR_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::mergeDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   MergeDialogImpl dlg(m_pTools->whatsThisIconSet(),
		       this, "MergeBranches", 
		       getCurrentTagList(), topModule, true);

   if (dlg.exec()) {

       QString command = "";
       if (!bRWPermission) command += "-r ";
       command += CvsOptions::cmprStr();
       if (dlg.m_bOnlyQuery->isChecked()) command += " -n";
       command += " update ";
       if (!dlg.m_bOnlyQuery->isChecked() && bKeywordSuppression) command += "-kk ";
       command += (CvsOptions::g_bPruneDirs ? "-P " : "");
       command += dlg.DOptionBox->isChecked() ? "-d " : "";
       command += "-j ";
       command += dlg.m_pFirstTag->currentText();
       if (dlg.m_bUseSecondTag->isChecked()) command += " -j " + dlg.m_pSecondTag->currentText();
       
       QString dir = m_pCurCvsDir->fullName();
       QString files = "";

       int CMD;
       if (CvsOptions::g_bPruneDirs) CMD = MERGEDIR_CMD;
       else CMD = MERGEDIR_NO_PRUNE_CMD;

       callInteractiveCmd( topModule, dir, command,
                           files, CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::commitAny() {
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         commitFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      commitDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::commitDir()
{
   QString lastComment = m_pCommitDlg->m_pComment->text();
   if (!(m_pCurCvsDir = getSelectedDir())) {
      return;
   }

   m_pCommitDlg->updateTemplate();
   m_pCommitDlg->reset();

   if (m_pCommitDlg->exec()) {//ok
      
      if (!(m_pCurCvsDir = getSelectedDir())) {
	  return;
      }
      
      //save new template file
      if ( (lastComment.stripWhiteSpace() != m_pCommitDlg->m_pComment->text().stripWhiteSpace())
	   && (!m_pCommitDlg->m_pComment->text().isEmpty())) {
	  writeUserHstFile(m_pCommitDlg->m_pComment->text());
      }

      m_curCommentFileName = writeTmpFile(m_pCommitDlg->m_pComment->text());

      QString command = "";
      if (!bRWPermission) command += "-r ";
      if (m_curCommentFileName.length())
      {
	  command += CvsOptions::cmprStr() +
	      " commit -F " + masqWs(m_curCommentFileName);
      }
      else
      {
	// fallback to option -m
	command += CvsOptions::cmprStr() + " commit -m \"";
	command += I18n::g_pTextEncoder->encode(m_pCommitDlg->m_pComment->text());
	command += "\"";
      }

      if (m_pCommitDlg->m_pRevisionCheckBox->isChecked()) {
	command += " -r " + m_pCommitDlg->m_pRevision->text();
      }

      QString dir = m_pCurCvsDir->fullName();
      QString files = "";
      QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
      callInteractiveCmd( topModule, dir, command,
                          files, COMMIT_CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateAny() 
{
    //only for test
    //allDirPopupMenuItems->changeItem(6, findEmbeddedPixmap ("Delete16x16"), "any");
    //M_COMMIT->setIcon(findEmbeddedPixmap ("Delete16x16"));

  // check if item in listview is selected
	QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      annotateDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLine() 
{
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateGrepLineFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      annotateGrepLineDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLines() 
{
  // check if item in listview is selected
   QListViewItem *myChild = m_pCvsFileList->firstChild();
   while(myChild) {
      if(myChild->isSelected()) {
         // item selected
         annotateGrepLinesFile();
         return;
      }
      myChild = myChild->itemBelow();
   }

   // check if item in directory is selected
   if( static_cast<Directory*>(m_pWorkBenchTree->currentItem()) ) {
      annotateGrepLinesDir();
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = "annotate";

   QString dirBranch = m_pCurCvsDir->getDirTag();
   QString dirType = "";
   if (!dirBranch.isEmpty()) {
     dirType = dirBranch.left(1);
     dirBranch = dirBranch.mid(1);
     if (dirType=="T" || dirType=="N") {
       command += " -r " + dirBranch;
     } else if (dirType=="D") {
       command += " -D " + dirBranch;
     } else {
       Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
     }
   }

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, ANNOTATE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLineDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   AnnotateGrepLineDialogImpl dlg(m_pTools->whatsThisIconSet(),
				  this, "AnnotateGrepLine", true);
   if (dlg.exec()) {

       m_searchLine = dlg.m_SearchLine->currentText();
       m_caseSensitive = dlg.m_bCaseSensitive->isChecked();

       QString command = "annotate";

       QString dirBranch = m_pCurCvsDir->getDirTag();
       QString dirType = "";
       if (!dirBranch.isEmpty()) {
	 dirType = dirBranch.left(1);
	 dirBranch = dirBranch.mid(1);
	 if (dirType=="T" || dirType=="N") {
	   command += " -r " + dirBranch;
	 } else if (dirType=="D") {
	   command += " -D " + dirBranch;
	 } else {
	   Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	 }
       }

       QString dir = m_pCurCvsDir->fullName();
       QString files = "";
       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
       callInteractiveCmd( topModule, dir, command,
                           files, ANNOTATE_GREP_LINE_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLinesDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   m_searchLines.clear();

   AnnotateGrepLinesDialogImpl dlg(m_pTools->whatsThisIconSet(),
				   this, "AnnotateGrepLines", true);
   if (dlg.exec()) {

       QListViewItem *myChild = dlg.m_SearchLines->firstChild();

       while(myChild) {
	   m_searchLines.append(myChild->text(0));
	   myChild = myChild->nextSibling();
       }

       m_caseSensitive = dlg.m_bCaseSensitive->isChecked();

       QString command = "annotate";

       QString dirBranch = m_pCurCvsDir->getDirTag();
       QString dirType = "";
       if (!dirBranch.isEmpty()) {
	 dirType = dirBranch.left(1);
	 dirBranch = dirBranch.mid(1);
	 if (dirType=="T" || dirType=="N") {
	   command += " -r " + dirBranch;
	 } else if (dirType=="D") {
	   command += " -D " + dirBranch;
	 } else {
	   Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
	 }
       }

       QString dir = m_pCurCvsDir->fullName();
       QString files = "";
       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
       callInteractiveCmd( topModule, dir, command,
                           files, ANNOTATE_GREP_LINES_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::logDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = CvsOptions::cmprStr();
   command += " log";

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, LOG_DIR_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::historyDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();

   QString cvsRoot = CvsOptions::cmprStr();
   cvsRoot += " history ";
   if (CvsOptions::g_historyPeriod != -1) {
     cvsRoot +=
       "-D " +
       masqWs(QDateTime::currentDateTime().addDays(-CvsOptions::g_historyPeriod).toString("yyyy-MM-dd hh:mm")) +
       " ";
   }
   cvsRoot += "-e -a -p ";
   cvsRoot += masqWs(m_pCurCvsDir->repository());
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, HISTORY_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::loginDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   m_pMessages->setText("");
   m_pMessages->setCursorPosition(0, 0);
   m_pCurCvsDir->loginOk(m_pMessages, true);
}

//----------------------------------------------------------------------------

void CvsControl::logoutDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   m_pMessages->setText("");
   m_pMessages->setCursorPosition(0, 0);
   m_pCurCvsDir->removeLogin(m_pMessages);
}

//----------------------------------------------------------------------------

void CvsControl::editFile( QString FN) {

  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }

  QStringList nameList;
  nameList.append(FN);

  FileAppItem item = getMatchForWildcard(nameList,openFileAppList);
  if (!item.params.isEmpty()) {
    
    QStringList::Iterator it;
    for (it=nameList.begin(); it != nameList.end(); it++) {
      (*it) = m_pCurCvsDir->fullName()+"/"+(*it);
    }
    QString cmd = masqWs(item.app);
    QString params = " " + item.params;
    if (masqedFilenamesForPlaceholders(params,nameList)) {
      cmd += params;
      runExternal(cmd);
    } else {
      showWarning(tr("Warning"), tr("There are no matching placeholders specified,")
		  +"\n"
		  +tr("Check and adjust 'Options/Open file mapping ...' to proceed."));
    }
  } else {
    showWarning(tr("Warning"), tr("There is no application specified to open the selected file with,")
		+"\n"
		+tr("Check and adjust 'Options/Open file mapping ...' to proceed."));
  }
}

//----------------------------------------------------------------------------

void CvsControl::viewFileWithPath( QString FN) {

  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }

  int pos = FN.findRev("/");
  QString path = FN.mid(0,pos);
  FN = FN.mid(pos+1);

  QStringList nameList;
  nameList.append(FN);

  FileAppItem item = getMatchForWildcard(nameList,viewFileAppList);
  if (!item.params.isEmpty()) {
    
    QStringList::Iterator it;
    for (it=nameList.begin(); it != nameList.end(); it++) {
      (*it) = path+"/"+(*it);
    }
    QString cmd = masqWs(item.app);
    QString params = " " + item.params;
    if (masqedFilenamesForPlaceholders(params,nameList)) {
      cmd += params;
      runExternal(cmd);
    } else {
      showWarning(tr("Warning"), tr("There are no matching placeholders specified,")
		  +"\n"
		  +tr("Check and adjust 'Options/View file mapping ...' to proceed."));
    }
  } else {
    showWarning(tr("Warning"), tr("There is no application specified to view the selected file with,")
		+"\n"
		+tr("Check and adjust 'Options/View file mapping ...' to proceed."));
  }
}

//----------------------------------------------------------------------------

void CvsControl::browseDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString cmd = "\"" + ExtApps::g_localBrowser.path + "\" ";
   QStringList nameList;
   nameList.append(m_pCurCvsDir->fullName());//the startup dir is not sufficiant -- some browsers need the startup dir specified
   QString options = ExtApps::g_localBrowser.options;
   masqedFilenamesForPlaceholders(options, nameList);
   cmd += options;
   QDir * dir = new QDir(m_pCurCvsDir->fullName());
   runExternal(cmd, dir);
   delete dir;
}

//----------------------------------------------------------------------------

void CvsControl::openShellDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString cmd = "\"" + ExtApps::g_localShell.path + "\" ";
   QStringList nameList;
   nameList.append(m_pCurCvsDir->fullName());//the startup dir is not sufficiant -- some consoles might need the startup dir specified
   QString options = ExtApps::g_localShell.options;
   masqedFilenamesForPlaceholders(options, nameList);
   cmd += options;
   QDir * dir = new QDir(m_pCurCvsDir->fullName());
   runExternal(cmd, dir);
   delete dir;
}

//----------------------------------------------------------------------------

void CvsControl::openFile()
{
  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }

  QStringList nameList = getSelectedFilesList();
  if (!nameList.empty()) {//dir is selected, no further check needed

    FileAppItem item = getMatchForWildcard(nameList,openFileAppList);
    if (!item.params.isEmpty()) {
      
      QStringList::Iterator it;
      for (it=nameList.begin(); it != nameList.end(); it++) {
	(*it) = m_pCurCvsDir->fullName()+"/"+(*it);
      }
      QString cmd = masqWs(item.app);
      QString params = " " + item.params;
      if (masqedFilenamesForPlaceholders(params,nameList)) {
	cmd += params;
	runExternal(cmd);
      } else {
	showWarning(tr("Warning"), tr("There are no matching placeholders specified,")
		    +"\n"
		    +tr("Check and adjust 'Options/Open file mapping ...' to proceed."));
      }
    } else {
      showWarning(tr("Warning"), tr("There is no application specified to open the selected files with,")
		  +"\n"
		  +tr("Check and adjust 'Options/Open file mapping ...' to proceed."));
    }
  }
}



//----------------------------------------------------------------------------

void CvsControl::viewFile()
{
  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }

  QStringList nameList = getSelectedFilesList();
  if (!nameList.empty()) {//dir is selected, no further check needed

    FileAppItem item = getMatchForWildcard(nameList,viewFileAppList);
    if (!item.params.isEmpty()) {
      
      QStringList::Iterator it;
      for (it=nameList.begin(); it != nameList.end(); it++) {
	(*it) = m_pCurCvsDir->fullName()+"/"+(*it);
      }
      QString cmd = masqWs(item.app);
      QString params = " " + item.params;
      if (masqedFilenamesForPlaceholders(params,nameList)) {
	cmd += params;
	runExternal(cmd);
      } else {
	showWarning(tr("Warning"), tr("There are no matching placeholders specified,")
		    +"\n"
		    +tr("Check and adjust 'Options/View file mapping ...' to proceed."));
      }
    } else {
      showWarning(tr("Warning"), tr("There is no application specified to view the selected files with,")
		  +"\n"
		  +tr("Check and adjust 'Options/View file mapping ...' to proceed."));
    }
  }
}
//----------------------------------------------------------------------------

void CvsControl::editorsFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();
   QString cvsRoot;
   cvsRoot = "editors";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, SHOWEDITORS_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::editorsDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString cvsRoot;
   cvsRoot = "editors";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, SHOWEDITORS_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::watchersFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();
   QString cvsRoot;
   cvsRoot = "watchers";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, SHOWWATCHERS_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::watchersDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString cvsRoot;
   cvsRoot = "watchers";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, SHOWWATCHERS_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------


void CvsControl::addSelectedFiles()
{
   addSelectedFiles(false);
}

//----------------------------------------------------------------------------

void CvsControl::addSelectedBinFiles()
{   addSelectedFiles(true);
}

//----------------------------------------------------------------------------

void CvsControl::addSelectedFiles(bool bAsBinary) {

  // in what directory are the files
  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }
  
  QString files = "";
  QString St;
  m_FirstPosAddedDir = "";
  QFileInfo FI;
  m_AddedFilesList = getSelectedFilesList();
  
  QStringList::Iterator it;
  for (it = m_AddedFilesList.begin(); it != m_AddedFilesList.end(); it++) {
    St = m_pCurCvsDir->fullName()+"/"+(*it);
    files += " " + masqWs( (*it));
    if( m_FirstPosAddedDir.isEmpty() ) {
      // check if St is directory 
      FI.setFile( St );
      if( FI.isDir() ) {
	m_FirstPosAddedDir = St;
      }
    }
  }
  
  if(!m_AddedFilesList.isEmpty()){
    QString dir = m_pCurCvsDir->fullName();
    QString command;
    command = CvsOptions::cmprStr() + " add";
    if (bAsBinary) command += " -kb";
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    
    callInteractiveCmd( topModule, dir, command, files, 
			ADD_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT, this);
  }
}

//----------------------------------------------------------------------------

void CvsControl::addCvsFiles(const QStringList list, bool binary) {

  QStringList fileList;
  Directory * dir = NULL;
  int pos = -1;
  QStringList::ConstIterator it;
  for (it = list.begin(); it != list.end(); it++) {
    if (!dir) {
      pos = (*it).findRev("/");
      dir = m_pWorkBenchTree->find( (*it).left(pos) );
      ++pos;
    }
    if (dir) {
      fileList.append( (*it).mid(pos) );
    }
  }
  addCvsFiles(dir,fileList,binary);
}

//----------------------------------------------------------------------------

void CvsControl::addCvsFiles(Directory * directory, const QStringList list, bool binary) {

  QString files = "";
  QString St;
  m_FirstPosAddedDir = "";
  QFileInfo FI;
  
  QStringList::ConstIterator it;
  for (it = list.begin(); it != list.end(); it++) {
    files += " " + masqWs( (*it));
  }
  
  if(!list.isEmpty()){
    QString dir = directory->fullName();
    QString command;
    command = CvsOptions::cmprStr() + " add";
    if (binary) command += " -kb";
    QString topModule = directory->topCvsDir()->relativeName();
    
    callInteractiveCmd( topModule, dir, command, files, 
			NOT_INTERACTIVE_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT);
  }

}

//----------------------------------------------------------------------------

/**
 * chained cvs call for cvs removing all files in 'list' (containing fullName())
 * these can be located in different subdirs
 * dropDir is fullName() of dir to add the files in 'list'
 */
void CvsControl::moveCvsFiles(const QString dropDir,const QStringList list) {

  static Directory * dir = NULL;
  static QStringList removeList;
  static QString removeDir = QString::null;
  static QStringList fileList;
  static QStringList::ConstIterator staticIt;
  static QStringList dropAsciiFiles;
  static QStringList dropBinaryFiles;

  if (!removeDir.isNull()) {
    Directory * tmpDir = m_pWorkBenchTree->find(removeDir);
    if (tmpDir) tmpDir->parseCvsCallResult(m_pCvsBuffer,REMOVE_CMD,&fileList);
    removeDir = QString::null;
    fileList.clear();
  }

  if (!dropDir.isNull()) {//init all values if called with params
    dir = m_pWorkBenchTree->find(dropDir);
    assert(dir);
    removeList = list;
    removeDir = QString::null;
    fileList.clear();
    staticIt = removeList.begin();
    dropAsciiFiles.clear();
    dropBinaryFiles.clear();
  }

  QString files;
  bool binary = FALSE;
  if (staticIt != removeList.end()) {
    removeDir = (*staticIt).left( (*staticIt).findRev("/") );
    Directory * tmpDir = m_pWorkBenchTree->find(removeDir);
    assert(tmpDir);

    QStringList::ConstIterator it;
    for (it = staticIt; it != removeList.end(); ++it) {
      int pos = (*it).findRev("/");
      QString nextRemoveDir = (*it).left(pos);
      if (removeDir==nextRemoveDir) {
	QString fileName = (*it).mid(pos+1);
	if (tmpDir->isBinary(fileName)) {
	  dropBinaryFiles.append(fileName);
	} else {
	  dropAsciiFiles.append(fileName);
	}
	fileList.append(fileName);
	staticIt = it;
	++staticIt;

      } else {

	staticIt = it;
	break;
      }
    }

    files = masqWs(fileList.join("\" \""));
    Directory * workDir = m_pWorkBenchTree->find(removeDir);
    QString command = "remove";
    QString topModule = workDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, removeDir, command,
                        files, MOVE_FILES_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT, this);
    return;

  } else if (!dropAsciiFiles.isEmpty()) {//add files ascii

    fileList = dropAsciiFiles;
    files = masqWs(dropAsciiFiles.join("\" \""));
    dropAsciiFiles.clear();
   
  } else if (!dropBinaryFiles.isEmpty()) {//add files binary

    fileList += dropBinaryFiles;
    files = masqWs(dropBinaryFiles.join("\" \""));
    binary = TRUE;
    dropBinaryFiles.clear();

  } else {
     dir->parseCvsCallResult(m_pCvsBuffer,ADD_CMD,&fileList);
     if (dir->isSelected()) dir->activateItem(FALSE);
     fileList.clear();
     return;
  }

  QString St;
  
  QString dirName = dir->fullName();
  QString command;
  command = CvsOptions::cmprStr() + " add";
  if (binary) command += " -kb";
  QString topModule = dir->topCvsDir()->relativeName();
  
  callInteractiveCmd( topModule, dirName, command, files, 
		      MOVE_FILES_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::createNewFile() {

    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString newFile = QInputDialog::getText("Create File", "Name: ",
					    QLineEdit::Normal ).stripWhiteSpace();
    if (!newFile.isEmpty()) {
	QFile file(m_pCurCvsDir->fullName()+"/"+newFile);
	if (file.exists()) {
	    showWarning( tr("Warning"), tr("This file allready exists.\n") );
	    return;
	} else {
	    file.open(IO_WriteOnly);
	    file.close();
	}
    }
    saveViewportPos(OVERRIDE);
    m_pCurCvsDir->activateItem(FALSE);
    recoverViewportPos();
}

//----------------------------------------------------------------------------

void CvsControl::createNewDir() {

    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString newDir = QInputDialog::getText("Create Dir", "Name: ",
					    QLineEdit::Normal ).stripWhiteSpace();
    if (!newDir.isEmpty()) {
	QDir dir(m_pCurCvsDir->fullName()+"/"+newDir);
	if (dir.exists()) {
	    showWarning( tr("Warning"), tr("This directory allready exists.\n") );
	    return;
	} else {
	    dir.mkdir(m_pCurCvsDir->fullName()+"/"+newDir,true);
	}
    }
    saveViewportPos(OVERRIDE);
    m_pCurCvsDir->activateItem(FALSE);
    recoverViewportPos();
}

//----------------------------------------------------------------------------

void CvsControl::resurrectSelectedFiles() {

	// in what directory are the files
        if (!(m_pCurCvsDir = getSelectedDir())) {
	    return;
	}

	QString files = "";
	m_AddedFilesList.clear();
	QString St;
	m_FirstPosAddedDir = "";
	QFileInfo FI;
	QListViewItem * It = m_pCvsFileList->firstChild();
	while( It ) {
	  if( It->isSelected() ) {
	    St = m_pCurCvsDir->fullName()+"/"+It->text(0);
	    m_AddedFilesList.append( It->text(0));
	    files += " " + masqWs(It->text(0));
	    if( m_FirstPosAddedDir.isEmpty() ) {
	      // check if St is directory 
	      FI.setFile( St );
	      if( FI.isDir() ) {
		m_FirstPosAddedDir = St;
	      }
	    }
	  }
	  It = It->itemBelow();
	}

	if(!m_AddedFilesList.isEmpty()){
	    QString dir = m_pCurCvsDir->fullName();
	    QString cvsRoot;
	    if (!bRWPermission) cvsRoot += "-r ";
	    cvsRoot += "add";
// 	    if (bAsBinary) cvsRoot += " -kb";
	    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
	    callInteractiveCmd( topModule, dir, cvsRoot,
                                files, RESURRECT_CMD,
                                ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                                NOROOT, this);
	}
}

//----------------------------------------------------------------------------

//returns the dir if successfully reopened, null otherwise
Directory * CvsControl::rereadProjectOfDir( Directory * dir ) {

	// have added directory
	QString dirToOpenAfterRescan = dir->fullName();

	Directory * topDir = dir->topDir();

	// keep project identifiers
	QString moduleName = topDir->shortName();
	QString topDirName = topDir->fullName();

	//keep open dirs
	QStringList openDirs;
	dir->recGetOpenDirsStringList(openDirs);

	if (topDirName == dirToOpenAfterRescan) {
	  delete dir;
	  dir = NULL;
	  performRereadProject( moduleName,topDirName);//calls addProject()
	} else {
	  Directory * parentDir = (Directory *)dir->parent();
	  parentDir->reloadChild(dirToOpenAfterRescan);
	}

	dir = (Directory *)m_pWorkBenchTree->find( dirToOpenAfterRescan );
	if( dir )  {
	  QStringList::Iterator it;
	  for (it = openDirs.begin(); it != openDirs.end(); ++it) {
	    openTreeAt( topDirName, (*it));
	  }
	  m_pWorkBenchTree->setSelected( dir, TRUE );
	  dir->activateItem(FALSE);
	}
	applyProjectSettings(moduleName);
	return dir;
}

//----------------------------------------------------------------------------

void CvsControl::queryUpdateFile()
{
    updateFile(true);
}

//----------------------------------------------------------------------------

void CvsControl::updateFile()
{
    updateFile(false);
}

//----------------------------------------------------------------------------

void CvsControl::updateFile(bool query)
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QStringList s;
   QString fileName;
   QString tmp = "";
 
   tmp = getNamesOfSelectedFiles();
   if (tmp.isEmpty()) {
       return;
   }

   QString queryString = (query ? " -n " : "" );
   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() +
       queryString + " update " + tmp;

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, (query ? QUERY_UPDATE_CMD : UPDATE_FILES_CMD),
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::replaceFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QStringList s;
   QString fileName;
   QString tmp = "";
 
   tmp = getNamesOfSelectedFiles();
   if (tmp.isEmpty()) {
       return;
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() + " update -C " + tmp;

   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, UPDATE_FILES_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::removeLocalOptionsFile()
{
    updateToDirBranch(true,true);
}

//----------------------------------------------------------------------------

void CvsControl::removeLocalOptionsDir()
{
    updateToDirBranch(false,true);
}

//----------------------------------------------------------------------------

void CvsControl::updateFilesToDirBranch()
{
    updateToDirBranch(true);
}

//----------------------------------------------------------------------------

void CvsControl::updateToDirBranch(bool fileCollection, bool removeLocalOptions)
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString dirBranch = m_pCurCvsDir->getDirTag();
    QString dirType = "";
    if (!dirBranch.isEmpty()) {
      dirType = dirBranch.left(1);
      dirBranch = dirBranch.mid(1);
    }

    if (removeLocalOptions) {
	if (dirBranch.isEmpty()) {//check that no file has any tag -> would be removed otherwise
	    bool cancel = false;
	    QListViewItem *myChild = m_pCvsFileList->firstChild();
	    while(myChild) {
		if (fileCollection) {
		    if( (myChild->isSelected()) && ( !(myChild->text(2).isEmpty())) ) {
			cancel = true;
			break;
		    }
		} else if ( !(myChild->text(2).isEmpty())) {
		    cancel = true;
		    break;
		}
		myChild = myChild->nextSibling();
	    }
	    if (cancel) {
		showWarning( tr("Warning"), tr("There are tags set that would be removed too.\n"
					       "If this is what you want, run:\n"
					       "'update other revision'\n"
					       "with option 'reset sticky tag' selected") );
		return;
	    }
	} else {//check that no file has a tag different to dirBranch ->would be removed otherwise
	    bool cancel = false;
	    QListViewItem *myChild = m_pCvsFileList->firstChild();
	    while(myChild) {
		if (fileCollection) {
		    if( (myChild->isSelected()) && ( myChild->text(2)!=dirBranch) ) {
			cancel = true;
			break;
		    }
		} else if( myChild->text(2)!=dirBranch) {
		    cancel = true;
		    break;
		}
		myChild = myChild->nextSibling();
	    }
	    if (cancel) {
		showWarning( tr("Warning"), tr("There are tags set that differ from the branch\n"
					       "set for this dir. They would be removed too\n"
					       "If this is what you want, run:\n"
					       "'update other revision'\n"
					       "with option 'reset sticky tag' selected") );
		return;
	    }
	}
    } else if (dirBranch.isEmpty()) {
	showWarning( tr("update to branch of dir"), tr("The selected dir has\nno branch tag set") );
	return;
    }

    QString fileName;
    QString tmp = "";

    if (fileCollection) { 
	tmp = getNamesOfSelectedFiles();
	if (tmp.isEmpty()) {
	    return;
	}
    }

    bool query = false;
    QString queryString = (query ? " -n " : "" );
    QString command = "";
    if (!bRWPermission) command += "-r ";
    command += CvsOptions::cmprStr() +
	queryString + " update " + (removeLocalOptions ? "-A " : "");
    if (!dirBranch.isEmpty()) {
      if (dirType=="T" || dirType=="N") {
	command += "-r " + dirBranch + " ";
      } else if (dirType=="D") {
	command += "-D " + dirBranch + " ";
      } else {
	Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown dirBranchType: ->"+dirType+"<-");
      }
    }
    command += tmp;

    QString dir = m_pCurCvsDir->fullName();
    QString files = "";
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, dir, command,
                        files, (query ? QUERY_UPDATE_CMD : UPDATE_FILES_CMD),
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::mergeFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   MergeDialogImpl dlg(m_pTools->whatsThisIconSet(),
		       this, "MergeBranches", 
		       getCurrentTagList(), topModule, true);

   if (dlg.exec()) {

       QString files = getNamesOfSelectedFiles();
       if (files.isEmpty()) {
	 return;
       }


       QString command = "";
       if (!bRWPermission) command += "-r ";
       if (dlg.m_bOnlyQuery->isChecked()) command += "-n ";
       command += CvsOptions::cmprStr() + " update ";
       if (!dlg.m_bOnlyQuery->isChecked() && bKeywordSuppression) command += "-kk ";
       command += "-j ";
       command += dlg.m_pFirstTag->currentText();
       if (dlg.m_bUseSecondTag->isChecked()) command += " -j " + dlg.m_pSecondTag->currentText();
     
       QString dir = m_pCurCvsDir->fullName();
       callInteractiveCmd( topModule, dir, command,
                           files, MERGEFILES_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::watchAdd()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString command = "watch add";
   QString dir = m_pCurCvsDir->fullName();
   QString files = getNamesOfSelectedFilesWithoutSelCheck();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);

}

//----------------------------------------------------------------------------

void CvsControl::watchRemove()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString command = "watch remove";
   QString dir = m_pCurCvsDir->fullName();
   QString files = getNamesOfSelectedFilesWithoutSelCheck();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);

}

//----------------------------------------------------------------------------

void CvsControl::watchOn()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString command = "watch on";
   QString dir = m_pCurCvsDir->fullName();
   QString files = getNamesOfSelectedFilesWithoutSelCheck();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::watchOff()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString command = "watch off";
   QString dir = m_pCurCvsDir->fullName();
   QString files = getNamesOfSelectedFilesWithoutSelCheck();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);

}

//----------------------------------------------------------------------------

void CvsControl::cvsEditFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }
      
   QString command = "edit";
   QString dir = m_pCurCvsDir->fullName();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::cvsEditDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString command = "edit";
   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::cvsUneditFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }
      
   QString dir = m_pCurCvsDir->fullName();

   QString cvsRoot = m_pCurCvsDir->getCvsRoot();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files,
                       UNEDIT_CMD,
                       ExtApps::g_cvsRsh.path);  //additional options of cvsRsh not supported yet
}

//----------------------------------------------------------------------------

void CvsControl::cvsUneditDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
      
   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString cvsRoot = m_pCurCvsDir->getCvsRoot();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, UNEDIT_CMD,
                       ExtApps::g_cvsRsh.path);  //additional options of cvsRsh not supported yet
}

//----------------------------------------------------------------------------

void CvsControl::cvsLockFile()
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString file;
    QString rev;
    getNameAndRevOfFirstSelectedFile(&file, &rev);
    rev.truncate(rev.findRev(".")+1);
    rev += "1";

    if (file.isEmpty()) {
	return;
    }

    QString command = "admin -l" + rev;
    QString dir = m_pCurCvsDir->fullName();
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, dir, command,
                        file, LOCKFILE_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::cvsLockDir()
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString command = "admin -l";
    QString dir = m_pCurCvsDir->fullName();
    QString files = "";
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, dir, command,
                        files, NOT_INTERACTIVE_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::cvsUnlockFile()
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString file;
    QString rev;
    getNameAndRevOfFirstSelectedFile(&file, &rev);
    rev.truncate(rev.findRev(".")+1);
    rev += "1";

    if (file.isEmpty()) {
	return;
    }

    QString command = "admin -u" + rev;
    QString dir = m_pCurCvsDir->fullName();
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, dir, command,
                        file, UNLOCKFILE_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::cvsUnlockDir()
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    QString command = "admin -u";
    QString dir = m_pCurCvsDir->fullName();
    QString files = "";
    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    callInteractiveCmd( topModule, dir, command,
                        files, NOT_INTERACTIVE_CMD,
                        ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                        NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::expandDir() {//only called from lincvs.cpp
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   bool blocked = isGuiBlocked();
   if (!blocked) blockGUI(true);
   m_pCurCvsDir->expandAllDir();
   slot_checkStatusOfTree(m_pCurCvsDir);
   if (!blocked) blockGUI(false);
}

//----------------------------------------------------------------------------

void CvsControl::collapsDir() {//only called from lincvs.cpp
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   blockGUI(true,BLOCKNOCONTROLS);
   m_pCurCvsDir->collapsAllDir();
   QApplication::restoreOverrideCursor();
   blockGUI(false,BLOCKNOCONTROLS);
}

//----------------------------------------------------------------------------

void CvsControl::updateRevFile(QString cvsTag)
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString tmp = getNamesOfSelectedFiles();
   if (tmp.isEmpty()) {
       return;
   }

   QString command = "";
   if (!bRWPermission) command += "-r ";
   command += CvsOptions::cmprStr() + " update ";
   if (!cvsTag.isEmpty()) {
     command += cvsTag + " ";
   }
   command += tmp;
   QString dir = m_pCurCvsDir->fullName();
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, UPDATE_FILES_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::revertToRevFile(QString param /*=QString::null*/) {

  static QString topModule;
  static QString dir;
  static QString selectedFiles;
  static QString module;
  static QString tag;
  static QString updateParam;
  static QString cvsRoot;
  static int state;

  QString command;
  QString files = "";
  if (!param.isNull()) {
    if (!(m_pCurCvsDir = getSelectedDir())) {
      return;
    }
    dir = m_pCurCvsDir->fullName();
    selectedFiles = getNamesOfSelectedFiles();
    module = m_pCurCvsDir->repository();
    topModule = m_pCurCvsDir->topCvsDir()->relativeName();
    cvsRoot = m_pCurCvsDir->getCvsRoot();
    updateParam = param;
    if (updateParam.startsWith(":")) state = 0;
    else state = 1;
    tag = "TMP-TAG-"+m_pCurCvsDir->userName()+"-"+QDateTime::currentDateTime().toString("yyyyMMddhhmmzzz")+"-";
    command = "tag "+tag+"CURRENT";
    files = selectedFiles;
  } else {
    switch( state) {
      case 1: {
	command = "-d \""+cvsRoot+"\" rtag -D \""+updateParam.mid(1)+"\" "+tag+"TO \""+module+"\"";
	break;
      }
      case 2: {
	command = "";
	if (!bRWPermission) command += "-r ";
	command += CvsOptions::cmprStr() + " update ";
	if (updateParam.startsWith(":")) command += "-j \""+tag+"CURRENT\" -j \"" + tag + "TO\"";
	else {
	  command += "-j \""+tag+"CURRENT\" -j \"" + updateParam + "\"";
	  ++state;
	}
	files = selectedFiles;
	break;
      }
      case 3: {
	command = "-d \""+cvsRoot+"\" rtag -d \""+tag+"TO\" \""+module+"\"";
	break;
      }
      case 4: {
	command = "tag -d "+tag+"CURRENT";
	files = selectedFiles;
	break;
      }
      case 5: {
	m_pCurCvsDir = m_pWorkBenchTree->findBestMatch(dir);
	if (m_pCurCvsDir) {
	  m_pCurCvsDir->recRemoveTmpEntries(FALSE);
	  m_pWorkBenchTree->setSelected(m_pCurCvsDir,true);
	  m_pCurCvsDir->activateItem(TRUE);
	  recoverViewportPos();
	  return;
	}
	break;
      }
      default: {
	qDebug("revertToRevDir: should never reach this line");
	return;
      }
    }
  }
  ++state;
  callInteractiveCmd( topModule, dir, command,
                      files, REVERT_FILES_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::statusFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }
   
   QString dir = m_pCurCvsDir->fullName();
   QString cvsRoot = "status -v";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, STATUS_FILES_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::renameFile() {

	QString command;

	if (!(m_pCurCvsDir = getSelectedDir())) {
	    return;
	}
	QString fname =  getNameOfFirstSelectedFile();

	if( fname.isEmpty() ) {
	    return;

	}

	QString NewName = QInputDialog::getText("Rename", "New name : ",
						QLineEdit::Normal,
						fname ).stripWhiteSpace();

	if( fname == NewName ) {//not detected by system(command.ascii())
	    return;
	}

        if( ! NewName.isEmpty() ) {
	    m_AddedFilesList.clear();
	    m_AddedFilesList.append(fname);

	    QString path = m_pCurCvsDir->fullName();
	    QString src = path + "/" + fname;
	    QString dst = path + "/" + NewName;
	    if (QFileInfo(dst).exists()) {
	      if (!showOkCancelQuestion( tr("Rename"),
				 tr("Cannot rename file") + "\n"
				 + tr("File exists:") + "\n"
				 + dst + "\n\n"
				 + tr("Overwrite?") )) {
		return;
	      }
	    }
	    if (!rename(src,dst)) {
		showWarning( tr("Rename"), tr("Cannot rename file") );
		m_AddedFilesList.clear();
		return;
	    }

	   m_newFileName = NewName;
	   command = "remove";
	   QString dir = m_pCurCvsDir->fullName();
	   QString file = masqWs(fname);
	   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
	   callInteractiveCmd( topModule, dir, command,
                               file, RENAME_CMD,
                               ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                               NOROOT, this);

        }
}

//----------------------------------------------------------------------------

//works only on toplevel dirs!!!!!!!!!!!!!!!!!!!!
void CvsControl::renameCurrentDir(QString newName) {

  QStringList projectNameList;
  projectSettings->getProjects(projectNameList);
  unsigned int i;
  for(i=0; i < projectNameList.count(); i++) {
    QString fullName;
    projectSettings->get(projectNameList[i],WORKDIR,fullName);
    if ( newName.startsWith(fullName)) {
      int pos = fullName.length()-projectNameList[i].length();
      QString newModule = newName.mid(pos,newName.find('/',pos)-pos);
      if (projectNameList[i] == newModule) {
	showWarning(tr("Warning: rename aborted")
		    ,tr("This project is already in workbench:")+" "
		    +projectNameList[i]
		    +"\nat: "+fullName);
	setStatusText( tr("Rename aborted"), 4000 );
	return;
      }
    }
  }

  QString oldTopName = m_pCurCvsDir->topDir()->fullName();
  QString topModule = m_pCurCvsDir->topDir()->shortName();
  QString newTopModule = newName.mid( newName.findRev("/")+1);

  if (renameProject(topModule,newTopModule)) {
    if (!rename(m_pCurCvsDir->fullName(),newName)) {
      renameProject(newTopModule,topModule);
      showWarning(tr("Warning: rename aborted")
		  ,tr("Can't rename:\n")
		  +oldTopName
		  +",\n"+tr("check permissions"));
      setStatusText( tr("Rename aborted"), 4000 );
      return;
    }
  } else {
    showWarning(tr("Warning: rename aborted")
		,tr("Can't rename:\n")
		+oldTopName
		+",\n"+tr("the new name: ")+newTopModule+"\n"+tr("is allready in use"));
    setStatusText( tr("Rename aborted"), 4000 );
    return;
  }

  QStringList openDirs;
  QStringList::Iterator it;
  m_pCurCvsDir->recGetOpenDirsStringList(openDirs);
  for (it = openDirs.begin(); it != openDirs.end(); ++it) {
    if ( (*it).startsWith(oldTopName)) {
      (*it) = newName + (*it).mid(oldTopName.length());
    }
  }

  delete m_pCurCvsDir;
  m_pCurCvsDir = NULL;
  addProject(newName,TRUE);

  m_pCurCvsDir = m_pWorkBenchTree->find(newName);
  if (m_pCurCvsDir) {
    for (it = openDirs.begin(); it != openDirs.end(); ++it) {
      openTreeAt( newName, (*it));
    }
    m_pWorkBenchTree->setSelected( m_pCurCvsDir, TRUE );
    m_pCurCvsDir->activateItem(FALSE);
  }
}

//----------------------------------------------------------------------------

bool CvsControl::moveProject(const QString& project, const QString& oldPath, const QString& newPath) {//FIXME unused ...
  projectSettings->set(project,WORKDIR,newPath);
  QStringList::Iterator it;
  QStringList list;
  projectSettings->get(project,DISABLED,list);
  for (it = list.begin(); it != list.end(); ++it) {
    if ( (*it).startsWith(oldPath)) {
      (*it) = newPath + (*it).mid(oldPath.length());
    }
  }
  projectSettings->set(project,DISABLED,list);
  projectSettings->getSubProjects(project,list);
  for (it = list.begin(); it != list.end(); ++it) {
    QString path;
    projectSettings->get(*it,WORKDIR,path);
    if ( path.startsWith(oldPath)) {
      path = newPath + path.mid(oldPath.length());
    }
    projectSettings->set(*it,WORKDIR,path);
  }
  return true;
}

//----------------------------------------------------------------------------

bool CvsControl::renameProject(const QString& oldProject, const QString& newProject) {

  if (!projectSettings->renameProject(oldProject,newProject)) return false;
  QString oldWorkdir;
  projectSettings->get(newProject,WORKDIR,oldWorkdir);
  QString newWorkdir = oldWorkdir.left(oldWorkdir.length()-oldProject.length()) + newProject;
  projectSettings->set(newProject,WORKDIR,newWorkdir);

  QStringList::Iterator it;
  QStringList list;
  projectSettings->get(newProject,DISABLED,list);
  QStringList oldDisabledList = list;
  for (it = list.begin(); it != list.end(); ++it) {
    if ( (*it).startsWith(oldWorkdir)) {
      (*it) = newWorkdir + (*it).mid(oldWorkdir.length());
    }
  }
  projectSettings->set(newProject,DISABLED,list);

  list.clear();
  projectSettings->getSubProjects(newProject,list);
  for (it = list.begin(); it != list.end(); ++it) {

    QString newName = newProject + (*it).mid(oldProject.length());
    if (renameProject(*it,newName)) {
      *it = newName;
    } else {//rollback
      QStringList::Iterator rit = list.begin();
      while( rit != it) {
	QString oldName = oldProject + (*it).mid(newProject.length());
	renameProject(*it,oldName);
	++rit;
      }
      projectSettings->renameProject(newProject,oldProject);
      projectSettings->set(oldProject,WORKDIR,oldWorkdir);
      projectSettings->set(oldProject,DISABLED,oldDisabledList);
      return false;
    }

  }
  projectSettings->setSubProjects(newProject,list);
  return true;
}

//----------------------------------------------------------------------------

void CvsControl::diffFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }

   QString command = CvsOptions::cmprStr() + " diff";
   //if (bKeywordSuppression) command += " -kk";// commented because diff compares to local file
	                                        // which is not necessarily checked out with -kk
   QString dir = m_pCurCvsDir->fullName();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::diffAdvancedFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }
      
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   MergeDialogImpl dlg(m_pTools->whatsThisIconSet(),
		       this, "MergeBranches", 
		       getCurrentTagList(), topModule, true);
   dlg.setCaption(tr("Advanced diff"));
   dlg.hideQueryNUpdateDirs();

   if (dlg.exec()) {

       QString command = CvsOptions::cmprStr() + " diff";
       //if (bKeywordSuppression) command += " -kk";// commented because diff compares to local file
                                                    // which is not necessarily checked out with -kk
       
       command += " -r ";
       command += dlg.m_pFirstTag->currentText();
       if (dlg.m_bUseSecondTag->isChecked()) command += " -r " + dlg.m_pSecondTag->currentText();

       QString dir = m_pCurCvsDir->fullName();
       callInteractiveCmd( topModule, dir, command,
                           files, NOT_INTERACTIVE_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT);
   }
}

//----------------------------------------------------------------------------

void CvsControl::diffDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString command = CvsOptions::cmprStr() + " diff";
//    if (bKeywordSuppression) command += " -kk";// commented because diff compares to local file
	                                         // which is not necessarily checked out with -kk
   QString dir = m_pCurCvsDir->fullName();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   QString files = "";
   callInteractiveCmd( topModule, dir, command,
                       files, NOT_INTERACTIVE_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT);
}

//----------------------------------------------------------------------------

void CvsControl::diffAdvancedDir()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   MergeDialogImpl dlg(m_pTools->whatsThisIconSet(),
		       this, "MergeBranches", 
		       getCurrentTagList(), topModule, true);
   dlg.setCaption(tr("Advanced diff"));
   dlg.hideQueryNUpdateDirs();

   if (dlg.exec()) {

       QString command = CvsOptions::cmprStr() + " diff";
       //if (bKeywordSuppression) command += " -kk";// commented because diff compares to local file
                                                    // which is not necessarily checked out with -kk
       
       command += " -r ";
       command += dlg.m_pFirstTag->currentText();
       if (dlg.m_bUseSecondTag->isChecked()) command += " -r " + dlg.m_pSecondTag->currentText();

       QString dir = m_pCurCvsDir->fullName();
       QString files = "";
       callInteractiveCmd( topModule, dir, command,
                           files, NOT_INTERACTIVE_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT);
   }
}

//----------------------------------------------------------------------------

void CvsControl::diffFileSideBySide()
{
    m_bExternalDiffIsRunning = false;

    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }
    QString fname =  getNameOfFirstSelectedFile();

    if( fname.isEmpty() ) {

	return;
    }

    QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

    if (bUSEEXTERNALDIFFFORSIDEBYSIDE) {
       if (m_curTmpDiffFileName.length()) QFile::remove(m_curTmpDiffFileName);

       m_curDiffFileName = m_pCurCvsDir->fullName() + "/" + fname;
       m_curTmpDiffFileName = createTempFile() + "-" + fname;

        QString command = CvsOptions::cmprStr() + " -Q update ";
	//if (bKeywordSuppression) command += "-kk ";// commented because diff compares to local file
	                                             // which is not necessarily checked out with -kk
	command += "-p " + masqWs(fname) + " > " + m_curTmpDiffFileName;

	QString dir = m_pCurCvsDir->fullName();
	QString es = "";
	callInteractiveCmd( topModule, dir, command,
                            es, EXT_DIFF_CMD,
                            ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                            NOROOT, this);

    } else {	      
	QString dir = m_pCurCvsDir->fullName();
	QString hlp;
	QString cvsRoot = CvsOptions::cmprStr() + " -Q diff -N ";
	fname = masqWs(fname);
	if (bDiffIgnoreWhiteSpace) cvsRoot += "-b -B ";
	//if (bKeywordSuppression) cvsRoot += "-kk ";// commented because diff compares to local file
	                                             // which is not necessarily checked out with -kk
	cvsRoot += "--side-by-side -t -W "+hlp.setNum(SIDEWIDTH);
	callInteractiveCmd( topModule, dir, cvsRoot,
                            fname, DIFF_CMD,
                            ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                            NOROOT, this);
    }
}

//----------------------------------------------------------------------------

void CvsControl::commitFile()
{
   QString lastComment = m_pCommitDlg->m_pComment->text();
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }

   m_pCommitDlg->updateTemplate();
   m_pCommitDlg->reset();

   if(m_pCommitDlg->exec()) {//ok

      if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
      }

      //save template file
      if ( (lastComment.stripWhiteSpace() != m_pCommitDlg->m_pComment->text().stripWhiteSpace())
	   && (!m_pCommitDlg->m_pComment->text().isEmpty())) {
	  writeUserHstFile(m_pCommitDlg->m_pComment->text());
      }

      m_curCommentFileName = writeTmpFile(m_pCommitDlg->m_pComment->text());

      QString command = "";
      if (!bRWPermission) command += "-r ";
      if (m_curCommentFileName.length())
      {
	  command += CvsOptions::cmprStr() +
	      " commit -F " + masqWs(m_curCommentFileName);
      }
      else
      {
	// fallback to option -m
	command += CvsOptions::cmprStr() + " commit -m \"";
	command += I18n::g_pTextEncoder->encode(m_pCommitDlg->m_pComment->text());
	command += "\"";
      }
      
      if (m_pCommitDlg->m_pRevisionCheckBox->isChecked()) {
	command += " -r " + m_pCommitDlg->m_pRevision->text();
      }

      QString dir = m_pCurCvsDir->fullName();
      QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
      callInteractiveCmd( topModule, dir, command,
                          files, COMMIT_CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

/*
   QString files = getNamesOfSelectedFiles();

   if( ! files.isEmpty() ) {
      QString dir = m_pCurCvsDir->fullName();
      QString command = "annotate";
      QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
      callInteractiveCmd( topModule, dir, command,
                          files, ANNOTATE_CMD,
                          ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                          NOROOT, this);
   }
*/

   QString file;
   QString rev;
   getNameAndRevOfFirstSelectedFile(&file, &rev);
   if (file.isEmpty()) {
      return;
   }
   file = "-r " + rev + " " + masqWs(file);

   QString command = "annotate";
   QString dir = m_pCurCvsDir->fullName();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command, file, ANNOTATE_CMD, ExtApps::g_cvsRsh.path, NOROOT, this);

}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLineFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString file;
   QString rev;
   getNameAndRevOfFirstSelectedFile(&file, &rev);
   if (file.isEmpty()) {
       return;
   }
   file = "-r " + rev + " " + masqWs(file);

   AnnotateGrepLineDialogImpl dlg(m_pTools->whatsThisIconSet(),
				  this, "AnnotateGrepLine", true);
   if (dlg.exec()) {

       m_searchLine = dlg.m_SearchLine->currentText();
       m_caseSensitive = dlg.m_bCaseSensitive->isChecked();

       QString command = "annotate";
       QString dir = m_pCurCvsDir->fullName();
       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
       callInteractiveCmd( topModule, dir, command,
                           file, ANNOTATE_GREP_LINE_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::annotateGrepLinesFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }

   QString file;
   QString rev;
   getNameAndRevOfFirstSelectedFile(&file, &rev);
   if (file.isEmpty()) {
       return;
   }
   file = "-r " + rev + " " + masqWs(file);

   m_searchLines.clear();
   AnnotateGrepLinesDialogImpl dlg(m_pTools->whatsThisIconSet(),
				   this, "AnnotateGrepLines", true);
   if (dlg.exec()) {

       QListViewItem *myChild = dlg.m_SearchLines->firstChild();
       while(myChild) {
	   m_searchLines.append(myChild->text(0));
	   myChild = myChild->nextSibling();
       }

       m_caseSensitive = dlg.m_bCaseSensitive->isChecked();

       QString command = "annotate";
       QString dir = m_pCurCvsDir->fullName();
       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
       callInteractiveCmd( topModule, dir, command,
                           file, ANNOTATE_GREP_LINES_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::logFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString Fname = getNamesOfSelectedFiles();

   if( ! Fname.isEmpty() ) {

       QString dir = m_pCurCvsDir->fullName();
       QString cvsRoot = CvsOptions::cmprStr();
       cvsRoot += " log";
       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
       callInteractiveCmd( topModule, dir, cvsRoot,
                           Fname, LOG_CMD,
                           ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                           NOROOT, this);
   }
}

//----------------------------------------------------------------------------

void CvsControl::historyFile()
{
   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   QString files = getNamesOfSelectedFiles();
   if (files.isEmpty()) {
       return;
   }
   
   QString dir = m_pCurCvsDir->fullName();
   QString cvsRoot = CvsOptions::cmprStr();
   cvsRoot += " history ";
   if (CvsOptions::g_historyPeriod != -1) {
     cvsRoot +=
       "-D " +
       masqWs(QDateTime::currentDateTime().addDays(-CvsOptions::g_historyPeriod).toString("yyyy-MM-dd hh:mm")) +
       " ";
   }
   cvsRoot += "-e -a";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, cvsRoot,
                       files, HISTORY_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::removeFile()
{
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }
    
    if(!m_pCurCvsDir->loginOk(m_pMessages)) return;
    
    doRemoveFileFromRepOrDisk( 1, 1 );
}

//----------------------------------------------------------------------------

void CvsControl::removeFileFromRep() {
	// backup file
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }

    if(!m_pCurCvsDir->loginOk(m_pMessages)) return;
    doRemoveFileFromRepOrDisk( 1, 0 );
}

//----------------------------------------------------------------------------

void CvsControl::removeFileFromDisk() {

       if (!(m_pCurCvsDir = getSelectedDir())) {
	   return;
       }

       doRemoveFileFromRepOrDisk( 0, 1 );
}

//----------------------------------------------------------------------------

void CvsControl::removeNonCvsFileFromDisk() {
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }
    WarningDlg *dlg = new WarningDlg(
				     tr("Warning"), 
				     tr("This command will remove file(s) from disk!"), this);
	   
    if(!dlg->exec()) {
	delete dlg;
	return;
    }

    QString CurDir = m_pCurCvsDir->fullName() + "/";
    QListViewItem *myChild;
    myChild = m_pNonCvsFileList->firstChild();
    while(myChild) {
	if(myChild->isSelected()) {
	    QString fileName = myChild->text(0).stripWhiteSpace();
	    QFileInfo info( CurDir + fileName);
	    
	    if (info.isDir()) {
	      if (removeDirTreeFromDisk(CurDir + fileName, FALSE)) {
		if (!removeDirTreeFromDisk(CurDir + fileName, TRUE)) {
		  myChild = myChild->nextSibling();
		  continue;
		}
	      }
            } else {
		QFile f( CurDir + fileName);
		setPermission(f,READABLE | WRITEABLE);
		f.remove();
	    }
	    QListViewItem *childToDelete;
	    childToDelete = myChild;
	    myChild = myChild->nextSibling();
	    delete childToDelete;
	    continue;
	}
	myChild = myChild->nextSibling();
    }
    saveViewportPos(OVERRIDE);
    m_pCurCvsDir->activateItem(TRUE);
    recoverViewportPos();
}

//----------------------------------------------------------------------------

void CvsControl::removeCvsIgnoreFileFromDisk() {
    if (!(m_pCurCvsDir = getSelectedDir())) {
	return;
    }
    WarningDlg *dlg = new WarningDlg(
				     tr("Warning"), 
				     tr("This command will remove file(s) from disk!"), this);
	   
    if(!dlg->exec()) {
	delete dlg;
	return;
    }

    QString CurDir = m_pCurCvsDir->fullName() + "/";
    QListViewItem *myChild;
    myChild = m_pCvsIgnoreFileList->firstChild();
    while(myChild) {
	if(myChild->isSelected()) {
	    QString fileName = myChild->text(0).stripWhiteSpace();
	    QFileInfo info( CurDir + fileName);
	    
	    if (info.isDir()) {
	      if (removeDirTreeFromDisk(CurDir + fileName, FALSE)) {
		if (!removeDirTreeFromDisk(CurDir + fileName, TRUE)) {
		  myChild = myChild->nextSibling();
		  continue;
		}
	      }
            } else {
		QFile f( CurDir + fileName);
		setPermission(f,READABLE | WRITEABLE);
		f.remove();
	    }
	    QListViewItem *childToDelete;
	    childToDelete = myChild;
	    myChild = myChild->nextSibling();
	    delete childToDelete;
	    continue;
	}
	myChild = myChild->nextSibling();
    }
    saveViewportPos(OVERRIDE);
    m_pCurCvsDir->activateItem(TRUE);
    recoverViewportPos();
}

//----------------------------------------------------------------------------

bool CvsControl::removeDirTreeFromDisk(QString dir,bool remove) {

  QDir D( dir);
  D.setFilter( QDir::Dirs); //should list all dirs
  QStringList AllDirEntries = D.entryList();
  AllDirEntries.remove( "." );
  AllDirEntries.remove( ".." );

  if (!remove && (AllDirEntries.find( "CVS") != AllDirEntries.end())) return FALSE;

  QStringList::Iterator it;
  for ( it = AllDirEntries.begin(); it != AllDirEntries.end(); it++ ) {
    if (!removeDirTreeFromDisk( dir+"/"+(*it),remove)) return FALSE;
  }

  dir += "/";
  D.setFilter( QDir::Files | QDir::Hidden); //should list everything
  QStringList AllEntriesInDir = D.entryList();

  for ( it = AllEntriesInDir.begin(); it != AllEntriesInDir.end(); it++ ) {
    QString file = dir + (*it);
    if (remove) {
      QFile f( file);
      setPermission(f,READABLE | WRITEABLE);
      if (!f.remove()) {
	showWarning( tr("Warning"), tr("Cannot remove:\n")+file);
	return FALSE;
      }
    }
  }
  QFileInfo info( D.path());
  if (info.permission( QFileInfo::WriteUser)) {
    if (remove) {
      if (D.rmdir(D.path())) return TRUE;
      else {
	showWarning( tr("Warning"), tr("Cannot remove:\n")+D.path()+tr("\nno permission") );
	return FALSE;
      }
    } else return TRUE;
  } else {
      showWarning( tr("Warning"), tr("Cannot remove:\n")+D.path()+tr("\nno permission") );
      return FALSE;
  }
}

//----------------------------------------------------------------------------

bool CvsControl::removeFilesInTreeFromDisk(QString dir,bool remove) {

  QDir D( dir);
  D.setFilter( QDir::Dirs); //should list all dirs
  QStringList AllDirEntries = D.entryList();
  AllDirEntries.remove( "." );
  AllDirEntries.remove( ".." );

  if (!remove && (AllDirEntries.find( "CVS") != AllDirEntries.end())) return FALSE;
  AllDirEntries.remove( "CVS");

  QStringList::Iterator it;
  for ( it = AllDirEntries.begin(); it != AllDirEntries.end(); it++ ) {
    if (!removeFilesInTreeFromDisk( dir+"/"+(*it),remove)) return FALSE;
  }

  dir += "/";
  D.setFilter( QDir::Files | QDir::Hidden); //should list everything
  QStringList AllEntriesInDir = D.entryList();

  for ( it = AllEntriesInDir.begin(); it != AllEntriesInDir.end(); it++ ) {
    QString file = dir + (*it);
    if (remove) {
      QFile f( file);
      setPermission(f,READABLE | WRITEABLE);
      if (!f.remove()) {
	showWarning( tr("Warning"), tr("Cannot remove:\n")+file);
	return FALSE;
      }
    }
  }
  return TRUE;
}

//----------------------------------------------------------------------------

void CvsControl::doRemoveFileFromRepOrDisk( int RmFromRep, int RmFromDisk)
{
   QString tmp = "";
   QStringList AllFiles;
   QString fileName;
   m_AddedFilesList.clear();
   m_newFileNameList.clear();
   QString CurDir = m_pCurCvsDir->fullName() + "/";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   if( RmFromDisk ) {
       if( m_showWarning) {
	   //create dialog  
	   WarningDlg *dlg = new WarningDlg(
					    tr("Warning"), 
					    tr("This command will remove file(s) from disk!"), this);
	   
	   if(!dlg->exec()) {
	       delete dlg;
	       return;
	   }
	   m_showWarning = dlg->showWarningNextTime();
	   delete dlg;
       }
   }
   
   // we are sure we can delete files from disk if needed
   QListViewItem * myChild = m_pCvsFileList->firstChild();
   
   while(myChild) {
      if(myChild->isSelected()) {
         fileName = myChild->text(0).stripWhiteSpace();
         QString quotedFileName = masqWs(fileName);
	 m_AddedFilesList.append(fileName);
         QFile f( CurDir + fileName);
         
         if(f.exists()) {
            if( RmFromDisk ) {
		setPermission(f,READABLE | WRITEABLE);
		f.remove();
            } else {
               // make backup
               QString S = "__" + fileName;
               AllFiles.append( S );
               S = CurDir + S;
               // rename file
	       rename(f.name(),S);
            }
         }
         tmp += " " + quotedFileName;
      }
      myChild = myChild->nextSibling();
   }

   if ( tmp.isEmpty()) { //check if there are files selected
       showWarning( tr("Warning"), tr("There is no file selected") );
       return;
   }

   if( RmFromRep ) {
       // remove from repository
       QString dir = m_pCurCvsDir->fullName();
       QString command = "remove";
       QString files = tmp;
       if ( ! RmFromDisk) {
	   m_newFileNameList = AllFiles;
	   callInteractiveCmd( topModule, dir, command,
                               files, REMOVE_AND_RENAME_CMD,
                               ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                               NOROOT, this);
       }
       else callInteractiveCmd( topModule, dir, command,
                                files, REMOVE_CMD,
                                ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                                NOROOT, this);
   }
   else {
       if (RmFromDisk) {
	   for( QStringList::Iterator it = m_AddedFilesList.begin(); it != m_AddedFilesList.end(); ++ it ) {
	       m_pCurCvsDir->setEntry(*it,ES_missing);
	   }
       }
       m_AddedFilesList.clear();
       bool blocked = isGuiBlocked();
       if (!blocked) blockGUI(true);//FIXME: should be moved to the calling methods but first the warning dialog must be moved
       slot_checkStatusOfTree(m_pCurCvsDir,OVERRIDE);
       if (!blocked) blockGUI(false);
   }
}

//----------------------------------------------------------------------------

void CvsControl::stopCurAction()
{
   globalStopAction = true;
   
   if(m_interactiveCmdThreadIsRunning) {//nur fuer cvs thread
       m_interactiveCmdThreadIsRunning = false;   //"prellen" verhindern
       Timer::singleShot(100, this, SLOT(killInteractiveCmdThread()));
   }

}

//----------------------------------------------------------------------------

void CvsControl::killInteractiveCmdThread()
{
    //m_pInteractiveCmdThread can be 0 after cancelling cvs action by user
    if (m_pInteractiveCmdThread) m_pInteractiveCmdThread->exit();
}

//----------------------------------------------------------------------------

//write config
void CvsControl::writeCfg()
{
    conf->writeCfg();
}

//----------------------------------------------------------------------------

void CvsControl::removeDir() {

   if (!(m_pCurCvsDir = getSelectedDir())) {
       return;
   }
   if( m_showWarning) {
     //create dialog  
     WarningDlg *dlg = new WarningDlg( tr("Warning"), 
				       tr("This command will recursively remove\nfile(s) from disk and repository!"), this);
     
     if(!dlg->exec()) {
       delete dlg;
       return;
     }
     m_showWarning = dlg->showWarningNextTime();
     delete dlg;
   }
   
   QString dir = m_pCurCvsDir->fullName();

   if ( !(removeDirTreeFromDisk(dir, FALSE)) ) {
     if ( !( showYesNoQuestion( tr("Warning"), 
				tr("The directory contains files under CVS control:\n")+
				dir + tr("\nDelete anyway?") )
	     && removeFilesInTreeFromDisk(dir, TRUE)) ) {

       showWarning( tr("Error"),
		    tr("Failure during removal of:\n")+dir);
       return;
     }
   }

   QString command = "remove -f";
   QString files = "";
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
   callInteractiveCmd( topModule, dir, command,
                       files, REMOVEDIR_CMD,
                       ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                       NOROOT, this);
}

//----------------------------------------------------------------------------

void CvsControl::removeDirIfNoCvsRegisteredFiles() {

   Directory *dir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
   if(!dir) {
       showWarning( tr("Warning"), tr("There is no directory selected") );
       return;
   }

   if(!dir->loginOk(m_pMessages)) return;
   if( m_showWarning) {
       //create dialog  
       WarningDlg *dlg = new WarningDlg(
					tr("Warning"), 
					tr("This command will remove file(s) from disk!"), this);
       
       if(!dlg->exec()) {
	   delete dlg;
	   return;
       }
       m_showWarning = dlg->showWarningNextTime();
       delete dlg;
   }

   bool blocked = isGuiBlocked();
   if (!blocked) blockGUI(true);
   bool isTopLevel = (dir == dir->topDir());

   if ( !(removeDirTreeFromDisk(dir->fullName(), FALSE)) ) {

     if ( !( showYesNoQuestion( tr("Warning"),
				tr("The directory contains files under CVS control:\n")+
				dir->fullName() + tr("\nDelete anyway?") )
	     && removeDirTreeFromDisk(dir->fullName(), TRUE)) ) {

       showWarning( tr("Error"),
		    tr("Failure during removal of:\n")+dir->fullName() );
       if (!blocked) blockGUI(false);
       return;
     }
   }
   if (!isTopLevel) rereadProjectOfDir( (Directory*)dir->parent() );
   if (!blocked) blockGUI(false);
}

//----------------------------------------------------------------------------

void CvsControl::exportProject( QString dir,
				QString module,
				QString checkoutAs,
				QString /*checkoutAsDOption*/,
				QString repository,
				QString tmpCvsRsh,
				QString tmpRevTagDate,
				QString server,
				QString user,
				int sshAccess,
				int mode,
				int rshMode,
				bool tmpRWPermission)
{

  bool needcvsroot = false;
  QString cvsRoot;
  QString files = "";

  //check ssh access settings
  bUseSsh = false;
  bUseSshAgent = false;
  bUseSshAgentVars = false;
  switch( sshAccess) {
    case USESSH: {
      bUseSsh = true;
      break;
    }
    case USESSHAGENT: {
      bUseSshAgent = true;
      break;
    }
    case USESSHAGENTVARS: {
      bUseSshAgentVars = true;
      break;
    }
  }

  //modify application-wide lists
  adaptQStringList(module,&MODULEHISTORYLIST,HistorySize::g_profiles);
  adaptQStringList(checkoutAs,&CHECKOUTASLIST,HistorySize::g_profiles);
  adaptQStringList(m_coImportLocalDirectory,&WORKDIRHISTORYLIST,HistorySize::g_workdirs);
  adaptQStringList(user, &cvsUsers, HistorySize::g_profiles);
  adaptQStringList(server, &cvsServers, HistorySize::g_profiles);
  adaptQStringList(repository, &cvsRepositories, HistorySize::g_profiles);

  //set CVSROOT      
  if(mode == MODE_PSERVER) { //password server
    //set CVSROOT for pserver mode
    cvsRoot = ":pserver:";
    needcvsroot = true;
  } else if(mode == MODE_RSH) {//remote shell access
    needcvsroot = true;
    if(rshMode == RSH_EXT) {
      cvsRoot = ":ext:";
    } else {
      cvsRoot = ":server:";
    }
  }
  if (server.find(':') == -1) server += ":";
  if (needcvsroot) cvsRoot += user + "@" + server;
  cvsRoot +=  repository;
  
  if (tmpRWPermission) {
    files = "-r ";
  }
  files += CvsOptions::cmprStr();
  files += " export -N ";
  
  if (tmpRevTagDate.isNull()) {
    files += "-r HEAD ";
  } else if (!tmpRevTagDate.isEmpty()) {
    files += tmpRevTagDate + " ";
  }
  
  if (!checkoutAs.isEmpty()) {
    m_coModule =  checkoutAs;
    files += "-d " + masqWs(checkoutAs) + " ";
    files += masqWs(module);
  } else {
    m_coModule =  module;
    files += masqWs(m_coModule);
  }
  
  QString topModule = QString::null;
  callInteractiveCmd( topModule, dir, cvsRoot,
                      files, EXPORT_CMD, tmpCvsRsh, false);
}

//----------------------------------------------------------------------------

void CvsControl::createDirTreePatch(Directory * workDir,const QString& patchFile) {

  QString hlp;
  QString cvsRoot = CvsOptions::cmprStr() + " -Q diff -N ";
  if (bDiffIgnoreWhiteSpace) cvsRoot += "-b -B ";
  if (CvsOptions::g_bUnifiedDiffOutput) cvsRoot += "-u ";
  cvsRoot +=  "> " + patchFile;
  QString file = "";
  QString dir = workDir->fullName();
  QString topModule = workDir->topCvsDir()->relativeName();
  callInteractiveCmd( topModule, dir, cvsRoot,
                      file, NOT_INTERACTIVE_CMD,
                      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                      NOROOT);

}

//----------------------------------------------------------------------------

/*
 * This funktion should export the workdir the user klicked on to export.
 * Since the cvs export funktion only exports cvs registered files from the
 * repository, we need here the ability to only export the cvs registered
 * files as well, but maybe in modified state, just as they are currently
 * in the workdir. If a file for example is missing, we don't export it, and
 * so on. So we have here a funktion that exports our working dir, but without
 * ignored or not cvs registered files. We could build a tarbal without .o and
 * whatever files, but wouldn't need to run make clean before (very anoying in
 * big projects since it takes quite long to recompile).
 */
void CvsControl::exportFromModuleDir() {

    if (!(m_pCurCvsDir = getSelectedDir())) {
      return;
    }

    QString exportDir = m_pCurCvsDir->shortName()+"_export";
    QString parentDir = m_pCurCvsDir->fullName();
    parentDir = parentDir.left(parentDir.findRev("/"));
    QFileDialog* fd = new QFileDialog( parentDir, tr("Directories"), this, "file dialog", TRUE );
    fd->setCaption("Export project: "+m_pCurCvsDir->shortName());
    fd->setMode( QFileDialog::AnyFile );
    fd->setSelection(exportDir);
    if ( fd->exec() == QDialog::Accepted) {
      exportDir = fd->selectedFile();
      if ( exportDir.isEmpty()) return;
    } else {
      return;
    }

    QFile file(exportDir);
    if (file.exists()) {
      if (!showYesNoQuestion(tr("Warning"),
			     tr("The directory: ")
			     + exportDir
			     + tr(" allready exists.\nOverwrite it?"))) return;
      file.remove();
    }

    QApplication::setOverrideCursor(waitCursor);
    if (!m_pCurCvsDir->recCopyCvsFiles(exportDir)) showWarning( (tr("Error")+", "+tr("export stopped")),
								tr("Couldn't create directories"));
    QApplication::restoreOverrideCursor();
}

//----------------------------------------------------------------------------

void CvsControl::callCVS(QString topModule,
			 QString dir,
			 QString command,
			 QString files,
			 int cmd)
{
  callInteractiveCmd( topModule, dir, command,
		      files, cmd,
		      ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
		      NOROOT, this);
}

/*****************************************************************************
 *
 * Call InteractiveCmdThread!
 *      Params: topModule -        the toplevel (dir-)name of the module
 *              dir -              the dir to invoke the command in
 *              cvsRootOrCommand - the CVS/Root or command ==> noroot
 *              files -            the files to invoke the command with
 *              icmd -             Command that should be called (ie.CommandInterface.h)
 *              cvsRsh -           the remote shell to use
 *              noroot -           if noroot is true the command except 'cvs'
 *                                 has to be appended to cvsRootOrCommand, no -d option will be specified
 *              instance -         the instance (for ex. a dialog implementing CommandInterface.h)
 *                                 if instance is given with 'this', afterCall(icmd) in 'this' instance
 *                                 will be called when cvs-call has finished
 *
 *      to append additional interactive commands, InteractiveCmdThread.cpp and 
 *              InteractiveCmdThread.h have to be edited and the additional rules
 *              must be put to the switch-clauses
 *
 *      to call cvs from external classes, inherit CommandInterface, implement void afterCall(int cmd)
 *      and start cvscall: void callInteractive( dir, cvsRootOrCommand, files, icmd, cvsRsh, noroot);
 *      calling cvsCallFailed(icmd,instance) makes shure that afterCall gets called in error situation too.
 *
 ****************************************************************************/
void CvsControl::callInteractiveCmd( QString& topModule,
				     QString& dir,
				     QString& cvsRootOrCommand,
				     QString& files,
				     int icmd,
				     QString& cvsRsh,
				     bool noroot /*=FALSE*/,
				     CommandInterface *instance /*=NULL*/) {

    if (m_interactiveCmdThreadIsRunning) {//make shure no two commands run simultaneous
        QString msg = "THREAD WARNING: InteractiveCmdThread is allready running, call: "+QString::number(icmd)+" canceled";
	if (Debug::g_pLog) Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,msg);
	cvsCallFailed(icmd,instance);//makes shure automatic recursive calls don't get mixed up
	return;
    }

    QString command;
    setStatusText("");
    m_pLastCvsCallDir = m_pWorkBenchTree->find(dir.stripWhiteSpace());
    QDir workDir(dir.stripWhiteSpace());
    if (!workDir.exists()) {//would run in LinCVS's workdir otherwise
	cvsCallFailed(icmd,instance);
	return;
    }

    // check for required login
    QString cvsRootLine = "";
    QString method;
    QString user;
    QString passwd;
    QString host;
    int port;
    QString rootDir;

    if (noroot) { // We have to check the cvs connection method
	QFile f;
	f.setName(dir + "/CVS/Root");
	if(f.open(IO_ReadOnly)) {
	    QTextStream textStream(&f); 
	    cvsRootLine = textStream.readLine();
	    f.close();
	}
    }
    else cvsRootLine = umasqWs(cvsRootOrCommand);//only umasq for local use, not for cmd call!!!
    if (!extractCVSROOT( cvsRootLine, method, user, passwd, host, port, rootDir)) {
        if (Debug::g_pLog) Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"ERROR in cvsRootLine: ->"+cvsRootLine+"<-");
	cvsCallFailed(icmd,instance);
	return;
    }

    globalStopAction = false;

    if( method == "pserver") { // password server

	if (!isInCvsPass(cvsRootLine))
	{

	    bool ok = false;

	    bool blocked = isGuiBlocked();
	    if (blocked) blockGUI(false);

#ifdef QT_NEED_PWDIALOGPATCH
	    QString pwd;
	    PasswordDialogImpl *pwDialog = new PasswordDialogImpl(this);
	    pwDialog->Label->setText( user + "@" + host + ":" + QString::number(port) + rootDir);
	    if (pwDialog->exec()) {
	      pwd = pwDialog->TextField->text();
	      ok = true;
	    }
	    delete pwDialog;
#else
	    QString pwd = QInputDialog::getText("Enter CVS password",
						user + "@" + host + ":" + QString::number(port) + rootDir, QLineEdit::Password,
						QString::null, &ok, this, "Password Dialog");
#endif

	    if (blocked) blockGUI(true);

	    if (!ok) {
	      cvsCallFailed(icmd,instance);
	      return;
	    }

	    globalStopAction = false;

	    if (!blocked) blockGUI(true);

	    Login *l = new Login( m_pMessages, user, host, port, rootDir);
	    l->doLogin(pwd);
	    delete l;

	    if (!blocked) blockGUI(false);

            if(!isInCvsPass(cvsRootLine)) {
		if (!KILLED) {
		    showWarning(tr("Warning"), tr("Login failed.") );
		}
		cvsCallFailed(icmd,instance);
		return;
            }
	}

	//export proxy settings
	QString proxyString = getProxy(user+"@"+host);
	if (!proxyString.isEmpty()) {
	    if ( (icmd==RELEASE_D_CMD) || (icmd==RELEASE_CMD)) {//cvs bug prevents proxy usage
		showWarning( tr("Warning"), tr("can't release through proxy tunnel.") );
		cvsCallFailed(icmd,instance);
		return;
	    }
	    command += "export PSERVER_PROXY=" + proxyString + " && ";
	}

    } else if ( method == "ext") { // external connect method

	command += "export CVS_RSH=" + cvsRsh + " && ";

	//need project specific settings here:
	QString cvsServerCmd;
	if ( projectSettings->get(topModule,CVSSERVER,cvsServerCmd)) {
	  if (!cvsServerCmd.isEmpty()) {
	    command += "export CVS_SERVER=" + cvsServerCmd + " && ";
	  }
	}

	int sshAccess = NOSSH;
	if ( projectSettings->get(topModule,SSHACCESS,sshAccess)) {
	  bUseSsh = false;
	  bUseSshAgent = false;
	  bUseSshAgentVars = false;
	  switch( sshAccess) {
	    case USESSH: {
	      bUseSsh = true;
	      break;
	    }
	    case USESSHAGENT: {
	      bUseSshAgent = true;
	      break;
	    }
	    case USESSHAGENTVARS: {
	      bUseSshAgentVars = true;
	      break;
	    }
	  }
	}

	if (bUseSsh) {//use modified ssh with graphical login
	    if (WINVERSION) {

	      bool ok = false;

	      bool blocked = isGuiBlocked();
	      if (blocked) blockGUI(false);

#ifdef QT_NEED_PWDIALOGPATCH
	      QString pwd;
	      PasswordDialogImpl *pwDialog = new PasswordDialogImpl(this);
	      pwDialog->Label->setText( user + "@" + host + ":" + QString::number(port) + rootDir);
	      if (pwDialog->exec()) {
		pwd = pwDialog->TextField->text();
		ok = true;
	      }
	      delete pwDialog;
#else
		QString pwd = QInputDialog::getText("Enter CVS password",
			user + "@" + host + ":" + QString::number(port) + rootDir, QLineEdit::Password,
			QString::null, &ok, this, "Password Dialog");
#endif

	        if (blocked) blockGUI(true);

		if (!ok) {
		  cvsCallFailed(icmd,instance);
		  return;
		}

		command += "export SSH_LinCVS_PW=";
		command += pwd;
		command += " && ";

	    } else {

		command += "export SSH_LinCVS=LinCVSMagic && ";

	    }

	} else if ( (cvsRsh.find("ssh",0,false) >= 0) && (bUseSshAgent || bUseSshAgentVars) ) { // special for ssh
	    if (!sshAgentStarted) {
	        bool blocked = isGuiBlocked();
	        if (!blocked) blockGUI(true);
		setStatusText(tr("Starting ssh-agent.  Please wait..."));
		startSshAgent();//(check for ssh agent variables if bUseSshAgentVars) and start ssh agent
		if (!blocked) blockGUI(false);
		if (!sshAgentStarted) {
		    setStatusText(tr("couldn't start Ssh-agent"));
		    cvsCallFailed(icmd,instance);
		    return;
		} else {
		    setStatusText(tr("Ssh-agent started.  Please wait..."));
		}
            }
	    
	    if (!sshAgentIsRunning) checkSshAgent();
	    if (!sshAgentIsRunning) {//cancel the command if ssh agent could not be started
		cvsCallFailed(icmd,instance);
		return;
	    }

	    if (WINVERSION) {//todo: setup for win ssh-agent

	    } else {

	      command += "export ";
	      command += envSSH_AGENT_PID;
	      command += " && ";
	      command += "export ";
	      command += envSSH_AUTH_SOCK;
	      command += " && ";

	      command += "export SSH_LinCVS=LinCVSMagic && ";

	    }

	} else {
	  showWarning( tr("Warning"),
		       tr("Connect method is 'ext'")
		       +",\n"
		       +tr("but there is no remote shell specified, aborting") );
	  cvsCallFailed(icmd,instance);
	  return;
	}

    } else { // local connect method

    }

    //append cvsRoot (ev. including additional commands) to command
    command += "cvs ";
    if (!noroot) command +="-d ";
    command += cvsRootOrCommand;

    //complete the command
    switch(icmd) {
      case RELEASE_CMD: {
	  command += " release";
	  break;
      }
      case RELEASE_D_CMD: {
	  command += " release -d";
	  break;
      }
      case UNEDIT_CMD: {
	  command += " unedit";
	  break;
      }
      default: {
	  break;
      }
    }

    //change to absolute cvs path if given
    if (!CVSPATH.isEmpty()) {
	command.replace( QRegExp("^cvs\\s"), masqWs(CVSPATH)+" ");
	command.replace( QRegExp("&&\\scvs\\s"), "&& "+masqWs(CVSPATH)+" ");
    }

    if (!m_chainedCommand) {//don't clear output window on chained calls
      m_pMessages->reset();
    } else {
      m_pMessages->insert("\n========================== "+tr("next command")+" ===========================\n\n");
    }
    m_pMessages->setMode(icmd);
    m_pCvsBuffer->clear();

    if (!isGuiBlocked()) blockGUI(true);
    setStatusText(tr("Operation in progress.  Please wait ..."));
    suspendDirWatch();
    m_pPixmapTimer->slotStart();

    m_debugCvsCallString = "cvscall:\ndir: ->"+
      workDir.absPath()+"<-\ncommand: ->"+
      command+"<-\nicmd: ->"+
      QString::number(icmd)+"<-\nfiles: ->"+
      files+"<-\n";

    if (m_pInteractiveCmdThread) delete m_pInteractiveCmdThread;
    m_pInteractiveCmdThread = new InteractiveCmdThread( this, m_pCvsBuffer, m_pMessages, workDir, command, icmd, files, instance);

    connect(m_pInteractiveCmdThread,SIGNAL(requestReceived(MESSAGE)),this,SLOT(handleMessage(MESSAGE)));
    connect(m_pInteractiveCmdThread,SIGNAL(cvsCommandFinished()),this,SLOT(cvsCommandFinished()));

    if (m_chainedCommand) {//delay chained cvs call, to make CVS dir timestamp detectable as beeing changed
	int from = QTime::currentTime().second();
	while (from == QTime::currentTime().second()) {
	    qApp->processEvents();
	    wait(100);
	}
	m_chainedCommand = false;
    }

    if (m_pInteractiveCmdThread->start()) {
	m_interactiveCmdThreadIsRunning = true;
    }
}

//----------------------------------------------------------------------------

void CvsControl::cvsCallFailed(int cmd, CommandInterface* instance) {

  if (Debug::g_pLog) Debug::g_pLog->log(Debug::LL_INFO, "cvs call failed");
  m_pLastCvsCallDir = NULL;
  if (instance) {
    instance->afterCall( cmd, m_pCvsBuffer, true);
  }
}

//----------------------------------------------------------------------------

void CvsControl::cvsCommandFinished() {

//   qDebug("-->"+m_pCvsBuffer->readAll()+"<--");
    if (!globalStopAction) {
	finishInteractiveCmd();
    } else {
	m_pMessages->append("\n" + tr("Action interrupted by user!") + "\n");
	finishInteractiveCmd();
    }
}

//----------------------------------------------------------------------------

void CvsControl::handleMessage(MESSAGE m) {
    m_pPixmapTimer->slotStop();
    setStatusText(tr("waiting for your answer ..."));
    switch( m.type) {
       case Y_N: {
	 blockGUI(false);
	 if (showYesNoQuestion(tr("cvs asks:"), m.request + "\n\n") ) {
	   m.reply = "y\n\0";
	   m_pInteractiveCmdThread->sendReply( m);
	 } else {
	   m.reply = "n\n\0";
	   m_pInteractiveCmdThread->sendReply( m);
	 }
	 blockGUI(true);
	 break;
       }
      default: {
	Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"unknown cvs message to handle: ->"+m.request+"<-");
	break;
      }
    }
    m_pPixmapTimer->slotStart();
    setStatusText(tr("Operation in progress.  Please wait ..."));
}

//----------------------------------------------------------------------------

void CvsControl::finishInteractiveCmd() {

   CommandInterface *afterCallInstance = m_pInteractiveCmdThread->getInstance();
   int cmd = m_pInteractiveCmdThread->getCommand();

   if (m_curCommentFileName.length()) QFile::remove(m_curCommentFileName);
   
   bool callFailed = !m_interactiveCmdThreadIsRunning;
   m_interactiveCmdThreadIsRunning = false;

   if (m_pLastCvsCallDir) {
     QDateTime t = QDateTime::currentDateTime();
     m_pLastCvsCallDir->setCvsCallTime(t);
     m_pLastCvsCallDir = NULL;
   }

   if ( Debug::g_logLevel >= Debug::LL_INFO) {
     QString str;
     QTextStream stream(&str,IO_WriteOnly);
     fillCmdDebugStream(stream);
     Debug::g_pLog->log(Debug::LL_INFO, str);
   }

   if (afterCallInstance) {
       m_chainedCommand = true;
       afterCallInstance->afterCall(cmd,m_pCvsBuffer,callFailed);
       afterCallInstance = NULL;
       if (m_interactiveCmdThreadIsRunning) {// for loop commands, (i.e. commands calling another cvs command) don't continue
	   return;
       } else {
	   m_chainedCommand = false;
       }
   } else {
     m_pCurCvsDir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
     if (m_pCurCvsDir) {
       slot_checkStatusOfTree(m_pCurCvsDir,OVERRIDE);
     }
   }

   expurgateProjectSettings();

   m_pPixmapTimer->slotStop();
   globalStopAction = false;
   blockGUI(false);
   resumeDirWatch();

   //set status info
   m_pCurCvsDir = static_cast<Directory*>(m_pWorkBenchTree->currentItem());
   if (m_pCurCvsDir && m_pCurCvsDir->isSelected() ) {
     setStatusText(m_pCurCvsDir->fullName());
   }
   setStatusText( tr("Ready"), 2000);
}

//----------------------------------------------------------------------------

void CvsControl::release() {
    doRelease( false);
}

void CvsControl::release_and_remove() {
    doRelease( true);
}

void CvsControl::doRelease(bool remove) {

  if (!(m_pCurCvsDir = getSelectedDir())) {
    return;
  }

   QString dir = m_pCurCvsDir->fullName();
   QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();

   QString cvsRoot = m_pCurCvsDir->getCvsRoot();
   cvsRoot = masqWs(cvsRoot);
   QString projectname = masqWs(dir.right(dir.length()-dir.findRev( "/",-2)-1));//local workdir name might be different from module name
   dir = dir.left( dir.findRev( "/",-2)); // one dir up

   if ( !remove) callInteractiveCmd( topModule, dir, cvsRoot,
                                     projectname, RELEASE_CMD,
                                     ExtApps::g_cvsRsh.path);  //additional options of cvsRsh not supported yet
   else callInteractiveCmd( topModule, dir, cvsRoot,
                            projectname, RELEASE_D_CMD,
                            ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                            false, this);
}

//----------------------------------------------------------------------------

void CvsControl::afterCall(int cmd,CvsBuffer*,bool failed) {
    saveViewportPos(OVERRIDE);
    switch(cmd) {
       case LOG_CMD: {
	   if (failed) return;
	   QString Fname;
	   QString Frevision;
	   QListViewItem *myChild = m_pCvsFileList->firstChild();
	   int count = 0;
	   while(myChild) {
	       if(myChild->isSelected())  {
		   Fname = myChild->text(0).stripWhiteSpace();
		   Frevision = myChild->text(1).stripWhiteSpace();
		   count++;
	       }
	       myChild = myChild->nextSibling();
	   }
           if (count != 1) {
	     Fname = "";
	     Frevision = "";
	   }

	   LogDialogImpl * DD = new LogDialogImpl (m_pTools->whatsThisIconSet(),
						   this, "LogDlg", false,
                                                   Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
                                                   Qt::WStyle_DialogBorder | Qt::WType_Dialog |
                                                   WDestructiveClose | Qt::WStyle_ContextHelp);
           
	   connect( DD, SIGNAL(editFile(QString)), this, SLOT(viewFileWithPath(QString)));
	   DD->parseCvsLog( (QDir*)m_pCurCvsDir->topCvsDir(), m_pCurCvsDir->fullName(), Fname, Frevision , m_pCvsBuffer );
	   DD->show();//must use WDestructiveClose to avoid memory leak
	   break;
       }
      case LOG_DIR_CMD: {
	   if (failed) return;
	   LogDialogImpl * DD = new LogDialogImpl (m_pTools->whatsThisIconSet(),
						   this, "LogDlg", false,
                                                   Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
                                                   Qt::WStyle_DialogBorder | Qt::WType_Dialog |
                                                   WDestructiveClose | Qt::WStyle_ContextHelp);
           
	   connect( DD, SIGNAL(editFile(QString)), this, SLOT(viewFileWithPath(QString)));
	   DD->parseCvsLog( (QDir*)m_pCurCvsDir->topCvsDir(), m_pCurCvsDir->fullName(), m_pCurCvsDir->shortName(), "", m_pCvsBuffer );
	   DD->show();//must use WDestructiveClose to avoid memory leak
	break;
      }
       case STATUS_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case STATUS_FILES_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case HISTORY_CMD: {
	   if (failed) return;
           HistoryDialogImpl *pHistDlg = new HistoryDialogImpl(
               m_pTools->whatsThisIconSet(),
               this, "HistoryDlg",
	       false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
	       WDestructiveClose | Qt::WStyle_ContextHelp);
	   connect( pHistDlg, SIGNAL(editFile(QString)), this, SLOT(viewFileWithPath(QString)));
           pHistDlg->parseHistory (m_pCvsBuffer, m_pCurCvsDir->repository(), (QDir*)m_pCurCvsDir->topCvsDir());
           pHistDlg->show();
           break;
       }
       case DIFF_CMD: {
	   if (failed) return;
           QString fname;
           QString rev;
           getNameAndRevOfFirstSelectedFile( &fname, &rev);
           DiffDialogImpl *l = new DiffDialogImpl(
               m_pTools->whatsThisIconSet(),
               this, "DiffDlg", false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
	       WDestructiveClose | Qt::WStyle_ContextHelp);
               
           l->parseCvsDiff(m_pCvsBuffer, fname, rev);
           l->show();
           break;
       }
       case CHECKOUT_NO_PRUNE_CMD:
       case CHECKOUT_CMD: {
	   if (failed) return;
           finishCheckOut(m_pCvsBuffer);
           break;
       }
       case RELEASE_D_CMD: {
	   if (failed) return;
           if (m_pCurCvsDir == m_pCurCvsDir->topDir()) {
	       m_pWorkBenchTree->setSelected( m_pCurCvsDir, TRUE );
	       removeProjectSlot();
	   } else {
	       Directory * dir = (Directory *)m_pCurCvsDir->parent();
	       if (dir) {
		 rereadProjectOfDir(dir);
	       }
	   }
	   break;
       }
       case RENAME_CMD: {
	   if (failed) return;
	   if (!m_newFileName.isEmpty()) {
	       QString dir = m_pCurCvsDir->fullName();
	       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
	       QString command = CvsOptions::cmprStr() + " add";
	       bool asBinary = m_pCurCvsDir->isBinary(m_AddedFilesList.first());
	       if ( asBinary) command += " -kb";
	       m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, REMOVE_CMD, &m_AddedFilesList);//set parse cmd to REMOVE_CMD for proper state
	       m_pCurCvsDir->activateItem(FALSE);
	       recoverViewportPos();
	       m_AddedFilesList.clear();
	       m_AddedFilesList.append(m_newFileName);
	       m_newFileName = masqWs(m_newFileName);
	       //don't call RENAME_CMD -> endless loop!
	       callInteractiveCmd( topModule, dir, command,
                                   m_newFileName, ADD_CMD,
                                   ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                                   NOROOT, this);
	       m_newFileName = "";
	   }
	   else {
	       m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, STATUS_CMD);
	       m_pCurCvsDir->activateItem(TRUE);
	       recoverViewportPos();
	   }
	   break;
       }
       case EXT_DIFF_CMD: {
           if (failed) return;
           QString diffPrg = "\"" + ExtApps::g_diffProg.path + "\" ";
           QStringList nameList;
           nameList.append(m_curTmpDiffFileName);
           nameList.append(m_curDiffFileName);
           nameList.append("");  //3way diff from within logdlg --> there we can use 3 parameters
           QString options = ExtApps::g_diffProg.options;
           if (masqedFilenamesForPlaceholders(options, nameList)) diffPrg += options;
           else {// no options, we add filenames assuming the user want it so ;-)
              diffPrg += masqWs(m_curTmpDiffFileName.latin1()) + " ";
              diffPrg += masqWs(m_curDiffFileName.latin1());
           }
           runExternal(diffPrg);
           m_curDiffFileName = "";
           break;
       }
       case REMOVE_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd, &m_AddedFilesList);
	   m_pCurCvsDir->activateItem(FALSE);
           m_AddedFilesList.clear();
           recoverViewportPos();
	   break;
       }
       case REMOVE_AND_RENAME_CMD: {
	   if (failed) return;
	   // restor backup files 
	   QString CurDir = m_pCurCvsDir->fullName() + "/";
	   for( QStringList::Iterator it = m_newFileNameList.begin(); it != m_newFileNameList.end(); ++ it ) {
	       QString S1 = CurDir + (*it);
	       QString S2 = CurDir + (*it).mid(2);
	       // remove __
	       rename(S1,S2);
	   }
	   m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, REMOVE_CMD, &m_AddedFilesList);
	   m_pCurCvsDir->activateItem(TRUE);
	   m_AddedFilesList.clear();
	   m_newFileNameList.clear();
	   recoverViewportPos();
	   break;
       }
       case ADD_CMD: {
	   if (failed) return;
	   m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd, &m_AddedFilesList);
	   m_pCurCvsDir->activateItem(TRUE);
	   if( !m_FirstPosAddedDir.isEmpty()) {
	       m_pCurCvsDir = rereadProjectOfDir( m_pCurCvsDir);//selects the given dir and opens from top to parent if available
	       if( m_pCurCvsDir) {
		 // make directory active and open so that content of directory is shown
		 m_pCurCvsDir->setOpen(TRUE);
		 Directory *pOldCurCvsDir = m_pCurCvsDir;
		 m_pCurCvsDir = m_pCurCvsDir->searchDirOfPath( m_FirstPosAddedDir );
		 if (m_pCurCvsDir) {
		   m_pWorkBenchTree->setSelected( m_pCurCvsDir, TRUE );
		   m_pCurCvsDir->activateItem(FALSE);
		 } else {//adding of dir failed
		   m_pCurCvsDir = pOldCurCvsDir;
		   m_pCurCvsDir->activateItem(FALSE);
		   QString caption = tr("LinCVS - Error");
		   QString msg = tr("Adding of directory %1 failed!\nMaybe permission denied.");
		   showWarning( caption, msg.arg(m_FirstPosAddedDir) );
		 }
	       }
	   } else {
	       recoverViewportPos();
	   }
	   m_FirstPosAddedDir = "";
	   m_AddedFilesList.clear();
	   break;
       }
       case SHOWEDITORS_CMD: {
	   if (failed) return;
           EditorsDialogImpl *pEditorsDlg = new EditorsDialogImpl (
               m_pTools->whatsThisIconSet(),
	       this, "EditorsDlg",
	       false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
               WDestructiveClose | Qt::WStyle_ContextHelp);
           
           pEditorsDlg->parseEditors (m_pCvsBuffer, m_pCurCvsDir->repository());
           pEditorsDlg->show();
	   break;
       }
       case SHOWWATCHERS_CMD: {
	   if (failed) return;
           WatchersDialogImpl *pWatchersDlg = new WatchersDialogImpl(
               m_pTools->whatsThisIconSet(),
	       this, "WatchersDlg",
               false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
	       WDestructiveClose | Qt::WStyle_ContextHelp);

           pWatchersDlg->parseWatchers (m_pCvsBuffer, m_pCurCvsDir->repository());
           pWatchersDlg->show();
           break;
       }
       case ANNOTATE_CMD: {
	   if (failed) return;
           QString fname;
           QString rev;
           getNameAndRevOfFirstSelectedFile( &fname, &rev);
           if (fname.isEmpty()) {
               fname = m_pCurCvsDir->repository();
           }
           QString tmp = "";
           AnnotateDialogImpl *pAnnotateDlg = new AnnotateDialogImpl (
		m_pTools->whatsThisIconSet(),
		this, "AnnotateDlg",
               false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
               WDestructiveClose | Qt::WStyle_ContextHelp);

           pAnnotateDlg->parseCvsAnnotate (tmp, fname, m_pCvsBuffer, rev);
           pAnnotateDlg->show();
           break;
       }
       case ANNOTATE_GREP_LINE_CMD: {
	   if (failed) return;
           QString fname;
           QString rev;
           getNameAndRevOfFirstSelectedFile( &fname, &rev);
           if (fname.isEmpty()) {
               fname = m_pCurCvsDir->repository();
           }
           QString tmp = "";
           AnnotateDialogImpl *pAnnotateDlg = new AnnotateDialogImpl (
	       m_pTools->whatsThisIconSet(),
               this, "AnnotateGrepLineDlg",
               false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
               WDestructiveClose | Qt::WStyle_ContextHelp);

           pAnnotateDlg->parseCvsAnnotateGrepLine (tmp, fname, m_pCvsBuffer, rev, m_searchLine, m_caseSensitive);
           pAnnotateDlg->show();
           break;
       }
       case ANNOTATE_GREP_LINES_CMD: {
	   if (failed) return;
           QString fname;
           QString rev;
           getNameAndRevOfFirstSelectedFile( &fname, &rev);
           if (fname.isEmpty()) {
               fname = m_pCurCvsDir->repository();
           }
           QString tmp = "";
           AnnotateDialogImpl *pAnnotateDlg = new AnnotateDialogImpl (
	       m_pTools->whatsThisIconSet(),
	       this, "AnnotateGrepLinesDlg",
               false,
               Qt::WStyle_SysMenu | Qt::WStyle_MinMax | 
               Qt::WStyle_DialogBorder | Qt::WType_Dialog |
               WDestructiveClose | Qt::WStyle_ContextHelp);

           pAnnotateDlg->parseCvsAnnotateGrepLines (tmp, fname, m_pCvsBuffer, rev, &m_searchLines, m_caseSensitive);
           pAnnotateDlg->show();
           break;
       }
       case LOCKFILE_CMD: {
	   if (failed) return;
           QString s = m_pCvsBuffer->textLine(m_pCvsBuffer->numLines()-1);
           if (s.find("done")>-1) {
               QString dir = m_pCurCvsDir->fullName();
	       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
               QString fname;
               QString rev;
               getNameAndRevOfFirstSelectedFile( &fname, &rev);
               fname = masqWs(fname);
               QString command = "admin -l" + rev;
               callInteractiveCmd( topModule, dir, command,
                                   fname, NOT_INTERACTIVE_CMD,
                                   ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                                   NOROOT);
           }
           break;
       }
       case UNLOCKFILE_CMD: {
	   if (failed) return;
           QString s = m_pCvsBuffer->textLine(m_pCvsBuffer->numLines()-1);
           if (s.find("done")>-1) {
               QString dir = m_pCurCvsDir->fullName();
	       QString topModule = m_pCurCvsDir->topCvsDir()->relativeName();
               QString fname;
               QString rev;
               getNameAndRevOfFirstSelectedFile( &fname, &rev);
               fname = masqWs(fname);
               QString command = "admin -u" + rev;
               callInteractiveCmd( topModule, dir, command,
                                   fname, NOT_INTERACTIVE_CMD,
                                   ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
                                   NOROOT);
           }
           break;
       }
       case AUTOUPDATE_DIR_NO_PRUNE_CMD:
       case AUTOUPDATE_DIR_CMD: {// no "if (failed) return;" because we don't want to interrupt the recursive loop
	   Timer::singleShot(0,this,SLOT(slot_checkAutoUpdate()));//need singleShot, since otherwise errors would run endless
           break;
       }
       case MERGEFILES_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, UPDATE_FILES_CMD);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case MERGEDIR_NO_PRUNE_CMD:
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, UPDATE_DIR_CMD);
	   m_pCurCvsDir->activateItem(FALSE);
           recValidateCvsFilesStatus();
           recoverViewportPos();
           break;
       case MERGEDIR_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, UPDATE_DIR_CMD);
	   m_pCurCvsDir->activateItem(FALSE);
           recValidateCvsFilesStatus();
           recoverViewportPos();
           break;
       }
       case REMOVEDIR_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
	   break;
       }
       case RESURRECT_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case MERGE_REV_INTO_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }  
       case COMMIT_CMD: {
	   if (failed) return;
           m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	   m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case UPDATE_FILES_CMD: {
	 if (failed) return;
	 m_pCurCvsDir->removeTmpEntries();
	 if (!m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd)) {
	   showWarning(tr("CVS Info"),m_pCurCvsDir->getLastError());
	 }
	 m_pCurCvsDir->activateItem(FALSE);
           recoverViewportPos();
           break;
       }
       case UPDATE_VIRTUAL_DIR_CMD: {
	  if (failed) return;
	  bool fromVirtualDir = false;
	  QString name = "";
	  Directory * pOldDir = m_pCurCvsDir;

	  QStringList::Iterator iter;
	  for (iter = m_newFileNameList.begin(); iter != m_newFileNameList.end(); iter++) {
	    name = (*iter);
	    if (name.contains("/")) {//called from virtual dir in workbench
	      QString tmpName = name.mid(pOldDir->fullName().length()+1);
	      m_pCurCvsDir->removeTmpEntries(tmpName.left(tmpName.find("/")));
	      fromVirtualDir = true;
	      break;
	    } else {
	      m_pCurCvsDir->removeTmpEntries(name);
	    }
	  }
	  m_newFileNameList.clear();
	  if (!m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, UPDATE_DIR_CMD)) {
	    showWarning(tr("CVS Info"),m_pCurCvsDir->getLastError());
	  }
	  if (fromVirtualDir) {
	    Directory * tmpDir = NULL;
	    if ( (tmpDir = openTreeAndActivate(m_pCurCvsDir->topDir()->fullName(),name)) ) {
	      m_pCurCvsDir = tmpDir;
	    } else {
	      tmpDir = m_pWorkBenchTree->findBestMatch(name);
	      if (tmpDir) {
		m_pCurCvsDir = tmpDir;
		m_pWorkBenchTree->setSelected(m_pCurCvsDir,true);
	      } else {
		m_pCurCvsDir = NULL;
	      }
	    }
	  } else {
	    m_pCurCvsDir->activateItem(FALSE);
	    recoverViewportPos();
	  }
	  break;
       }
       case UPDATE_DIR_CMD: {
	 if (failed) return;
	 QString name = m_pCurCvsDir->fullName();
	 m_pCurCvsDir->recRemoveTmpEntries(FALSE);
	 if (!m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd)) {
	     showWarning(tr("CVS Info"),m_pCurCvsDir->getLastError());
	 }
	 m_pCurCvsDir = m_pWorkBenchTree->findBestMatch(name);
	 if (m_pCurCvsDir) {
	   m_pWorkBenchTree->setSelected(m_pCurCvsDir,true);
	   m_pCurCvsDir->activateItem(FALSE);
	 }
	 recoverViewportPos();
	 break;
       }
       case QUERY_UPDATE_CMD: {
	 if (failed) return;
	 m_pCurCvsDir->recRemoveTmpEntries(FALSE);
	 m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	 m_pCurCvsDir->activateItem(FALSE);
	 recoverViewportPos();
	 break;
       }
       case QUERY_UPDATE_ALL_CMD: {
	 if (failed) return;
	 m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd);
	 m_pCurCvsDir->activateItem(FALSE);
	 recoverViewportPos();
	 break;
       }
       case MOVE_FILES_CMD: {
	 if (failed) {
	   showWarning(tr("LinCVS ERROR"),tr("couldn't run cvs command"));
	   return;
	 }
	 moveCvsFiles();
	 break;
       }
       case REVERT_DIR_CMD: {
	 if (failed) {
	   showWarning(tr("LinCVS ERROR"),tr("couldn't run cvs command"));
	   return;
	 }
	 revertToRevDir();
	 break;
       }
       case REVERT_FILES_CMD: {
	 if (failed) {
	   showWarning(tr("LinCVS ERROR"),tr("couldn't run cvs command"));
	   return;
	 }
	 revertToRevFile();
	 break;
       }
       case GET_TAG_INFO_PROJECT_CMD: {
	 if (failed) {
	   showWarning(tr("LinCVS ERROR"),tr("couldn't run cvs command"));
	   return;
	 }
	 QStringList projectTagList;
	 projectSettings->get(m_pCurCvsDir->topCvsDir()->relativeName(),PROJECTTAGLIST,projectTagList);
	 m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd, &projectTagList);
	 projectTagList.sort();
	 projectSettings->set(m_pCurCvsDir->topCvsDir()->relativeName(),PROJECTTAGLIST,projectTagList);
	 emit tagListFetched();
	 break;
       }
       case GET_TAG_INFO_FILES_CMD: {
	 if (failed) {
	   showWarning(tr("LinCVS ERROR"),tr("couldn't run cvs command"));
	   return;
	 }
	 m_pCurCvsDir->parseCvsCallResult( m_pCvsBuffer, cmd, &m_currentTagList);
	 m_currentTagList.sort();
	 //now add to projectTagList
	 if (!m_currentTagList.isEmpty()) {
	   QStringList projectTagList;
	   projectSettings->get(m_pCurCvsDir->topCvsDir()->relativeName(),PROJECTTAGLIST,projectTagList);
	   for (QStringList::Iterator it=m_currentTagList.begin(); it != m_currentTagList.end(); ++it) {
	     if (projectTagList.find(*it) == projectTagList.end()) {
	       projectTagList.append(*it);
	     }
	   }
	   projectTagList.sort();
	   projectSettings->set(m_pCurCvsDir->topCvsDir()->relativeName(),PROJECTTAGLIST,projectTagList);
	 }
	 emit tagListFetched();
	 break;
       }
       default: {
           recoverViewportPos();
	   Debug::g_pLog->log(Debug::LL_THE_OLD_MAN_AND_THE_SEA,"there is no afterCall implemented for cmd id: "+QString::number(cmd));
       }
    }
}

//----------------------------------------------------------------------------

CvsControl::FileAppItem CvsControl::getMatchForWildcard(QStringList& nameList, FileAppList& appList) {

   QRegExp rx;
   FileAppList::Iterator appIt;
   for ( appIt = appList.begin(); appIt != appList.end(); ++appIt ) {
     rx.setWildcard(!(*appIt).isRegExp);
     rx.setPattern( (*appIt).wildcard);
     QStringList::Iterator nameIt;
     bool match = TRUE;
     for ( nameIt = nameList.begin(); nameIt != nameList.end(); ++nameIt ) {
       if (!rx.exactMatch( (*nameIt))) {
	 match = FALSE;
	 break;
       }
     }
     if (match) {
       return FileAppItem((*appIt));
     }
   }
   FileAppItem item;
   item.wildcard = "";
   item.app = "";
   item.params = "";
   item.isRegExp = FALSE;
   return item;
}

//----------------------------------------------------------------------------
