/* $Id: gtkedit-meaning.c,v 1.17 2005/05/10 17:18:40 marcusva Exp $
 *
 *  This file is part of LingoTeach, the Language Teaching program
 *  Copyright (C) 2004-2005 Marcus von Appen. All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * TODO: Add edited highlighting
 */

#include <string.h>
#include <libxml/xmlstring.h>

#include "../li18n.h"
#include "../errors.h"
#include "hig-widgets.h"
#include "lingmeaningtree.h"

#include "gtkconfig.h"
#include "util.h"
#include "gtkedit.h"
#include "gtkedit-meaning.h"

extern lingGtkPrefs *settings;
extern lingGtkMainWindow *main_app;

/* edit functions */
static void language_edit (GtkCellRendererCombo *cell, gchar *path,
                           gchar *new_text, GtkTreeView *view);
static void translation_edit (GtkCellRendererText *cell, gchar *path,
                              gchar *new_text, GtkTreeView *view);
static void add_image (GtkTreeView *view);
static void remove_image (GtkTreeView *view);
static void load_image (GtkTreeView *view, GtkWidget *image);
static void edit_information (GtkTreeView *view);
static void edit_information_cb (GtkAction *act, lingGtkEditor *editor);
static void lesson_set_edited (lingLesson *lesson);

static gboolean button_pressed (GtkTreeView *view, GdkEventButton *event,
                                gpointer data);
static void popup_menu (GtkTreeView *view, GdkEventButton *event,
                        gpointer data);
static void insert_meaning (GtkTreeView *view);
static void insert_meaning_cb (GtkAction *act, lingGtkEditor *editor);
static void insert_child_meaning (GtkTreeView *view);
static void insert_child_meaning_cb (GtkAction *act, lingGtkEditor *editor);
static void delete_meaning (GtkTreeView *view);
static void delete_meaning_cb (GtkAction *act, lingGtkEditor *editor);

static void create_menu_entries (GtkTreeView *view, lingGtkEditor *editor);

static void
language_edit (GtkCellRendererCombo *cell, gchar *path, gchar *new_text,
               GtkTreeView *view)
{
     lingMeaning *meaning = NULL;
     GtkTreeIter iter;
     lingchar *orig = NULL;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     debug ("Editing language...\n");

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          /* get the meaning pointer - does this lead to refs or
           * inconsistency? */
          gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
          
          orig = ling_lang_get_trans_language (settings->prefs->config,
                                               new_text, settings->edit_lang);

          if (!orig || g_utf8_strlen (orig, -1) == 0) /* invalid input */
               return;
          
          if (meaning->language
              && g_utf8_collate (meaning->language, orig) == 0)
          {
               ling_free (orig);
               return;
          }

          if (meaning->language)
               ling_free (meaning->language);
          meaning->language = NULL;
          meaning->language = ling_malloc (strlen (orig) + 1);
          strncpy (meaning->language, orig, strlen (orig) + 1);
          ling_free (orig);

          lesson_set_edited (meaning->lesson);
     }
     return;
}

static void
translation_edit (GtkCellRendererText *cell, gchar *path, gchar *new_text,
                  GtkTreeView *view)
{
     lingMeaning *meaning = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     debug ("Editing translation...\n");

     if (gtk_tree_model_get_iter_from_string (model, &iter, path))
     {
          /* get the meaning pointer - does this lead to refs or
           * inconsistency? */
          gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
          
          if (meaning->translation
              && g_utf8_collate (meaning->translation, new_text) == 0)
               return;

          if (meaning->translation)
               ling_free (meaning->translation);
          meaning->translation = NULL;
          meaning->translation = ling_malloc (strlen (new_text) + 1);
          strncpy (meaning->translation, new_text, strlen (new_text) + 1);
          
          lesson_set_edited (meaning->lesson);
     }
     return;
}

static void
add_image (GtkTreeView *view)
{
     GdkPixbuf *buf = NULL;
     GError *error = NULL;
     gchar *file = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     file = util_run_fileselection (_("Select image"),
                                    GTK_FILE_CHOOSER_ACTION_OPEN);
     if (!file)
          return;

     /* add image */
     buf = gdk_pixbuf_new_from_file (file, &error);
     if (!buf)
     {
          /* broken images will not be added */
          error_warning (_("Image file broken!"), error->message,
                         ERR_NOT_AVAILABLE);
          g_free (file);
          return;
     }

     /* clean the buf */
     g_object_unref (buf);

     gtk_list_store_append (GTK_LIST_STORE (model), &iter);
     gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, file, -1);

     return;
}

static void
remove_image (GtkTreeView *view)
{
     GtkTreeIter iter;
     GtkTreeModel *model;
     GtkTreeSelection *select = gtk_tree_view_get_selection (view);

     if (!gtk_tree_selection_get_selected (select, &model, &iter))
          util_info (_("Please select the row, you want to remove, first."));
     gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
     return;
}

static void
load_image (GtkTreeView *view, GtkWidget *image)
{
     GdkPixbuf *buf = NULL;
     GError *error = NULL;
     gchar *path = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model;
     GtkTreeSelection *sel = gtk_tree_view_get_selection (view);

     if (gtk_tree_selection_get_selected (sel, &model, &iter))
     {
          gtk_tree_model_get (model, &iter, 0, &path, -1);

          /* load the image into the widget */
          buf = gdk_pixbuf_new_from_file_at_size (path, 100, 100, &error);
          if (!buf)
          {
               error_warning (_("Image could not be loaded!"), error->message,
                              ERR_NOT_AVAILABLE);
               g_error_free (error);
               gtk_image_set_from_stock (GTK_IMAGE (image),
                                         GTK_STOCK_MISSING_IMAGE,
                                         GTK_ICON_SIZE_DIALOG);
          }
          else
          {
               gtk_image_set_from_pixbuf (GTK_IMAGE (image), buf);
               g_object_unref (buf);
          }
          g_free (path);
     }
     return;
}

static void
edit_information (GtkTreeView *view)
{
     GtkTreeSelection *select;
     GtkTreeIter iter;
     GtkTreeModel *model;

     GtkWidget *dlg_info;
     GtkWidget *notebook;
     
     GtkWidget *tbl_info;
     GtkWidget *lbl_info;
     GtkWidget *lbl_desc;
     GtkWidget *txt_desc;
     GtkWidget *lbl_phon;
     GtkWidget *txt_phon;

     GtkWidget *lbl_image;
     GtkWidget *box_images;
     GtkWidget *frm_preview;
     GtkWidget *align;
     GtkWidget *view_image;
     GtkWidget *scr_win;
     GtkWidget *box_btns;
     GtkWidget *btn_add;
     GtkWidget *btn_rem;
     GtkWidget *image;

     GtkListStore *list;
     GtkCellRenderer *renderer;
     gint i = 0;

     gint result = 0;
     lingchar *tmp = NULL;
     lingchar *tmp1 = NULL;
     lingchar **tmp_images = NULL;
     gint amount = 0;
     gchar *path = NULL;

     lingMeaning *meaning = NULL;
     lingMInfo *info = NULL;

     /* retrieve meaning information */
     select = gtk_tree_view_get_selection (view);
     if (!gtk_tree_selection_get_selected (select, &model, &iter))
          gtk_tree_model_get_iter_first (model, &iter);

     gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
     info = meaning->info;
     if (!info)
          info = ling_meaning_get_information (meaning);

     /* start dialog */
     dlg_info = gtk_dialog_new_with_buttons (_("Edit information"),
                                             GTK_WINDOW (main_app->editor->window),
                                             GTK_DIALOG_MODAL |
                                             GTK_DIALOG_DESTROY_WITH_PARENT,
                                             GTK_STOCK_CANCEL,
                                             GTK_RESPONSE_CANCEL,
                                             GTK_STOCK_OK,
                                             GTK_RESPONSE_ACCEPT,
                                             NULL);
     gtk_dialog_set_has_separator (GTK_DIALOG (dlg_info), FALSE);
     gtk_container_set_border_width (GTK_CONTAINER (dlg_info), WIDGET_BORDER);
     gtk_window_set_resizable (GTK_WINDOW (dlg_info), FALSE);

     notebook = gtk_notebook_new ();
     gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg_info)->vbox), notebook,
                         TRUE, TRUE, 0);
     gtk_container_set_border_width (GTK_CONTAINER (notebook), WIDGET_BORDER);
     
     lbl_info = gtk_label_new (_("General"));
     tbl_info = hig_table_new (2, 2);
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), tbl_info, lbl_info);

     /* description */
     lbl_desc = gtk_label_new_with_mnemonic (_("_Description:"));
     gtk_table_attach (GTK_TABLE (tbl_info), lbl_desc, 0, 1, 0, 1,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_desc), 0.0, 0.5);

     txt_desc = gtk_entry_new ();
     gtk_entry_set_text (GTK_ENTRY (txt_desc), info->description);
     gtk_table_attach (GTK_TABLE (tbl_info), txt_desc, 1, 2, 0, 1,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_desc), txt_desc);
     gtk_tooltips_set_tip (main_app->tips, txt_desc,
                           _("A more detailled description"),
                           NULL);

     /* phonetic */
     lbl_phon = gtk_label_new_with_mnemonic (_("_Phonetic transcription:"));
     gtk_table_attach (GTK_TABLE (tbl_info), lbl_phon, 0, 1, 1, 2,
                       GTK_FILL, GTK_FILL, 0, 0);
     gtk_misc_set_alignment (GTK_MISC (lbl_phon), 0.0, 0.5);

     txt_phon = gtk_entry_new ();
     gtk_entry_set_text (GTK_ENTRY (txt_phon), info->phonetic);
     gtk_table_attach (GTK_TABLE (tbl_info), txt_phon, 1, 2, 1, 2,
                       GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
     gtk_label_set_mnemonic_widget (GTK_LABEL (lbl_phon), txt_phon);
     gtk_tooltips_set_tip (main_app->tips, txt_phon,
                           _("The phonetic transcription of the meaning"),
                           NULL);

     /* image list */
     box_images = hig_vbox_new ();
     lbl_image = gtk_label_new (_("Images"));
     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box_images, lbl_image);

     list = gtk_list_store_new (1, G_TYPE_STRING);
     if (info->images)
          for (i = 0; info->images[i]; i++)
          {
               gtk_list_store_append (list, &iter);
               gtk_list_store_set (list, &iter, 0, info->images[i], -1);
          }
     view_image = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list));
     g_object_unref (list);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view_image),
                                                  0, _("Image"),
                                                  renderer, "text", 0, NULL);
     scr_win = hig_scrolled_window_new ();
     gtk_container_set_border_width (GTK_CONTAINER (scr_win), 0);
     gtk_container_add (GTK_CONTAINER (scr_win), view_image);
     gtk_box_pack_start (GTK_BOX (box_images), scr_win, FALSE, TRUE, 0);

     /* image buttons */
     box_btns = gtk_hbutton_box_new ();
     gtk_button_box_set_layout (GTK_BUTTON_BOX (box_btns),
                                GTK_BUTTONBOX_SPREAD);
     gtk_box_pack_start (GTK_BOX (box_images), box_btns, FALSE, TRUE, 0);

     btn_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
     g_signal_connect_swapped (G_OBJECT (btn_add), "clicked",
                               G_CALLBACK (add_image), view_image);
     gtk_container_add (GTK_CONTAINER (box_btns), btn_add);
     btn_rem = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
     g_signal_connect_swapped (G_OBJECT (btn_rem), "clicked",
                               G_CALLBACK (remove_image), view_image);
     gtk_container_add (GTK_CONTAINER (box_btns), btn_rem);

     frm_preview = hig_frame_new (_("Preview"));
     gtk_container_set_border_width (GTK_CONTAINER (frm_preview), 0);
     align = gtk_bin_get_child (GTK_BIN (frm_preview));
     image = gtk_image_new_from_stock (GTK_STOCK_MISSING_IMAGE,
                                       GTK_ICON_SIZE_DIALOG);
     gtk_container_add (GTK_CONTAINER (align), image);
     gtk_box_pack_start (GTK_BOX (box_images), frm_preview, TRUE, TRUE, 0);

     g_signal_connect (G_OBJECT (view_image), "cursor-changed",
                       G_CALLBACK (load_image), image);

     gtk_widget_show_all (notebook);
     result = gtk_dialog_run (GTK_DIALOG (dlg_info));
     switch (result)
     {
     case GTK_RESPONSE_ACCEPT:
          /* now copy the changed cotents back to the info */
          tmp = xmlCharStrdup (gtk_entry_get_text (GTK_ENTRY (txt_desc)));
          if (!tmp) /* error while duplicating! */
          {
               error_warning (ERR_MEM_INFO,
                              _("An error occured while saving the "
                                "information."), ERR_NOT_AVAILABLE);
               break;
          }

          tmp1 = xmlCharStrdup (gtk_entry_get_text (GTK_ENTRY (txt_phon)));
          if (!tmp1) /* error while duplicating! */
          {
               error_warning (ERR_MEM_INFO,
                              _("An error occured while saving the "
                                "information."), ERR_NOT_AVAILABLE);
               ling_free (tmp1);
               break;
          }

          ling_free (info->description);
          info->description = tmp;
          ling_free (info->phonetic);
          info->phonetic = tmp1;

          /* we have to add the images now */
          amount = gtk_tree_model_iter_n_children (GTK_TREE_MODEL (list),
                                                   NULL);
          if (amount > 0) /* are there any children */
          {
               tmp_images = ling_malloc (sizeof (lingchar *) * (amount + 1));
               for (i = 0; i <= amount; i++)
                    tmp_images[i] = NULL;

               i = 0;
               /* copy the entries to the image array */
               gtk_tree_model_get_iter_first (GTK_TREE_MODEL (list), &iter);
               do 
               {
                    gtk_tree_model_get (GTK_TREE_MODEL (list), &iter,
                                        0, &path, -1);
                    tmp_images[i] = xmlCharStrdup (path);
                    g_free (path);
               }
               while (gtk_tree_model_iter_next (GTK_TREE_MODEL (list), &iter));
          }
          if (info->images)
               ling_strings_free (info->images);
          info->images = tmp_images; /* either NULL or set */

          util_new_status (GTK_STATUSBAR (main_app->editor->status),
                           _("Meaning information updated."));
          break;
     case GTK_RESPONSE_CANCEL:
     default:
          util_new_status (GTK_STATUSBAR (main_app->editor->status),
                           _("Editing aborted."));
          break;
     }

     gtk_widget_destroy (dlg_info);
     return;
}

static void
edit_information_cb (GtkAction *act, lingGtkEditor *editor)
{
     edit_information (GTK_TREE_VIEW (editor->view_meanings));
     return;
}

static void
lesson_set_edited (lingLesson *lesson)
{
     lingLesson *match = NULL;
     gboolean found = FALSE;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model
                             (GTK_TREE_VIEW (main_app->editor->view_lesson));

     debug ("Edited lesson...\n");

     if (gtk_tree_model_get_iter_first (model, &iter))
     {
          do
          {
               gtk_tree_model_get (model, &iter,
                                   GTKEDIT_LESSON_POINTER, &match, -1);
               if (match == lesson)
               {
                    found = TRUE;
                    break;
               }
          } while (gtk_tree_model_iter_next (model, &iter));
          
          if (!found) /* this should not happen */
               return;

          gtk_list_store_set (GTK_LIST_STORE (model), &iter,
                              GTKEDIT_LESSON_EDITED, TRUE,
                              -1);
     }
     return;
}

static gboolean
button_pressed (GtkTreeView* view, GdkEventButton *event, gpointer data)
{
     GtkTreePath *path;
     GtkTreeSelection *select;

     if (event->type == GDK_BUTTON_PRESS && event->button == 3)
     {
          /* select the node under the mouse cursor, if none is selected */
          select = gtk_tree_view_get_selection (view);
          if (gtk_tree_selection_count_selected_rows (select) == 0)
               if (gtk_tree_view_get_path_at_pos(view, event->x, event->y,
                                                 &path, NULL, NULL, NULL))
               {
                    gtk_tree_selection_select_path(select, path);
                    gtk_tree_path_free(path);
               }
          popup_menu (view, event, data);
          return TRUE;
     }
     return FALSE;
}

static void
popup_menu (GtkTreeView *view, GdkEventButton *event, gpointer data)
{
     GtkTreeSelection *select;
     GtkTreeModel *model;
     GtkTreeIter iter;
     GtkTreeIter parent;
     GtkWidget *menu;
     GtkWidget *item;
     gboolean forced_sel = FALSE;

     /* check, if a model is available */
     select = gtk_tree_view_get_selection (view);
     if (!gtk_tree_selection_get_selected (select, &model, &iter))
     {
          if (!gtk_tree_model_get_iter_first (model, &iter))
               return;
          /* selection was forced, so do not activate deletion */
          forced_sel = TRUE;
     }
     menu = gtk_menu_new ();

     item = gtk_menu_item_new_with_label (_("Insert new meaning"));
     g_signal_connect_swapped (G_OBJECT (item), "activate",
                               G_CALLBACK (insert_meaning), view);
     gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

     /* add subnode inserter on demand */
     if (!gtk_tree_model_iter_parent (model, &parent, &iter))
     {
          item = gtk_menu_item_new_with_label (_("Add meaning as child"));
          g_signal_connect_swapped (G_OBJECT (item), "activate",
                                    G_CALLBACK (insert_child_meaning), view);
          gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
     }

     /* delete meaning */
     item = gtk_menu_item_new_with_label (_("Delete meaning"));
     if (forced_sel)
          gtk_widget_set_sensitive (item, FALSE);
     g_signal_connect_swapped (G_OBJECT (item), "activate",
                               G_CALLBACK (delete_meaning), view);
     gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

     /* seperator before editing entries */
     item = gtk_separator_menu_item_new ();
     gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

     /* edit descriptions, images, etc. */
     item = gtk_menu_item_new_with_label (_("Edit enhanced information"));
     gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
     g_signal_connect_swapped (G_OBJECT (item), "activate",
                               G_CALLBACK (edit_information), view);

     gtk_widget_show_all (menu);
     gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
                     (event) ? event->button : 0,
                     gdk_event_get_time ((GdkEvent *)event));
     return;
}

static void
insert_meaning (GtkTreeView *view)
{
     GtkTreeSelection *select;
     GtkTreeModel *model;
     GtkTreeIter parent;
     GtkTreeIter iter;
     GtkTreeIter sibling;
     lingMeaning *meaning;

     debug ("Inserting new meaning...\n");
     
     select = gtk_tree_view_get_selection (view);
     if (!gtk_tree_selection_get_selected (select, &model, &sibling))
          if (!gtk_tree_model_get_iter_first (model, &sibling))
          {
               /* special case: no meanings */
               ling_tree_store_insert_after (LING_TREE_STORE (model), &iter,
                                             NULL, NULL);
               gtk_tree_model_get (model, &iter, LING_TREE_POINTER,
                                   &meaning, -1);
               lesson_set_edited (meaning->lesson);
               return;
          }
     
     if (gtk_tree_model_iter_parent (model, &parent, &sibling))
          ling_tree_store_insert_after (LING_TREE_STORE (model), &iter,
                                        &parent, &sibling);
     else /* add toplevel node */
          ling_tree_store_insert_after (LING_TREE_STORE (model), &iter,
                                        &sibling, NULL);
     
     gtk_tree_model_get (model, &iter, LING_TREE_POINTER,&meaning, -1);
     lesson_set_edited (meaning->lesson);
     return;
}

static void
insert_meaning_cb (GtkAction *act, lingGtkEditor *editor)
{
     insert_meaning (GTK_TREE_VIEW (editor->view_meanings));
     return;
}

static void
insert_child_meaning (GtkTreeView *view)
{
     GtkTreeSelection *select;
     GtkTreeModel *model;
     GtkTreeIter iter;
     GtkTreeIter iter2;
     GtkTreePath *path;
     lingMeaning *meaning = NULL;

     select = gtk_tree_view_get_selection (view);
     if (!gtk_tree_selection_get_selected (select, &model, &iter))
          gtk_tree_model_get_iter_first (model, &iter);

     debug ("Adding child...\n");
     
     ling_tree_store_insert_after (LING_TREE_STORE (model), &iter2,
                                   &iter, &iter);

     /* expand the row afterwards */
     path = gtk_tree_model_get_path (model, &iter);
     gtk_tree_view_expand_row (view, path, FALSE);
     gtk_tree_path_free (path);

     gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
     lesson_set_edited (meaning->lesson);
     return;
}

static void
insert_child_meaning_cb (GtkAction *act, lingGtkEditor *editor)
{
     insert_child_meaning (GTK_TREE_VIEW (editor->view_meanings));
     return;
}

static void
delete_meaning (GtkTreeView *view)
{
     GtkTreeSelection *select;
     GtkTreeModel *model;
     GtkTreeIter iter;
     lingMeaning *meaning = NULL;

     select = gtk_tree_view_get_selection (view);
     if (!gtk_tree_selection_get_selected (select, &model, &iter))
          return; /* no selection, so do not do anything */

     if (gtk_tree_model_iter_has_child (model, &iter)
         && !util_ask_for_confirmation (_("Confirm deletion"),
                                        _("The meaning has several children, "
                                          "really delete them all?")))
          return;

     gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
     lesson_set_edited (meaning->lesson);

     ling_tree_store_remove (LING_TREE_STORE (model), &iter);
     
     return;
}

static void
delete_meaning_cb (GtkAction *act, lingGtkEditor *editor)
{
     delete_meaning (GTK_TREE_VIEW (editor->view_meanings));
     return;
}

static void
create_menu_entries (GtkTreeView *view, lingGtkEditor *editor)
{
     GtkToolItem *button;
     GError *error = NULL;
     const gchar* menu = 
          "<ui>"
          "  <menubar name='EditorMenu'>"
          "    <menu action='Edit'>"
          "      <separator/>"
          "      <menuitem action='Add'/>"
          "      <menuitem action='AddChild'/>"
          "      <menuitem action='Remove'/>"
          "      <separator/>"
          "      <menuitem action='EditInfo'/>"
          "    </menu>"
          "  </menubar>"
          "</ui>";
     
     GtkActionEntry entries[] = {
          { "Add", GTK_STOCK_ADD, N_("Add _meaning"), NULL, NULL, 
            G_CALLBACK (insert_meaning_cb) },
          { "AddChild", GTK_STOCK_ADD, N_("Add _child meaning"), NULL, NULL, 
            G_CALLBACK (insert_child_meaning_cb) },
          { "Remove", GTK_STOCK_REMOVE, N_("_Remove meaning"), NULL, NULL,
            G_CALLBACK (delete_meaning_cb) },
          { "EditInfo", GTK_STOCK_EDIT, N_("_Edit information"), NULL, NULL,
            G_CALLBACK (edit_information_cb) },
     };

     /* toolbar entries */
     button = gtk_tool_button_new_from_stock (GTK_STOCK_ADD);
     gtk_toolbar_insert (GTK_TOOLBAR (editor->toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Add a new meaning"),
                                NULL);
     g_signal_connect_swapped (G_OBJECT (button), "clicked",
                               G_CALLBACK (insert_meaning), view);
     
     button = gtk_tool_button_new_from_stock (GTK_STOCK_REMOVE);
     gtk_toolbar_insert (GTK_TOOLBAR (editor->toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Delete the selected meaning"),
                                NULL);
     g_signal_connect_swapped (G_OBJECT (button), "clicked",
                               G_CALLBACK (delete_meaning), view);

     
     button = gtk_tool_button_new_from_stock (GTK_STOCK_EDIT);
     gtk_toolbar_insert (GTK_TOOLBAR (editor->toolbar), button, -1);
     gtk_tool_item_set_tooltip (button, main_app->tips,
                                _("Edit enhanced information"),
                                NULL);
     g_signal_connect_swapped (G_OBJECT (button), "clicked",
                               G_CALLBACK (edit_information), view);

     /* create menu entries */
     gtk_action_group_add_actions (editor->actions, entries,
                                   G_N_ELEMENTS (entries), editor);
     if (!gtk_ui_manager_add_ui_from_string (editor->menu, menu, -1, &error))
     {
          error_critical (_("Editor menu could not be created!"),
                          error->message, ERR_NOT_AVAILABLE);
          g_error_free (error);
     }
     return;
}

lingLesson*
gtkedit_meaning_create_lesson_from_list (GtkTreeView *view)
{
     lingMeaning *meaning = NULL;
     GtkTreeIter iter;
     GtkTreeModel *model = gtk_tree_view_get_model (view);

     if (gtk_tree_model_get_iter_first (model, &iter))
     {
          gtk_tree_model_get (model, &iter, LING_TREE_POINTER, &meaning, -1);
          return ling_meaning_create_lesson (meaning,
                                             settings->prefs->config,
                                             meaning->lesson->type,
                                             meaning->lesson->sound);
     }
     return NULL;
}

GtkWidget*
gtkedit_meaning_create_view (lingGtkEditor *editor, const gchar *labels[])
{
     LingTreeStore *tmp;
     GtkWidget *view;
     GtkCellRenderer *renderer;

     GtkTreeSelection *select;

     GtkListStore *list_lang;
     lingchar *language = NULL;
     GtkTreeIter iter;
     gint i = 0;

     tmp = ling_tree_store_new ();
     view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (tmp));
     g_object_unref (tmp);
     g_signal_connect (G_OBJECT (view), "button-press-event",
                       G_CALLBACK (button_pressed), NULL);
     g_signal_connect (G_OBJECT (view), "popup-menu",
                       G_CALLBACK (popup_menu), NULL);

     renderer = gtk_cell_renderer_text_new ();
     /*g_object_set (G_OBJECT (renderer), "weight", PANGO_WEIGHT_BOLD, NULL);*/
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
                                                  LING_TREE_ID,
                                                  labels[LING_TREE_ID],
                                                  renderer, "text",
                                                  LING_TREE_ID,
                                                  /*"weight-set",EDIT_EDITED,*/
                                                  NULL);

     renderer = gtk_cell_renderer_combo_new ();

     /* create language list */
     list_lang = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
     while (settings->prefs->languages[i])
     {
          if (settings->edit_lang)
          {
               gtk_list_store_append (list_lang, &iter);
               language =
                    ling_lang_get_lang_translation (settings->prefs->config,
                                                    settings->prefs->languages[i],
                                                    settings->edit_lang);

               gtk_list_store_set (list_lang, &iter,
                                   0, (language) ? language :
                                   settings->prefs->languages[i],
                                   /* the next one is not visible */
                                   1, settings->prefs->languages[i],
                                   -1);
               if (language)
                    free (language);
          }
          else
               gtk_list_store_set (list_lang, &iter, 0,
                                   settings->prefs->languages[i], -1);
          i++;
     }
     g_object_set (G_OBJECT (renderer), "model", list_lang,
                   "text-column", 0, "editable", TRUE, NULL);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
                                                  LING_TREE_LANGUAGE,
                                                  labels[LING_TREE_LANGUAGE],
                                                  renderer, "text",
                                                  LING_TREE_LANGUAGE,
                                                  /*"weight-set",EDIT_EDITED,*/
                                                  NULL);
     g_signal_connect (G_OBJECT (renderer), "edited",
                       G_CALLBACK (language_edit), GTK_TREE_VIEW (view));

     renderer = gtk_cell_renderer_text_new ();
     g_object_set (G_OBJECT (renderer), "editable", TRUE,
                   /*"weight", PANGO_WEIGHT_BOLD,*/ NULL);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view),
                                                  LING_TREE_TRANS,
                                                  labels[LING_TREE_TRANS],
                                                  renderer, "text",
                                                  LING_TREE_TRANS,
                                                  /*"weight-set",EDIT_EDITED,*/
                                                  NULL);
     g_signal_connect(G_OBJECT (renderer), "edited",
                      G_CALLBACK (translation_edit), GTK_TREE_VIEW (view));

     /* add selection callbacks */
     select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
     gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
     g_signal_connect_swapped (G_OBJECT (select), "changed",
                               G_CALLBACK (gtkedit_meaning_update_menus),
                               editor);

     /* create add, remove, etc. entries in the editor toolbar */
     create_menu_entries (GTK_TREE_VIEW (view), editor);
     gtkedit_meaning_update_menus (editor);
     return view;
}

GtkTreeModel*
gtkedit_meaning_create_entry_list (lingLesson *lesson)
{
     LingTreeStore *list;
     lingMeaning *meaning;

     /* TODO: a popup with state indicator would be nice,
        if there were a matching signal fir this */

     meaning = ling_lesson_create_tree (lesson);
     if (!meaning)
     {
          error_warning (_("Tree could not be created!"),
                         _("The meaning tree could not be created from the "
                           "lesson file."), ERR_NOT_AVAILABLE);
          return NULL;
     }
     list = ling_tree_store_new_with_meanings (meaning);
     
     debug ("LingTreeStore created!\n");

     return GTK_TREE_MODEL (list);
}

void
gtkedit_meaning_apply_search (GtkEntry *entry, GtkTreeView *tree)
{
     gint id;
     gpointer data;
     gboolean result = FALSE;
     GtkTreeIter iter;
     GtkTreeIter parent;
     GtkTreePath *path;
     GtkTreeSelection *select;
     GtkTreeModel *model = gtk_tree_view_get_model (tree);
     gint column = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (entry),
                                                       GTKEDIT_SEARCH_ID));
     const gchar *search = gtk_entry_get_text (entry);


     /* only apply anything, if there is data */
     if (strlen (search) <= 0 ||!gtk_tree_model_get_iter_first (model, &iter))
          return;

     debug ("Applying search...\n");

     /* start from the selected node or from the beginning */
     select = gtk_tree_view_get_selection (tree);
     if (!gtk_tree_selection_get_selected (select, &model, &iter))
          gtk_tree_model_get_iter_first (model, &iter);

     /* search */
     if (column == LING_TREE_ID)
     {
          id = (gint) g_strtod (search, NULL);
          if (id != 0)
          {
               data = GINT_TO_POINTER (id);
               result = ling_tree_store_search (model, &iter, column, data);
          }
     }
     else
          result = ling_tree_store_search (model, &iter, column,
                                           (gpointer) search);
     
     if (!result)
          gtk_tree_selection_unselect_all (select);
     else
     {
          /* expand node on demand */
          if (gtk_tree_model_iter_parent (model, &parent, &iter))
          {
               path = gtk_tree_model_get_path (model, &parent);
               gtk_tree_view_expand_row (tree, path, FALSE);
               gtk_tree_path_free (path);
          }
          /* select row and focus */
          gtk_tree_selection_select_iter (select, &iter);
          path = gtk_tree_model_get_path (model, &iter);
          gtk_tree_view_scroll_to_cell (tree, path, NULL, FALSE, 0.0, 0.0);
          gtk_tree_path_free (path);
     }
     return;
}

void
gtkedit_meaning_update_menus (lingGtkEditor *editor)
{
     GtkTreeIter iter;
     GtkTreeModel *model;
     GtkTreeSelection *select;
     gboolean rem = FALSE;
     gboolean deac = FALSE;
     GtkWidget *entry;
     
     deac = !editor->view_meanings;
     if (!deac)
     {
          select = 
               gtk_tree_view_get_selection
               (GTK_TREE_VIEW (editor->view_meanings));
          rem = gtk_tree_selection_get_selected (select, &model, &iter);
     }

     gtkedit_set_toolbar_item_sensitive (editor,
                                         GTKEDIT_TOOL_LAST
                                         + GTKEDITMEANING_TOOL_REM, rem);
     gtkedit_set_toolbar_item_sensitive (editor,
                                         GTKEDIT_TOOL_LAST
                                         + GTKEDITMEANING_TOOL_ADD, !deac);
     gtkedit_set_toolbar_item_sensitive (editor,
                                         GTKEDIT_TOOL_LAST
                                         + GTKEDITMEANING_TOOL_EDIT, rem);

     /* set menu entries */
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/Add");
     gtk_widget_set_sensitive (entry, !deac);
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/AddChild");
     gtk_widget_set_sensitive (entry, rem);
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/Remove");
     gtk_widget_set_sensitive (entry, rem);
     entry = gtk_ui_manager_get_widget (editor->menu,
                                        "/EditorMenu/Edit/EditInfo");
     gtk_widget_set_sensitive (entry, rem);
     
     return;
}
