/*
def MACRO key_code

set MACRO command
meta MACRO command

*.key file example
-------------------
#comment
def KEY_UP 24
def KEY_DOWN 2

set KEY_UP <up>
meta KEY_LEFT <history>
*/

#include "keybind.h"
#include "StringTokenizer.h"
#include "define.h"
#include "strutil.h"
#include <termios.h>
#include "keydef.h"

#include <iostream>
#include <fstream>

using namespace std;
using namespace strutil;

namespace MLS {

/// @brief		파일에서 바인딩 파일을 읽어들입니다.
/// @param fn   읽어올 파일
/// @return		SUCCESS or ERROR.
bool KeyBind::Load( const string &fn	)
{
	string   line;	
	ifstream in(fn.c_str());
	
	if (!in) return false;
	bool 	 bVersion = false;

	while(getline(in, line))
	{
		if (line.empty()) continue;
		if (tolower(line.substr(0, 10)) == "#!version ")
		{
			_sVersion = tolower(chop(line.substr(10)));
			continue;
		}
		if (line[0] == '#') continue;

		StringTokenizer st(line, "\t\r\n ");
	
		if (!st.Next())
		{
			continue;
		}

		string cmd = st.Get();
		
		if (!st.Next())
		{
			continue;
		}
		string macro = strutil::tolower(st.Get());

		if (cmd == "version")
		{
			_sVersion = macro;
			continue;
		}
		
		string param = chop(st.Rest());
		
		if (param.empty())
		{
			continue;
		}

		if (cmd == "def")
		{
			// backspace 확장
			if (macro == "backspace") 
			{
				struct termios mode;
				memset(&mode, 0, sizeof(mode));
				
				tcgetattr(0, &mode);				
				_definitions[macro] = mode.c_cc[VERASE];
				_definitions[macro] = KEY_BS;
			}
			else
			{
				_definitions[macro] = atoi(param.c_str());
			}
		}
		else if (cmd == "set")
		{
			map<string, int>::iterator i;
			
			LOG("Set [%s]", macro.c_str());

			if (macro.size() >= 3 && macro[0]=='\'' && macro[2]=='\'') 
			{
				_bindings[(int)macro[1]] = param;
			}
			else if (macro.size() <= 3 && macro[0] == 'f')
			{
				string func = macro.substr(1, macro.size());
				int		nFunc = atoi(func.c_str());
				_func_bindings[nFunc+10000] = param;
			}
			else if ((i = _definitions.find(macro)) != _definitions.end())
			{
				_bindings[(*i).second] = param;
			}
		}
		else if (cmd == "meta")
		{
			map<string, int>::iterator i;

			if (macro.size() >= 3 && macro[0]=='\'' && macro[2]=='\'')
			{
				_meta_bindings[(int)macro[1]] = param;
			}
			else if ((i = _definitions.find(macro)) != _definitions.end())
			{
				_meta_bindings[(*i).second] = param;
			}
		}
		else if (cmd == "help")
		{
			map<string, int>::iterator i;
			LOG("HELP [%s] [%s]", macro.c_str(), getbetween(param, '[', ']').c_str());
			_help_bindings[macro] = getbetween(param, '[', ']');
		}
	}
	in.close();
	return true;
}

static string EmptyString;

int KeyBind::GetDefinition(const string &var)
{
	map<string, int>::iterator i;
	
	if ( (i = _definitions.find(tolower(var))) != _definitions.end() ) return (*i).second;
	
	return -1;
}

string KeyBind::GetCommand( int key )
{
	map<int, string>::iterator i;

	if ( (i = _bindings.find(key)) != _bindings.end() ) return (*i).second;
	if ( 'a' <= key && key <= 'z'&& (i = _bindings.find(key-'a'+'A')) != _bindings.end()) return (*i).second;
	if ( 'A' <= key && key <= 'Z'&& (i = _bindings.find(key-'A'+'a')) != _bindings.end()) return (*i).second;

	return EmptyString;
}

string KeyBind::GetMetaCommand( int key )
{
	map<int, string>::iterator i;

	if ( (i = _meta_bindings.find(key)) != _meta_bindings.end() ) return (*i).second;
	if ( 'a' <= key && key <= 'z'&& (i = _meta_bindings.find(key-'a'+'A')) != _meta_bindings.end()) return (*i).second;
	if ( 'A' <= key && key <= 'Z'&& (i = _meta_bindings.find(key-'A'+'a')) != _meta_bindings.end()) return (*i).second;

	return EmptyString;
}

string KeyBind::GetKeyStr(string sCommand)
{
	int 	nKey = 0;
	char	cKeyChar;
	string 	sKeyName = "";
	map<int, string>::iterator i;
	
	for (i = _meta_bindings.begin(); i != _meta_bindings.end(); ++i)
	{
		if (i->second == sCommand)
		{
			nKey = i->first;
			nKey = nKey + 1000;
		}
	}
	
	for (i = _bindings.begin(); i != _bindings.end(); ++i)
	{
		if (i->second == sCommand)
		{
			nKey = i->first;
		}
	}

	for (i = _func_bindings.begin(); i != _func_bindings.end(); ++i)
	{
		if (i->second == sCommand)
		{
			nKey = i->first;
		}
	}
	
	if (nKey >= 10000)
	{
		sKeyName = "F" + strutil::itoa(nKey-10000);
	}
	else if (nKey < 10000 && nKey > 1000)
	{
		switch(nKey - 1000)
		{
			case 13: case 10:	sKeyName = "Enter"; break;
			case 27:			sKeyName = "ESC"; break;
			case 9:				sKeyName = "TAB"; break;
			case 362: 			sKeyName = "Home"; break;
			case 385: 			sKeyName = "End"; break;
			case 263: 			sKeyName = "Back"; break;
			case 331: 			sKeyName = "Ins"; break;
			case 330: 			sKeyName = "Del"; break;
			case 339: 			sKeyName = "PgUp"; break;
			case 338: 			sKeyName = "PgDn"; break;
			case 259: 			sKeyName = "Up"; break;
			case 258: 			sKeyName = "Down"; break;
			case 260: 			sKeyName = "Left"; break;
			case 261: 			sKeyName = "Right"; break;
			default:
				cKeyChar = (char)(nKey - 1000);
				sKeyName.append(1, cKeyChar);
				sKeyName = strutil::toupper(sKeyName);
				break;
		}
		sKeyName = "Alt+" + sKeyName;
	}
	else
	{
		if (nKey != 0 && nKey < 33)
		{
			switch(nKey)
			{
				case 13: case 10:	sKeyName = "Enter"; 	break;
				case 27:			sKeyName = "ESC";	break;
				case 9:				sKeyName = "TAB";	break;
				default:
					cKeyChar = (char)('A' + nKey - 1);
					sKeyName = "Ctrl+";
					sKeyName.append(1, cKeyChar);
			}
		}
		else
		{
			switch(nKey)
			{
				case 362: sKeyName = "Home"; break;
				case 385: sKeyName = "End"; break;
				case 263: sKeyName = "Back"; break;
				case 331: sKeyName = "Ins"; break;
				case 330: sKeyName = "Del"; break;
				case 339: sKeyName = "PgUp"; break;
				case 338: sKeyName = "PgDn"; break;
				case 259: sKeyName = "Up"; break;
				case 258: sKeyName = "Down"; break;
				case 260: sKeyName = "Left"; break;
				case 261: sKeyName = "Right"; break;
				case 407: sKeyName = "Ctrl+Z"; break;
				case 0: break;
				default:
					cKeyChar = (char)nKey;
					sKeyName.append(1, cKeyChar);
					break;
			}
		}
	}
	return sKeyName;
}

string KeyBind::GetHelpCmd(string sCommand)
{
	string 	sHelpName = "";
	map<string, string>::iterator i;
	
	for (i = _help_bindings.begin(); i != _help_bindings.end(); ++i)
		if (i->first == sCommand)
			sHelpName = i->second;
	return sHelpName;
}

string KeyBind::GetHelpMetaKey(int Key)
{
	string 	sHelpName = "";
	map<string, string>::iterator i;
	
	for (i = _help_bindings.begin(); i != _help_bindings.end(); ++i)
		if (i->first == GetMetaCommand(Key))
			sHelpName = i->second;
	
	return sHelpName;
}

string KeyBind::GetHelpKey(int Key)
{
	string 	sHelpName = "";
	map<string, string>::iterator i;
	
	for (i = _help_bindings.begin(); i != _help_bindings.end(); ++i)
	{
		if (i->first == GetCommand(Key))
		{
			sHelpName = i->second;
		}
	}
	return sHelpName;
}

string KeyBind::GetFuncKey(int Key)
{
	string 	sFuncName = "";
	map<int, string>::iterator i;
	
	for (i = _func_bindings.begin(); i != _func_bindings.end(); ++i)
	{
		if (i->first == Key+10000)
		{
			sFuncName = i->second;
		}
	}
	return sFuncName;
}

void KeyBind::SetFuncKey(int Key, string& sCmd)
{
	_func_bindings[Key+10000] = sCmd;
}

void KeyBind::SetFuncKey(map<int, string>& mapFunc)
{
	_func_bindings = mapFunc;
}

KeyBind &KeyBind::GetInstance()
{
	static KeyBind instance;
	return instance;
}

KeyBind::KeyBind()
{
}

KeyBind::~KeyBind( )
{
}

}; // namespace
