///	\file	menu.h
///	\brief	menu.h
#ifndef Menu_h_
#define Menu_h_

#include <string>
#include <vector>
#include <ncurses.h>
#include "keybind.h"

namespace MLS {

///	\brief	MenuItem class
class MenuItem
{
	bool		bEnable;
	std::string name;		///< 화면에 출력될 MenuItem 이름
	std::string command;	///< 실행될 MenuItem 명령어
	
public:
	
	///	\brief	생성자
	///	\param	n		MenuItem 이름
	///	\param	c		MenuItem 명령어
	MenuItem(const std::string &n, const std::string &c, bool bEn = true) 
		: name(n), command(c), bEnable(bEn)
	{}
	
	// setter	===================================
	///	\brief	이름 설정함수
	///	\param	n		설정할 화면에 출력될 메뉴 Item
	void SetName(const std::string &n) {name = n;}

	///	\brief	명령 설정 함수
	///	\param	n		  설정할 명령어
	void SetCmd(const std::string &n) {command = n;}	
	// ============================================
	
	// getter	===================================
	///	\brief	화면에 출력될 이름 얻기 함수
	///	\return	화면에 출력될 이름
	const std::string &GetName() {return name;}

	///	\brief	명령어 얻기 함수
	///	\return	명령어
	const std::string &GetCmd() {return command;}		
	// ============================================

	///	\brief	명령 활성화 
	///	\param	bEn		Enable이면 true
	bool	SetEnable(bool bEn) { bEnable = bEn; }
	
	///	\brief	활성화 여부를 돌려준다.
	///	\return	Enable
	bool	GetEnable() { return bEnable; }
};

// 메뉴의 분류
///	\brief	MenuCategory class[메뉴의 분류]
class MenuCategory
{
	std::string _name;		///< 메뉴 이름
	
	int _index;     			///< 메뉴 Category 내에서 현재 커서 위치
	int _location;			///< 상위 메뉴 내에서의 Category 위치
	
	std::vector<MenuItem> _item; ///< 서브메뉴
	
	WINDOW *_win;			///< Main WINDOW(ncurses)	

public:
	///	\brief	생성자
	///	\param	p		MenuCategory 이름
	///	\param	l		MenuCategory location
	MenuCategory(const std::string &p, const int &l = 0)
		: _name(p) , _index(0), _location(l), _win(NULL)
	{}

	///	\brief	소멸자
	~MenuCategory()
	{
		if (_win)
		{
			delwin(_win);
		}
	}
			
	///	\brief	이름  얻기 함수
	///	\return	이름
	const std::string &GetName(){return _name;}
	
	///	\brief	MenuItem 추가 함수
	///	\param	n		MenuItem 이름
	///	\param	c		MenuItem Command
	///	\param	b		MenuItem Enable 여부 (기본 Enable)
	void AddItem(const std::string &n, const std::string &c, bool b = true)
	{
		_item.push_back(MenuItem(n,c,b));
	}
	
	///	\brief	위치 설정 함수
	///	\param	x	설정될 위치
	void SetLocation(const int &x){ _location = x;}

	///	\brief	위치 얻기 함수
	///	\return	위치
	int  GetLocation() { return _location;}
	
	///	\brief	MenuItem Command 얻기함수
	///	\return	MenuItem Command
	const std::string &GetCmd() { return _item[_index].GetCmd(); }

	///	\brief	MenuItem Command 얻기함수
	///	\return	MenuItem Command
	bool	GetEnable() { return _item[_index].GetEnable(); }

	void	SetDisable(std::string sStr)
	{
		for(int c=0; c<_item.size(); c++)
		{
			if (_item[c].GetCmd() == sStr) _item[c].SetEnable(false);
		}
	}
	
	int  GetItemSize() { return _item.size(); }
	void SetCursor(int index);

	void Draw(KeyBind* pKeyBind = NULL);
	void Create();
	
	void CursorUp();
	void CursorDown();	
};

///	\brief	Menu class
class Menu
{
protected:
	int _num;						///< 현재 선택되어 있는 item index
	std::vector<MenuCategory> 	_item; ///< 메뉴 뭉치들(예.윈도우에서 File 메뉴)
	std::vector<std::string>			_vCmdDisable; // Disable될 Cmd list
	
	WINDOW *_win;					///< nCurses WINDOW

	bool MouseProc();
	void DiableList();
public:
	Menu();
	~Menu();
	
	///	\brief	menuCategory를 추가한다.
	///	\param	p	추가할 MenuCategory
	void AddItem(MenuCategory p)
	{
		// . 아무런 메뉴(예. File 메뉴)가 없으면 초기위치(x)를 4로한다.
		if (_item.empty()) p.SetLocation(4);
		// . 메뉴가 있다면 전 메뉴가 위치한 자리와 글자수를 참고하여 위치를 설정
		else p.SetLocation(_item.back().GetLocation() + _item.back().GetName().size() + 2);

		// . 메뉴 뭉치 추가
		_item.push_back(p);
	}
	
	///	\brief	menuCategory를 추가한다.
	///	\param	p	추가할 메뉴 문자열
	void AddItem(const std::string &p)
	{
		AddItem(MenuCategory(p));
	}

	///	\brief	Disable 될 명령 list 를 추가한다.
	///	\param	vCmd	disable 할 메뉴 List (command)
	void SetDefault(void)	{ _vCmdDisable.clear(); }
	void SetDisable(std::vector<std::string> vCmd) { _vCmdDisable = vCmd; }
	
	void Create();
	void CreateWindow();

	void Draw();

	// 실행
	std::string Process(KeyBind*  pKeyBind = NULL);

	// Reflash 할때 기타 화면을 Reflash 꼭 넣어야 제대로 화면 나옴.
	virtual void MenuRefresh() {};
	
	// 각 메뉴를 넣는 가상함수
	virtual void MenuAddCatecory() {};
};

}; // namespace

#endif 
