/**
 * @file	mainframe.cpp
 * @brief	mainframe source file
 * @author	la9527
 * @date	2004-08-22
 */
#include <sys/types.h>
#include <signal.h>

#include "define.h"
#include "mainframe.h"
#include "mlslocale.h"
#include "dialog.h"
#include "configure.h"

#include "editor.h"

namespace MLS {

MainFrame *g_MainFrame;

MainFrame::MainFrame()
{
	_bSplit  = false;
	_nActive = 0;
	_bHints = false;
	UpdateConfig();
}

MainFrame::~MainFrame()
{
	SaveConfig();
}

void MainFrame::UpdateConfig( )
{
	_Func[0].addEntry(config.GetValue("FuncF1"));
	_Func[1].addEntry(config.GetValue("FuncF2"));
	_Func[2].addEntry(config.GetValue("FuncF3"));
	_Func[3].addEntry(config.GetValue("FuncF4"));
	_Func[4].addEntry(config.GetValue("FuncF5"));
	_Func[5].addEntry(config.GetValue("FuncF6"));
	_Func[6].addEntry(config.GetValue("FuncF7"));
	_Func[7].addEntry(config.GetValue("FuncF8"));
	_Func[8].addEntry(config.GetValue("FuncF9"));
	_Func[9].addEntry(config.GetValue("FuncF10"));
	_Func[10].addEntry(config.GetValue("FuncF11"));
	_Func[11].addEntry(config.GetValue("FuncF12"));

	_bSplit = config.GetBool("SplitWindow");
	_bHints = config.GetBool("Hint");
	_sLastPath = config.GetValue("LastPath");
}

void MainFrame::SaveConfig()
{
	_sLastPath = _tPanel[_nActive].GetCurrentPath();
	config.SetBool("SplitWindow", _bSplit);
	config.SetBool("Hint", _bHints);
	config.SetValue("LastPath", _sLastPath);
}

/// @brief		consol 초기화 작업.
bool MainFrame::Init()
{
	initscr();     // nCurses 시작
	start_color(); // 컬러 가능하게

	nonl();        // 엔터 처리?
	raw();		   // Ctrl+C : Signal무시 가능하게
	//cbreak();      // nCurses : 즉시 입력이 가능하도록 설정

	noecho();	   // nCurses : echo모드 해제
	keypad(stdscr, TRUE);  // keypad 활성화

	curs_set(0); // 커서를 보이지 않게 한다.

	use_default_colors();
	
	// 색깔 쌍을 초기화 한다. 디폴트 포함 0 ~ 64개 까지
	for (int t=0; t<8; t++)
		for (int t2=0; t2<8; t2++)
	{
		if (t==-1 && t2==-1) continue;
		init_pair(t*8+t2, t, t2);
	}

	// 디폴트 백그라운드를 위한 내용 (투명콘솔)
	init_pair(70, COLOR_WHITE, -1);

	// curses 마우스 설정
	mousemask( 	BUTTON1_CLICKED | BUTTON2_CLICKED | BUTTON3_CLICKED | 
				BUTTON_SHIFT | BUTTON_CTRL, NULL);
	//mousemask(ALL_MOUSE_EVENTS, NULL);
	mouseinterval(10);
	_beforeEvent.x = 0; _beforeEvent.y = 0; // 더블클릭을 위한
	
	// line 설정
	Set_Line(e_nBoxLineCode);

	_bShell = false;
	bkgd(COLOR(g_Color.Default.font, g_Color.Default.back));

	// Menu 에 넣을 도움말
	for (int n=0; n<12; n++)
		_tKeyBind.SetFuncKey(n+1, _Func[n][0]);

	// 커맨드를 초기화
	g_Command.Init();

	_tMenu.Create();
	return true;
}

void MainFrame::Destroy()
{
	clear();
	refresh();

	// . keypad 비활성화
	keypad(stdscr, FALSE);  // . 라인단위로 입력이 이루어지도록 설정
	//nocbreak();
	noraw();
	curs_set(1); // 커서를 다시 보이게 한다.
	noecho();
	endwin();   // . nCurses 해제
}

void MainFrame::SetSplitWindow( )
{
	if (_bSplit)
	{
		_tPanel[0].SetWindow(_win1, (g_nCOLS-2)/2, g_nLINES-7);
		_tPanel[1].SetWindow(_win2, (g_nCOLS-1)/2, g_nLINES-7);
	}
	else
	{
		_tPanel[_nActive].SetWindow(_win1, g_nCOLS, g_nLINES-7);
	}
}

void MainFrame::Draw()
{
	int i;
	//////////////////. Function Draw

	setcol(g_Color.Func);
	move(0,0);
	hline(' ', g_nCOLS);
	
	// Funtion panel draw
	for (i=0; i<8; i++)
	{
		int pos = GetColumnIndex(g_nCOLS, 8, i);
		mvprintw(0, 2+pos, "%s", gettext(_Func[i+1].getName()));
	}

	setcol(g_Color.FuncA);
	mvprintw (0, 0, "F");
	for (i=0; i<8; i++)
	{
		int pos = GetColumnIndex(g_nCOLS, 8, i);
		mvprintw(0, 1+pos, "%d", i+2);
	}

	setcol(COLOR_BLACK, g_Color.Func.back);
	for (i=1; i<8;i++)
	{
		int pos = GetColumnIndex(g_nCOLS, 8, i);
		mvaddch(0, pos, VLINE);
	}

	////////////////// . PATH Draw
	setcol(g_Color.Stat);
	move(1, 0); hline(' ', g_nCOLS);
	
	if (_bSplit)
	{
		mvprintw(1, 0, "%s", _tPanel[0].GetPathView().c_str());
		mvprintw(1, g_nCOLS/2+1, "%s", _tPanel[1].GetPathView().c_str());
		
		_tPanel[0].Draw(_nActive == 0);
		_tPanel[1].Draw(_nActive == 1);
	}
	else
	{
		mvprintw(1, 0, "%s", _tPanel[_nActive].GetPathView().c_str());
		_tPanel[_nActive].Draw(true);
	}

	//////// 상태선 그리기..
	_tPanel[_nActive].DrawStatus();

	//////// 2단일 경우. 중간의 꺾은선 출력.
	if (_bSplit)
	{
		setcol(g_Color.Line);
		for (int t=0; t<g_nLINES-7; t++)
		{
			move(3 + t, (g_nCOLS-2)/2);
			addch(VLINE);
			addch(VLINE);
		}

		move(2, (g_nCOLS-2)/2);
		addch(TTEE);
		addch(TTEE);

		move(g_nLINES-4, (g_nCOLS-2)/2);
		addch(BTEE);
		addch(BTEE);
	}

	if (_bHints) HintDraw();
	refresh();
}

void MainFrame::HintDraw()
{
	//////////////////. Hint Info
	std::map<std::string, std::string> 	mapHints;
	std::map<std::string, std::string>::iterator	 i;
	
	setcol(g_Color.Default);
	move(g_nLINES-2,0);
	hline(' ', g_nCOLS);

	mapHints[_tKeyBind.GetKeyStr("<shell>")] = gettext("Shell");
	mapHints[_tKeyBind.GetKeyStr("<exit>")] = gettext("Quit");
	mapHints["F1"] = gettext(_Func[0].getName());
	mapHints["F10"] = gettext(_Func[9].getName());
	mapHints["F11"] = gettext(_Func[10].getName());
	mapHints["F12"] = gettext(_Func[11].getName());

	int		pos = 0;

	mvprintw(g_nLINES-2, 0, "Hint: ");
	for (i = mapHints.begin(); i!=mapHints.end(); i++)
	{
		setcol(COLOR_CYAN, g_Color.Default.back);
		mvprintw(g_nLINES-2, 6+pos, "%s", i->first.c_str());
		pos = pos + strutil::krstrlen(i->first);
		setcol(g_Color.Default);
		mvprintw(g_nLINES-2, 6+pos, "-");
		setcol(g_Color.Default);
		pos = pos + 1;
		mvprintw(g_nLINES-2, 6+pos, "%s ", i->second.c_str());
		pos = pos + strutil::krstrlen(i->second)+1;
	}
}

void MainFrame::Refresh()
{
	if (g_nLINES - 7 <= 0) return;
	if ((g_nCOLS-2)/2 <= 0) return;

	if (_bSplit)
	{
		wresize(_win1, g_nLINES-7, (g_nCOLS-2)/2);
		wresize(_win2, g_nLINES-7, (g_nCOLS-1)/2);
		mvwin(_win1, 3, 0);
		mvwin(_win2, 3, (g_nCOLS+2)/2);
	}
	else
	{
		//wresize(_win0, g_nLINES-7, g_nCOLS);
		wresize(_win1, g_nLINES-7, g_nCOLS);
		mvwin(_win1, 3, 0);
	}

	SetSplitWindow();
	Draw();
}

///	@brief 메인프레임 수행
/// @param bMcd Mcd실행 여부.
void MainFrame::Do( bool bMcd )
{
	//_win0 = newwin(g_nLINES-7, g_nCOLS, 3, 0);
	//_win1 = newwin(g_nLINES-7, g_nCOLS, 3, 0);
	_win1 = newwin(g_nLINES-7, (g_nCOLS-2)/2, 3, 0);
	_win2 = newwin(g_nLINES-7, (g_nCOLS-1)/2, 3, (g_nCOLS+2)/2);

	_bExit = false;

	_tPanel[0].Read(".");
	if (_sLastPath.empty())
		_tPanel[1].Read(".");
	else
		_tPanel[1].Read(_sLastPath);	

	g_Command.SetPanel(&_tPanel[0]);

	// 디렉토리 이동
	chdir(_tPanel[_nActive].GetPathView().c_str());
	
	ESCDELAY = 10;  // ESC 딜레이 줄이기.
	int nErrno = 0; // 에러 번호 체크
	bool bDraw = true;
	
	if (g_Config._sVersion != VERSION)
	{
		Refresh();
		Draw();

		int nYN = YNBox(gettext("configuration files are not compatible. configuration files copy ?"), 1);

		if (nYN == YN_Y)
		{
			system(	"mkdir ~/.mls/back 2> /dev/null > /dev/null; "
					"cp ~/.mls/* ~/.mls/back 2> /dev/null > /dev/null; "
					"cp /etc/mls/* ~/.mls 2> /dev/null > /dev/null");

			g_Config.Load((g_Config.GetValue("cfghome") + "mls.cfg").c_str());
			g_Color.Load((g_Config.GetValue("cfghome") + "mls.col").c_str());
			_tKeyBind.Load((g_Config.GetValue("cfghome") + "mls.key").c_str());
			g_pEditor->KeyLoad((g_Config.GetValue("cfghome") +
								g_Config.GetValue("EditorKeyFile")).c_str());
			UpdateConfig();
		}
	}

	if (bMcd) g_Command.Execute("<mcd>");
	Refresh();

	while(!_bExit)
	{
		// 패널 화면을 그린다.
		if (bDraw) Draw();
		bDraw = true;
		
		// 쉘을 실행하면 계속 실행하게 한다.
		if (_bShell == true)
		{
			g_Command.Execute("<shell>");
			continue;
		}

		// 키를 입력 받는다.
		int nKey = getch();

		LOG("KEY [%d]", nKey);

		// 콘솔에 따라서 백그라운드로 남아 있는 것을 잡기 위한 방법
		if (nKey == -1)
		{
			nErrno++;
			if (nErrno == 100)
			{
				kill(getpid(), 3); // Signal Quit로 종료
			}
			continue;
		}
		else
		{
			nErrno = 0;
		}

		if (nKey == 3)
		{
			//nKey = getch(); // raw로 바꾼 후에는 한번더 입력 받지 않는다.
			g_Command.Execute("<copyclip>");
			continue;
		}

		// 리사이즈
		if (nKey == KEY_RESIZE)
		{
			g_nLINES = LINES;
			g_nCOLS = COLS;
			Refresh();
			bDraw = false;
			continue;
		}

		// 마우스
		if (nKey == KEY_MOUSE)
		{
			MouseProc();
			continue;
		}

		// Searching..
		if (_tPanel[_nActive].SearchProcess(nKey))
		{
			continue;
		}

		// Function key
		if (KEY_F(1) <= nKey && nKey <=KEY_F(12))
		{
			if (_Func[nKey-KEY_F(1)].size()>=1)
			{
				g_Command.Execute(_Func[nKey-KEY_F(1)][0]);
			}
			continue;
		}

		// Execute
		if (nKey != 27)
		{
			string command = _tKeyBind.GetCommand(nKey);
			g_Command.Execute(command);
		}
		else
		{
			// Alt + 문자는 이렇게 받는다.
			nodelay(stdscr, TRUE);
			nKey = getch();
			int nKey2 = getch();
			int nKey3 = getch();
			int nKey4 = getch();
			nodelay(stdscr, FALSE);

			// 27, 79, 72 (linux) ::  27, 91, 72 (vt100)
			if (nKey == 91 || nKey == 79)
			{
				// 리모트 접속 End, Home 키
				if (nKey2 == 72) { g_Command.Execute("<home>"); continue; }
				if (nKey2 == 70) { g_Command.Execute("<end>"); continue; }
				// Konsole. End Home 키
				if (nKey2 == 49 && nKey3 == 126) { g_Command.Execute("<home>"); continue; }
				if (nKey2 == 52 && nKey3 == 126) { g_Command.Execute("<end>"); continue; }
			}
			if (nKey3 >= 'A' && nKey3 <= 'E') // F1~F5 (konsole :: console)
			{
				g_Command.Execute(_Func[nKey3-'A'][0]); continue;
			}
			if (nKey2 == 49 && nKey3 >= 49 && nKey3 <= 52) // F1~F4 (konsole :: xterm xfree 3.x)
			{
				g_Command.Execute(_Func[nKey3-49][0]);	continue;
			}
			if (nKey == 79 && nKey2 >= 80 && nKey2 <= 83) // Remote F1~F4
			{
				g_Command.Execute(_Func[nKey2-80][0]);	continue;
			}

			if (nKey != ERR)
			{
				g_Command.Execute(_tKeyBind.GetMetaCommand(nKey));
			}
			else
			{
				g_Command.Execute(_tKeyBind.GetCommand(27));
			}
		}
	}
}

/// @brief	NextWindow
void MainFrame::NextWindow()
{
	if (_bSplit)
	{
		_nActive = (_nActive + 1) %2;
		g_Command.SetPanel(&_tPanel[_nActive]);

		// 디렉토리 이동
		chdir(_tPanel[_nActive].GetPathView().c_str());
	}
}

void MainFrame::RefreshPanel(bool bReload)
{
	if (_bSplit)
	{
		_tPanel[0].Refresh(bReload);
		_tPanel[1].Refresh(bReload);
	}
	else
	{
		_tPanel[_nActive].Refresh(bReload);
	}
	chdir(_tPanel[_nActive].GetPathView().c_str());
	Refresh();
}

#define	MENU_XPOS				0		///< 화면상의 Menu Y 위치
#define	FUNCTIONKEY_TOTALCOUNT	8		///< 화면상의 Functionkey 개수
#define	FILELIST_MIN_POS		3		///< FileList 시작 y 위치.

/**
 * @brief	마우스 이벤트 처리
 */
void MainFrame::MouseProc()
{
	MEVENT event;
	getmouse(&event);

	LOG("Event [%ld]",event.bstate);
	
	// 마우스가 메뉴를 눌렸을 경우
	if(event.y == MENU_XPOS)
	{
		int functionkey;
		// 어느 메뉴를 눌렀는지 확인하여 실행
		functionkey = XPositionToFunctionkey(event.x);
		if(KEY_F(1) < functionkey && functionkey <= KEY_F(9))
		{
			g_Command.Execute(_Func[functionkey-KEY_F(1)][0]);
		}	
	}
	// 마우스가 파일List를 눌렀을경우
	else if(FILELIST_MIN_POS <= event.y)
	{
		int	panelIndex;
		// 해당 패널 구하기
		panelIndex = XPositionToFilePanel(event.x);

		// x, y 위치를 패널에 맞게 지정
		SetXYLocalToPanel(event.x, event.y, panelIndex);

		// 해당 패널 활성화.
		if(panelIndex != _nActive)
		{
			NextWindow();
		}

		if ((event.bstate & BUTTON1_CLICKED))
		{
			LOG("Event.x [%d] Event.y [%d] BeforeX [%d] BeforeY [%d]",
					event.x, event.y, _beforeEvent.x, _beforeEvent.y);

			if (event.x == _beforeEvent.x && event.y == _beforeEvent.y)
			{
				_tPanel[panelIndex].MouseProc(event);
				_tPanel[panelIndex].Key_Enter();
				_beforeEvent.x = 0; _beforeEvent.y = 0;
				return;
			}
			else
			{			
				_tPanel[panelIndex].MouseProc(event);
				_beforeEvent.x = event.x; _beforeEvent.y = event.y;
			}
		}

		if ((event.bstate & BUTTON3_CLICKED))
		{
			_tPanel[panelIndex].MouseProc(event);
			_tPanel[panelIndex].ToggleSelect();
		}
	}
	return;
}

/**
 * @brief	화면상의 X position을 가지고 Function key을 얻는다
 * @param xPosition 화면상의 X position
 * @return	Function key (KEY_F(2) ~ KEY_F(8))
 **/
int MainFrame::XPositionToFunctionkey( const int xPosition )
{
	int index = GetIndex(g_nCOLS, FUNCTIONKEY_TOTALCOUNT, xPosition);
	int x     = GetColumnIndex(g_nCOLS, FUNCTIONKEY_TOTALCOUNT, index);
	
	if (x && xPosition == x) return -1;
	return KEY_F(index + 2);
}

/// @brief	window split switching Function.
void MainFrame::SwitchSplit()
{
	_bSplit = !_bSplit;
}

/// @brief	x 값으로 해당 패널을 찾는 함수.
/// @param x 화면 X position
/// @return	해당 패널 index.
int MainFrame::XPositionToFilePanel(const int x)
{
	if(!_bSplit || x < g_nCOLS/2)
		return 0;

	return 1;
}

/// @brief	패널에 맞게 X, Y위치를 변경한다.
/// @param	x			[OUT]	x위치.
/// @param	y			[OUT]	y위치.
/// @param	panelindex	패널 위치. 
void MainFrame::SetXYLocalToPanel(int & x, int & y, const int panelindex)
{
	x = x - panelindex * ((int)g_nCOLS/2);

	//	두번째 패널이면 가운데 구분선을 빼어 준다.
	x = panelindex == 1 ? x-1 : x;
	y = y - FILELIST_MIN_POS;
}

}  // namespace

