///	@file	panel.cpp
///	@brief	panel.cpp
#include <ncurses.h>
#include <map>
#include <sys/stat.h>

#include "mls.h"
#include "panel.h"
#include "mainframe.h"

//using namespace strutil;
using namespace MLS;

/// @brief	생성자
Panel::Panel()
{
	_cur=0;
	_page=0;

	_sel_size=0;
	_sel_num=0;

	// Zip 정의
	_nZip = 0;
	_sZipFile = "";
	_zipFullPath = "/";

	_bExit=false;
	_bSearch = false;

	_lines = g_nLINES;
	_cols = g_nCOLS;
	_win = NULL;
	_sort = 0;

	UpdateConfig();
}

/// @brief	소멸자
Panel::~Panel()
{
	SaveConfig();
	if (_win) delwin(_win);
}

void Panel::UpdateConfig()
{
	_bShowHidden = config.GetBool("ShowHidden");
	_bDirSort    = config.GetBool("DirSort");
	_bFixedRow   = config.GetBool("FixedRow");

	_column      = atoi(config.GetValue("Column").c_str());
	if (_column > 4) _column = 0;

	switch(atoi(config.GetValue("Sort").c_str()))
	{
	case 1: _eSortMethod = SORT_NAME; break;
	case 2: _eSortMethod = SORT_EXT; break;
	case 3: _eSortMethod = SORT_SIZE; break;
	case 4: _eSortMethod = SORT_TIME; break;
	case 5: _eSortMethod = SORT_COLOR; break;
	default:
	case 0: _eSortMethod = SORT_NONE; break;
	}

}

void Panel::SaveConfig()
{
	config.SetBool("ShowHidden", _bShowHidden);
	config.SetBool("DirSort", _bDirSort);
	config.SetBool("FixedRow", _bFixedRow);

	config.SetValue("Column", strutil::itoa(_column));

	switch(_eSortMethod)
	{
	case SORT_NONE:  config.SetValue("Sort", "0"); break;
	case SORT_NAME:  config.SetValue("Sort", "1"); break;
	case SORT_EXT:   config.SetValue("Sort", "2"); break;
	case SORT_SIZE:  config.SetValue("Sort", "3"); break;
	case SORT_TIME:  config.SetValue("Sort", "4"); break;
	case SORT_COLOR: config.SetValue("Sort", "5"); break;
	}

}

/// @brief	비슷한 이름을 가진 파일을 찾는다.
///	@param	str		찾을 파일 앞자리나 전체 파일명
/// @param  s_index 파일을 찾을 첫번째 인덱스
///	@return	찾은 파일 index, 실패시 -1
int
Panel::Search(const string &str, int s_index)
{
	if (_file_list.empty())
		return -1;

	int l = _file_list.size(), len = str.size();
	string p=strutil::tolower(str);

	//if (l < s_index) s_index = s_index % l;

	for (int t=0; t < l ; t++)
	{
		int x = (t + s_index) % l;

		if ( _file_list[x]->name.size() < str.size() )
			continue;
		if ( strutil::tolower(_file_list[x]->name.substr(0, len)) == p)
		{
			return x;
	 	}
	}
	// 없으면
	return -1;
}


/// @brief 엔터키
void
Panel::Key_Enter(bool bExe)
{
	int key, bkey;
	bool bSearch = false;

	string tTmp;
	string::size_type	tSizeType;

	if (_nZip == 0)
	{
		if (Read(_file_list[_cur]->name))
		{
			LOG("Read Success");
			return;
		}

		if (_file_list[_cur]->bDir)
		{
			string sDirPath = _path + _file_list[_cur]->name;	
			if (access(sDirPath.c_str(), R_OK | X_OK) == -1)
			{
				MsgBox(	gettext("Error"), strerror(errno));
				return;
			}
		}
	}
	else
	{
		if (_file_list[_cur]->FullName == "/")
		{
			// new된 Zip 을 삭제
			string sZipFile = _sZipFile;
			
			_tReader->Destroy();
			_sZipFilePath = "";
			_sZipFile = "";
			_nZip = 0;

			Read(GetCurrentPath());
			// tar.gz 위치로
			int cur = Search(sZipFile, 0);
			if (cur != -1) _cur = cur;
			return;
		}

		if (_file_list[_cur]->bDir)
		{
			// 압축 파일은 FullName 으로 한다.
			LOG("Read Zip :: %s", _file_list[_cur]->FullName.c_str());
			if (_file_list[_cur]->name == "..") // .. 구현
			{
				tTmp = _file_list[_cur]->FullName;
				tTmp = tTmp.substr(0, tTmp.size() - 1);

				tSizeType = tTmp.rfind("/");

				string sBefPath;
				if (tSizeType != string::npos)
					sBefPath = tTmp.substr(tSizeType+1, tTmp.size() - tSizeType);
				else
					sBefPath = tTmp;

				tTmp = tTmp.substr(0, tSizeType+1);
				_zipPath = _sZipFile + " :: /" + tTmp;
				_zipFullPath = "/" + tTmp;
				
				if (Read(tTmp))
				{
					LOG("READ [%s]", sBefPath.c_str());
					int cur = Search(sBefPath, 0);
					if (cur != -1) _cur = cur;
					return;
				}
			}
			else
			{
				_zipPath = _sZipFile + " :: /" + _file_list[_cur]->FullName;
				_zipFullPath = "/" + _file_list[_cur]->FullName;
				if (Read(_file_list[_cur]->FullName))
					return;
			}
			return;
		}
	}

	if (!_file_list[_cur]->bDir)
	{
		WINDOW*	pWin = NULL;

		if (_nZip == 0)
		{
			// 파일 실행
			string sDirPath = _path + _file_list[_cur]->name;

			if (access(sDirPath.c_str(), R_OK) == -1)
			{
				MsgBox(gettext("Error"), strerror(errno));
				return;
			}

			if (bExe == true)
			{
				std::string cmd;
				cmd = _path + _file_list[_cur]->name;
				vector<string> q;
				q.push_back(gettext("Run"));
				q.push_back(gettext("Run(root)"));
				q.push_back(gettext("Edit"));
				q.push_back(gettext("ExtEditor"));
				q.push_back(gettext("Parameter"));
				q.push_back(gettext("Parameter(root)"));
				q.push_back(gettext("Cancel"));
	
				ListBox tListBox(gettext("Execute select"));
				tListBox.SetItem(q);
				int 	nSelect = 0;
				if ((nSelect = tListBox.Do()) == -1) return;
				switch(nSelect)
				{
					case 0:
						g_Command.ParseAndRun(cmd + " %W");
						return;
					case 1:
						g_Command.ParseAndRun(cmd + " %R %W");
						return;
					case 2:
						g_Command.Execute("<view>");
						return;
					case 3:
						g_Command.Execute("<exteditor>");
						return;
					case 4:
					case 5:
					{
						std::string sParam;
						if (InputBox(gettext("Input parameter."), sParam)>0)
							if (nSelect == 4)
								g_Command.ParseAndRun(cmd + " %W " + sParam);
							else
								g_Command.ParseAndRun(cmd + " %R %W " + sParam);
						return;
					}
				}
				return;
			}

			std::map<std::string, std::string>::iterator i;
			i = config.ext_bind.find( strutil::tolower( _file_list[_cur]->ext()) );
			
			if (i != config.ext_bind.end())
			{
				if (!(*i).second.empty())
				{
					// 확장자가 압축화일이면 내용을 보여준다.
					if (((*i).second) == "<zip>")
					{
						try
						{
							_nZip = 1;
							_sZipFilePath = _path;
							_sZipFile = _file_list[_cur]->name;
							_zipPath = _sZipFile + " :: /";
							if (Read("") == true)	return;
							else
							{	_nZip = 0; return; }
						}
						catch(Exception& ex)
						{
							MsgBox(1, gettext("Error"), "%s", (char*)ex);
						}
					} 
					else if (((*i).second) == "<view>")
					{
						g_Command.Execute("<view>");
						return;
					}
					else 
					{
						g_Command.Execute(((*i).second));
						return;
					}
				}
			} else {
				i = config.name_bind.find(strutil::tolower(_file_list[_cur]->name));
				if (i != config.name_bind.end()) {
					if (!(*i).second.empty()) {
						g_Command.Execute(((*i).second));
						return;
					}
				}
			}

			if (_file_list[_cur]->attr[3] == 'x')
			{
				std::string cmd;
				cmd = _path + _file_list[_cur]->name;
				if (access(cmd.c_str(), X_OK) == -1)
				{
					MsgBox(	gettext("Error"), strerror(errno));
					return;
				}
				g_Command.ParseAndRun(cmd + " %W");
			}
		}
		else
		{
			string	sTmp;
			File*	pFile	= _file_list[_cur];

			sTmp = pFile->name;

			try
			{
				// 압축 파일 /tmp에 풀고 filename 리턴
				LOG("DataName :: [%s]", sTmp.c_str());

				// 파일 실행
				std::map<std::string, std::string >::iterator i;

				i = config.ext_bind.find( strutil::tolower( pFile->ext()) );

				if (i != config.ext_bind.end())
				{
					if (!(*i).second.empty()) {
						g_Command.Execute(((*i).second));
						return;
					}
				} else {
					i = config.name_bind.find(strutil::tolower(pFile->name));
					if (i != config.name_bind.end()) {
						if (!(*i).second.empty()) {
							g_Command.Execute(((*i).second));
							return;
						}
					}
				}

				MsgBox(	gettext("Error"), gettext("Not execute file. "));
			}
			catch(Exception& ex)
			{
				MsgBox(1, gettext("Error"), "%s", (char*)ex);
			}
			return;
		}
	}
}

/// @brief RPM, Deb, Zip Install or Compress
void
Panel::Key_ZipInstall()
{
	string	sPath;
	string	sFilename;
	
	if (_nZip == true)
	{
		sPath = _sZipFilePath;
		sFilename = _sZipFile;
	}
	else
	{
		sPath = _path;
		sFilename = _file_list[_cur]->name;

		if (sFilename == "..") 
		{
			MsgBox(1, gettext("Error"), "parent directory can't compress."); 
			return;
		}
	}

	int			nList, nRt;
	TargzUtil	tTargzUtil(sPath, sFilename, 0);

	string sMsg = gettext("Compress the select files. ");

	int nFileinfo = 0;
	vector<string> q;
	ListBox tListBox(gettext("Achive Menu"));

	nFileinfo = tTargzUtil.GetFileInfo();

	if (_nZip == false)
	{
		q.push_back(sMsg + "(tar.gz)");
		q.push_back(sMsg + "(tar.bz2)");
		q.push_back(sMsg + "(zip)");
		
		if (nFileinfo == ERROR)
		{
			q.push_back(gettext("Cancel"));
			tListBox.SetItem(q);
			if ((nRt = tListBox.Do()) == -1) return;
			FileCompress(nRt);
			return;
		}
	}
	
	sMsg = gettext("Current file install.");
	sMsg = sMsg + "[" + sFilename + "]";
	q.insert(q.begin(), sMsg);
	q.push_back(gettext("Cancel"));
	tListBox.SetItem(q);

	if ((nRt = tListBox.Do()) == -1) return;
	if (_nZip == false && nFileinfo == SUCCESS)
	{
		if (nRt != 0) { FileCompress(nRt-1); return; }
	}
	else
	{
		if (nRt == q.size()-1) return;
	}

	tTargzUtil.GetFileInfo();
	ZIPTYPE 		nZipType = tTargzUtil.GetZipType();

	LOG("FILENAME [%s] [%s]", sPath.c_str(), sFilename.c_str());

	switch(nZipType)
	{
		case TAR_GZ:
		case TAR_BZ:
		case ZIP:
		case ALZ:
		case TAR:
		{
			string 			sTargetDir;

			try
			{
				Progress 	tProgress(gettext("Wait"), gettext(	"Please wait !!! - Cancel Key [ESC]"));

				MLS::DirMcd mcd("", gettext("Extract to..."));
				
				tProgress.Start((Counter*)&mcd, JOIN);

				mcd.SetCounter(0);
				mcd.AddDirectory(GetCurrentPath());
				mcd.setCur(GetCurrentPath());
				mcd.SetCounter(100);
				tProgress.SetRun(PTHREAD_END);
				tProgress.End();
				if (mcd.proc())
					sTargetDir = mcd.getCur();
			}
			catch(Exception& ex)
			{
				MsgBox(1, gettext("Error"), "%s", (char*)ex);
			}

			if (sTargetDir != "")
			{
				try
				{
					string sView = gettext("Please wait !!! - Uncompress file [") + _file_list[_cur]->name + "]";

					Progress 	tProgress(gettext("Wait"), sView.c_str());
					tProgress.Start((Counter*)&tTargzUtil, JOIN);
					tTargzUtil.SetCounter(0);
					tTargzUtil.Uncompress(sTargetDir);
					tTargzUtil.SetCounter(100);
					tProgress.SetRun(PTHREAD_END);
					tProgress.End();
				}
				catch(Exception& ex)
				{
					MsgBox(1, gettext("Error"), "%s", (char*)ex);
				}

				// 압축을 풀면 풀린디렉토리로 이동
				if (_nZip == true)
				{
					// new된 Zip 을 삭제
					_tReader->Destroy();
					_sZipFilePath = "";
					_sZipFile = "";
					_nZip = 0;
				}
				Read(sTargetDir);
			}
			break;
		}

		case RPM:
		{
			vector<string> q;
			q.push_back(gettext("Install"));
			q.push_back(gettext("Install Nodeps"));
			q.push_back(gettext("Cancel"));

			int nSelect = SelectBox(sFilename.c_str(), q, 0);

			string sCommand;

			switch(nSelect)
			{
				case 0:	// root 계정으로 실행
					sCommand = "rpm -Uvh " + sPath + sFilename + " %R %W"; 
					break;
				case 1: // root 계정으로 실행
					sCommand = "rpm -Uvh --nodeps " + sPath + sFilename + " %R %W";
					break;
				case 2:	return;	break;
			}

			LOG("RPM Install [%s]", sCommand.c_str());
			//g_Command.Run(sCommand, true);
			g_Command.ParseAndRun(sCommand);
			break;
		}

		case DEB:
		{
			string sMsg = gettext("Do you want install current file ?");
			sMsg = sMsg + "[" + sFilename + "]";
			if (YNBox(sMsg.c_str(), YN_N) == YN_Y)
			{
				string sCommand = "dpkg -i " + sPath + sFilename + " %R %W";
				LOG("DEB Install [%s]", sCommand.c_str());
				g_Command.ParseAndRun(sCommand);
			}
			break;
		}
	}
}

/// @brief	각 type 에 맞게 현재 선택된 파일을 압축해준다.
///	@param	nType	0 : tar.gz, 1 : tar.bz2, 2 : zip
void
Panel::FileCompress(int nType)
{
	
	string sTitle = gettext("Input name of the compressed file. [filename]");
	string	sFilename;

	switch(nType)
	{
		case 0:
			InputBox((sTitle + ".tar.gz").c_str(), sFilename);
			if (sFilename == "") return;
			sFilename = sFilename + ".tar.gz";
			break;
		case 1:
			InputBox((sTitle + ".tar.bz2").c_str(), sFilename);
			if (sFilename == "") return;
			sFilename = sFilename + ".tar.bz2";
			break;
		case 2:
			InputBox((sTitle + ".zip").c_str(), sFilename);
			if (sFilename == "") return;
			sFilename = sFilename + ".zip";
			break;
		default:
			return;
			break;
	}

	if (access(sFilename.c_str(), R_OK) == 0)
    {
		MsgBox(gettext("Error"), gettext("input file used."));
        return;
    }

	if (access(_path.c_str(), W_OK) == -1)
	{
		MsgBox(gettext("Error"), strerror(errno));
		return;
	}

	Progress 	tProgress(gettext("Wait"));

	try
	{
		vector<string>	tFileList = GetSelection().GetData();
	
		TargzUtil	tTargzUtil(_path, sFilename, true);

		ZIPTYPE 		nZipType = tTargzUtil.GetZipType();

		LOG("FILENAME [%s] [%d]", sFilename.c_str(), tTargzUtil.GetZipType());

		string sView = gettext("Please wait !!! - Compress file [") + _file_list[_cur]->name + "]";
	
		tProgress.SetMsg(sView);
		tProgress.Start((Counter*)&tTargzUtil, JOIN);
		tTargzUtil.SetCounter(0);
		tTargzUtil.Compress(tFileList, TAR_APPEND);
		tTargzUtil.SetCounter(100);		
	}
	catch(Exception& ex)
	{
		MsgBox(1, gettext("Error"), "%s", (char*)ex);
	}

	tProgress.SetRun(PTHREAD_END);
	tProgress.End();

	Refresh();
}

/// @brief	왼쪽 방향키
void
Panel::Key_Left()
{
	if (_cur < (unsigned)_nrow)
	{
		SetCur(0);
	}
	else
	{
		SetCur(_cur-_nrow);
	}
}

/// @brief	오른쪽 방향 키
void
Panel::Key_Right()
{
	int cur;
	cur = _cur+_nrow;
	if (cur > _file_list.size()-1)
		cur=_file_list.size()-1;
	SetCur(cur);
}

/// @brief	위쪽 방향키
void
Panel::Key_Up()
{
	if (_cur > 0)
	{
		SetCur(_cur-1);
	}
}

/// @brief	아래쪽 방향키
void
Panel::Key_Down()
{
	if (_cur < _file_list.size()-1)
	{
		SetCur(_cur+1);
	}
}

/// @brief	page up key
void
Panel::Key_PageUp()
{
	if (_cur < (unsigned)_nrow*_ncol)
		SetCur(0);
	else
		SetCur(_cur-(_nrow*_ncol));

}

/// @brief	page down key
void
Panel::Key_PageDown()
{
	int cur;
	cur = _cur + (_nrow*_ncol);
	if (cur > _file_list.size()-1)
		cur=_file_list.size()-1;

	SetCur(cur);
}

/// @brief	home key
void
Panel::Key_Home()
{
	SetCur(0);
}

/// @brief	Key End
void
Panel::Key_End()
{
	SetCur(_file_list.size()-1);
}

/// @brief  키를 입력받아 파일 찾기
/// @param
/// @return 찾았는가 못찾았는가
bool
Panel::SearchProcess(int key)
{
	int tmp;

	if (!_bSearch && ( isalnum(key) || key == '.' ) ) _bSearch = true;

	if (_bSearch && key == '\t')
	{// 탭이 나오면 다음 것을 찾는다.
		if ((tmp = Search(_strSearch, _cur+1)) != -1)
		{
			SetCur(tmp);
		}
		return true;
	}
		// .. 파일을 찾아 커서를 파일위에 둔다
	if (_bSearch &&  key != '/' && ( (32<= key && key <=126) || key == KEY_BS ) )
	{
		// search mode;
		// 뒷부분에 key를 넣고

		if (key == KEY_BS)
		{
			_strSearch.erase(_strSearch.size()-1, 1);
			if (_strSearch.empty())
			{
				_bSearch = false;
				return true;
			}
		}
		else
		{
			_strSearch += (char) key;
		}

		// _strSearch로 시작하는 문자를 찾는다..
		if ((tmp = Search(_strSearch, _cur))== -1)
		{
			// 없으면 처음에 넣고 다시 찾는다.
			// bugfix : gcc 2.96
			string sKey;
			sKey = (char)key;
			if ((tmp = Search(sKey)) != -1)
			{
				_strSearch = (char)key;
				SetCur(tmp);
			}
			else
			{
				_strSearch.erase(_strSearch.size()-1, 1);
			}
		}
		else
		{
			SetCur(tmp);
		}

		return true;
	}

	_bSearch = false;
	_strSearch.erase();

	return false;
}

void
Panel::SetWindow( WINDOW *win, int w, int h)
{
	_win = win;

	_lines=h;
	_cols=w;
}

///	\brief	선택된 파일 목록을 가져오는 함수
///
/// 내부static 변수를 리턴함 쓰레드에 대해서 안전하지 않음
/// 가져온 스트링의 마지막이 / 로 끝나면 디렉토리 입니다.
/// zip 파일일때는 bDirExpansion은 true가 되선 않된다.
///
///	\param	size			파일 목록 개수(반환값)
///	\param	bDirExpansion	Directory 확장여부
///	\return	선택된 파일 목록
Selection Panel::GetSelection( )
{
	Selection ret;

	if (_sel_num == 0)
	{
		if (_file_list[_cur]->name != "..")
		{
			if (_nZip == 0)
			{
				if (_file_list[_cur]->bDir)
					ret.Add(_file_list[_cur]->name + '/');
				else
					ret.Add(_file_list[_cur]->name);
			}
			else
			{
				// zip 이면 TarFullName 을 넣는다.
				ret.SetZipFile(_Zip_Reader.GetZipPath()+_sZipFile);
				ret.Add(_file_list[_cur]->TarFullName);
			}
		}
	}
	else
	{
		vector<string> directories;

		for (pFileIterator i= _file_list.begin(); i!=_file_list.end(); ++i)
		{
			if ((*i)->selected)
			{
				if (_nZip == 0)
				{
					if ((*i)->bDir)
						directories.push_back((*i)->name + '/');
					else
						ret.Add((*i)->name);
				}
				else
				{
					// zip 이면 FullName을 넣는다.
					ret.SetZipFile(_Zip_Reader.GetZipPath()+_sZipFile);
					ret.Add((*i)->TarFullName);
					if (!(*i)->bDir)
					{
						//size += (*i)->size;
					}
				}
			}
		}

		for (int t=0; t<directories.size(); t++)
		{
			ret.Add(directories[t]);
		}
	}

	ret.SetBaseDir(_path);
	return ret;
}

///	\brief	파일을 선택한다.(반전 시킴)
///	\param	f	선택할 파일
void Panel::Select(File &f)
{
	if (f.name == "..")
		return;

	if (f.selected)
		return;

	// 인덱스를 저장
	f.selected = true;
	_sel_size += f.size;
	_sel_num++;
}

///	\brief	파일선택을 해제한다.
///	\param	f	선택 해제할 파일
void Panel::Deselect(File &f)
{
	if (!f.selected)
		return;

	f.selected = false;
	_sel_size -= f.size;
	_sel_num--;
}

/// @brief	파일선택
void Panel::ToggleSelect()
{
	if (_file_list[_cur]->selected)
	{
		Deselect(*_file_list[_cur]);
	}
	else
	{
		Select(*_file_list[_cur]);
	}

	if (_cur < _file_list.size()-1)
	{
		_cur++;
	}
}

/// @brief	마우스 이벤트 처리.
/// @param	event	Event
void Panel::MouseProc(const MEVENT&		event)
{
	int		fileRowIndex;
	int		fileColIndex;
	int		fileIndex;
	int		fileMaxIndex;

	// x 위치로 해당 컬럼위치를 구한다.
	fileColIndex = event.x / (_cols / _ncol);

	fileRowIndex = event.y;

	fileIndex = fileColIndex * _nrow + fileRowIndex + (_page * _ncol * _nrow);

	fileMaxIndex =  _file_list.size() - 1;
	if(fileIndex < fileMaxIndex)
	{
		_cur = fileIndex;
	}
	else
	{
		_cur = fileMaxIndex;
	}
}

