/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441Decoder.h"

#include "common/SFFT.h"
#include "common/SoundFile.h"
#include "common/Exception.h"

#include <math.h>

enum {
	FSK441_SEEKING,
	FSK441_LOCKED
};

int main(int argc, char **argv)
{
	if (argc < 3) {
		::fprintf(stderr, "Usage: FSK441Decoder <filename> <ratio>\n");
		return 1;
	}

	wxString fileName  = wxString(argv[1]);
	double       ratio = ::atof(argv[2]);

	try {
		CFSK441Decoder decoder(fileName, ratio);
		decoder.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "Error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "An exception has occurred\n");
		return 1;
	}

	return 0;
}

CFSK441Decoder::CFSK441Decoder(const wxString& fileName, double ratio) :
m_fileName(fileName),
m_ratio(ratio),
m_noise(),
m_burstData()
{
	for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++)
		m_burstData[i].setRatio(ratio);
}

CFSK441Decoder::~CFSK441Decoder()
{
}

void CFSK441Decoder::run()
{
	CSoundFile* file = new CSoundFile();

	file->openRead(m_fileName, FSK441_SAMPLE_RATE, 16);

	CSFFT sfft(FSK441_FFT_LENGTH, FSK441_BIN0, FSK441_BIN3 + 1);

	double* in = new double[FSK441_FFT_LENGTH];

	int tim   = 0;
	int state = FSK441_SEEKING;

	while (true) {
		int len = FSK441_FFT_LENGTH;
		if (!file->read(in, len))
			break;

		for (int i = 0; i < len; i++) {
			double* bins = sfft.process(in[i]);

			double v[4];
			v[0] = bins[FSK441_BIN0];
			v[1] = bins[FSK441_BIN1];
			v[2] = bins[FSK441_BIN2];
			v[3] = bins[FSK441_BIN3];

			if (tim >= FSK441_FFT_LENGTH) {
				switch (state) {
					case FSK441_SEEKING:
						state = seekBurst(tim, v);
						break;
					case FSK441_LOCKED:
						state = storeBurst(tim, v);
						break;
					default:
						::printf("Unknown state\n");
						state = FSK441_SEEKING;
						break;
				}
			}

			tim++;
		}
	}

	file->close();

	delete file;
	delete[] in;

	// End of the data and still receiving a burst, display the collected data
	if (state == FSK441_LOCKED)
		decodeBurst();
}

/*
 * Correlator for the four FSK441 tones held in the bins represented by
 * v0, v1, v2 and v3 with the correlation values returned in c0, c1, c2
 * and c3. If there is zero or negative correlation than a false is
 * returned and the values in c0 to c3 are not normalised, a true
 * indicates a positive correlation with normalised values returned in
 * c0 to c3.
 */
bool CFSK441Decoder::correlate(double* v, double* c) const
{
	wxASSERT(v != NULL);
	wxASSERT(c != NULL);

	c[0] = 3.0 * v[0] - v[1] - v[2] - v[3];
	c[1] = 3.0 * v[1] - v[0] - v[2] - v[3];
	c[2] = 3.0 * v[2] - v[0] - v[1] - v[3];
	c[3] = 3.0 * v[3] - v[0] - v[1] - v[2];

	// Not interesting, return false and don't do the expensive sqrt() function
	if (c[0] < m_ratio && c[1] < m_ratio && c[2] < m_ratio && c[3] < m_ratio)
		return false;

	double div = ::sqrt(12.0 * (v[0] * v[0] + v[1] * v[1] + v[2] * v[2] + v[3] * v[3]));

	c[0] /= div;
	c[1] /= div;
	c[2] /= div;
	c[3] /= div;

	return true;
}

/*
 * This is the normal state when a burst has not been detected. Every
 * fifth FFT reading is processed here and if any data with a suitable
 * correlation is found then its value is stored and the state is set
 * LOCKING. We are at the beginning of a burst.
 */
int CFSK441Decoder::seekBurst(int tim, double* v)
{
	wxASSERT(v != NULL);

	double c[4];
	bool ret = correlate(v, c);

	// Nothing to do
	if (!ret) {
		m_noise.addValue(v[0]);
		m_noise.addValue(v[1]);
		m_noise.addValue(v[2]);
		m_noise.addValue(v[3]);

		return FSK441_SEEKING;
	}

	bool found = false;

	for (int i = 0; i < 4; i++) {
		if (c[i] > m_ratio)
			found = true;
	}

	if (found) {
		for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
			m_burstData[i].setStartBurst();
			m_burstData[i].setStartTime(tim);
		}

		int n = tim % FSK441_SYMBOL_LENGTH;
		m_burstData[n].setData(v, c);

		return FSK441_LOCKED;
	}

	return FSK441_SEEKING;
}

/*
 * We now have a burst of over one symbol in length and a more or
 * less optimum sample location for it. From now on we only process
 * every 25 sample times (one symbol length) and find the best
 * correlation values and store the appropriate symbol in m_data.
 * Once the burst ends then we call decodeBurst() to process and
 * display the burst data.
 */
int CFSK441Decoder::storeBurst(int tim, double* v)
{
	wxASSERT(v != NULL);

	double c[4];
	bool ret = correlate(v, c);

	if (!ret) {
		m_noise.addValue(v[0]);
		m_noise.addValue(v[1]);
		m_noise.addValue(v[2]);
		m_noise.addValue(v[3]);
	}

	int n = tim % FSK441_SYMBOL_LENGTH;

	m_burstData[n].setData(v, c);

	for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++) {
		if (m_burstData[i].getInBurst())
			return FSK441_LOCKED;
	}

	decodeBurst();

	return FSK441_SEEKING;
}

/*
 * This is where the raw burst data is processed. We look for a
 * space character, the synchronisation key in FSK441, and failing
 * that we look for single tone characters. If none is found then
 * we have a very strange burst indeed !
 * The burst data is held as a character string containing the
 * numbers representing the different tones so we can use string
 * manipulation methods on it.
 */
void CFSK441Decoder::decodeBurst()
{
	int maxLen = 0;
	int      n = -1;

	for (int i = 1; i < FSK441_SYMBOL_LENGTH; i++) {
		bool valid = m_burstData[i].processBurst(false) > 0;
		int    len = m_burstData[i].getLength();

		if (valid && len > maxLen && len >= 9) {
			maxLen = len;
			n      = i;
		}
	}

	if (n == -1)
		return;

	wxString    text = m_burstData[n].getMessage();
	double  strength = m_burstData[n].getStrength(m_noise.getAverage());
	double startTime = double(m_burstData[n].getStartTime()) * 0.0000907;
	int       length = int(double(m_burstData[n].getLength()) * 2.2676);

	::printf("n:%d Start:%d/%.1fs Length:%d/%dms Strength:%.0fdB Text:%s\n", n, m_burstData[n].getStartTime(), startTime, m_burstData[n].getLength(), length, strength, text.mb_str());
}
