package Lire::DlfConverterManager;

use strict;

use vars qw/ $instance /;

use Lire::Config;
use Lire::OldDlfAdapter;
use Lire::Utils qw/file_content/;

use Carp;

$instance = new Lire::DlfConverterManager();

sub new {
    return bless { '_converters' => {}, }, shift;
}

=pod

=head1 NAME

Lire::DlfConverterManager - Manages the available DLF conveters

=head1 SYNOPSIS

  use Lire::DlfConverterManager;

  my $mgr = Lire::DlfConverterManager->instance;
  $mgr->register_converter( $dlf_converter );

  # Another common case

  use Lire::DlfConverterManager;

  my $mgr = Lire::DlfConverterManager->instance;
  $mgr->register_default_converters();
  my $converter = $mgr->get_converter( "combined")

=head1 DESCRIPTION

This singleton object will take care of maitaining the list of
available converters and instantiating them.

The DlfConverterManager also takes care of registering the standard
available set of DLF converters.

=head2 instance()

Returns the singleton instance of the DlfConverterManager

=cut

sub instance {
    return $instance;
}

=pod

=head2 converter_names()

Returns the names of all available DLF converters available.

=cut

sub converter_names {
    my ( $self ) = @_;

    return keys %{$self->{'_converters'}};
}

=pod

=head2 converters()

Returns all the available DLF converters (as objects).

=cut

sub converters {
    my ( $self ) = @_;

    return values %{$self->{'_converters'}}
}

=pod

=head2 get_converter( $name )

Returns the Lire::DlfConverter object named $name. Returns undef when
no DLF converter named $name was registered.

=cut

sub get_converter {
    my ( $self, $name ) = @_;

    return $self->{'_converters'}{$name};
}

=pod

=head2 register_converter( $converter )

Adds a Lire::DlfConverter to the list of available DlfConverters. This
method can be used by the configuration framework to register
available converters.

This method will die if there is already another converter registered
under the same name.

=cut

sub register_converter {
    my ( $self, $dlf_converter) = @_;

    croak( "$dlf_converter isn't a instance of Lire::DlfConverter")
      unless UNIVERSAL::isa( $dlf_converter, "Lire::DlfConverter" );

    my $name = $dlf_converter->name;

    croak( "there is already a DLF converter registered under '$name'" )
      if exists $self->{'_converters'}{$name};

    $self->{'_converters'}{$name} = $dlf_converter;
}

=pod

=head2 unregister_converter( $name )

Mark a Lire::DlfConverter as unavailable. This method will die if
there is no converter registered under that name. Returns the instance
that was registered under that name.

=cut

sub unregister_converter {
    my ( $self, $name) = @_;

    croak ( "there is no DLF converter '$name' registered" )
      unless exists $self->{'_converters'}{$name};

    my $converter = $self->{'_converters'}{$name};
    delete $self->{'_converters'}{$name};

    $converter;
}

=pod 

=head2 register_default_converters()

This method will loads initialize all the converters available. For old
style script-based DLF converters, an OldDlfAdapter DlfConverter will
be created based on the content of the files I<lr_old_address_file>.

The set of new module based DlfConverters to instanciated is taken
from the I<lr_converters_init_path> configuration variable. Each files
present in these directories will be interpreted as a perl script
which should return one or more instances of Lire::DlfConverter which
will be registered with the DlfConverterManager.

=cut

sub register_default_converters {
    my $self = $_[0];

    $self->_load_dlf_adapters();
    $self->_create_old_dlf_adapters();

    return;
}

sub _create_old_dlf_adapters {
    my $self = $_[0];

    my $convertors_dir = Lire::Config->get( 'lr_old_convertors_dir' );
    my $address_file = Lire::Config->get( 'lr_old_address_file' );
    my $service2schema = $self->_parse_old_map_file( $address_file );
    while ( my ($service, $schema) = each %$service2schema ) {
        unless ( Lire::DlfSchema->has_superservice( $schema )) {
            warn "invalid superservice '$schema' assigned to service '$service'\n";
            next;
        }
        my $script = $convertors_dir .'/' . $service . '2dlf';
        if ( -x $script ) {
            my $adapter = new Lire::OldDlfAdapter( $schema, $script );
            $self->register_converter( $adapter );
        } else {
            warn "can't find executable $service" . "2dlf in $convertors_dir\n";
        }
    }
}

sub _parse_old_map_file {
    my ($self, $file) = @_;

    my %map = ();

    open my $fh, $file
      or croak "can't open file '$file': $!";
    my $line;
    while ( defined($line = <$fh>) ) {
        next if $line =~ /^#/;       # Skip comments
        next if $line =~ /^\s*$/;    # Skip empty lines
        my ($key, $value) = $line =~ /^(\S+)\s+(\S+)\s*(#.*)?$/
          or warn "can't parse line $. of file '$file'\n";
        $map{$key} = $value
          if defined $key;
    }
    close $fh;

    return \%map;
}

sub _load_dlf_adapters {
    my $self = $_[0];

    foreach my $init_file ( $self->_converters_init_files() ) {
        my $initializer = eval {  file_content( $init_file ) };
        if ( $@ ) {
            warn "error reading DLF converter initializer file '$init_file': $@\n";
            next;
        }
        my @converters = eval $initializer;
        if ( $@ ) {
            warn "error while running initializer in '$init_file': $@\n";
            next;
        }
        foreach my $c ( @converters ) {
            if ( UNIVERSAL::isa( $c, 'Lire::DlfConverter' ) ) {
                $self->register_converter( $c );
            } else {
                warn "initializaer '$init_file' didn't return a Lire::DlfConverter object: $c\n";
            }
        }
    }
}

sub _converters_init_files {
    my $self = $_[0];

    my @initializers = ();
    my %dirs;
    foreach my $dir ( @{ Lire::Config->get( 'lr_converters_init_path' )} ) {
        next if exists $dirs{$dir};
        $dirs{$dir} = 'done';
        opendir my $dh, $dir
          or croak "failed to opendir '$dir': $!";
        foreach my $file ( map { "$dir/$_" } readdir $dh ) {
            next unless -f $file;
            push @initializers, $file;
        }
        closedir $dh;
    }

    return @initializers;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfConverter(3pm)

=head1 VERSION

$Id: DlfConverterManager.pm,v 1.11 2004/03/31 18:39:00 flacoste Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002-2003 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

