package Lire::DlfStream;

use strict;

use Lire::DlfSchema;
use Lire::DlfQuery;

use Lire::Utils qw/ sql_quote_name check_param check_object_param /;
use Lire::I18N qw/ ensure_utf8  /;

use Carp;

=pod

=head1 NAME

Lire::DlfStream - Interface to DLF data stream

=head1 SYNOPSIS

  use Lire::DlfStore;

  my $store = Lire::DlfStore->open( "mystore" );
  my $dlf_stream = $store->open_dlf_stream( "www", "r" );

  print "Data begins on ", scalar localtime $dlf_stream->start_time(), "\n";
  print "Data ends on   ", scalar localtime $dlf_stream->end_time(), "\n";
  while ( my $dlf = $dlf_stream->read_dlf() ) {
    ...
  }

=head1 DESCRIPTION

This object encapsulates DLF stream.

=cut

#------------------------------------------------------------------------
# Constructor new( $store, $name, $mode )
#
# Used by Lire::DlfStore
sub new {
    my ( $pkg, $store, $name, $mode, $sort_spec ) = @_;

    check_object_param( $store, 'store', 'Lire::DlfStore' );
    check_param( $mode, 'mode', sub { $mode eq 'r' ||
                                      $mode eq 'w' },
                 "'mode' should be one of 'r' or 'w'" );

    croak "only 'r' mode stream can use a sort_spec"
      if $mode ne 'r' && $sort_spec;

    my $self = bless { '_store'     => $store,
                       '_name'      => $name,
                       '_mode'      => $mode,
                       '_sort_spec' => $sort_spec,
                     }, $pkg;
    $self->_init_schema_infos();
    $self->_check_table_schema();

    if ( $mode eq "w" ) {
        $self->{'_sth'} = $self->{'_store'}->_dbh()->prepare_cached( $self->_insert_query );
    } else {
        $self->{'_dlf_reader'} = $self->_select_query()->execute( $store );
    }

    return $self;
}

sub _init_schema_infos {
    my $self = $_[0];

    my $schema = Lire::DlfSchema::load_schema( $self->{'_name'} );
    $self->{'_fields'} = [ map { $_->name } $schema->fields ];
    $self->{'_ts_field'} = $schema->timestamp_field->name;
}

=pod

=head2 name

Returns the schema's name of the DlfStream.

=cut

sub name {
    $_[0]{'_name'};
}

sub _sql_table {
    my ( $self, $prefix, $suffix ) = @_;

    $prefix = ""
      unless defined $prefix;
    $suffix = ""
      unless defined $suffix;

    return sql_quote_name( $prefix . 'dlf_' . $_[0]->{'_name'} . $suffix );
}

=pod

=head2 mode()

Returns the mode in which the DlfStream was opened.

=cut

sub mode {
    $_[0]{'_mode'};
}

=pod

=head2 sort_spec()

Returns the sort specification that is set on the stream.

=cut

sub sort_spec {
    $_[0]{'_sort_spec'};
}

=pod

=head2 close()

This method should be called when the Lire::DlfStream isn't needed
anymore, otherwise the same DlfStream cannot be opened until then.

=cut

sub close {
    my $self = $_[0];

    if ( $self->{'_sth'} ) {
        $self->{'_sth'}->finish;
        delete $self->{'_sth'};
    } elsif ( $self->{'_dlf_reader'} ) {
        $self->{'_dlf_reader'}->dlf_query()->release();
        delete $self->{'_dlf_reader'};
    }
}

sub DESTROY {
    $_[0]->close;
}

=pod

=head2 nrecords()

Returs the number of DLF records in the stream.

=cut

sub nrecords {
    my $self = $_[0];
    $self->_get_stat( "nrecords", "count()");
}

=pod

=head2 start_time()

Returns the timestamp of the oldest DLF record in the stream in
seconds since the epoch.

=cut

sub start_time {
    my $self = $_[0];

    return $self->_get_stat( 'start_time',
                             sprintf( "min(%s)", $self->{'_ts_field'} ) );
}

=pod

=head2 end_time()

Returns the timestamp of the newest DLF record in the stream in
seconds since the epoch.

=cut

sub end_time {
    my $self = $_[0];

    return $self->_get_stat( 'end_time',
                             sprintf( "max(%s)", $self->{'_ts_field'} ) );
}

sub _get_stat {
    my ( $self, $name, $expr ) = @_;

    my $query = new Lire::DlfQuery( $self->{'_name'} );
    $query->add_aggr_field( $name, $expr );
    my $result = $query->execute( $self->{'_store'} );
    my $stat = $result->next_row()->{$name};
    $query->release();

    return $stat;
}

sub _check_table_schema {
    my $self = $_[0];

    my $create_sql = $self->_create_dlf_table_query();

    # SQLite will remove trailing newline
    chomp $create_sql;

    my $sql = "SELECT sql FROM sqlite_master WHERE type = 'table' AND name=?";
    my $sth = $self->{'_store'}->_dbh()->prepare( $sql );
    $sth->execute( "dlf_$self->{'_name'}" );
    my $table_def = $sth->fetchrow_arrayref();
    $sth->finish();

    if ( $table_def && $table_def->[0] ne $create_sql ) {
        $self->_migrate_dlf_schema();
    } elsif ( !defined $table_def ) {
        $self->_create_dlf_schema();
    }
}

sub _migration_insert_query {
    my ( $self, $common_fields ) = @_;

    # New fields will be set to NULL
    # Old fields aren't migrated

    my $field_list = join (", ", map { sql_quote_name( $_ ) } @$common_fields);
    return sprintf( "INSERT INTO %s (%s) SELECT %s FROM %s",
                    $self->_sql_table,
                    $field_list, $field_list,
                    $self->_sql_table( "temp_" ) );
}

sub _migrate_dlf_schema {
    my $self = $_[0];

    if ( ! $self->nrecords ) {
        $self->_create_dlf_schema(1);
        return
    }

    # SQLite doesn't support ALTER TABLE

    # Find the list of fields in common
    my @fields = $self->_find_fields_in_schema();

    # Create temporary backup
    my $sql = sprintf( "CREATE TEMP TABLE %s AS SELECT * FROM %s",
                       $self->_sql_table( "temp_" ),
                       $self->_sql_table );
    $self->{'_store'}->_dbh()->do( $sql  );

    # Recreate the schema
    $self->_create_dlf_schema(1);

    # Migrate the data
    my @common_fields = ();
    foreach my $f ( @fields ) {
        push @common_fields, $f
          if grep { $_ eq $f } @{$self->{'_fields'}};
    }

    $sql = $self->_migration_insert_query(\@common_fields);
    $self->{'_store'}->_dbh()->do( $sql );
    $self->{'_store'}->_dbh()->do( "DROP TABLE " . $self->_sql_table( "temp_" ) );
}

sub _find_fields_in_schema {
    my $self = $_[0];

    # Since DBD::SQLite doesn't support the column_info API
    # we run a query to find the fields present in the schema.
    my $sth = $self->{'_store'}->_dbh()->prepare( "SELECT * FROM " . $self->_sql_table . " LIMIT 1" );
    $sth->execute;

    my @fields = @{$sth->{'NAME'}};
    $sth->finish;

    return @fields;
}

sub _create_dlf_schema {
    my ($self, $remove) = @_;

    $self->{'_store'}->_dbh()->do( "DROP TABLE " . $self->_sql_table() )
      if $remove;
    $self->{'_store'}->_dbh()->do( $self->_create_dlf_table_query() );

    my $idx_sql = sprintf( "CREATE INDEX %s ON %s ( %s )",
                           $self->_sql_table( "", "_" . $self->{'_ts_field'} . "_idx"),
                           $self->_sql_table(),
                           $self->{'_ts_field'} );
    $self->{'_store'}->_dbh()->do( $idx_sql )
}

=pod

=head2 read_dlf()

Returns an hash reference containing the next DLF record in the
stream. It returns undef once the end of the stream is reached.

This method will throw an exception if the DlfStream isn't open in 'r'
mode or if there is an error reading the DLF record.

=cut
sub read_dlf {
    my $self = $_[0];

    croak "read_dlf() can't be called on a DlfStream open in 'w' mode"
      if $self->{'_mode'} ne 'r';

    return $self->{'_dlf_reader'}->next_row();
}

=pod

=head2 read_dlf_aref()

Returns the next DLF record in the stream as an array reference. The
fields are in the order specified by the schema.

This method will throw an exception if the DlfStream isn't open in 'r'
mode or if there is an error reading the DLF record.

=cut
sub read_dlf_aref {
    my $self = $_[0];

    croak "read_dlf_aref() can't be called on a DlfStream open in 'w' mode"
      if $self->{'_mode'} ne 'r';

    return $self->{'_dlf_reader'}->next_row_aref();
}

=pod

=head2 write_dlf( $r )

Writes the fields contained in the hash ref $r to the DLF stream.

This method will throw an exception if there is an error writing the
DLF record or if the stream isn't opened in 'w' mode.

=cut

sub write_dlf {
    my ( $self, $r ) = @_;

    croak "write_dlf() can't be called on a DlfStream open in 'r' mode"
      if $self->{'_mode'} ne 'w';

    croak "missing DLF hash reference"
      unless defined $r;

    croak "DLF record isn't an hash reference: $r"
      unless ref $r eq 'HASH';

    my @values = ();
    foreach my $f ( @{$self->{'_fields'}} ) {
        my $v = $r->{$f};
        #   replace control chars with ?
        $v =~ tr/\000-\037/?/
          if defined $v;
        push @values, ensure_utf8( $v );
    }
    $self->{'_sth'}->execute( @values);
}

=pod

=head2 clean( [ $time ] )

This method will remove all DLF records older than $time. It $time is
omitted, all Dlf records will be removed.

=cut

sub clean {
    my ( $self, $time ) = @_;

    croak "clean() can't be called on a DlfStream open in 'r' mode"
      if $self->{'_mode'} ne 'w';

    my $table = $self->_sql_table();
    if ( defined $time ) {
        my $ts_field = sql_quote_name( $self->{'_ts_field'} );
        $self->{'_store'}->_dbh()->do( qq{DELETE FROM $table WHERE $ts_field < ?}, {}, $time );
    } else {
        $self->{'_store'}->_dbh()->do( "DELETE FROM $table" );
    }

    return;
}

sub _select_query {
    my $self = $_[0];

    my $query = new Lire::DlfQuery( $self->{'_name'} );
    foreach my $f ( @{$self->{'_fields'}} ) {
        $query->add_field( $f );
    }

    $query->set_sort_spec( $self->{'_sort_spec'} )
      if $self->{'_sort_spec'};

    return $query;
}

sub _insert_query {
    my $self = $_[0];

    return "INSERT INTO " . $self->_sql_table . " ("
      . join( ", ", map { sql_quote_name( $_ ) } @{$self->{'_fields'}} ) . ") VALUES ("
      . join( ",", ("?") x scalar @{$self->{'_fields'}}  ) . ")";
}

sub _create_dlf_table_query {
    my $self = $_[0];

    my $schema = Lire::DlfSchema::load_schema( $self->{'_name'} );
    my $sql = "CREATE TABLE " . $self->_sql_table() . " "
      . $schema->sql_fields_def();

    return $sql;
}

1;
