package Lire::Param;

use strict;

use Carp;

use Lire::DataTypes qw/ :xml :special /;
use Lire::I18N qw/ dgettext_para /;

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;


    my %params = @_;
    my $name	= $params{'name'};
    croak "invalid name for a parameter: $name"
      unless check_xml_name( $name );

    my $type	= $params{'type'};
    croak "invalid type for a parameter: $type"
      unless check_type( $type );

    my $self	= bless {
			 'name'	=> $name,
			 'type'	=> $type,
                         'i18n_domain' => $params{'i18n_domain'} || 'lire',
			 'check'  => $Lire::DataTypes::VALIDATORS{$type},
			 'description' => $params{'description'},
			}, $class;

    if (exists $params{'default'}) {
	$self->default( $params{'default'} );
    }

    if (exists $params{'value'}) {
	$self->default( $params{'value'} );
    }

    return $self;
}

sub name {
    return $_[0]->{'name'};
}

sub value {
    my ($self, $value ) = @_;

    if ( @_ == 2 ) {
	if ( defined $value ) {
	    croak "invalid value for a $self->{'type'}: $value"
	      unless $self->{'check'}->( $value );

	    $self->{'value'} = $value;
	} else {
	    $self->{'value'} = undef;
	}
    }

    # Check for default
    if ( defined $self->{'value'}) {
	return $self->{'value'};
    } else {
	return $self->default();
    }
}

sub type {
    return $_[0]->{'type'};
}

sub default {
    my ( $self, $default ) = @_;

    if ( defined $default ) {
	croak "invalid default value for a $self->{'type'}: $default"
	  unless $self->{'check'}->( $default );

	$self->{'default'} = $default;
    }

    return $self->{'default'};
}

sub description {
    my ( $self, $desc ) = @_;

    if ( @_ == 2 ) {
	$self->{'description'} = $desc;
    }

    return dgettext_para( $self->{'i18n_domain'},
                          $self->{'description'} );
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Param -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Param.pm,v 1.8 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
