package Lire::ReportParser::ReportBuilder;

use strict;

use base qw/  Lire::ReportParser /;

use Lire::Report;
use Lire::Report::Image;
use Lire::Report::Note;
use Lire::Report::Section;
use Lire::Report::Subreport;

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my $self = $class->SUPER::new( @_ );
    my %args = @_;
    $self->{'only_head'} = $args{'only_head'};

    return $self;
}

sub parse_start {
    my ( $self ) = @_;

    $self->{'curr_section'} = undef;
    $self->{'curr_subreport'} = undef;
    $self->{'curr_image'} = undef;
    $self->{'curr_desc'}  = undef;
    $self->{'curr_note'}  = undef;
}

sub parse_end {
    my ( $self ) = @_;

    return $self->{'report'};
}

sub report_start {
    my ( $self, %attr ) = @_;

    $self->SUPER::report_start( %attr );

    # 0, 0 will be replaced by proper values once the timespan
    # element is encountered
    $self->{'report'} = new Lire::Report( $attr{'superservice'}, 0, 0 );

    $self->{'report'}->generator( __PACKAGE__ . "(3pm)" );
}

sub handle_title {
    my ( $self, $title ) = @_;

    if ( $self->{'curr_image'} ) {
        $self->{'curr_image'}->title( $title );
    } elsif ( $self->{'curr_subreport'} ) {
        $self->{'curr_subreport'}->title( $title );
    } elsif ( $self->{'curr_section'} ) {
        $self->{'curr_section'}->title( $title );
    } else {
        $self->{'report'}->title( $title );
    }
}

sub handle_date {
    my ( $self, $date, $date_epoch ) = @_;

    if ( $self->{'curr_subreport'} ) {
        $self->{'curr_subreport'}{'date'} = $date_epoch;
    } else {
        $self->{'report'}{'date'} = $date_epoch;
    }
}

sub handle_timespan {
    my ( $self, $timespan, $start, $end ) = @_;

    if ( $self->{'curr_subreport'} ) {
        $self->{'curr_subreport'}{'timespan_start'} = $start;
        $self->{'curr_subreport'}{'timespan_end'} = $end;
    } else {
        $self->{'report'}{'timespan_start'} = $start;
        $self->{'report'}{'timespan_end'} = $end;
    }
}

sub handle_hostname {
    my ( $self, $hostname ) = @_;

    if ( $self->{'curr_subreport'} ) {
        $self->{'curr_subreport'}->hostname( $hostname );
    } else {
        $self->{'report'}->hostname( $hostname );
    }
}

sub description_start {
    my ( $self, %attr )= @_;

    $self->{'curr_desc'} = "";
    $self->{'curr_desc_context'} = $self->depth;

}

sub description_end {
    my ( $self )= @_;

    my $o;
    if ( $self->{'curr_subreport'} ) {
        $o = $self->{'curr_subreport'};
    } elsif ( $self->{'curr_section'} ) {
        $o = $self->{'curr_section'};
    } else {
        $o = $self->{'report'};
    }
    $o->description( $self->{'curr_desc'} );
    $self->{'curr_desc'} = undef;
    $self->{'curr_desc_context'} = undef;
}

sub note_start {
    my ( $self, %attr )= @_;

    $self->{'curr_note'} = "";
    $self->{'curr_note_xref'} = $attr{'xref'};
    $self->{'curr_note_context'} = $self->depth;

}

sub note_end {
    my ( $self )= @_;

    my $o;
    if ( $self->{'curr_subreport'} ) {
        $o = $self->{'curr_subreport'};
    } elsif ( $self->{'curr_section'} ) {
        $o = $self->{'curr_section'};
    } else {
        $o = $self->{'report'};
    }
    my $note = new Lire::Report::Note( $self->{'curr_note_xref'} );
    $note->content( $self->{'curr_note'} );
    $o->add_note( $note );
    $self->{'curr_note'} = undef;
    delete $self->{'curr_note_xref'};
    delete $self->{'curr_note_context'};
}

sub element_start {
    my ( $self, $name, %attr ) = @_;

    # In description collect everything
    if ( defined $self->{'curr_desc'} ) {
        $self->{'curr_desc'} .= $self->original_string;
    } elsif ( defined $self->{'curr_note'} ) {
        $self->{'curr_note'} .= $self->original_string;
    }

    return 0; # Continue normal processing
}

sub element_end {
    my ( $self, $name ) = @_;

    # In description collect everything
    if ( defined $self->{'curr_desc'} && $self->depth > $self->{'curr_desc_context'} )
    {
        $self->{'curr_desc'} .= $self->original_string;
    } elsif ( defined $self->{'curr_note'} && $self->depth > $self->{'curr_note_context'} )
    {
        $self->{'curr_note'} .= $self->original_string;
    }


    return 0; # Continue normal processing
}

sub pcdata {
    my ( $self, $str ) = @_;

    # In description collect everything
    if ( defined $self->{'curr_desc'} ) {
        $self->{'curr_desc'} .= $str
    } elsif ( defined $self->{'curr_note'} ) {
        $self->{'curr_desc'} .= $str
    }
    return 0; # Continue normal processing
}

sub ignorable_ws {
    my ( $self, $str ) = @_;

    # In description collect everything
    if ( defined $self->{'curr_desc'} ) {
        $self->{'curr_desc'} .= $str
    } elsif ( defined $self->{'curr_note'} ) {
        $self->{'curr_note'} .= $str
    }

    return 0; # Continue normal processing
}

sub section_start {
    my ( $self, %attr ) = @_;

    $self->SUPER::section_start( %attr );

    if ( $self->{'only_head'} ) {
        $self->expat->finish;
        return;
    }

    $self->{'curr_section'} = new Lire::Report::Section();
    $self->{'report'}->add_section ( $self->{'curr_section'} );
}

sub section_end {
    my ( $self ) = @_;
    $self->SUPER::section_end();

    $self->{'curr_section'} = undef;
}

sub subreport_start {
    my ( $self,  %attr ) = @_;
    $self->SUPER::subreport_start( %attr );

    my $subreport = new Lire::Report::Subreport( $self->{'report'}, $attr{'type'});

    $subreport->{'superservice'} = $attr{'superservice'}
      if exists $attr{'superservice'};

    $self->{'curr_subreport'} = $subreport;
    $self->{'curr_section'}->add_subreport( $self->{'curr_subreport'} );

    $self->{'group_stack'} = [];
    $self->{'entry_stack'} = [];
    push @{$self->{'group_stack'}}, $subreport;
}

sub subreport_end {
    my ( $self ) = @_;
    $self->SUPER::subreport_end();

    $self->{'curr_subreport'} = undef;
    pop @{$self->{'group_stack'}};
    die "subreport_end(): assertion failed: group_stack should be empty"
      if @{$self->{'group_stack'}};
    die "subreport_end(): assertion failed: entry_stack should be empty"
      if @{$self->{'entry_stack'}};
}

sub missing_subreport_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::missing_subreport_start( %attr );

    my $subreport = 
      new_missing Lire::Report::Subreport( $self->{'report'}, $attr{'type'},
                                           $attr{'reason'} );
    $subreport->{'superservice'} = $attr{'superservice'}
      if exists $attr{'superservice'};

    $self->{'curr_section'}->add_subreport( $subreport );
}

sub table_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::table_start( %attr );

    $self->{'curr_subreport'}->charttype( $attr{'charttype'} );
    $self->{'curr_subreport'}->show( $attr{'show'} );

}

sub group_summary_start {
    my ( $self, %attr ) = @_;

    $self->SUPER::group_summary_start( %attr );
    $self->curr_group->nrecords( $attr{'nrecords'} );
    $self->curr_group->missing_cases( $attr{'missing-cases'} );
    $self->curr_group->row_idx( $attr{'row-idx'} )
      if defined $attr{'row-idx'};

    return;
}

sub group_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::group_start( %attr );

    my $group = $self->curr_entry->create_group;
    $group->show( $attr{'show'} )
      if defined $attr{'show'};

    push @{$self->{'group_stack'}}, $group;
}

sub group_end {
    my ( $self ) = @_;
    $self->SUPER::group_end();

    pop @{$self->{'group_stack'}};
}

sub curr_group {
    $_[0]{'group_stack'}[$#{$_[0]{'group_stack'}}];
}

sub table_info_end {
    my ( $self ) = @_;

    $self->SUPER::table_info_end();
    $self->{'curr_subreport'}->table_info( $self->current_table_info );
}

sub entry_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::entry_start( %attr );

    # Sets the TableInfo object on first entry
    # in case we are running in compatibility mode
    # and the TableInfo is created from the report specification 
    # instead of being included in the XML report.
    $self->{'curr_subreport'}->table_info( $self->current_table_info() )
      unless ( $self->{'curr_subreport'}->table_info() );

    my $entry = $self->curr_group()->create_entry();
    $entry->row_idx( $attr{'row-idx'} )
      if defined $attr{'row-idx'};

    push @{$self->{'entry_stack'}}, $entry;

    return;
}

sub entry_end {
    my ( $self ) = @_;
    $self->SUPER::entry_end();

    pop @{$self->{'entry_stack'}};
}

sub curr_entry {
    $_[0]{'entry_stack'}[$#{$_[0]{'entry_stack'}}];
}

sub handle_name {
    my ( $self, $name ) = @_;

    $self->curr_entry->add_name( $name->{'content'}, $name->{'value'},
                                 $name->{'range'} );
}

sub handle_value {
    my ( $self, $value ) = @_;

    $self->curr_entry->add_value( 'content' => $value->{'content'},
                                  'value' => $value->{'value'},
                                  'total' => $value->{'total'},
                                  'n' => $value->{'n'},
                                  'missing_cases' => $value->{'missing-cases'},
                                );
}

sub handle_summary_value {
    my ( $self, $value ) = @_;

    $self->curr_group->set_summary_value( $value->{'col_info'}->name,
                                          'content' => $value->{'content'},
                                          'value' => $value->{'value'},
                                          'total' =>$value->{'total'},
                                          'n' => $value->{'n'},
                                          'missing_cases' => $value->{'missing-cases'},
                                         );
}

sub image_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::image_start( %attr );

    $self->{'curr_image'} = new Lire::Report::Image;
}

sub image_end {
    my ( $self ) = @_;
    $self->SUPER::image_end();

    $self->{'curr_subreport'}->image( $self->{'curr_image'} );
    $self->{'curr_image'} = undef;
}

sub file_start {
    my ( $self, %attr ) = @_;
    $self->SUPER::image_start( %attr );

    $self->{'curr_image'}->format( $attr{'format'} );
}

sub handle_file {
    my ( $self, $file ) = @_;
    $self->{'curr_image'}->file( $file );
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportParser::ReportBuilder - Creates Lire::Report objects from XML files

=head1 SYNOPSIS

    use Lire::ReportParser::ReportBuilder;
    my $parser = new Lire::ReportParser::ReportBuilder;
    my $report = $parser->parsefile( "report.xml" );

=head1 DESCRIPTION

This is a subclass of Lire::ReportParser that creates an object
representation of a Lire report contained in a XML file.

=head1 SEE ALSO

Lire::Report(3pm)

=head1 VERSION

$Id: ReportBuilder.pm,v 1.24 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
