package Lire::UI::FileWidget;

use strict;

use base qw/ Curses::UI::Container Lire::UI::Widget /;

use Curses::UI::Common;

use Carp;
use vars qw/@CARP_NOT/;
use Lire::Utils qw/ check_object_param /;

use Locale::TextDomain 'lire';

@CARP_NOT = qw/Curses::UI::Container/;

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase(\%userargs);

    check_object_param( $userargs{'value'}, 'value',
                        'Lire::Config::Scalar' );

    my %args = (
                '-height' => 2,
                %userargs,
                '-releasefocus' => 1,
                '-border' => 0,
               );
    my $label = __( '[Browse]' );
    my $btn_width = length( $label );
    my $self =  $class->Curses::UI::Container::new( %args );
    my $file_field = $self->add( 'file_field', 'TextEntry',
                                 '-sbborder' => 1,
                                 '-text' => $userargs{'value'}->get(),
                                 '-onchange' => \&_on_change_cb );
    my $browse_btn = $self->add( 'browse_btn', 'Buttonbox',
                                 '-y' => 1,
                                 '-width' => $btn_width,
                                 '-buttons' =>
                                 [ { '-label' => $label,
                                     '-onpress' => \&_browse_cb } ] );
    $self->add( 'warn_label', 'Label',
                '-x' => $btn_width + 2, '-y' => 1,
                '-width' => length( $self->invalid_label_str() ),
                '-text' => '', '-bold' => '1' );
    # Buttonbox as last focus in a releasefocus container
    # doesn't work too well with Curses::UI
    $self->set_focusorder( 'browse_btn', 'file_field' );
    $self->_refresh_warn_label();

    return $self;
}

sub refresh_view {
    my $self = $_[0];

    my $file_widget = $self->getobj( 'file_field' );
    $file_widget->text( $self->{'value'}->get() );
    $self->_refresh_warn_label();

    return;
}

sub _refresh_warn_label {
    my $self = $_[0];

    my $warn_label = $self->getobj( 'warn_label' );
    $warn_label->text( $self->{'value'}->spec()->is_valid( $self->{'value'}->get() ) ? ''
                       : $self->invalid_label_str() );

    return;
}

# template methods
sub invalid_label_str {
    return __( 'Invalid filename' );
}

sub browser_title_str {
    return __( 'Select a File');
}

sub browse_dialog {
    my $self = $_[0];

    my $file_field = $self->getobj( 'file_field' );

    return $self->root()->filebrowser( '-title' => $self->browser_title_str(),
                                       '-file' => $file_field->get(),
                                       '-editfilename' => 1,
                                       '-show_hidden' => 1 );
}

# callback functions
sub _on_change_cb {
    my $file_widget = $_[0];

    my $self = $file_widget->parent();
    $self->{'value'}->set( $file_widget->{'-text'} );
    $self->_refresh_warn_label();

    return;
}

sub _browse_cb {
    my $button = $_[0];

    my $self = $button->parent();
    my $file_field = $self->getobj( 'file_field' );

    my $new_file = $self->browse_dialog();

    if ( defined $new_file ) {
        $file_field->text( $new_file );
        _on_change_cb( $file_field );
    }

    return;
}

1;
