package Lire::UI::PluginWidget;

use strict;

use base qw/ Curses::UI::Container Lire::UI::Widget /;

use Curses::UI::Common;
use Locale::TextDomain 'lire';

use Lire::Utils qw/ item_index check_param check_object_param /;
use Lire::UI::Utils qw/ button_box_width /;

use Carp;
use vars qw/@CARP_NOT/;

@CARP_NOT = qw/Curses::UI::Container/;

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase(\%userargs);

    check_object_param( $userargs{'value'}, 'value', 'Lire::Config::Plugin' );

    my %args = ( %userargs, '-height' => 1, '-releasefocus' => 1 );
    my $self = $class->Curses::UI::Container::new( %args );
    $self->_new_add_list();
    $self->_update_button();

    return $self;
}

sub _update_button {
    my $self = $_[0];

    my $has_props = $self->{'value'}->has_properties();
    my $need = '';
    if ( defined $self->getobj( 'button' ) ) {
        $need = 'delete' unless $has_props;
    } else {
        $need = 'draw' if $has_props;
    }

    if ( $need eq 'draw' ) {
        my $list = $self->getobj( 'list' );
        $self->add( 'button', 'Buttonbox',
                    '-buttons' => [ { '-label' => '<...>',
                                      '-onpress' => sub {
                                          $self->_properties_cb_helper()
                                      } } ],
                    '-width' => 5,
                    '-x' => $list->{'-width'} );
        $self->intellidraw();
    } elsif ( $need eq 'delete' ) {
        $self->delete( 'button' );
        $self->intellidraw();
    }
}

sub _new_add_list {
    my $self = $_[0];

    my @options = map { $_->name() } $self->{'value'}->spec()->options();
    my $selected;
    my $focusable;

    if ( @options ) {
        my $plugin = $self->{'value'}->get_plugin();
        $selected = ( defined $plugin
                      ? item_index( \@options, $plugin )
                      : undef );
        $focusable = 1;
    } else {
        @options = ( __( '-- empty list --' ) );
        $selected = 0;
        $focusable = 0;
    }

    my $width = ( defined $self->{'-width'}
                  ? $self->{'-width'} - 5
                  : undef );
    my $popup_menu = $self->add( 'list', 'Popupmenu',
                                 '-onchange' => sub {
                                     $self->_on_change_cb_helper()
                                 },
                                 '-selected' => $selected,
                                 '-values' => \@options,
                                 '-focusable' => $focusable,
                                 '-width' => $width );

    return;
}

sub _on_change_cb_helper {
    my $self = $_[0];

    my $list = $self->getobj( 'list' );
    $self->{'value'}->set_plugin( $list->get() );
    $self->_update_button();

    return;
}

sub _properties_cb_helper {
    my $self = $_[0];

    my $plugin = $self->{'value'}->get_plugin();
    return unless ( defined $plugin
                    && $self->{'value'}->has_properties() );

    my $properties = $self->{'value'}->get_properties();

    my $win = $self->root();
    my $name = "properties_dialog_$self";
    my $dlg = $win->add( $name, 'Curses::UI::Window',
                         '-title' => __x( 'Properties for {plugin}',
                                          'plugin' => $plugin ),
                         '-border' => 1, '-pad' => 2, '-ipad' => 1 );
    my $widget = $dlg->add( 'widget', 'Lire::UI::Widget',
                            'value' => $properties->clone(),
                            '-releasefocus' => 1,
                            '-width' => $dlg->canvaswidth() - 1,
                            '-height' => $dlg->canvasheight() - 2 );

    my $buttons = [ {'-label' => __( '< Cancel >' ),
                     '-onpress' => sub { $dlg->loose_focus() } },
                    {'-label' => __( '< OK >' ),
                     '-onpress' => sub {$self->_properties_ok_cb_helper()}} ];
    my $bb_width = button_box_width( $buttons );
    $dlg->add( 'buttons', 'Curses::UI::Buttonbox',
               '-buttons' => $buttons,
               '-x' => $dlg->canvaswidth() - $bb_width - 10,
               '-y' => $dlg->canvasheight() - 1,
               '-width' => $bb_width, '-height' => 1 );
    $dlg->modalfocus();
    $win->delete( $name );

    $self->root()->draw( 1 );
    $self->parent()->draw( 1 );

    return;
}

sub _properties_ok_cb_helper {
    my $self = $_[0];

    my $window = $self->root();
    my $dlg = $window->getobj( "properties_dialog_$self" );
    my $widget = $dlg->getobj( 'widget' );

    my $properties = $self->{'value'}->get_properties();

    my @comps = map { $_->name() } $properties->spec()->components();
    foreach my $comp ( @comps ) {
        $properties->set( $widget->{'value'}->get( $comp ) );
    }
    $dlg->loose_focus();

    return;
}

1;
