package tests::CountTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Count;
use Lire::ReportSpec;
use Lire::Report;
use Lire::Report::Subreport;
use Lire::Report::TableInfo;
use tests::MockAggregator;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'test-count' );

    $self->_set_up_flat_subreport();
    $self->_set_up_nested_subreport();

    $self->{'cfg'}{'lr_scale_numbers'} = 1;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    return  $self;
}

sub _set_up_flat_subreport {
    my $self = $_[0];

    my $report = new Lire::Report( 'test', 0, 1 );
    my $mock = new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'count_op'} = new Lire::Count( 'report_spec' => $self->{'spec'},
                                           'parent' => $mock,
                                           'name' => 'countTest' );

    $self->{'flat_subreport'} = new Lire::Report::Subreport( $report, 'test' );
    my $info = new Lire::Report::TableInfo();
    $self->{'flat_subreport'}->table_info( $info );
    $info->create_column_info( 'countTest', 'numerical', 'int' );

    return;
}

sub _set_up_nested_subreport {
    my $self = $_[0];

    my $report = new Lire::Report( 'test', 0, 1 );
    my $mock = new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'ratio_op'} = new Lire::Count( 'report_spec' => $self->{'spec'},
                                           'parent' => $mock,
                                           'ratio' => 'table',
                                           'name' => 'ratioTest' );

    $self->{'nested_subreport'} =
      new Lire::Report::Subreport( $report, 'test' );
    my $info = new Lire::Report::TableInfo();
    $self->{'nested_subreport'}->table_info( $info );
    my $group1_info = $info->create_group_info( 'group1' );
    $group1_info->create_column_info( 'ratioTest', 'numerical', 'int' );
    return;
}

sub test_sql_aggr_expr {
    my $self = $_[0];

    $self->assert_equals( 'count(*)', $self->{'count_op'}->sql_aggr_expr());

    $self->{'count_op'}{'fields'} = [ 'connection-id', 'user' ];
    $self->assert_equals( 'lr_count_uniq("connection-id",user)',
                          $self->{'count_op'}->sql_aggr_expr());
}

sub test_sql_required_fields {
    my $self = $_[0];

    $self->assert_deep_equals( [],
                               $self->{'count_op'}->sql_required_fields() );

    $self->{'count_op'}{'fields'} = [ 'connection_id', 'user' ];
    $self->assert_deep_equals( [ 'connection_id', 'user' ],
                               $self->{'count_op'}->sql_required_fields() );
}

sub test_create_value {
    my $self = $_[0];

    my $table = $self->{'flat_subreport'};

    my $row = { 'countTest' => 1000,
                '_lr_countTest_mc' => 2 };
    my $value = $self->{'count_op'}->create_value( $table, $row );
    $self->assert_deep_equals( { 'content' => '1k',
                                 'value' => 1000,
                                 'missing_cases' => 2 }, $value );
}

sub test_create_value_table_ratio_bootstrap {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    my $table_value =
      $self->{'ratio_op'}->create_value( $table, { 'ratioTest' => 1000,
                                                    '_lr_ratioTest_mc' => 0 });
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 1000,
                                 'missing_cases' => 0 },
                               $table_value );
}

sub test_create_value_table_ratio {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    $table->set_summary_value( 'ratioTest', 'value' => 1000 );
    my $group1 = $table->create_entry()->create_group();

    $self->{'ratio_op'}{'ratio'} = 'table';
    my $table_value =
      $self->{'ratio_op'}->create_value( $table, { 'ratioTest' => 1000,
                                                    '_lr_ratioTest_mc' => 0 });
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 1000,
                                 'missing_cases' => 0 },
                               $table_value );

    my $group1_value =
      $self->{'ratio_op'}->create_value( $group1, { 'ratioTest' =>  100,
                                                     '_lr_ratioTest_mc' => 0});
    $self->assert_deep_equals( { 'content' => '10.0',
                                 'value' => 100,
                                 'missing_cases' => 0 },
                               $group1_value );
}

sub test_create_value_group_ratio {
    my $self = $_[0];

    my $table = $self->{'nested_subreport'};
    $table->set_summary_value( 'ratioTest', 'value' => 500 );
    my $group1 = $table->create_entry()->create_group();

    $self->{'ratio_op'}{'ratio'} = 'group';
    my $table_value =
      $self->{'ratio_op'}->create_value( $table,
                                         { 'ratioTest' =>  500,
                                           '_lr_ratioTest_mc' => 0 } );
    $self->assert_deep_equals( { 'content' => '100.0',
                                 'value' => 500,
                                 'missing_cases' => 0 },
                               $table_value );

    $group1->set_summary_value( 'ratioTest', 'value' => 100 );
    my $group1_value =
      $self->{'ratio_op'}->create_value( $group1,
                                         { 'ratioTest' =>  1,
                                           '_lr_ratioTest_mc' => 0 } );

    $self->assert_deep_equals( { 'content' => '1.0',
                                 'value' => 1,
                                 'missing_cases' => 0 },
                               $group1_value );
}

1;
