package tests::DlfConverterManagerTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::DlfConverterManager;
use Lire::Utils qw/tempdir create_file /;
use File::Path qw/mkpath/;
use File::Basename;
use Cwd qw/realpath/;
use Carp;

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->{'_converters'} = [];

    foreach my $name ( qw/dlf1 dlf2 dlf3 dlf4/ ) {
        push @{$self->{'_converters'}},
          new tests::DlfConverterManagerTest::DlfConverter( $name );
    }

    $self->{'tmpdir'} = tempdir( "DlfConverterManager_XXXXXX", 'CLEANUP' => 1 );
    mkpath( [ $self->{'tmpdir'} . "/dlf_converters1",
              $self->{'tmpdir'} . "/dlf_converters2",
            ], 0, 0755 ) == 2
              or croak "mkpath failed: $!\n";

    $self->{'cfg'}{'lr_schemas_path'} = [ realpath(dirname( __FILE__ )) . "/schemas" ];
    $self->{'cfg'}{'lr_converters_init_path'} =
      [ $self->{'tmpdir'} . "/dlf_converters1",
        $self->{'tmpdir'} . "/dlf_converters2" ];

    $self->{'mgr'} = bless { '_converters' => {} }, "Lire::DlfConverterManager";
}

sub test_instance {
    my $self = $_[0];

    my $mgr = Lire::DlfConverterManager->instance;
    $self->assert_not_null( $mgr, "instance() returned undef" );
    $self->assert_equals( 'Lire::DlfConverterManager', ref $mgr );
}

sub test_get_converters {
    my $self = $_[0];

    my $mgr = Lire::DlfConverterManager->instance;
    foreach my $c ( @{$self->{'_converters'}} ) {
        $mgr->register_converter( $c );
    }

    my @names = sort map { $_->name } @{$self->{'_converters'}};
    my @mgr_names = sort $mgr->converter_names;
    $self->assert_deep_equals( \@names, \@mgr_names );

    my @conv  = sort { $a->name cmp $b->name } @{$self->{'_converters'}};
    my @mgr_conv  = sort { $a->name cmp $b->name } $mgr->converters;
    $self->assert_deep_equals( \@conv, \@mgr_conv );

    foreach my $c ( @{$self->{'_converters'}} ) {
        $mgr->unregister_converter( $c->name );
    }
}

sub test_register_converter {
    my $self = $_[0];

    my $mgr = Lire::DlfConverterManager->instance;

    $mgr->register_converter( $self->{'_converters'}[0] );

    my $conv = $mgr->get_converter( $self->{'_converters'}[0]->name );
    $self->assert_not_null( $conv, "DLF converter wasn't registered" );
    $self->assert_equals( $self->{'_converters'}[0], $conv );

;
    $self->assert_died( sub { $mgr->register_converter( $conv ) },
                        qr/there is already a DLF converter registered/ );

    $mgr->unregister_converter( $conv->name );
    $self->assert_null( $mgr->get_converter( $conv->name ),
                        "unregister_converter failed" );
}

sub test_unregister_converter {
    my $self = $_[0];

    my $mgr = Lire::DlfConverterManager->instance;

    $self->assert_died( sub { $mgr->unregister_converter( "no_such_converter" ) },
                        qr/there is no DLF converter 'no_such_converter' registered/ );
}

sub test_converters_file {
    my $self = $_[0];

    my @files = ( $self->{'tmpdir'} . "/dlf_converters1/test",
                  $self->{'tmpdir'} . "/dlf_converters1/test2",
                  $self->{'tmpdir'} . "/dlf_converters2/test3",
                );


    foreach my $f ( @files ) {
        create_file( $f );
    }

    $self->assert_deep_equals( \@files, [ Lire::DlfConverterManager->instance->_converters_init_files() ] );
}

sub test_load_dlf_adapters {
    my $self = $_[0];

    create_file( $self->{'tmpdir'} . "/dlf_converters1/test1" );
    chmod 0000, $self->{'tmpdir'} . "/dlf_converters1/test1"
      or croak "chmod failed: $!\n";
    create_file( $self->{'tmpdir'} . "/dlf_converters1/test2", <<'EOF' );
use strict;

$x = 1;
EOF

    create_file( $self->{'tmpdir'} . "/dlf_converters1/test3", <<'EOF' );
use tests::DlfConverterManagerTest;

return ( new tests::DlfConverterManagerTest::DlfConverter( 'test1' ),
         new tests::DlfConverterManagerTest::DlfConverter( 'test2' ) );
EOF
    create_file( $self->{'tmpdir'} . "/dlf_converters2/test4", <<'EOF' );
return bless {}, "Test";
EOF

    my @msg = ();
    local $SIG{'__WARN__'} = sub { push @msg, join "", @_ };
    $self->{'mgr'}->_load_dlf_adapters();

    # Reset the permission, so it can be deleted with rmtree()
    chmod 0644, $self->{'tmpdir'} . "/dlf_converters1/test1"
      or croak "chmod failed: $!\n";

    $self->assert( @msg == 3, "expected 3 warnings" );
    $self->assert_matches( qr/error reading DLF converter/, $msg[0] );
    $self->assert_matches( qr/error while running initializer/, $msg[1] );
    $self->assert_matches( qr/didn't return a Lire::DlfConverter/, $msg[2] );

    $self->assert_deep_equals( [ "test1", "test2" ],
                               [ sort $self->{'mgr'}->converter_names()] );
}

sub test_parse_map_file {
    my $self = $_[0];

    my $address_file = $self->{'tmpdir'} . "/address.cf";
    create_file ( $address_file, <<EOF );
# A comment
   
service superservice # with comment
service1 schema1
line with an error
# More comment
EOF

    my @msg = ();
    local $SIG{'__WARN__'} = sub { push @msg, join "", @_ };
    my $services = Lire::DlfConverterManager->instance->_parse_old_map_file( $address_file );
    $self->annotate( join "\n", @msg );
    $self->assert( @msg == 1, "expected one warning, got " . scalar @msg );
    $self->assert_matches( qr/can't parse line 5/, $msg[0] );

    $self->assert_deep_equals( { 'service' => 'superservice',
                                 'service1' => 'schema1',
                               }, $services );
}

sub test_create_old_dlf_adapters {
    my $self = $_[0];

    my $address_file = $self->{'tmpdir'} . "/address.cfg";
    $self->{'cfg'}{'lr_old_convertors_dir'} = $self->{'tmpdir'};
    $self->{'cfg'}{'lr_old_address_file'} = $address_file;

    create_file( "$self->{'tmpdir'}/service12dlf" );
    create_file( "$self->{'tmpdir'}/service22dlf" );
    create_file( "$self->{'tmpdir'}/service42dlf" );
    chmod( 0755, map { "$self->{'tmpdir'}/" . $_ . "2dlf" } qw/service1 service2/) == 2
      or croak "chmod failed";
    create_file( $address_file, <<EOF );
service1 test       # Good
service2 unknown    # Bad superservice
service3 test       # service32dlf doesn't exist
service4 test       # not executable
EOF

    my @msg = ();
    local $SIG{'__WARN__'} = sub { my $m = join "", @_;
                                 $self->annotate( $m);
                                 push @msg, $m; };
    $self->{'mgr'}->_create_old_dlf_adapters();

    @msg = sort @msg;
    my @warnings = ( qr/can't find executable service32dlf/,
                     qr/can't find executable service42dlf/,
                     qr/invalid superservice/ 
                   );
    $self->assert( @msg == @warnings, "expected " . scalar @warnings .
                   " warnings, got " . scalar @msg );
    for ( my $i=0; $i < @warnings; $i++ ) {
        $self->assert_matches( $warnings[$i], $msg[$i] );
    }

    my @converters = ( "service1" );
    $self->assert_deep_equals( \@converters, 
                               [sort $self->{'mgr'}->converter_names()] );
}

package tests::DlfConverterManagerTest::DlfConverter;

use base qw/Lire::DlfConverter/;

sub new {
    bless { 'name' => $_[1] }, $_[0];
}

sub name { $_[0]{'name'} };

1;
