package tests::DlfSchemaTest;

use strict;

use base qw/Test::Unit::TestSuite/;

sub name {
    return "Configuration Specification Tests";
}

sub include_tests {
    return qw/tests::DlfSchemaTestBase tests::DlfSchemaTestI18N /;
}

package tests::DlfSchemaTestBase;

use base qw/ Lire::Test::TestCase /;

use Lire::DlfSchema;
use File::Temp qw/:mktemp tempdir /;
use Symbol;
use Carp;

my $test_schema = <<EOF;
<lire:dlf-schema superservice="test" timestamp="time_start"
 xmlns:lire="http://www.logreport.org/LDSML/">
<lire:field name="time_start" type="timestamp"/>
<lire:field name="field-1" type="string"/>
<lire:field name="field_2" type="string"/>
<lire:field name="int_3" type="int"/>
<lire:field name="url_4" type="url"/>
</lire:dlf-schema>
EOF

my $test_sql_fields_def = <<EOS;
(
    time_start TIMESTAMP,
    "field-1" VARCHAR(512),
    field_2 VARCHAR(512),
    int_3 NUMBER(10,0),
    url_4 VARCHAR(512)
)
EOS

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->{'tmpdir'} = tempdir( __PACKAGE__ . "XXXXXX",
                               'TMPDIR' => 1, CLEANUP => 1 );

    $self->{'schemapath'} = $self->{'tmpdir'} . "/test.xml";
    mkdir $self->{'tmpdir'} . "/schemas", 0755
      or croak "failed to make directory: $self->{'tmpdir'}/schemas";
    $self->create_schema( $test_schema, "test", $self->{'schemapath'} );
    $self->create_schema( $test_schema, "test-exttest",
                          "$self->{'tmpdir'}/schemas/test-exttest.xml" );

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'tmpdir'},
                                          $self->{'tmpdir'} . "/schemas",
                                        ];
}

sub create_schema {
    my ( $self, $template, $name, $path ) = @_;

    $template =~ s/superservice="test"/superservice="$name"/;
    my $fh = gensym;
    open $fh, "> $path"
      or die "can't create $name schema: $path: $!";
    print $fh $template;
    close $fh;
}

sub test_schema_load {
    my $self = $_[0];

    # To make sure that the change to the schema file
    # is noticed
    my $or_schema = Lire::DlfSchema::load_schema( "test" );
    $self->assert_not_null( $or_schema, "load_schema() returned undef" );

    # Should return cached copy
    my $schema = Lire::DlfSchema::load_schema( "test" );
    $self->assert_equals( $or_schema, $schema );

    # Modify mtime to force a reload
    my $now = time + 5;
    utime $now, $now, $self->{'schemapath'};

    $schema = Lire::DlfSchema::load_schema( "test" );
    $self->assert_not_equals( $self->{'schema'}, $schema );
}

sub test_has_schema {
    my $self = $_[0];

    $self->assert( Lire::DlfSchema->has_schema( "test"),
                   "has_schema( 'test' ) failed" );
    $self->assert( Lire::DlfSchema->has_schema( "test-exttest"),
                   "has_schema( 'test-exttest' ) failed" );
    $self->assert( ! Lire::DlfSchema->has_schema( "test2"),
                   "has_schema( 'test2' ) failed" );

    $self->assert_died( sub { Lire::DlfSchema->has_schema() },
                        qr/missing 'schema_name' parameter/ );

    $self->assert_died( sub { Lire::DlfSchema->has_schema( "../test" ) },
                        qr/invalid schema name/ );
}

sub test_has_superservice {
    my $self = $_[0];

    $self->assert( Lire::DlfSchema->has_superservice( "test" ),
                   "has_schema( 'test' ) failed" );

    $self->assert( ! Lire::DlfSchema->has_superservice( "test2" ),
                   "has_schema( 'test2' ) should have failed" );

    $self->assert_died( sub { Lire::DlfSchema->has_superservice() },
                        qr/missing 'superservice' parameter/ );


    $self->assert_died( sub { Lire::DlfSchema->has_superservice( "test-exttest" ) },
                        qr/invalid superservice schema name/ );

    $self->assert_died( sub { Lire::DlfSchema->has_superservice( "../test" ) },
                        qr/invalid superservice schema name/ );


}

sub test_superservices {
    my $self = $_[0];

    my @super = Lire::DlfSchema->superservices();
    $self->assert_deep_equals( [ "test" ], \@super );
}

sub test_schemas {
    my $self = $_[0];

    my @schemas = Lire::DlfSchema->schemas();
    $self->assert_deep_equals( [ "test", "test-exttest" ], \@schemas );
}

sub test_as_sql_schema {
    my $self = $_[0];

    my $schema = Lire::DlfSchema::load_schema( "test" );
    $self->assert_str_equals( $test_sql_fields_def,
                              $schema->sql_fields_def()
                            );
}

package tests::DlfSchemaTestI18N;

use base qw/Lire::Test::TestCase tests::TestStoreFixture /;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();
    $self->init_i18n();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    # This has to be done before set_up_locale() otherwise
    # the textdomain might be wrong (since first load of schema
    # will set it to a non-existent directory
    $self->{'schema'} = Lire::DlfSchema::load_schema( "test-extended" );

    $self->set_up_locale();

    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    $self->tear_down_locale();

    return $self;
}

sub test_title_i18n {
    my $self = $_[0];

    my $schema = $self->{'schema'};
    $self->assert_str_equals( 'Simple Extended Schema for Tests',
                              $schema->title() );
    $self->set_locale( 'fr_CA' );
    $self->assert_str_equals( "Dfinition simple d'un schma tendu pour test",
                              $schema->title() );
}

sub test_description_i18n {
    my $self = $_[0];

    my $schema = $self->{'schema'};
    local $schema->{'description'} = '<para>JPEG Files</para>';
    $self->assert_str_equals( "<para>JPEG Files</para>",
                              $schema->description() );
    $self->set_locale( 'fr_CA' );
    $self->assert_str_equals( "<para>Fichiers JPEG</para>",
                              $schema->description() );
}

1;
