package tests::ReportTest;

use strict;

use Data::Dumper;
$Data::Dumper::Terse = 1;
use base qw/Lire::Test::TestCase/;

use Lire::Report;
use File::Temp qw/ tempfile /;
use POSIX qw/strftime/;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_tz( 'UTC' );

    ( $self->{'tmp_fh'}, $self->{'tmp_file'} ) = tempfile();
    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_new {
    my $self = $_[0];


    foreach my $args ( [ "test", time, time + 10 ],
                       [ "test", undef, undef ],
                       [ "test", "", "" ]
                     )
    {
        my $report = new Lire::Report( @$args );
        $self->assert_not_null( $report, "new(", Dumper( $args ),
                                ") returned undef" );
        $self->assert( UNIVERSAL::isa( $report, "Lire::Report" ),
                       "new( ", Dumper( $args ),
                       ") didn't return a Lire::Report instance: $report");
    }

    foreach my $args ( [ qr/missing 'superservice' parameter/, 
                         undef, undef, undef],
                       [ qr/invalid 'superservice' parameter/,
                         "bad service", undef, undef ],
                       [ qr/'timespan_start' parameter should be an integer/,
                         "www", "bad_start", undef ],
                       [ qr/'timespan_end' parameter should be an integer/,
                         "www", time, "bad_end" ],
                       [ qr/'timespan_end' should be greater than 'timespan_start'/, "www", time(), time() - 30 ],
                     )
    {
        $self->assert_died( sub { new Lire::Report( @{$args}[1..3]) },
                            $args->[0] );
    }
}


my $empty_report = <<EOX;
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE lire:report PUBLIC
  "-//LogReport.ORG//DTD Lire Report Markup Language V2.0//EN"
  "http://www.logreport.org/LRML/2.0/lrml.dtd">
<lire:report xmlns:lire="http://www.logreport.org/LRML/" version="2.0" superservice="test">
 <!-- generated by Lire::Report(3pm) -->
 <lire:title>Title with an eacute: \xc3\xa9</lire:title>
 <lire:date time="&NOW&">&NOW_FMT&</lire:date>
 <lire:timespan start="" end="">Unknown Period</lire:timespan>
</lire:report>
EOX

sub test_write_report_empty {
    my $self = $_[0];

    my $warn = "";
    local $SIG{'__WARN__'} = sub { $warn .= join "", @_ };
    my $report = new Lire::Report( "test", undef, undef );
    $report->title( "Title with an eacute: \x{00e9}" );
    my $now = $report->date();
    my $now_string = strftime( '%Y-%m-%d %H:%M:%S %Z', localtime( $now ));
    $empty_report =~ s/&NOW&/$now/;
    $empty_report =~ s/&NOW_FMT&/$now_string/;

    $report->write_report( $self->{'tmp_fh'} );
    $self->annotate( $warn );
    $self->assert( !$warn, "There were warnings" );

    close $self->{'tmp_fh'};
    my $fh;
    open $fh, $self->{'tmp_file'}
      or die( "open failed: $!" );
    local $/ = undef;
    my $buf = <$fh>;
    $self->assert_str_equals( $empty_report, $buf );
}

1;

