package tests::TimegroupTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Timegroup;
use Lire::ReportSpec;
use Lire::Report::Group;
use Lire::Report::TableInfo;
use Lire::DlfQuery;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();

    $self->set_up_tz( 'UTC' );

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'timegroup-test' );

    foreach my $p ( qw/4h 1d 1w 3M 1y/ ) {
        $self->{ $p . '_timegroup'} =
          new Lire::Timegroup( 'report_spec' => $self->{'spec'},
                               'period' =>  $p,
                               'label' => 'Period',
                             );
    }

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_name {
    my $self = $_[0];

    $self->assert_equals( 'timegroup:time_start',
                          $self->{'1d_timegroup'}->name() );
}

sub test_create_categorical_info {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timegroup'}->create_categorical_info( $info );

    my @cols = $info->children();
    $self->assert_equals( 1, scalar @cols );

    my $col = $cols[0];
    $self->assert_equals( 'categorical', $col->class() );
    $self->assert_equals( 'timegroup:time_start', $col->name() );
    $self->assert_equals( 'timestamp', $col->type() );
    $self->assert_equals( 'Period', $col->label() );
}

sub test_period {
    my $self = $_[0];

    $self->{'spec'}->param( 'dur', new Lire::Param( 'name' => 'dur',
                                                    'type' => 'duration',
                                                    'default' => '1d' ) );
    $self->{'spec'}->param( 'int', new Lire::Param( 'name' => 'int',
                                                    'type' => 'int',
                                                    'default' => '100' ) );

    my $timegroup = $self->{'1d_timegroup'};
    $self->assert_equals( '1d', $timegroup->period() );

    $self->assert_died( sub { $timegroup->period( undef ) },
                        qr/missing 'period' parameter/ );
    $self->assert_died( sub { $timegroup->period( 'wawa' ) },
                        qr/'period' parameter isn't a valid duration: 'wawa'/);
    $self->assert_died( sub { $timegroup->period( '$wawa' ) },
                        qr/parameter 'wawa' isn't defined/ );
    $self->assert_died( sub { $timegroup->period( '2d' ) },
                        qr/can't use multiple with period of type 'd'/ );
    $self->assert_died( sub { $timegroup->period( '$int' ) },
                        qr/parameter 'int' isn't a 'duration' parameter: 'int'/ );

    $timegroup->period( '$dur' );
    $self->assert_equals( '$dur', $timegroup->{'period'} );

    $timegroup->period( '2h' );
    $self->assert_equals( '2h', $timegroup->{'period'} );
}

sub test_build_query {
    my $self = $_[0];

    foreach my $t ( [ '4h', 'lr_timegroup_sec("time-start",14400)' ],
                    [ '1d', 'lr_timegroup_day("time-start")' ],
                    [ '1w', 'lr_timegroup_week("time-start",1)' ],
                    [ '3M', 'lr_timegroup_month("time-start",3)' ],
                    [ '1y', 'lr_timegroup_year("time-start",1)' ] )
    {
        my $timegroup = $self->{ $t->[0] . '_timegroup' };
        $timegroup->{'field'} = 'time-start';
        my $e_query = new Lire::DlfQuery( 'test' );
        $e_query->add_aggr_field( '_lr_nrecords', 'count(*)' );
        $e_query->add_group_field( "timegroup:time-start", $t->[1] );
        $e_query->set_sort_spec( "timegroup:time-start" );

        my $query = new Lire::DlfQuery( 'test' );
        $timegroup->build_query( $query );
        $self->assert_deep_equals( $e_query, $query );
    }
}

sub test_create_entry {
    my $self = $_[0];

    $self->{'cfg'}{'lr_week_numbering'} = 'ISO';

    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timegroup'}->create_categorical_info( $info );

    foreach my $t ( [ '4h', '2003-10-14 20:00',  1066161600, 4*3600 ],
                    [ '1d', '2003-10-14', 1066089600, 24*3600 ],
                    [ '1w', '2003-W42', 1064966400, 7*24*3600 ],
                    [ '3M', 'April 2003', 1049155200, 30*24*3600*3 ],
                    [ '1y', '2003', 1041379200, 365*24*3600 ], )
    {
        my ( $period, $fmt_value, $value, $period_sec ) = @$t;

        my $timegroup = $self->{$period . '_timegroup'};
        my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                             $info );
        my $dlf = { 'timegroup:time_start' => $value };
        my $entry = $timegroup->create_entry( $group, $dlf );

        $self->assert_not_null( $entry, "create_entry() returned undef" );
        my @data = $entry->data();
        $self->assert_equals( 1, scalar @data );
        $self->assert_equals( $fmt_value, $data[0]->{'content'} );
        $self->assert_equals( $value, $data[0]->{'value'} );
        $self->assert_equals( $period_sec, $data[0]->{'range'} );
    }
}

sub test_create_entry_mc {
    my $self = $_[0];

    my $timegroup = $self->{'1d_timegroup'};
    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timegroup'}->create_categorical_info( $info );
    my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                         $info );

    my $entry = $timegroup->create_entry( $group,
                                          { 'timegroup:time_start' => undef,
                                            '_lr_nrecords' => 24 } );
    $self->assert_null( $entry, "create_entry() should have returned undef" );
    $self->assert_equals( 24, $group->missing_cases() );
}

sub test_create_entry_day_change {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timegroup'}->create_categorical_info( $info );
    my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                         $info );
    foreach my $t ( [ '2003-10-14 16:00', 1066147200 ],
                    [ '           20:00', 1066161600 ],
                    [ '2003-10-15 00:00', 1066176000 ],
                  )
    {
        my $timegroup = $self->{'4h_timegroup'};
        my $dlf = { 'timegroup:time_start' => $t->[1] };
        my $entry = $timegroup->create_entry( $group, $dlf );


        $self->assert_not_null( $entry, "create_entry() returned undef" );
        $self->assert_str_equals( $t->[0], ($entry->data())[0]->{'content'} );
    }
}

1;
