package tests::UICompoundWidgetTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::Config::TypeSpec;

use Curses::UI;
use Curses::UI::Widget;
use Curses::UI::Container;

use Lire::Test::CursesUIDriver;
use Lire::Utils qw/deep_copy/;

sub new {
    my $self = shift->SUPER::new ( @_ );

    $self->{'driver'} = new Lire::Test::CursesUIDriver();

    my $spec = new Lire::Config::ConfigSpec();
    my $compound = new Lire::Config::ConfigSpec( 'name' => 'compound' );
    $spec->add( $compound );
    $compound->add ( new Lire::Config::IntegerSpec( 'name' => 'integer' ) );
    my $list = new Lire::Config::ListSpec( 'name' => 'list',
                                           'summary' => 'List summary',
                                           'description' => '<para>List description</para>' );
    $compound->add ( $list );
    $list->add ( new Lire::Config::StringSpec( 'name' => 'string' ) );
    $compound->add ( new Lire::Config::StringSpec( 'name' => 'string' ) );

    my $empty = new Lire::Config::ConfigSpec( 'name' => 'empty' );
    $spec->add( $empty );

    $self->{'compound'} = $compound->instance();
    $self->{'empty'} = $empty->instance();

    return $self;
}

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();

    $self->{'driver'}->setup_curses_ui();

    $self->{'ui'} = new Curses::UI();
    $self->{'driver'}->set_curses_ui( $self->{'ui'} );
    $self->{'window'} = $self->{'ui'}->add( 'window', 'Window' );

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->{'driver'}->teardown_curses_ui();

    return;
}

sub test_new {
    my $self = $_[0];

    my $win = $self->{'window'};
    $self->assert_died( sub { my $widget = $win->add( 'widget',
                                                      'Lire::UI::CompoundWidget' ) },
                        qr{missing 'value' parameter} );
    $self->assert_died( sub { my $widget = $win->add( 'widget',
                                                      'Lire::UI::CompoundWidget',
                                                      'Value' => {} ) },
                        qr{'value' parameter should be a 'Lire::Config::Dictionary' instance, not 'HASH} ); #'} );#cperl-mode really sucks

    my $widget = $win->add( 'widget', 'Lire::UI::CompoundWidget',
                            'value' => $self->{'compound'} );
    $self->assert( UNIVERSAL::isa( $widget, 'Lire::UI::CompoundWidget' ),
                   "not a Lire::UI::CompoundWidget instance: $widget" );
    $self->assert_str_equals( $self->{'compound'}, $widget->{'value'} );

    my $help_viewer = $widget->getobj( 'help_viewer' );
    $self->assert( UNIVERSAL::isa( $help_viewer, 'Curses::UI::TextViewer' ),
                   "not a Curses::UI::TextViewer instance: $help_viewer" );
    my $help_summary = $widget->getobj( 'help_summary' );
    $self->assert( UNIVERSAL::isa( $help_summary, 'Curses::UI::Label' ),
                   "not a Curses::UI::Label instance: $help_summary" );

    $self->assert_null( $widget->getobj( 'empty_label' ),
                        'Unexpected empty label' );

    my $str_widget = $widget->getobj( 'string_widget' );
    $self->assert( UNIVERSAL::isa( $str_widget, 'Lire::UI::StringWidget' ),
                   "not a 'Lire::UI::StringWidget': $str_widget" );
    $self->assert_str_equals( $self->{'compound'}->get( 'string' ),
                              $str_widget->{'value'} );

    my $int_widget = $widget->getobj( 'integer_widget' );
    $self->assert( UNIVERSAL::isa( $int_widget, 'Lire::UI::IntegerWidget' ),
                   "not a 'Lire::UI::IntegerWidget: '$int_widget'" );
    my $list_widget = $widget->getobj( 'list_widget' );
    $self->assert( UNIVERSAL::isa( $list_widget, 'Lire::UI::ListWidget' ),
                   "not a 'Lire::UI::ListWidget': $list_widget" );

    foreach my $w ( ( $str_widget, $int_widget, $list_widget ) ) {
        $self->assert_str_equals( \&Lire::UI::CompoundWidget::_child_focus_cb,
                                  $w->{'-onfocus'},
                                  "wrong focuscb for $w" );
    }

    foreach my $name ( qw/ string integer list / ) {
        my $w = $widget->getobj( "${name}_label" );
        $self->assert( UNIVERSAL::isa( $w, 'Curses::UI::Label' ),
                       "bad label for '$name': $w" );
        $self->assert_str_equals( $name, $w->text() );
    }

    $self->check_layout( $widget );

    return;
}

sub test_new_empty {
    my $self = $_[0];

    my $win = $self->{'window'};
    my $widget = $win->add( 'widget', 'Lire::UI::CompoundWidget',
                            'value' => $self->{'empty'} );

    $self->assert( UNIVERSAL::isa( $widget, 'Lire::UI::CompoundWidget' ),
                   "not a Lire::UI::CompoundWidget instance: $widget" );
    $self->assert_str_equals( $self->{'empty'}, $widget->{'value'} );
    my $label = $widget->getobj( 'empty_label' ) || '< undef >';
    $self->assert( UNIVERSAL::isa( $label, 'Curses::UI::Label' ),
                   "not a Curses::UI::Label instance: $label" );
    $self->assert_str_equals( '-- This option does not have any attributes --',
                              $label->text() );
}

sub test_refresh_view {
    my $self = $_[0];

    my $win = $self->{'window'};
    my $widget = $win->add( 'widget', 'Lire::UI::CompoundWidget',
                            'value' => $self->{'compound'} );

    no warnings 'redefine';

    my ( $strw, $intw, $listw ) = ( 'undef' x 3 );
    local *Lire::UI::StringWidget::refresh_view = sub { $strw = shift };
    local *Lire::UI::IntegerWidget::refresh_view = sub { $intw = shift };
    local *Lire::UI::ListWidget::refresh_view = sub { $listw = shift };

    $widget->refresh_view();

    $self->assert_str_equals( $strw, $widget->getobj( 'string_widget' ) );
    $self->assert_str_equals( $intw, $widget->getobj( 'integer_widget' ) );
    $self->assert_str_equals( $listw, $widget->getobj( 'list_widget' ) );
}

sub test_child_focus_cb {
    my $self = $_[0];

    my $win = $self->{'window'};
    my $widget = $win->add( 'widget', 'Lire::UI::CompoundWidget',
                            'value' => $self->{'compound'} );
    my $help_summary = $widget->getobj( 'help_summary' );
    my $help_viewer = $widget->getobj( 'help_viewer' );

    $self->assert_str_equals( '', $help_summary->text() );
    $self->assert_str_equals( '', $help_viewer->text() );

    Lire::UI::CompoundWidget::_child_focus_cb( $widget->getobj('list_widget'));

    $self->assert_str_equals( 'List summary', $help_summary->text() );
    $self->assert_str_equals( 'List description', $help_viewer->text() );

    Lire::UI::CompoundWidget::_child_focus_cb( $widget->getobj('string_widget'));

    $self->assert_str_equals( 'string', $help_summary->text() );
    $self->assert_str_equals( 'No help available.', $help_viewer->text() );
}

sub check_layout {
    my ( $self, $widget ) = @_;

    $self->fail( 'Test requires a screen of 80 columns' )
      unless $ENV{'COLS'} == 80;

    my $integer_label = $widget->getobj( 'integer_label' );
    $self->assert_num_equals( 0, $integer_label->{'-y'}  );

    my $list_label = $widget->getobj( 'list_label' );
    $self->assert_num_equals( 2, $list_label->{'-y'} );

    my $string_label = $widget->getobj( 'string_label' );
    $self->assert_num_equals( 12, $string_label->{'-y'} );
    foreach my $label ( ( $list_label, $integer_label, $string_label ) ) {
        $self->annotate( "Checking $label->{'-text'}\n" );
        $self->assert_str_equals( 'right', $label->{'-textalignment'} );
        $self->assert_num_equals( 0, $label->{'-x'} );
        $self->assert_num_equals( 1, $label->{'-height'} );
        $self->assert_num_equals( 7, $label->{'-width'} );
    }

    my $integer_widget = $widget->getobj( 'integer_widget' );
    $self->assert_num_equals( 0, $integer_widget->{'-y'} );
    $self->assert_num_equals( 1, $integer_widget->{'-height'} );

    my $list_widget = $widget->getobj( 'list_widget' );
    $self->assert_num_equals( 2, $list_widget->{'-y'} );
    $self->assert_num_equals( 9, $list_widget->{'-height'} );

    my $string_widget = $widget->getobj( 'string_widget' );
    $self->assert_num_equals( 12, $string_widget->{'-y'} );
    $self->assert_num_equals( 1, $string_widget->{'-height'});

    foreach my $wid ( ( $list_widget, $integer_widget, $string_widget ) ) {
        $self->assert_num_equals( 8, $wid->{'-x'} );
        $self->assert_num_equals( 41, $wid->{'-width'} );
    }

    my $summary = $widget->getobj( 'help_summary' );
    $self->assert_num_equals( 50, $summary->{'-x'} );
    $self->assert_num_equals( 30, $summary->{'-width'} );
    $self->assert_num_equals( 0, $summary->{'-y'} );
    my $help_viewer = $widget->getobj( 'help_viewer' );
    $self->assert_num_equals( 50, $help_viewer->{'-x'} );
    $self->assert_num_equals( 30, $help_viewer->{'-width'} );
    $self->assert_num_equals( 2, $help_viewer->{'-y'} );

    return;
}

sub test_max_label_size {
    my $self = $_[0];

    my $new_compound = deep_copy( $self->{'compound'}->spec() );

    my $win = $self->{'window'};
    my $widget = $win->add( 'widget', 'Lire::UI::CompoundWidget',
                            'value' => $new_compound->instance() );

    $self->assert_num_equals( 7, $widget->_max_label_size() );

    $new_compound->get( 'string' )->{'name'} = "A very very very very long";
    $self->assert_num_equals( 20, $widget->_max_label_size() );

}

1;
