package tests::UIPluginWidgetTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Curses::UI;

use Lire::Config::TypeSpec;
use Lire::Test::CursesUIDriver;

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();
    $self->set_up_spec();
    $self->set_up_config();

    $self->{'driver'} = new Lire::Test::CursesUIDriver();
    $self->{'driver'}->setup_curses_ui();

    $self->{'ui'} = new Curses::UI();
    $self->{'driver'}->set_curses_ui( $self->{'ui'} );

    $self->{'window'} = $self->{'ui'}->add( 'window', 'Window' );

    return;
}

sub set_up_spec {
    my $self = $_[0];

    my $spec = new Lire::Config::ConfigSpec( 'name' => 'lire' );
    my $plugin_spec = new Lire::Config::PluginSpec( 'name' => 'plugin',
                                                    'class' => 'Wawa::Class' );
    $spec->add( $plugin_spec );
    foreach my $name ( qw/one two wawa/ ) {
        $plugin_spec->add( new Lire::Config::OptionSpec( 'name' => $name ) );
    }

    my $wawa_props =
      new Lire::Config::RecordSpec( 'name' => 'wawa_properties' );
    $spec->add( $wawa_props );
    $wawa_props->add( new Lire::Config::StringSpec( 'name' => 'dummy' ) );

    my $empty_plugin = new Lire::Config::PluginSpec('name' => 'empty_plugin',
                                                    'class' => 'Empty::Class');
    $spec->add( $empty_plugin );

    $self->{'spec'} = $spec;

    return;
}

sub set_up_config {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    my $config = Lire::Config->instance();
    $self->{'config'} = $config;
    $config->{'_lr_config_spec'} = $spec;

    my $plugin = $spec->get( 'plugin' )->instance( 'value' => 'wawa');
    $config->{'plugin'} = $self->{'plugin'} = $plugin;

    my $wawa_props = $plugin->get_properties();
    $wawa_props->get( 'dummy' )->set( 'wawa string' );

    $config->{'empty_plugin'} = $spec->get( 'empty_plugin' )->instance();
    $self->{'empty_plugin'} = $config->{'empty_plugin'};

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down();
    $self->{'driver'}->teardown_curses_ui();

    return;
}

sub test_new {
    my $self = $_[0];

    my $win = $self->{'window'};
    $self->assert_died( sub {my $widget = $win->add('test_widget',
                                                    'Lire::UI::PluginWidget')},
                       qr/missing \'value\' parameter/ );
    $self->assert_died( sub { my $widget = $win->add( 'test_widget',
                                                      'Lire::UI::PluginWidget',
                                                      'Value' => {} ) },
                        qr/\'value\' parameter should be a \'Lire::Config::Plugin\' instance, not \'HASH/ );

    $self->assert_isa( 'Lire::Config::Plugin', $self->{'plugin'} );

    my $widget = $win->add( 'testwidget', 'Lire::UI::PluginWidget',
                            'value' => $self->{'plugin'} );
    $self->assert_isa( 'Lire::UI::PluginWidget', $widget );
    $self->assert_isa( 'Curses::UI::Container', $widget );

    $self->assert_num_equals( 1, $widget->{'-releasefocus'} );

    my $list = $widget->getobj( 'list' );
    $self->assert_num_equals( 2, $list->{'-selected'} );

    $list->{'-selected'} = 1;
    $self->assert_isa( 'Curses::UI::Popupmenu', $list );
    $self->assert_str_equals( 'two', $list->get() );
    $self->assert_deep_equals( [ 'one', 'two', 'wawa' ],
                               $list->{'-values'} );
    $self->assert_num_equals( 1, $list->{'-selected'} );

    my $button = $widget->getobj( 'button' );
    $self->assert_isa( 'Curses::UI::Buttonbox', $button );
    $self->assert_str_equals( '<...>', $button->{'-buttons'}[0]{'-label'} ) ;
    $self->assert( defined $widget->getobj( 'button' ) );

    $widget = $win->add( 'empty_widget', 'Lire::UI::PluginWidget',
                         'value' => $self->{'empty_plugin'} );
    $list = $widget->getobj( 'list' );
    $self->assert_deep_equals( [ '-- empty list --' ], $list->{'-values'});
    $self->assert_num_equals( 0, $list->{'-selected'} );
    $self->assert_num_equals( 0, $list->{'-focusable'} );
    $self->assert_null( $widget->getobj( 'button' ) );

    $self->{'plugin'}->set_plugin( undef );
    $widget = $win->add( 'undefplugin', 'Lire::UI::PluginWidget',
                         'value' => $self->{'plugin'} );
    $list = $widget->getobj( 'list' );
    $self->assert_null( $list->{'-selected'} );
    $self->assert_null( $widget->getobj( 'button' ) );

    $self->{'plugin'}->set_plugin( 'one' );
    $widget = $win->add( 'nopropplugin', 'Lire::UI::PluginWidget',
                         'value' => $self->{'plugin'} );
    $self->assert_null( $widget->getobj( 'button' ) );

    return;
}

sub test_properties_cb {
    my $self = $_[0];

    my $win = $self->{'window'};
    my $widget = $win->add( 'testwidget', 'Lire::UI::PluginWidget',
                            'value' => $self->{'plugin'} );
    my $list = $widget->getobj( 'list' );
    my $button = $widget->getobj( 'button' );

    my $dialog;
    $self->{'driver'}->add_event_loop_handler( sub { $dialog = $_[1] } );

    my $properties = $self->{'plugin'}->get_properties();
    Lire::UI::PluginWidget::_properties_cb_helper( $widget );
    $self->assert_isa( 'Curses::UI::Window', $dialog );
    $self->assert_num_equals( 1, $dialog->{'-border'} );
    $self->assert_num_equals( 1, $dialog->{'-ipad'} );
    $self->assert_str_equals( "Properties for wawa", $dialog->{'-title'} );

    my $prop_widget = $dialog->getobj( 'widget' );
    $self->assert_isa( 'Lire::UI::CompoundWidget', $prop_widget );
    $self->assert_str_not_equals( $properties, $prop_widget->{'value'} );
    $self->assert_deep_equals( $properties, $prop_widget->{'value'} );

    my $prop_buttons = $dialog->getobj( 'buttons' );
    $self->assert_isa( 'Curses::UI::Buttonbox', $prop_buttons );
    my @btns = map { $_->{'-label'} } @{ $prop_buttons->{'-buttons'} };
    $self->assert_deep_equals( [ '< Cancel >', '< OK >' ],
                               \@btns );

    $self->{'driver'}->add_event_loop_handler( sub {$self->fail( "This code should not have been reached" ) } );
    $self->{'plugin'}->set_plugin( undef );
    Lire::UI::PluginWidget::_properties_cb_helper( $widget );

    $self->{'plugin'}->set_plugin( 'one' );
    Lire::UI::PluginWidget::_properties_cb_helper( $widget );

    return;
}

sub test_on_change_cb {
    my $self = $_[0];

    my $win = $self->{'window'};

    my $widget = $win->add( 'test_widget', 'Lire::UI::PluginWidget',
                            'value' => $self->{'plugin'} );
    my $list = $widget->getobj( 'list' );
    $self->assert_str_equals( 'wawa', $list->get() );
    $self->assert( defined $widget->getobj( 'button' ) );
    $list->{'-selected'} = 0;
    $list->{'-onchange'}->( $widget );
    $self->assert_str_equals( 'one', $self->{'plugin'}->get_plugin() );
    $self->assert_null( $widget->getobj( 'button' ) );

    return;
}

sub _button_cb_test_helper {
    my ( $self, $button_nbr, $e_string ) = @_;

    my $orig_props = $self->{'plugin'}->get_properties();
    my $dummy_spec = $orig_props->spec()->get( 'dummy' );
    my $dummy = $dummy_spec->instance( 'value' => 'wawa string' );
    $orig_props->set( $dummy );

    my $win = $self->{'window'};
    my $widget = $win->add( 'testwidget', 'Lire::UI::PluginWidget',
                            'value' => $self->{'plugin'} );
    my $cancel_sub = sub {
        my $dialog = $_[1];
        my $p_widget = $dialog->getobj( 'widget' );
        $p_widget->{'value'}->get( 'dummy' )->set( 'new string' );
        my $buttons = $dialog->getobj( 'buttons' );
        $buttons->{'-buttons'}[$button_nbr]{'-onpress'}->( $widget );
    };

    $self->{'driver'}->add_event_loop_handler( $cancel_sub );

    $self->assert_str_equals( 'wawa string',
                              $orig_props->get( 'dummy' )->as_value() );

    Lire::UI::PluginWidget::_properties_cb_helper( $widget );

    my $prop_window = $self->{'ui'}->getobj( 'plugin_properties' );
    $self->assert_null( $prop_window );
    my $props = $self->{'plugin'}->get_properties();
    $self->assert_str_equals( $orig_props, $props );
    $self->assert_str_equals( $e_string,
                              $props->get( 'dummy' )->as_value() );

    return;
}

sub test_properties_cancel_cb {
    my $self = $_[0];

    $self->_button_cb_test_helper( 0, 'wawa string' );

    return;
}

sub test_properties_ok_cb {
    my $self = $_[0];

    $self->_button_cb_test_helper( 1, 'new string' );

    return;
}

1;
