package tests::functional::DefaultsTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase/;

use File::Path;
use File::Basename;
use Lire::Config::Build qw/ac_info ac_path/;
use Lire::Utils qw/tempdir create_file/;
use Carp;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'defaults_file'} = ac_info( "sysconfdir") . "/lire/defaults";

    return $self;
}

sub test_tmpdir_unset {
    my $self = $_[0];

    local $ENV{'TMPDIR'} = '';

    my $result = $self->lire_run( ". $self->{'defaults_file'}; env" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_matches( qr!TMPDIR=/tmp!, $result->stdout );
}

sub test_tmpdir_set {
    my $self = $_[0];

    my $tmpdir = $self->homedir . "/tmp";
    local $ENV{'TMPDIR'} = $tmpdir;

    my $result = $self->lire_run( ". $self->{'defaults_file'}; env" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_matches( qr!TMPDIR=/tmp!, $result->stdout );

    mkdir $tmpdir,0700
      or croak "mkdir $tmpdir failed $!";
    $result = $self->lire_run( ". $self->{'defaults_file'}; env" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_matches( qr!TMPDIR=$tmpdir!, $result->stdout );
}

sub test_lr_environment_run {
    my $self = $_[0];
    my $result = $self->lire_run( ". $self->{'defaults_file'}; env" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_matches( qr/lr_environment info started with/,
                           $result->stderr );
}

sub test_home_directory_setup {
    my $self = $_[0];

    # Delete the .lire directory created by our parent
    # We create in the cmd to lire_run() because the $HOME/.lire/config
    # needs to be created when lire_run() is called.
    my $homedir = $self->homedir() . "/.lire";
    my $result = $self->lire_run( "rm -fr $homedir; . $self->{'defaults_file'}; env" );
    $self->annotate( $result->stderr() );
    $self->assert_equals( 0, $result->status() );

    foreach my $d ( qw/config converters reports schemas filters/ ) {
        $self->assert( -d $self->homedir() . "/.lire/$d",
                       "home directory '.lire/$d' wasn't created"
                     );
        $self->assert_matches( qr/creating.*$d directory/, $result->stderr );
    }
}

sub test_lr_tempfile {
    my $self = $_[0];

    my $result = $self->lire_run( <<EOS );
. $self->{'defaults_file'}
lr_tempfile test_XXXXXX.log
EOS
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_does_not_match( qr/ (crit|err|warning) / , $result->stderr,
                                  "There were warnings or error messages." );
    my $file = $result->stdout;
    chomp $file;
    $self->assert_matches( qr/test_[a-zA-Z0-9]{6}\.log$/, $file );
    $self->assert( -f $file, "file '$file' doesn't exists" );
}

sub test_lr_tempdir {
    my $self = $_[0];

    my $result = $self->lire_run( <<EOS );
. $self->{'defaults_file'}
lr_tempdir test_XXXXXX
EOS
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_does_not_match( qr/ (crit|err|warning) / , $result->stderr,
                                  "There were warnings or error messages." );

    my $dir = $result->stdout;
    chomp $dir;
    $self->assert_matches( qr/test_[a-zA-Z0-9]{6}/, $dir );
    $self->assert( -d $dir, "directory '$dir' doesn't exists" );
}

sub test_cleanup {
    my $self = $_[0];

    my $file = $self->rundir . "/test_file";
    create_file( $file, "This is a test\n" );
    my $dir = tempdir( "test_XXXXXX" );
    my $result = $self->lire_run( <<EOS );
. $self->{'defaults_file'}
lr_mark_for_cleanup $dir
lr_mark_for_cleanup $file
EOS
    $self->annotate( $result->stderr );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    $self->assert( !$result->stdout, "STDOUT should be empty" );
    $self->assert( ! -f $file, "$file wasn't removed" );
    $self->assert( ! -d $file, "$dir wasn't removed" );
}

sub test_keep_cleanup {
    my $self = $_[0];

    my $file = $self->rundir . "/test_file";
    $self->create_test_file( $file, "This is a test\n" );
    my $dir = tempdir( "test_XXXXXX" );
    my $result = $self->lire_run( <<EOS );
. $self->{'defaults_file'}
lr_keep=1
lr_mark_for_cleanup $dir $file
EOS
    $self->annotate( $result->stderr );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    $self->assert_matches( qr/ debug lr_keep is set, not deleting '$file'/,
                           $result->stderr );
    $self->assert_matches( qr/ debug lr_keep is set, not deleting '$dir'/,
                           $result->stderr );
    $self->assert( !$result->stdout, "STDOUT should be empty" );
    $self->assert( -f $file, "$file was removed even if with lr_keep=1" );
    $self->assert( -d $dir, "$dir was removed even if with lr_keep=1" );
}

1;
