/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/variable.h"
#include "include/symbol.h"
#include "include/environment.h"

class LWVariableWorldPiece: public LWVariable {

LWPiece *piece;
	
protected:
	
gboolean isSpecial ()
{
	return TRUE;
}

public:

LWVariableWorldPiece (LWPiece *a_piece)
: piece(a_piece) {}

void clear ()
{
	piece->clear();
}

gboolean canBeIndexed ()
{
	return FALSE;
}

void set (LWValue *value)
{
	piece->setBackgroundPixmap (value->getPixmap());
}

LWValue *get ()
{
	return new LWValue(piece->getBackgroundPixmap());
}

guint count ()
{
	return 1;
}

};

class LWVariableWorldRow: public LWVariable {

LWRow *row;
GPtrArray *array;
guint len;

protected:
	
gboolean isSpecial ()
{
	return TRUE;
}

public:

LWVariableWorldRow (LWRow *a_row): row(a_row)
{
	len = row->getWidth ();
	array = g_ptr_array_sized_new (len);

	for (guint i = 0; i < len; i++) {
		LWPiece *piece = row->getPieceNth (i);
		g_ptr_array_add (array, (gpointer) new LWVariableWorldPiece(piece));
	}
}

~LWVariableWorldRow ()
{
	for (guint i = 0; i < len; i++)
		delete (LWVariableWorldPiece *) g_ptr_array_index (array, i);
		
	g_ptr_array_free (array, TRUE);
}


void clear ()
{
	GSList *plist = row->getPieceList();
	
	for (GSList *l = plist ; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece *) l->data;
		piece->clear();
	}

	g_slist_free (plist);
}

LWValue *getIndex (guint n)
{
	if (n > len) return NULL;
	return (LWVariable *) g_ptr_array_index (array, n-1);
}

void set (LWValue *value)
{
	if (FALSE == value->canBeIndexed()) return;
		
	for (guint i = 0; i < len; i++)
		((LWVariableWorldPiece *) g_ptr_array_index (array, i))
			->set(value->getIndex(i+1));
}

LWValue* get ()
{
	LWValue *v = new LWValue();
	
	GSList *plist = row->getPieceList();
	
	for (GSList *l = plist ; l != NULL; l = l->next) {
		LWPiece *piece = (LWPiece *) l->data;
		LWPixmap *pixmap = piece->getBackgroundPixmap();
		v->append (pixmap);
	}
	
	g_slist_free (plist);	
	return v;
}

guint count ()
{
	return len;
}

};

class LWVariableWorld: public LWVariable, public LWSymbol {

GPtrArray *array;
guint len;

protected:
gboolean isSpecial ()
{
	return TRUE;
}

public:
LWVariableWorld (): array(NULL), len(0) {}
	
gchar *getName ()
{
	return "world";
}	

gchar *getHint ()
{
	return _("World as an array");
}

gboolean isValue ()
{
	return TRUE;
}

gboolean canClone ()
{
	return FALSE;	
}


void clear ()
{
	if (array == NULL) return;
	
	for (guint i = 0; i < len; i++)
		delete (LWVariableWorldRow *) g_ptr_array_index (array, i);	
		
	g_ptr_array_free (array, TRUE);
	
	array = NULL;
}

void init ()
{
	LWProgram *program = LWEnvironment::getProgram();
	LWBoard *world = program->getWorld();
	
	len = world->getHeight();
	array = g_ptr_array_sized_new (len);

	for (guint i = 0; i < len; i++) {
		LWRow *row = world->getRowNth (i);
		g_ptr_array_add (array, (gpointer) new LWVariableWorldRow(row));
	}
}

LWValue *getIndex (guint n)
{
	if (array == NULL) init();
	if (n > len) return NULL;
	
	return (LWVariable *) g_ptr_array_index (array, n-1);
}

void set (LWValue *value)
{
	if (FALSE == value->canBeIndexed()) return;
	if (array == NULL) init();
		
	for (guint i = 0; i < len; i++)
		((LWVariableWorldRow *) g_ptr_array_index (array, i))
			->set(value->getIndex(i+1));
}

LWValue* get ()
{
	if (array == NULL) init();

	LWValue *v = new LWValue();

	for (guint i = 0; i < len; i++) {
		LWValue *t = ((LWVariable *) g_ptr_array_index (array, i))->get();
		v->getIndex (i+1)->set(t);
		delete t;
	}

	return v;	
}

guint count ()
{
	if (array == NULL) init();

	return len;
}

};

LWSymbol *new_LWVariableWorld()
{
	return new LWVariableWorld();
}
