/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/message.h"

/*! \brief Konstruktor komunikatu

	Tworzy komunikat na podstawie typu.
	\param a_message Typ wiadomości
*/
LWMessage::LWMessage (LWMsg a_message): piece(NULL), program(NULL)
{
	message = a_message;
}

/*! \brief Przypisanie klocka

	Przypisuje klocek, z którym związany jest komunikat
	\param a_piece Klocek
*/
void LWMessage::setPiece (LWPiece *a_piece)
{
	g_return_if_fail (a_piece != NULL);
	g_return_if_fail (a_piece->getRow() != NULL);
	g_return_if_fail (a_piece->getRow()->getBoard() != NULL);
	
	if (piece == NULL) {
		piece = a_piece;
		a_piece->getRow()->getBoard()->markPiece (a_piece);
	}
}

/*! \brief Przypisanie programu

	Przypisuje program, w którym wystąpił błąd. Po zamknięciu
	okienka z wiadomością przez użytkownika, program zostanie
	zniszczony.
	\param a_program Program
*/
void LWMessage::setProgram (LWProgram *a_program)
{
	g_return_if_fail (a_program != NULL);
	g_return_if_fail (program == NULL);
	
	program = a_program;
}

/*! \brief Wyświetlenie komunikatu

	Wyświetla komunikat. Przed wywołaniem tej metody należy skojarzyć
	z wiadomością klocek, którego tyczy komunikat, ponieważ wewnątrz
	tej metody zostanie on oznaczony.
*/
void LWMessage::show ()
{
	gchar *msg=NULL;
	GtkMessageType type=GTK_MESSAGE_ERROR;
	GtkWidget *dialog;

	switch (message) {
		case LW_INFO_ProgramInterrupted:
			msg = _("Program interrupted");
			type = GTK_MESSAGE_INFO;
			break;
		case LW_ERROR_WrongNumberOfArguments:
			if (piece != NULL)
				msg = _("Wrong number of arguments");
			else
				msg = _("Begin program with command for wizard");
			break;
		case LW_ERROR_LValueIsNotVariable:
			msg = _("Left side of operator is not variable");
			break;
		case LW_ERROR_UnmatchedBrace:
			msg = _("Unmatched brace");
			break;
		case LW_ERROR_EmptyBrace:
			msg = _("Empty brace");
			break;
		case LW_ERROR_BadString:
			msg = _("Bad string. Don't mix up digits and variables.");
			break;
		case LW_ERROR_DivisionByZero:
			msg = _("Division by zero");
			break;
		case LW_ERROR_BadIndexOfArray:
			msg = _("Bad index of array. Arrays are indexed beginning with 1.");
			break;
		case LW_ERROR_ForButWhereIsToOrDownto:
			msg = _("There is command 'for', but there is no command 'to' or 'downto'.");
			break;
		case LW_ERROR_ToButWhereIsFor:
			msg = _("There is command 'to', but there is no command 'for'.");
			break;
		case LW_ERROR_DowntoButWhereIsFor:
			msg = _("There is command 'downto', but there is no command 'for'.");
			break;
		case LW_ERROR_RepeatButWhereIsUntil:
			msg = _("There is command 'repeat', but there is no command 'until'.");
			break;
		case LW_ERROR_UntilButWhereIsRepeat:
			msg = _("There is command 'until', but there is no command 'repeat'.");
			break;
		case LW_ERROR_NeedVariableAsArgument:
			msg = _("That command need a variable as an argument");
			break;
		case LW_ERROR_IsNotArray:
			msg = _("You can't use index here. It's not an array.");
			break;
		case LW_ERROR_IndexOutOfRange:
			msg = _("Index of array is out of range");
	}
	
	dialog = gtk_message_dialog_new (NULL,
					GTK_DIALOG_MODAL,
					type,
					GTK_BUTTONS_OK,
				   "%s", msg);
					
  	gtk_dialog_run (GTK_DIALOG (dialog));
  	gtk_widget_destroy (dialog);
	
	if (program != NULL)
		delete program;
}
