/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include <stdlib.h>
#include "include/support.h"
#include "include/variable.h"
#include "include/environment.h"

#define MAKE_TWIN(v) const LWValue *twin = (FALSE == v->isSpecial()) ? v : v->get();
#define FREE_TWIN if (TRUE == isSpecial()) delete twin;

static int pixmapToInt (LWPixmap *pixmap)
{
	g_return_val_if_fail (pixmap != NULL, 0);
	
	gchar *s = pixmap->getName ();
	return atoi (s);
}

static LWPixmap *intToPixmap (gint n)
{
	g_return_val_if_fail (NULL != LWEnvironment::getPixmapSet(), NULL);
	g_return_val_if_fail (n >= 0, NULL);
	
	gchar *s = g_strdup_printf ("%i", n);
	LWPixmap *pixmap = LWEnvironment::getPixmapSet()->getPixmap(s);
	g_free (s);
	return pixmap;
}

static LWPixmap *getMinusSign ()
{
	g_return_val_if_fail (NULL != LWEnvironment::getPixmapSet(), NULL);
	return LWEnvironment::getPixmapSet()->getPixmap("-");
}

void LWValue::convertToList ()
{
	g_return_if_fail (TRUE == canBeIndexed());
	
	if (is_list == FALSE) {
		list = g_slist_append (NULL, new LWValue (pixmap));
		is_list = TRUE;
	}
}

	
/*! \brief Konstruktor pustej wartości

	Tworzy pustą wartość
*/
LWValue::LWValue () : is_list(TRUE), list(NULL)
{
}

/*! \brief Konstruktor wartości z ikoną

	Tworzy wartość zawierającą pojedynczą ikonę
*/
LWValue::LWValue (LWPixmap *a_pixmap): is_list(FALSE), pixmap(a_pixmap)
{
}

/*! \brief Konstruktor kopiujący

	Konstruktor kopiujący wartości

	\param value Wartość na podstawie której ma powstać duplikat
*/
LWValue::LWValue (const LWValue *value): is_list(TRUE), list(NULL)
{
	set ( (LWValue *) value);
}

/*! \brief Konstruktor wartości na podstawie liczby

	Tworzy wartość zawierającą listę ikon będącą odpowiednikiem liczby
	\param ivalue Liczba 
*/
LWValue::LWValue (gint ivalue) : is_list(TRUE), list(NULL)
{
	gboolean hasminus = (ivalue < 0) ? TRUE : FALSE;
	
	if (hasminus == TRUE)
		ivalue = - ivalue;
	
	do {
		LWPixmap *pixmap = intToPixmap(ivalue % 10);
		list = g_slist_prepend (list, (gpointer) new LWValue(pixmap));
		ivalue /= 10;
	} while (ivalue > 0);
	
	if (hasminus == TRUE) {
		LWPixmap *pixmap = getMinusSign ();
		list = g_slist_prepend (list, (gpointer) new LWValue(pixmap));
	}
}

/*! \brief Destruktor

	Niszczy wartość
*/	
LWValue::~LWValue ()
{
	clear();
}

/*! \brief Czyszczenie wartości

	Czyści wartość niszcząc przy tym wszystkie ikony wchodzące w jej skład
*/	
void LWValue::clear ()
{
	if (TRUE == isSpecial()) return;

	if (is_list == TRUE) {
		
		for (GSList *l=list; l != NULL; l = l->next) {
			LWValue *value = (LWValue *) l->data;
			delete value;
		}
		
		g_slist_free (list);
	}
	
	is_list = TRUE;
	list = NULL;
}

/*! \brief Dołączenie ikony

	Dołącza ikonę do końca listy przechywowanej w wartości
	\param pixmap Ikona
*/
void LWValue::append (LWPixmap *pixmap)
{
	g_return_if_fail (FALSE == isSpecial());
	
	if (TRUE == canBeIndexed ())
		convertToList();
	
	if (is_list == TRUE)
		list = g_slist_append (list, 
			(TRUE == isVariable()) ? 
				new LWVariable(pixmap) : new LWValue(pixmap));
}

/*! \brief Łączenie wartości

	Dołącza listę przechywaną w value na koniec listy przechywowanej w wartości
	\param value Warość dołączana
*/
void LWValue::concat (LWValue *value)
{
	g_return_if_fail (value != NULL);
	g_return_if_fail (FALSE == value->isSpecial());
	
	if (TRUE == canBeIndexed ())
		convertToList();
	
	if (is_list == FALSE) 
		return;

	if (value->is_list == FALSE)
		list = g_slist_append (list, (TRUE == isVariable()) ?
			new LWVariable(pixmap) : new LWValue(pixmap));
		
	for (GSList *l= value->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		
		if (TRUE == isVariable())
			list = (v->is_list == TRUE) ? 
				g_slist_append (list, new LWValue (v->pixmap)) :
					g_slist_append (list, new LWValue (v));
		else
			list = (v->is_list == TRUE) ? 
				g_slist_append (list, new LWVariable (v->pixmap)) :
					g_slist_append (list, new LWVariable (v));
	}
}

/*! \brief Interpretacja jako liczba

	Interpretuje wartość jako liczbę
	\return Liczba
*/
gint LWValue::toInt ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE)
		if (twin->pixmap != NULL) {
			FREE_TWIN
			return pixmapToInt (twin->pixmap);
		} else {
			FREE_TWIN
			return 0;
		}
			
	gboolean isminus = FALSE;
	gint ivalue = 0;
	GSList *l;
	
	for (l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE) return 0;
		if (getMinusSign () == v->pixmap) isminus = TRUE; 
		else if (pixmapToInt( (LWPixmap *) v->pixmap) > 0) break;
		}
		
	for (; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE) break;
		ivalue *= 10; 
		ivalue += pixmapToInt( (LWPixmap *) v->pixmap);
	}

	FREE_TWIN
	
	return (isminus == FALSE) ? ivalue : -ivalue;
}

/*! \brief Czy wartość indeksowana

	Czy wartość może być indeksowana,
	zwraca prawdę. Metoda może być przeciążona
	w implementacji specjalnych wartości 
	\return Czy wartość może być indeksowana
*/
gboolean LWValue::canBeIndexed ()
{
	return TRUE;
}

/*! \brief Pobierz indeks

	Pobiera wartość znajdującą sie pod indeksem
	\param n Indeks do pobrania (liczony od 1)
	\return Pobrany indeks
*/
LWValue *LWValue::getIndex (guint n)
{
	g_return_val_if_fail (n > 0, new LWValue());
	g_return_val_if_fail (TRUE == canBeIndexed(), new LWValue());
	
	if (is_list == FALSE) 
		return (n == 1) ? new LWValue (pixmap) : new LWValue (); //this);
		
	LWValue *value = (LWValue *) g_slist_nth_data (list, n-1);
	return (value != NULL) ? new LWValue (value) : new LWValue();
}

/*! \brief Pobiera listę ikon

	Pobiera listę ikon znajdującą się w wartości
	\return Pobrana lista
*/
GSList *LWValue::getListPixmap ()
{
	MAKE_TWIN(this)
	
	if (twin->is_list == FALSE) 
		return g_slist_append (NULL, twin->pixmap);
	
	GSList *list_pixmap=NULL;
	
	for (GSList *l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		if (v->is_list == TRUE) break;
		list_pixmap = g_slist_append (list_pixmap, v->pixmap);
	}
	
	FREE_TWIN
	
	return list_pixmap;	
}


/*! \brief Pobranie piksmapy

	Pobranie piksmapy zawartej w wartości
	Wartość nie może być jest listą, lecz pojedynczą piksamapą,
	ewentualnie wartość może przechowywać listę jednoelementową
	\return Pobrana piksamapa
*/
LWPixmap *LWValue::getPixmap ()
{
LWPixmap *p=NULL;
	
	MAKE_TWIN(this)
	
	if (FALSE == twin->is_list) {
		p = twin->pixmap;
	} else
	if (twin->list != NULL)
		if (twin->list->next == NULL) {
			LWValue *v = (LWValue *) twin->list->data;
			if (FALSE == v->is_list)
				p = v->pixmap;
		}

	FREE_TWIN
	return p;
}

/*! \brief Ustawia wartość listy

	Ustawia wartość na podstawie innej wartości
	\param value Wartość na podstawie, której ma być dokonane podstawienie
*/
void LWValue::set (LWValue *value)
{
	g_return_if_fail (value != NULL);
	
	if (this == value) return;
	
	MAKE_TWIN(value)
	
	if (FALSE == twin->is_list) {
		pixmap = twin->pixmap;
		is_list = FALSE;
		FREE_TWIN
		return;
	}

	if (twin->list != NULL)
		if (twin->list->next == NULL) {
			LWValue *v = (LWValue *) twin->list->data;
			if (FALSE == v->is_list) {
				pixmap = v->pixmap;
				is_list = FALSE;
				FREE_TWIN
				return;
			}
		}
				
	clear();
	
	for (GSList *l = twin->list; l != NULL; l = l->next) {
		LWValue *v = (LWValue *) l->data;
		list = g_slist_append (list, 
			(gpointer) ((TRUE == isVariable()) ?
				new LWVariable(v) : new LWValue (v)));
	}
	
	FREE_TWIN
}

/*! \brief Pobranie kopi wartości

	Pobiera kopię wartości
	\return Pobrana wartość
*/
LWValue *LWValue::get()
{
	return new LWValue(this);
}

/*! \brief Zliczanie elementów w liści
	
	Zlicza ilość elementów w liście, jeśli wartość jest piksmapą
	zwracane jest 1.
	\return Ilość elementów w liście
*/
guint LWValue::count ()
{
	MAKE_TWIN(this)
	
	guint count = (twin->is_list == TRUE) ? g_slist_length (twin->list) : 1;
	
	FREE_TWIN
	
	return count;
}

/*! \brief Czy wartość stanowi zmienną

	Czy wartość stanowi zmienną
	\return Nie przeciążona metoda zawsze zwraca fałsz
*/
gboolean LWValue::isVariable ()
{
	return FALSE;
}

/*! \brief Czy wartość specjalna
	
	Czy wartość jest wartością specjalną, zwraca FAŁSZ.
	Metoda przeciążana w implementacji wartości specjalnej.

	\return PRAWDA jeśli wartość jest specjalna, w przeciwnym
	razie FAŁSZ
*/
gboolean LWValue::isSpecial ()
{
	return FALSE;
}
