//===-- InstrTypes.cpp - Implement Instruction subclasses -------*- C++ -*-===//
// 
//                     The LLVM Compiler Infrastructure
//
// This file was developed by the LLVM research group and is distributed under
// the University of Illinois Open Source License. See LICENSE.TXT for details.
// 
//===----------------------------------------------------------------------===//
//
// This file implements 
//
//===----------------------------------------------------------------------===//

#include "llvm/Instructions.h"
#include "llvm/Function.h"
#include "llvm/SymbolTable.h"
#include "llvm/Constant.h"
#include "llvm/Type.h"
#include <algorithm>  // find
using namespace llvm;

//===----------------------------------------------------------------------===//
//                            TerminatorInst Class
//===----------------------------------------------------------------------===//

TerminatorInst::TerminatorInst(Instruction::TermOps iType, Instruction *IB) 
  : Instruction(Type::VoidTy, iType, "", IB) {
}

TerminatorInst::TerminatorInst(Instruction::TermOps iType, BasicBlock *IAE)
  : Instruction(Type::VoidTy, iType, "", IAE) {
}



//===----------------------------------------------------------------------===//
//                               PHINode Class
//===----------------------------------------------------------------------===//

PHINode::PHINode(const PHINode &PN)
  : Instruction(PN.getType(), Instruction::PHI) {
  Operands.reserve(PN.Operands.size());
  for (unsigned i = 0; i < PN.Operands.size(); i+=2) {
    Operands.push_back(Use(PN.Operands[i], this));
    Operands.push_back(Use(PN.Operands[i+1], this));
  }
}

// removeIncomingValue - Remove an incoming value.  This is useful if a
// predecessor basic block is deleted.
Value *PHINode::removeIncomingValue(unsigned Idx, bool DeletePHIIfEmpty) {
  assert(Idx*2 < Operands.size() && "BB not in PHI node!");
  Value *Removed = Operands[Idx*2];
  Operands.erase(Operands.begin()+Idx*2,     // Erase Value and BasicBlock
                 Operands.begin()+Idx*2+2);

  // If the PHI node is dead, because it has zero entries, nuke it now.
  if (getNumOperands() == 0 && DeletePHIIfEmpty) {
    // If anyone is using this PHI, make them use a dummy value instead...
    replaceAllUsesWith(Constant::getNullValue(getType()));
    getParent()->getInstList().erase(this);
  }
  return Removed;
}
