/*
 * midi_file.cpp - support for importing/exporting MIDI-files
 *
 * Copyright (c) 2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * This file partly contains code from Fluidsynth, Peter Hanappe
 *
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "midi_file.h"
#include "track_container.h"
#include "channel_track.h"
#include "pattern.h"


#ifdef QT4

#include <QMessageBox>
#include <Qt/QtXml>
#include <QApplication>
#include <QProgressDialog>

#else

#include <qmessagebox.h>
#include <qdom.h>
#include <qapplication.h>
#include <qprogressdialog.h>

#define ungetChar ungetch
#define seek at 
#define pos at
#define setValue setProgress
#define fileName name

#endif

#define makeID(_c0, _c1, _c2, _c3) \
	( ( _c0 ) | ( ( _c1 ) << 8 ) | ( ( _c2 ) << 16 ) | ( ( _c3 ) << 24 ) )


midiFile::midiFile( const QString & _file ) :
	m_file( _file )
{
}




midiFile::~midiFile()
{
}




void FASTCALL midiFile::importToTrackContainer( trackContainer * _tc )
{
#ifdef QT4
	if( m_file.open( QFile::ReadOnly ) == FALSE )
#else
	if( m_file.open( IO_ReadOnly ) == FALSE )
#endif
	{
		QMessageBox::critical( NULL,
			trackContainer::tr( "Could not open file" ),
			trackContainer::tr( "Could not open file %1 "
						"for reading.\nPlease make "
						"sure you have read-"
						"permission to the file and "
						"the directory containing the "
						"file and try again!" ).arg(
							m_file.fileName() ),
					QMessageBox::Ok,
					QMessageBox::NoButton );
		return;
	}

	switch( readID() )
	{
		case makeID( 'M', 'T', 'h', 'd' ):
			readSMF( _tc );
			break;

		case makeID( 'R', 'I', 'F', 'F' ):
			readRIFF( _tc );
			break;

		default:
			printf( "midiFile::importToTrackContainer(): not a "
						"Standard MIDI file\n" );
			break;
	}
}




void FASTCALL midiFile::exportFromTrackContainer( const trackContainer * _tc )
{
#ifdef QT4
	if( m_file.open( QFile::WriteOnly | QFile::Truncate ) == FALSE )
#else
	if( m_file.open( IO_WriteOnly | IO_Truncate ) == FALSE )
#endif
	{
		QMessageBox::critical( NULL,
			trackContainer::tr( "Could not open file" ),
			trackContainer::tr( "Could not open file %1 "
						"for writing.\nPlease make "
						"sure you have write-"
						"permission to the file and "
						"the directory containing the "
						"file and try again!" ).arg(
							m_file.fileName() ),
					QMessageBox::Ok,
					QMessageBox::NoButton );
		return;
	}
}




bool FASTCALL midiFile::readSMF( trackContainer * _tc )
{
	// the curren position is immediately after the "MThd" id
	int header_len = readInt( 4 );
	if( header_len < 6 )
	{
invalid_format:
		printf( "midiFile::readSMF(): invalid file format\n" );
		return( FALSE );
	}

	int type = readInt( 2 );
	if( type != 0 && type != 1 )
	{
		printf( "midiFile::readSMF(): type %d format is not "
							"supported\n", type );
		return( FALSE );
	}

	int num_tracks = readInt( 2 );
	if( num_tracks < 1 || num_tracks > 1000 )
	{
		printf( "midiFile::readSMF(): invalid number of tracks (%d)\n",
								num_tracks );
		num_tracks = 0;
		return( FALSE );
	}
#ifdef LMMS_DEBUG
	printf( "tracks: %d\n", num_tracks );
#endif

	int time_division = readInt( 2 );
	if( time_division < 0 )
	{
		goto invalid_format;
	}
#ifdef LMMS_DEBUG
	printf( "time-division: %d\n", time_division );
#endif
	m_smpteTiming = !!( time_division & 0x8000 );

#ifdef QT4
	QProgressDialog pd( trackContainer::tr( "Importing MIDI-file..." ),
				trackContainer::tr( "Cancel" ), 0, num_tracks );
#else
	QProgressDialog pd( trackContainer::tr( "Importing MIDI-file..." ),
				trackContainer::tr( "Cancel" ), num_tracks,
								0, 0, TRUE );
#endif
	pd.setWindowTitle( trackContainer::tr( "Please wait..." ) );
	pd.show();

        // read tracks
	for( int i = 0; i < num_tracks; ++i )
	{
		pd.setValue( i );
#ifdef QT4
		qApp->processEvents( QEventLoop::AllEvents, 100 );
#else
		qApp->processEvents( 100 );
#endif

		if( pd.wasCanceled() )
		{
			return( FALSE );
		}

		int len;

                // search for MTrk chunk
		while( 1 )
		{
			Sint32 id = readID();
			len = readInt( 4 );
			if( m_file.atEnd() )
			{
				printf( "midiFile::readSMF(): unexpected end "
								"of file\n" );
				return( FALSE );
			}
			if( len < 0 || len >= 0x10000000 )
			{
				printf( "midiFile::readSMF(): invalid chunk "
							"length %d\n", len );
				return( FALSE );
			}
			if( id == makeID( 'M', 'T', 'r', 'k' ) )
			{
				break;
			}
			skip( len );
		}
		if( len <= 0 )
		{
			continue;
		}

		if( !readTrack( m_file.pos() + len ) )
		{
			return( FALSE );
		}
		if( i == 0 )
		{
			continue;
		}

		// now create new channel-track for reading track
		channelTrack * ct = dynamic_cast<channelTrack *>(
						track::create(
							track::CHANNEL_TRACK,
								_tc ) );
#ifdef LMMS_DEBUG
		assert( ct != NULL );
#endif
		// TODO: load midi-out instead and setup program, channel etc.
		ct->loadInstrument( "tripleoscillator" );
		ct->toggledChannelButton( FALSE );

		// now create pattern to store notes in
		pattern * p = dynamic_cast<pattern *>( ct->createTCO( 0 ) );
#ifdef LMMS_DEBUG
		assert( p != NULL );
#endif
		ct->addTCO( p );

		// init keys
		int keys[NOTES_PER_OCTAVE * OCTAVES][2];
		for( int j = 0; j < NOTES_PER_OCTAVE * OCTAVES; ++j )
		{
			keys[j][0] = -1;
		}

		// now process every event
		for( eventVector::const_iterator it = m_events.begin();
						it != m_events.end(); ++it )
		{
			const int tick = it->first;
			const midiEvent & ev = it->second;
			switch( ev.m_type )
			{
				case NOTE_ON:
					if( ev.key() >=
						NOTES_PER_OCTAVE * OCTAVES )
					{
						continue;
					}
					if( ev.velocity() > 0 )
					{
						keys[ev.key()][0] = tick;
						keys[ev.key()][1] =
								ev.velocity();
						break;
					}

				case NOTE_OFF:
					if( ev.key() <
						NOTES_PER_OCTAVE * OCTAVES &&
							keys[ev.key()][0] >= 0 )
					{
			note n( midiTime( ( tick - keys[ev.key()][0] ) / 10 ),
				midiTime( keys[ev.key()][0] / 10 ),
				(tones)( ev.key() % NOTES_PER_OCTAVE ),
				(octaves)( ev.key() / NOTES_PER_OCTAVE ),
				keys[ev.key()][1] * 100 / 128 );
						p->addNote( n );
						keys[ev.key()][0] = -1;
					}
					break;

				default:
/*					printf( "Unhandled event: %#x\n",
								ev.m_type );*/
					break;
			}
		}
        }
        return( TRUE );
}




bool FASTCALL midiFile::readRIFF( trackContainer * _tc )
{
        // skip file length
	skip( 4 );

        // check file type ("RMID" = RIFF MIDI)
        if( readID() != makeID( 'R', 'M', 'I', 'D' ) )
	{
invalid_format:
                printf( "midiFile::readRIFF(): invalid file format\n" );
                return( FALSE );
        }
        // search for "data" chunk
        while( 1 )
	{
                int id = readID();
                int len = read32LE();
                if( m_file.atEnd() )
		{
data_not_found:
                        printf( "midiFile::readRIFF(): data chunk not "
								"found\n" );
                        return( FALSE );
                }
                if( id == makeID( 'd', 'a', 't', 'a' ) )
		{
                        break;
		}
                if( len < 0 )
		{
                        goto data_not_found;
		}
                skip( ( len + 1 ) & ~1 );
        }
        // the "data" chunk must contain data in SMF format
        if( readID() != makeID( 'M', 'T', 'h', 'd' ) )
	{
                goto invalid_format;
	}
        return( readSMF( _tc ) );
}




bool FASTCALL midiFile::readTrack( int _track_end )
{
        int tick = 0;
        unsigned char last_cmd = 0;
//        unsigned char port = 0;

	m_events.clear();
        // the current file position is after the track ID and length
        while( (int) m_file.pos() < _track_end )
	{
		unsigned char cmd;
		int len;

		int delta_ticks = readVar();
		if( delta_ticks < 0 )
		{
			break;
		}
		tick += delta_ticks;

		int c = readByte();
		if( c < 0 )
		{
			break;
		}
		if( c & 0x80 )
		{
			// have command
			cmd = c;
			if( cmd < 0xf0 )
			{
				last_cmd = cmd;
			}
		}
		else
		{
			// running status
			m_file.ungetChar( c );
			cmd = last_cmd;
			if( !cmd )
			{
				error();
				return( FALSE );
			}
		}
                switch( cmd & 0xF0 )
		{
			// channel msg with 2 parameter bytes
			case NOTE_OFF:
			case NOTE_ON:
			case KEY_PRESSURE:
			case CONTROL_CHANGE:
			case PITCH_BEND:
			{
				int data1 = readByte() & 0x7F;
				int data2 = readByte() & 0x7F;
				m_events.push_back( qMakePair( tick,
					midiEvent( static_cast<midiEventTypes>(
								cmd & 0xF0 ),
							cmd & 0x0F,
							data1,
							data2 ) ) );
				break;
			}
			// channel msg with 1 parameter byte
			case PROGRAM_CHANGE:
			case CHANNEL_PRESSURE:
				m_events.push_back( qMakePair( tick,
					midiEvent( static_cast<midiEventTypes>(
								cmd & 0xF0 ),
							cmd & 0x0F,
							readByte() & 0x7F ) ) );
				break;

			case MIDI_SYSEX:
				switch( cmd )
				{
					case MIDI_SYSEX:
					case MIDI_EOX:
					{
						len = readVar();
						if( len < 0 )
						{
							error();
							return( FALSE );
						}
						if( cmd == MIDI_SYSEX )
						{
							++len;
						}
						char * data = new char[len];
						if( cmd == MIDI_SYSEX )
						{
							data[0] = MIDI_SYSEX;
						}
						for( ; c < len; ++c )
						{
							data[c] = readByte();
						}
						m_events.push_back(
							qMakePair( tick,
				midiEvent( MIDI_SYSEX, data, len ) ) );
						break;
					}

					case MIDI_META_EVENT:
						c = readByte();
						len = readVar();
/*						if( len < 0 )
						{
							error();
							return( FALSE );
						}*/
						switch( c )
						{
						case 0x21: // port number
							if( len < 1 )
							{
								error();
								return( FALSE );
							}
/*							port = readByte() %
								port_count;
							skip( len - 1 );*/
							skip( len );
							break;

						case 0x2F: // end of track
						//track->end_tick = tick;
							skip( _track_end -
								m_file.pos() );
							return( TRUE );

						case 0x51: // tempo
							if( len < 3 )
							{
								error();
								return( FALSE );
							}
							if( m_smpteTiming )
							{
								// SMPTE timing
								// doesnt change
								skip( len );
							}
							else
							{
/*			event = new_event(track, 0);
			event->type = SND_SEQ_EVENT_TEMPO;
			event->port = port;
			event->tick = tick;
			event->data.tempo = read_byte() << 16;
			event->data.tempo |= read_byte() << 8;
			event->data.tempo |= read_byte();
			skip( len -3 );*/
								skip( len );
							}
							break;

						default:// ignore all other
							// meta events
							skip( len );
							break;
						}
						break;

					default: // invalid Fx command
						error();
						return( FALSE );
				}
				break;

			default: // cannot happen
                	        error();
				return( FALSE );
                }
        }
	error();
	return( FALSE );
}




void midiFile::error( void )
{
	printf( "midiFile::readTrack(): invalid MIDI data (offset %#x)\n",
						(unsigned int) m_file.pos() );
}




#undef ungetChar
#undef seek
#undef pos
#undef setValue
#undef fileName
