/*
 * channel_track.cpp - implementation of channel-track-class
 *                     (window + data-structures)
 *
 * Copyright (c) 2004-2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "qt3support.h"

#ifdef QT4

#include <QFileDialog>
#include <QDir>
#include <QFile>
#include <QApplication>
#include <Qt/QtXml>
#include <QLineEdit>
#include <QCloseEvent>
#include <QLayout>
#include <QLabel>

#else

#include <qfiledialog.h>
#include <qdir.h>
#include <qfile.h>
#include <qapplication.h>
#include <qdom.h>
#include <qlineedit.h>
#include <qbuttongroup.h>
#include <qlayout.h>
#include <qlabel.h>

#define setChecked setOn
#define isChecked isOn

#endif


#include "channel_track.h"
#include "instrument.h"
#include "song_editor.h"
#include "effect_board.h"
#include "envelope_tab_widget.h"
#include "arp_and_chords_tab_widget.h"
#include "midi_tab_widget.h"
#include "embed.h"
#include "note_play_handle.h"
#include "paths.h"
#include "surround_area.h"
#include "piano_widget.h"
#include "pixmap_button.h"
#include "knob.h"
#include "mmp.h"
#include "midi_client.h"
#include "midi_port.h"
#include "pattern.h"
#include "config_mgr.h"
#include "debug.h"
#include "lcd_spinbox.h"
#include "buffer_allocator.h"
#include "tooltip.h"
#include "tab_widget.h"
#include "string_pair_drag.h"
#include "audio_port.h"


const char * volume_help = QT_TRANSLATE_NOOP( "channelTrack",
						"With this knob you can set "
						"the volume of the opened "
						"channel.");
const char * surroundarea_help = QT_TRANSLATE_NOOP( "channelTrack",
						"Within this rectangle you can "
						"set the position where the "
						"channel should be audible. "
						"You should have a soundcard "
						"supporting at least surround "
						"4.0 for enjoying this "
						"feature." );


const int CHANNEL_WIDTH		= 250;
const int INSTRUMENT_WIDTH	= CHANNEL_WIDTH;
const int INSTRUMENT_HEIGHT	= INSTRUMENT_WIDTH;
const int PIANO_HEIGHT		= 84;



channelTrack::channelTrack( trackContainer * _tc ) :
	QWidget( lmmsMainWin::inst()->workspace() ),
 	track( _tc ),
	midiEventProcessor(),
	m_trackType( CHANNEL_TRACK ),
	m_midiPort( mixer::inst()->getMIDIClient()->addPort( this,
						tr( "unnamed_channel" ) ) ),
	m_audioPort( new audioPort( tr( "unnamed_channel" ) ) ),
	m_baseTone( A ),
	m_baseOctave( OCTAVE_4 ),
	m_instrument( NULL )
{
	for( int i = 0; i < NOTES_PER_OCTAVE * OCTAVES; ++i )
	{
		m_notes[i] = NULL;
	}


#ifdef QT4
	lmmsMainWin::inst()->workspace()->addWindow( this );
#endif

	setAcceptDrops( TRUE );

	hide();

	getTrackWidget()->setFixedHeight( 32 );


	// creation of widgets for track-settings-widget
	m_tswVolumeKnob = new knob( knobSmall_17, getTrackSettingsWidget(),
						tr( "Channel volume" ) );
	m_tswVolumeKnob->setRange( MIN_VOLUME, MAX_VOLUME, 1.0f );
	m_tswVolumeKnob->setValue( DEFAULT_VOLUME, TRUE );
	m_tswVolumeKnob->setHintText( tr( "Channel volume:" ) + " ", "%" );
	m_tswVolumeKnob->move( 4, 4 );
	m_tswVolumeKnob->setLabel( tr( "VOL" ) );
	m_tswVolumeKnob->show();
	connect( m_tswVolumeKnob, SIGNAL( valueChanged( float ) ), this,
					SLOT( volValueChanged( float ) ) );
#ifdef QT4
	m_tswVolumeKnob->setWhatsThis(
#else
	QWhatsThis::add( m_tswVolumeKnob,
#endif
		tr( volume_help ) );

	m_tswSurroundArea = new surroundArea( getTrackSettingsWidget() );
	m_tswSurroundArea->move( 32, 2 );
	m_tswSurroundArea->setFixedSize( 28, 28 );
	m_tswSurroundArea->show();
	connect( m_tswSurroundArea, SIGNAL( valueChanged( const QPoint & ) ),
			this,
			SLOT( surroundAreaPosChanged( const QPoint & ) ) );
#ifdef QT4
	m_tswSurroundArea->setWhatsThis(
#else
	QWhatsThis::add( m_tswSurroundArea,
#endif
		tr( surroundarea_help ) );

	m_tswChannelButton = new channelButton( this );
	m_tswChannelButton->setCheckable( TRUE );
	m_tswChannelButton->setGeometry( 64, 2, 156, 28 );
	m_tswChannelButton->show();
	connect( m_tswChannelButton, SIGNAL( toggled( bool ) ), this,
					SLOT( toggledChannelButton( bool ) ) );



	// init own layout + widgets
#ifdef QT4
	setFocusPolicy( Qt::StrongFocus );
#else
	setFocusPolicy( StrongFocus );
#endif
	QVBoxLayout * vlayout = new QVBoxLayout( this );
	vlayout->setMargin( 0 );
	vlayout->setSpacing( 0 );

	m_generalSettingsWidget = new tabWidget( tr( "GENERAL SETTINGS" ),
									this );
	m_generalSettingsWidget->setFixedHeight( 90 );

	// setup line-edit for changing channel-name
	m_channelNameLE = new QLineEdit( m_generalSettingsWidget );
	m_channelNameLE->setFont( pointSize<8>( m_channelNameLE->font() ) );
	m_channelNameLE->setGeometry( 10, 16, 230, 18 );
	connect( m_channelNameLE, SIGNAL( textChanged( const QString & ) ),
				this, SLOT( textChanged( const QString & ) ) );


	// setup volume-knob
	m_volumeKnob = new knob( knobBright_26, m_generalSettingsWidget,
						tr( "Channel volume" ) );
	m_volumeKnob->move( 10, 44 );
	m_volumeKnob->setRange( MIN_VOLUME, MAX_VOLUME, 1.0f );
	m_volumeKnob->setValue( DEFAULT_VOLUME, TRUE );
	m_volumeKnob->setHintText( tr( "Channel volume:" ) + " ", "%" );
	m_volumeKnob->setLabel( tr( "VOLUME" ) );

#ifdef QT4
	m_volumeKnob->setWhatsThis(
#else
	QWhatsThis::add( m_volumeKnob,
#endif
		tr( volume_help ) );
	connect( m_volumeKnob, SIGNAL( valueChanged( float ) ), this,
					SLOT( volValueChanged( float ) ) );


	// setup surround-area
	m_surroundArea = new surroundArea( m_generalSettingsWidget );
	m_surroundArea->move( 20 + m_volumeKnob->width(), 38 );
	m_surroundArea->show();
#ifdef QT4
	m_surroundArea->setWhatsThis(
#else
	QWhatsThis::add( m_surroundArea,
#endif
		tr( surroundarea_help ) );

	connect( m_surroundArea, SIGNAL( valueChanged( const QPoint & ) ),
			this,
			SLOT( surroundAreaPosChanged( const QPoint & ) ) );



	// setup spinbox for selecting FX-channel
	m_effectChannelNumber = new lcdSpinBox( MIN_EFFECT_CHANNEL,
						MAX_EFFECT_CHANNEL, 2,
						m_generalSettingsWidget );
	m_effectChannelNumber->setValue( DEFAULT_EFFECT_CHANNEL );
	m_effectChannelNumber->setLabel( tr( "FX CHNL" ) );
	m_effectChannelNumber->move( m_surroundArea->x() +
					m_surroundArea->width() + 16, 40 );


	m_saveSettingsBtn = new QPushButton( embed::getIconPixmap(
							"project_save" ), "",
						m_generalSettingsWidget );
	m_saveSettingsBtn->setGeometry( m_effectChannelNumber->x() +
					m_effectChannelNumber->width() + 20, 40,
					32, 32 );
	connect( m_saveSettingsBtn, SIGNAL( clicked() ), this,
					SLOT( saveSettingsBtnClicked() ) );
	toolTip::add( m_saveSettingsBtn, 
		tr( "Save current channel settings in a preset-file" ) );
#ifdef QT4
	m_saveSettingsBtn->setWhatsThis(
#else
	QWhatsThis::add( m_saveSettingsBtn,
#endif
		tr( "Click here, if you want to save current channel settings "
			"in a preset-file. Later you can load this preset by "
			"double-clicking it in the preset-browser." ) );


	setVolume( DEFAULT_VOLUME );
	setSurroundAreaPos( QPoint() );
	setName( tr( "Default" ) );


	m_tabWidget = new tabWidget( "", this );
	m_tabWidget->setFixedHeight( INSTRUMENT_HEIGHT + 12 );


	// create other tab-widgets
	m_envWidget = new envelopeTabWidget( this );
	m_arpWidget = new arpAndChordsTabWidget( this );
	m_midiWidget = new midiTabWidget( this, m_midiPort );
	m_tabWidget->addTab( m_envWidget, tr( "ENV/LFO/FILTER" ), 1 );
	m_tabWidget->addTab( m_arpWidget, tr( "ARP/CHORD" ), 2 );
	m_tabWidget->addTab( m_midiWidget, tr( "MIDI" ), 3 );

	// setup piano-widget
	m_pianoWidget = new pianoWidget( this );
	m_pianoWidget->setFixedSize( CHANNEL_WIDTH, PIANO_HEIGHT );


	vlayout->addWidget( m_generalSettingsWidget );
	vlayout->addWidget( m_tabWidget );
	vlayout->addWidget( m_pianoWidget );


	// set window-icon
	setWindowIcon( embed::getIconPixmap( "channel_track" ) );


	_tc->updateAfterTrackAdd();
}






channelTrack::~channelTrack()
{
	invalidateAllMyNPH();
	delete m_audioPort;
	mixer::inst()->getMIDIClient()->removePort( m_midiPort );
}







void channelTrack::saveSettingsBtnClicked( void )
{
#ifdef QT4
	QFileDialog sfd( this, tr( "Save channel-settings in file" ), "",
				tr( "Channel-Settings-File (*.cs.xml)" ) );
#else
	QFileDialog sfd( this, "", TRUE );
	sfd.setWindowTitle( tr( "Save channel-settings in file" ) );
	sfd.setFilter( tr( "Channel-Settings-File (*.cs.xml)" ) );
#endif

	QString preset_root = configManager::inst()->presetsDir();
	if( !QDir( preset_root ).exists() )
	{
		QDir().mkdir( preset_root );
	}
	if( !QDir( preset_root + pluginName() ).exists() )
	{
		QDir( preset_root ).mkdir( pluginName() );
	}

	sfd.setDirectory( preset_root + pluginName() );
	sfd.setFileMode( QFileDialog::AnyFile );
	if( sfd.exec () == QDialog::Accepted &&
#ifdef QT4
		!sfd.selectedFiles().isEmpty() && sfd.selectedFiles()[0] != ""
#else
		sfd.selectedFile() != ""
#endif
	)
	{
		multimediaProject mmp( multimediaProject::CHANNEL_SETTINGS );
		saveTrackSpecificSettings( mmp, mmp.content() );
#ifdef QT4
		mmp.writeFile( sfd.selectedFiles()[0] );
#else
		mmp.writeFile( sfd.selectedFile() );
#endif
	}
}




float channelTrack::frequency( notePlayHandle * _n ) const
{
	return( BASE_FREQ * powf( 2.0f, (float)( _n->tone() - m_baseTone +
			songEditor::inst()->masterPitch() ) / 12.0f +
				(float)( _n->octave() - m_baseOctave ) ) );
}




Uint32 channelTrack::beatLen( notePlayHandle * _n ) const
{
#ifdef LMMS_DEBUG
	assert( m_instrument != NULL );
#endif
	Uint32 len = m_instrument->beatLen( _n );
	if( len > 0 )
	{
		return( len );
	}
	return( m_envWidget->envFrames() );
}






void channelTrack::processAudioBuffer( sampleFrame * _buf, Uint32 _frames,
							notePlayHandle * _n )
{
	// we may not play the sound if this channelTrack is muted...
	if( muted() )
	{
		return;
	}
	float v_scale = (float) getVolume() / DEFAULT_VOLUME;

	// instruments using instrument-play-handles will call this method
	// without any knowledge about notes, so they pass NULL for _n, whicj
	// is no problem for us since we just bypass the envelopes+LFO's
	if( _n != NULL )
	{
		m_envWidget->processAudioBuffer( _buf, _frames, _n );
		v_scale *= ( (float) _n->getVolume() / DEFAULT_VOLUME );
	}

	volumeVector v = m_surroundArea->getVolumeVector( v_scale );

	mixer::inst()->bufferToPort( _buf, _frames,
				( _n != NULL ) ? _n->framesAhead() : 0, v,
								m_audioPort );
}




void channelTrack::processInEvent( const midiEvent & _me,
							const midiTime & _time )
{
	switch( _me.m_type )
	{
		case NOTE_ON: 
			if( _me.velocity() > 0 )
			{
				if( m_notes[_me.key()] == NULL )
				{
					// create temporary note
					note n;
					n.setKey( _me.key() );
					n.setVolume( _me.velocity() * 100 /
									127 );
					// create (timed) note-play-handle
					notePlayHandle * nph = new
						notePlayHandle( this,
			_time.frames( songEditor::inst()->framesPerTact() ),
								~0, &n );
					mixer::inst()->addPlayHandle( nph );
					m_notes[_me.key()] = nph;
				}
				break;
			}

		case NOTE_OFF:
			if( m_notes[_me.key()] != NULL )
			{
				notePlayHandle * n = m_notes[_me.key()];
				// create dummy-note which has the same length
				// as the played note for sending it later
				// to all slots connected to signal noteDone()
				// this is for example needed by piano-roll for
				// recording notes into a pattern
				note done_note( midiTime( static_cast<Uint32>(
					n->totalFramesPlayed() * 64 /
					songEditor::inst()->framesPerTact() ) ),
					0, n->tone(), n->octave(),
					n->getVolume(), n->getPanning() );
				n->noteOff();
				m_notes[_me.key()] = NULL;
				emit noteDone( done_note );
			}
			break;

		case KEY_PRESSURE:
			if( m_notes[_me.key()] != NULL )
			{
				m_notes[_me.key()]->setVolume( _me.velocity() *
								100 / 128 );
			}
			break;

/*		case PITCH_BEND:
			if( m_pitchBendKnob != NULL )
			{
				float range = tAbs(
						m_pitchBendKnob->maxValue() -
						m_pitchBendKnob->minValue() );
				m_pitchBendKnob->setValue(
						m_pitchBendKnob->minValue() +
						_me.m_data.m_param[0] *
								range / 16384 );
			}
			break;*/

		default:
			printf( "channel-track: unhandled MIDI-event %d\n",
								_me.m_type );
			break;
	}
}




void channelTrack::processOutEvent( const midiEvent & _me,
							const midiTime & _time )
{
	// if appropriate, midi-port does futher routing
	m_midiPort->processOutEvent( _me, _time );
}




void channelTrack::playNote( notePlayHandle * _n )
{
	// arpeggio- and chord-widget has to do its work, that means
	// adding sub-notes for chords/arpeggios
	m_arpWidget->processNote( _n );

	if( _n->arpBaseNote() == FALSE )
	{
#ifdef LMMS_DEBUG
		assert( m_instrument != NULL );
#endif
		// all is done, so now lets play the note!
		m_instrument->playNote( _n );
	}
}




const QString & channelTrack::pluginName( void ) const
{
	if( m_instrument != NULL )
	{
		return( m_instrument->publicName() );
	}
	static const QString empty = "";
	return( empty );
}




void channelTrack::deleteNotePluginData( notePlayHandle * _n )
{
#ifdef LMMS_DEBUG
	assert( m_instrument != NULL );
#endif
	m_instrument->deleteNotePluginData( _n );
}




void channelTrack::setName( const QString & _new_name )
{
	// when changing name of channel, also change name of those patterns,
	// which have the same name as the channel
	for( csize i = 0; i < numOfTCOs(); ++i )
	{
		pattern * p = dynamic_cast<pattern *>( getTCO( i ) );
		if( p != NULL && p->name() == m_name || p->name() == "" )
		{
			p->setName( _new_name );
		}
	}

	m_name = _new_name;
	setWindowTitle( m_name );

	if( m_channelNameLE->text() != _new_name )
	{
		m_channelNameLE->setText( m_name );
	}
#ifdef LMMS_DEBUG
	assert( m_tswChannelButton != NULL );
#endif
	m_tswChannelButton->setText( m_name );
	m_midiPort->setName( m_name );
	m_audioPort->setName( m_name );
}




void channelTrack::setVolume( volume _new_volume )
{
	if( _new_volume <= MAX_VOLUME )
	{
		m_volumeKnob->setValue( _new_volume );
		m_tswVolumeKnob->setValue( _new_volume );
	}
}




volume channelTrack::getVolume( void ) const
{
	return( static_cast<volume>( m_volumeKnob->value() ) );
}




void channelTrack::setSurroundAreaPos( const QPoint & _p )
{
	if( m_surroundArea->value() != _p )
	{
		m_surroundArea->setValue( _p );
	}
	if( m_tswSurroundArea->value() != _p )
	{
		m_tswSurroundArea->setValue( _p );
	}
}




const QPoint & channelTrack::surroundAreaPos( void ) const
{
	return( m_surroundArea->value() );
}




void channelTrack::setBaseTone( tones _new_tone )
{
	if( _new_tone >= C && _new_tone <= H )
	{
		m_baseTone = _new_tone;
	}
	songEditor::inst()->setModified();
}




void channelTrack::setBaseOctave( octaves _new_octave )
{
	if( _new_octave >= MIN_OCTAVE && _new_octave <= MAX_OCTAVE )
	{
		m_baseOctave = _new_octave;
	}
	songEditor::inst()->setModified();
}




int channelTrack::masterKey( notePlayHandle * _n ) const
{
	int key = baseTone() + baseOctave() * NOTES_PER_OCTAVE +
					songEditor::inst()->masterPitch();
	return( _n->key() - ( A + DEFAULT_OCTAVE * NOTES_PER_OCTAVE - key ) );
}




bool FASTCALL channelTrack::play( const midiTime & _start, Uint32 _start_frame,
					Uint32 _frames, Uint32 _frame_base,
							Sint16 _tco_num )
{
	// calculate samples per tact; need that later when calculating
	// sample-pos of a note
	float frames_per_tact = songEditor::inst()->framesPerTact();

	vlist<trackContentObject *> tcos;
	if( _tco_num >= 0 )
	{
		tcos.push_back( getTCO( _tco_num ) );
	}
	else
	{
		getTCOsInRange( tcos, _start, _start +
				static_cast<Sint32>( _frames * 64 /
							frames_per_tact ) );
	}
	
	if ( tcos.size() == 0 )
	{
		return( FALSE );
	}

	bool played_a_note = FALSE;	// will be return variable

	// calculate the end of the current sample-frame
	const Uint32 end_frame = _start_frame+_frames-1;

	for( vlist<trackContentObject *>::iterator it = tcos.begin();
							it != tcos.end(); ++it )
	{
		pattern * p = dynamic_cast<pattern *>( *it );
		if( p == NULL )
		{
			continue;
		}
		midiTime cur_start = _start.getTact() * 64 -
					( ( _tco_num < 0 ) ?
						p->startPosition() :
							midiTime( 0 ) );
		if( p->frozen() && songEditor::inst()->exporting() == FALSE )
		{
			volumeVector v = m_surroundArea->getVolumeVector(
									1.0f );
			// volume-vector was already used when freezing
			// pattern, but only in stereo-mode, so front speakers
			// are already setup
			v.vol[0] = 1.0f;
			v.vol[1] = 1.0f;
			sampleFrame * buf = bufferAllocator::alloc<sampleFrame>(
								_frames );

			p->playFrozenData( buf, _start_frame +
						static_cast<Uint32>(
							cur_start.getTact() *
							frames_per_tact ),
						_frames );
			mixer::inst()->bufferToPort( buf, _frames, _frame_base +
				static_cast<Uint32>(
					p->startPosition().getTact64th() *
							frames_per_tact /
								64.0f ),
							v, m_audioPort );
			bufferAllocator::free( buf );
			continue;
		}

		// get all notes from the given pattern...
		noteVector & notes = p->notes();
		// ...and set our index to zero
		noteVector::iterator it = notes.begin();

		// very effective algorithm for playing notes that are
		// posated within the current sample-frame


		if( cur_start > 0 )
		{
			// skip notes which are posated before start-tact
			while( it != notes.end() && ( *it )->pos() < cur_start )
			{
				++it;
			}
		}

		// skip notes before sample-frame
		while( it != notes.end() &&
			( *it )->pos( cur_start ).frames( frames_per_tact ) <
								_start_frame )
		{
			++it;
		}

		note * cur_note;
		while( it != notes.end() &&
			( ( cur_note = *it )->pos( cur_start ).frames(
					frames_per_tact ) ) <= end_frame )
		{
			if( cur_note->length() != 0 )
			{
				const Uint32 frames_ahead = _frame_base -
								_start_frame +
			cur_note->pos( cur_start ).frames( frames_per_tact );

				const Uint32 note_frames =
					cur_note->length().frames(
							frames_per_tact );

/*				// generate according MIDI-events
				processOutEvent( midiEvent( NOTE_ON,
						m_midiPort->outputChannel(),
							cur_note->key(),
							cur_note->getVolume() *
								128 / 100 ),
						midiTime::fromFrames(
							frames_ahead,
							frames_per_tact ) );

				processOutEvent( midiEvent( NOTE_OFF,
						m_midiPort->outputChannel(),
							cur_note->key(), 0 ),
						midiTime::fromFrames(
							frames_ahead +
								note_frames,
							frames_per_tact ) );*/

				notePlayHandle * note_play_handle =
					new notePlayHandle( this,
								frames_ahead,
								note_frames,
								cur_note );
				note_play_handle->play();
				// could we play all within current number of
				// frames per audio-buffer?
				if( note_play_handle->done() == FALSE )
				{
					// no, then insert it into
					// play-handle-vector of mixer
					mixer::inst()->addPlayHandle(
							note_play_handle );
				}
				else
				{
					// otherwise just throw it away...
					delete note_play_handle;
				}
				played_a_note = TRUE;
			}
			++it;
		}
	}
	return( played_a_note );
}




trackContentObject * channelTrack::createTCO( const midiTime & )
{
	return( new pattern( this ) );
}




void channelTrack::saveTrackSpecificSettings( QDomDocument & _doc,
							QDomElement & _parent )
{
	QDomElement ct_de = _doc.createElement( nodeName() );
	ct_de.setAttribute( "name", name() );
	ct_de.setAttribute( "vol", QString::number( getVolume() ) );

	// make all coordinates positive
	unsigned int x = surroundAreaPos().x() + 2 * SURROUND_AREA_SIZE;
	unsigned int y = surroundAreaPos().y() + 2 * SURROUND_AREA_SIZE;
	ct_de.setAttribute( "surpos", QString::number(
				static_cast<unsigned int>( x & 0xFFFF ) +
							( y * 256 * 256 ) ) );

	ct_de.setAttribute( "fxch", QString::number(
					m_effectChannelNumber->value() ) );
	ct_de.setAttribute( "basetone", QString::number( m_baseTone ) );
	ct_de.setAttribute( "baseoct", QString::number( m_baseOctave ) );
	ct_de.setAttribute( "tab", QString::number(
						m_tabWidget->activeTab() ) );

#ifdef LMMS_DEBUG
	assert( m_instrument != NULL );
#endif
	m_instrument->saveSettings( _doc, ct_de );
	m_envWidget->saveSettings( _doc, ct_de );
	m_arpWidget->saveSettings( _doc, ct_de );
	m_midiWidget->saveSettings( _doc, ct_de );

	_parent.appendChild( ct_de );

}




void channelTrack::loadTrackSpecificSettings( const QDomElement & _this )
{
	//mixer::inst()->pause();

	invalidateAllMyNPH();

	setName( _this.attribute( "name" ) );
	setVolume( _this.attribute( "vol" ).toInt() );

	int i = _this.attribute( "surpos" ).toInt();
	setSurroundAreaPos( QPoint( ( i & 0xFFFF ) - 2 * SURROUND_AREA_SIZE,
			( i / ( 256 * 256 ) ) - 2 * SURROUND_AREA_SIZE) );

	m_effectChannelNumber->setValue( _this.attribute( "fxch" ).toInt() );
	m_baseTone = static_cast<tones>( _this.attribute(
							"basetone" ).toInt() );
	m_baseOctave = static_cast<octaves>( _this.attribute(
							"baseoct" ).toInt() );

	int tab = _this.attribute( "tab" ).toInt();

	QDomNode node = _this.firstChild();
	while( !node.isNull() )
	{
		if( node.isElement() )
		{
			if( m_envWidget->nodeName() == node.nodeName() )
			{
				m_envWidget->loadSettings( node.toElement() );
			}
			else if( m_arpWidget->nodeName() == node.nodeName() )
			{
				m_arpWidget->loadSettings( node.toElement() );
			}
			else if( m_midiWidget->nodeName() == node.nodeName() )
			{
				m_midiWidget->loadSettings( node.toElement() );
			}
			else
			{
				// if node-name doesn't match any known one,
				// we assume that it is an instrument-plugin
				// which we'll try to load
				delete m_instrument;
				m_instrument = instrument::instantiate(
							node.nodeName(), this );
				if( m_instrument->nodeName() ==
							node.nodeName() )
				{
					m_instrument->loadSettings(
							node.toElement() );
				}
				m_tabWidget->addTab( m_instrument,
							tr( "PLUGIN" ), 0 );
			}
		}
		node = node.nextSibling();
        }

	m_tabWidget->setActiveTab( tab );

	//mixer::inst()->play();

	m_pianoWidget->update();
	
}




instrument * channelTrack::loadInstrument( const QString & _plugin_name )
{
	invalidateAllMyNPH();

	delete m_instrument;
	m_instrument = instrument::instantiate( _plugin_name, this );

	m_tabWidget->addTab( m_instrument, tr( "PLUGIN" ), 0 );
	m_tabWidget->setActiveTab( 0 );

	m_tswChannelButton->update();

	return( m_instrument );
}




void channelTrack::volValueChanged( float _new_value )
{
	setVolume( (volume) _new_value );
}




void channelTrack::surroundAreaPosChanged( const QPoint & _p )
{
	setSurroundAreaPos( _p );
	songEditor::inst()->setModified();
}




void channelTrack::textChanged( const QString & _new_name )
{
	setName( _new_name );
	songEditor::inst()->setModified();
}




void channelTrack::toggledChannelButton( bool _on )
{
	if( m_tswChannelButton->isChecked() != _on )
	{
		m_tswChannelButton->setChecked( _on );
	}
	if( _on )
	{
		show();
		raise();
	}
	else
	{
		hide();
	}
}




void channelTrack::closeEvent( QCloseEvent * _ce )
{
	_ce->ignore();
	hide();
	m_tswChannelButton->setChecked( FALSE );
}




void channelTrack::focusInEvent( QFocusEvent * )
{
	m_pianoWidget->setFocus();
}




void channelTrack::dragEnterEvent( QDragEnterEvent * _dee )
{
	stringPairDrag::processDragEnterEvent( _dee, "instrument,presetfile" );
}




void channelTrack::dropEvent( QDropEvent * _de )
{
	QString type = stringPairDrag::decodeKey( _de );
	QString value = stringPairDrag::decodeValue( _de );
	if( type == "instrument" )
	{
		loadInstrument( value );
		songEditor::inst()->setModified();
		_de->accept();
	}
	else if( type == "presetfile" )
	{
		multimediaProject mmp( value );
		loadTrackSpecificSettings( mmp.content().firstChild().
								toElement() );
		songEditor::inst()->setModified();
		_de->accept();
	}
}




void channelTrack::invalidateAllMyNPH( void )
{
	// note-play-handles check track-type to determine whether their
	// channel-track is being deleted (if this is the case, they
	// invalidate themselves)
	m_trackType = NULL_TRACK;

	// invalidate all note-play-handles linked to this channel
	mixer::inst()->checkValidityOfPlayHandles();

	m_trackType = CHANNEL_TRACK;
}









channelButton::channelButton( channelTrack * _channel_track ) :
	QPushButton( _channel_track->getTrackSettingsWidget() ),
	m_channelTrack( _channel_track )
{
	setAcceptDrops( TRUE );
}




channelButton::~channelButton()
{
}




void channelButton::drawButtonLabel( QPainter * _p )
{
	QString pn = m_channelTrack->pluginName() + ":";
	int extra = isChecked() ? -1 : -3;
	_p->setFont( pointSize<7>( _p->font() ) );
#ifdef QT4
	_p->setPen( QApplication::palette().buttonText().color() );
#else
	_p->setPen( QApplication::palette().color( QPalette::Normal,
						QColorGroup::ButtonText ) );
#endif
	_p->drawText( ( width() - QFontMetrics( _p->font() ).width( pn ) ) / 2 +
					extra, height() / 2 + extra, pn );
	_p->setPen( QColor( 0, 0, 0 ) );
	_p->drawText( ( width() - QFontMetrics( _p->font() ).width( text() ) ) /
				2 + extra, height() / 2 +
				QFontMetrics( _p->font() ).height() + extra,
								text() );
}




void channelButton::dragEnterEvent( QDragEnterEvent * _dee )
{
	m_channelTrack->dragEnterEvent( _dee );
}




void channelButton::dropEvent( QDropEvent * _de )
{
	m_channelTrack->dropEvent( _de );
	setChecked( TRUE );
}



#undef setChecked
#undef isChecked




#include "channel_track.moc"

