# $Progeny: main.py,v 1.73 2002/06/11 19:08:47 branden Exp $

# Copyright (C) 2001 Progeny Linux Systems, Inc.
# Authors: Branden Robinson; Jeff Licquia; Eric Gillespie, Jr.
#
# This is free software; you may redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2,
# or (at your option) any later version.
#
# This is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with the Debian GNU/Linux system; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
# 02111-1307 USA

import configlet
import re

# list of XFree86 3.x servers
_xservers_v3 = [ "xserver-3dlabs", "xserver-8514",
                 "xserver-agx", "xserver-fbdev",
                 "xserver-i128", "xserver-mach32",
                 "xserver-mach64", "xserver-mach8",
                 "xserver-mono", "xserver-p9000",
                 "xserver-s3", "xserver-s3v",
                 "xserver-svga", "xserver-tga",
                 "xserver-vga16", "xserver-w32" ]

# location of glibc's supported locales file
SUPPORTED = "/usr/share/i18n/SUPPORTED"

class Kblang(configlet.Configlet):
    def __init__(self, attr):
        "Kblang class initialization method"
        # call base class ctor
        configlet.Configlet.__init__(self, attr)

        # leading part of debconf namespace for X server config
        self.configspace = ""

        # variables for what this configlet actually configures
        self.language = ""
        self.lang = ""
        self.select_lc_vars = [ ]
        self.lc_collate = ""
        self.lc_ctype = ""
        self.lc_messages = ""
        self.lc_monetary = ""
        self.lc_numeric = ""
        self.lc_time = ""
        self.keyboard = ""
        self.xkb_rules = ""
        self.xkb_model = ""
        self.xkb_layout = ""
        self.xkb_variant = ""
        self.xkb_options = ""
        self.use_lang = 1
        self.use_simple_keyboard = 1

        # list for simplified optionmenus
        self.keyboard_list = [ "English (U.S.)",
                               "English (U.K.)",
                               "English (Canada)",
                               "French (France)",
                               "French (Canada)",
                               "German",
                               "Spanish (Spain)",
                               "Spanish (Mexico)" ]

        # dictionary for language -> LANG
        # this is used as a two-way dictionary; the values must be
        # unique as well as the keys
        self.language_list = self.keyboard_list[:]
        self.language_values = [ "en_US ISO-8859-1",
                                 "en_GB ISO-8859-1",
                                 "en_CA ISO-8859-1",
                                 "fr_FR@euro ISO-8859-15",
                                 "fr_CA ISO-8859-1",
                                 "de_DE@euro ISO-8859-15",
                                 "es_ES@euro ISO-8859-15",
                                 "es_MX ISO-8859-1" ]

        self.language_dict = { }
        for i in range(len(self.language_list)):
            self.language_dict[self.language_list[i]] = self.language_values[i]
        # POSIX/C is used only for locales, not keyboards
        self.language_list.append("POSIX/C")
        self.language_dict["POSIX/C"] = "C"

        # dictionary for language -> XKB layout
        # XXX: these are not well-researched
        self.keyboard_values = [ "us",
                                 "gb",
                                 "ca",
                                 "fr",
                                 "ca",
                                 "de",
                                 "es",
                                 "us" ]
        self.keyboard_dict = { }
        for i in range(len(self.keyboard_list)):
            self.keyboard_dict[self.keyboard_list[i]] = self.keyboard_values[i]

        self.xkb_rules_list = [ "sun", "xfree86" ]
        self.xkb_model_list = [ "pc101", "pc102", "pc104", "pc105",
                                "macintosh", "macintosh_old", "type4",
                                "type5" ]
        self.xkb_layout_list = [ "us", "gb", "fr", "de", "es" ]
        self.xkb_variant_list = [ "", "nodeadkeys" ]
        self.xkb_options_list = [ "", "ctrl:nocaps", "ctrl:swapcaps" ]

        self.debug("self.keyboard_list \"%s\"." % (self.keyboard_list,))
        self.debug("self.language_list \"%s\"." % (self.language_list,))
        self.debug("self.keyboard_dict \"%s\"." % (self.keyboard_dict,))
        self.debug("self.language_dict \"%s\"." % (self.language_dict,))

        self.debug("Building self.locale_list.")

        # XXX: use everything from the SUPPORTED file until Configlet 2.0 comes
        # out and we can just grab the Choices field; this list is used only by
        # the advanced language combo boxes
        self.locale_list = [ "C" ]

        try:
            supported_file = open(SUPPORTED)
        except IOError, e:
            self.warn("Cannot open list of supported locales: %s" % (SUPPORTED,))
        else:
            for line in supported_file.readlines():
                line.strip()
                if re.match('^#', line):
                    # comment line
                    self.debug("skipped %s" % (line,))
                    pass
                elif line == "":
                    # blank line
                    self.debug("skipped blank line")
                    pass
                else:
                    # real beef
                    self.locale_list.append(line.strip())
            supported_file.close()

        self.debug("self.locale_list is %s" % (self.locale_list,))

    # widget callback functions

    def on_advanced_button_clicked(self, button):
        "advanced_button click event handler"
        self.wtree.get_widget("lc_collate_combo_entry").set_text(self.lc_collate)
        self.wtree.get_widget("lc_ctype_combo_entry").set_text(self.lc_ctype)
        self.wtree.get_widget("lc_messages_combo_entry").set_text(self.lc_messages)
        self.wtree.get_widget("lc_monetary_combo_entry").set_text(self.lc_monetary)
        self.wtree.get_widget("lc_numeric_combo_entry").set_text(self.lc_numeric)
        self.wtree.get_widget("lc_time_combo_entry").set_text(self.lc_time)
        self.wtree.get_widget("xkb_rules_combo_entry").set_text(self.xkb_rules)
        self.wtree.get_widget("xkb_model_combo_entry").set_text(self.xkb_model)
        self.wtree.get_widget("xkb_layout_combo_entry").set_text(self.xkb_layout)
        self.wtree.get_widget("xkb_variant_combo_entry").set_text(self.xkb_variant)
        self.wtree.get_widget("xkb_options_combo_entry").set_text(self.xkb_options)
        self.wtree.get_widget("lc_use_lang").set_active(self.use_lang)
        self.wtree.get_widget("use_simple_keyboard").set_active(self.use_simple_keyboard)

        self.wtree.get_widget("advanced_dialog").show()

    def on_advanced_ok_button_clicked(self, button):
        "advanced ok_button click event handler"
        self.wtree.get_widget("advanced_dialog").hide()

        self.use_lang = self.wtree.get_widget("lc_use_lang").get_active()
        self.lc_collate = self.wtree.get_widget("lc_collate_combo_entry").get_text()
        self.lc_ctype = self.wtree.get_widget("lc_ctype_combo_entry").get_text()
        self.lc_messages = self.wtree.get_widget("lc_messages_combo_entry").get_text()
        self.lc_monetary = self.wtree.get_widget("lc_monetary_combo_entry").get_text()
        self.lc_numeric = self.wtree.get_widget("lc_numeric_combo_entry").get_text()
        self.lc_time = self.wtree.get_widget("lc_time_combo_entry").get_text()

        self.use_simple_keyboard = self.wtree.get_widget("use_simple_keyboard").get_active()
        self.xkb_rules = self.wtree.get_widget("xkb_rules_combo_entry").get_text()
        self.xkb_model = self.wtree.get_widget("xkb_model_combo_entry").get_text()
        self.xkb_layout = self.wtree.get_widget("xkb_layout_combo_entry").get_text()
        self.xkb_variant = self.wtree.get_widget("xkb_variant_combo_entry").get_text()
        self.xkb_options = self.wtree.get_widget("xkb_options_combo_entry").get_text()

        self.wtree.get_widget("keyboard_optionmenu").set_sensitive(self.use_simple_keyboard)

    def on_advanced_cancel_button_clicked(self, button):
        "advanced cancel_button click event handler"
        self.wtree.get_widget("advanced_dialog").hide()

    def on_lc_use_lang_toggled(self, button):
        "lc_use_lang toggle event handler"
        is_checked = self.wtree.get_widget("lc_use_lang").get_active()
        self.wtree.get_widget("lc_frame").set_sensitive(not is_checked)

    def on_use_simple_keyboard_toggled(self, button):
        "use_simple_keyboard toggle event handler"
        is_checked = self.wtree.get_widget("use_simple_keyboard").get_active()
        self.wtree.get_widget("keyboard_frame").set_sensitive(not is_checked)

    # utility functions

    def get_optionmenu_index(self, optionmenu):
        """given a GTK+ option menu widget, return the current selected
        item within it as an integer"""
        menu = optionmenu.get_menu()
        children = menu.children()
        item = menu.get_active()
        for i in range(len(children)):
            if children[i] == item:
                break
        return i

    def append_debconf(self, name, value):
        "write debconf data collected from widgets to internal structure"
        if value:
            self.debconf_data.append(name + ' ' + name + ' ' + value)
        else:
            self.debconf_data.append(name + ' ' + name)

    # configlet interface functions

    def gnome_setup(self):
        "set up the configlet's graphical widgets"

        configlet.Configlet.gnome_setup(self)

        global gtk
        import gtk

        # initialize the option menus
        newmenu= gtk.GtkMenu()
        for item in self.keyboard_list:
            menuitem = gtk.GtkMenuItem(item)
            menuitem.show()
            newmenu.append(menuitem)
        newmenu.show()
        self.wtree.get_widget("keyboard_optionmenu").set_menu(newmenu)

        newmenu= gtk.GtkMenu()
        for item in self.language_list:
            menuitem = gtk.GtkMenuItem(item)
            menuitem.show()
            newmenu.append(menuitem)
        newmenu.show()
        self.wtree.get_widget("language_optionmenu").set_menu(newmenu)

        # initialize the combo boxes
        self.wtree.get_widget("xkb_rules_combo").set_popdown_strings(self.xkb_rules_list)
        self.wtree.get_widget("xkb_model_combo").set_popdown_strings(self.xkb_model_list)
        self.wtree.get_widget("xkb_layout_combo").set_popdown_strings(self.xkb_layout_list)
        self.wtree.get_widget("xkb_variant_combo").set_popdown_strings(self.xkb_variant_list)
        self.wtree.get_widget("xkb_options_combo").set_popdown_strings(self.xkb_options_list)

        self.wtree.get_widget("lc_collate_combo").set_popdown_strings(self.locale_list)
        self.wtree.get_widget("lc_ctype_combo").set_popdown_strings(self.locale_list)
        self.wtree.get_widget("lc_messages_combo").set_popdown_strings(self.locale_list)
        self.wtree.get_widget("lc_monetary_combo").set_popdown_strings(self.locale_list)
        self.wtree.get_widget("lc_numeric_combo").set_popdown_strings(self.locale_list)
        self.wtree.get_widget("lc_time_combo").set_popdown_strings(self.locale_list)

        # Connect signals
        dict = {}
        # XXX: This should work but does not; self.__class__ is a string rather
        # than a class.  Bug in Python?
        #for key in dir(self.__class__):
        global Kblang
        for key in dir(Kblang):
            dict[key] = getattr(self, key)
            self.wtree.signal_autoconnect(dict)

    def load_debconf(self, data):
        "set configlet widget state using debconf database values"
        # have to get current default X server first; assume 4.x unless we're
        # told otherwise
        self.configspace = "xserver-xfree86"
        for template in data:
            (name, question, value) = re.split(r'\s+', template, 2)
            if name == "shared/default-x-server":
                if value == "xserver-xfree86" or value == '""':
                    pass
                elif value in _xservers_v3:
                    self.configspace = "shared/xfree86v3"
                elif re.match(r'xsun.*', value):
                    # let them configure the keyboard, but it'll be a no-op for
                    # the server
                    pass
                else:
                    raise RuntimeError, "implausible value of shared/default-x-server"

        # set widgets according to corresponding debconf db info
        self.use_lang = 1
        for template in data:
            (name, question, value) = re.split(r'\s+', template, 2)
            if value == '""':
                value = ""

            # collect LANG, LC_* settings from localeconf package
            if name == ("localeconf/lang"):
                if value == "C":
                    real_value = value
                elif value == "":
                    real_value = "C"
                else:
                    (real_value, garbage) = re.split(r'\s+', value, 1)
                if real_value in self.language_values:
                    self.debug("setting LANG from debconf: %s" % real_value)
                    self.wtree.get_widget("language_optionmenu").set_history(self.language_values.index(real_value))
                    self.debug("set lang, current index is %d"
                               % (self.get_optionmenu_index(self.wtree.get_widget("language_optionmenu")),))

            if name == ("localeconf/select_lc_vars"):
                if value != "":
                    self.use_lang = 0
                    # The configlet doesn't need this template; its value is
                    # derived from self.use_lang and self.lc_*.
                    # self.select_lc_vars = value

            if re.match("^localeconf/lc_.*", name):
                if value != "":
                    self.__dict__[re.sub("^localeconf/", "", name)] = value

            # collect XKB settings from self.configspace + ...
            elif name == (self.configspace + "/config/inputdevice/keyboard/internal"):
                if value == "custom":
                    self.debug("custom keyboard is selected in debconf")
                    self.use_simple_keyboard = gtk.FALSE
                    self.wtree.get_widget("use_simple_keyboard").set_active(gtk.FALSE)
                    self.wtree.get_widget("keyboard_frame").set_sensitive(gtk.TRUE)
                    self.wtree.get_widget("keyboard_optionmenu").set_sensitive(gtk.FALSE)
                elif value in self.keyboard_list:
                    self.debug("setting keyboard from debconf: \"%s\"" % value)
                    self.wtree.get_widget("keyboard_optionmenu").set_history(self.keyboard_list.index(value))
                    self.debug("set keyboard, current index is %d"
                               % (self.get_optionmenu_index(self.wtree.get_widget("keyboard_optionmenu")),))

            elif name == (self.configspace + "/config/inputdevice/keyboard/rules"):
                self.xkb_rules = value

            elif name == (self.configspace + "/config/inputdevice/keyboard/model"):
                self.xkb_model = value

            elif name == (self.configspace + "/config/inputdevice/keyboard/layout"):
                self.xkb_layout = value

            elif name == (self.configspace + "/config/inputdevice/keyboard/variant"):
                self.xkb_variant = value

            elif name == (self.configspace + "/config/inputdevice/keyboard/options"):
                self.xkb_options = value

            else:
                pass

        if self.use_lang:
            self.debug("set_use_lang to be set")
            self.wtree.get_widget("lc_use_lang").set_active(gtk.TRUE)
            self.wtree.get_widget("lc_frame").set_sensitive(gtk.FALSE)
        else:
            self.debug("set_use_lang not to be set")
            self.wtree.get_widget("lc_use_lang").set_active(gtk.FALSE)
            self.wtree.get_widget("lc_frame").set_sensitive(gtk.TRUE)
            # set up combo boxes

    def validate(self):
        "validate widget states"

        self.wtree.get_widget("advanced_dialog").hide()
        return 1

    def on_gnome_close(self):
        "collect and store configlet widget states"

        self.lang = self.language_dict[self.language_list[self.get_optionmenu_index(self.wtree.get_widget("language_optionmenu"))]]

        if self.wtree.get_widget("use_simple_keyboard").get_active():
            self.keyboard = self.keyboard_list[self.get_optionmenu_index(self.wtree.get_widget("keyboard_optionmenu"))]
        else:
            self.keyboard = "custom"

        self.debug("self.lang = \"%s\"" % (self.lang,))
        self.debug("self.keyboard = \"%s\"" % (self.keyboard,))
        self.debug("self.use_lang = \"%s\"" % (self.use_lang,))
        self.debug("self.lc_collate = \"%s\"" % (self.lc_collate,))
        self.debug("self.lc_ctype = \"%s\"" % (self.lc_ctype,))
        self.debug("self.lc_messages = \"%s\"" % (self.lc_messages,))
        self.debug("self.lc_monetary = \"%s\"" % (self.lc_monetary,))
        self.debug("self.lc_numeric = \"%s\"" % (self.lc_numeric,))
        self.debug("self.lc_time = \"%s\"" % (self.lc_time,))
        self.debug("self.xkb_rules = \"%s\"" % (self.xkb_rules,))
        self.debug("self.xkb_model = \"%s\"" % (self.xkb_model,))
        self.debug("self.xkb_layout = \"%s\"" % (self.xkb_layout,))
        self.debug("self.xkb_variant = \"%s\"" % (self.xkb_rules,))
        self.debug("self.xkb_options = \"%s\"" % (self.xkb_options,))

        self.debconf_data = [ ]

        # if they're using the configlet, they want it to do something
        self.append_debconf("localeconf/manage_config_with_debconf", "true")
        self.append_debconf("localeconf/backup_existing_nondebconf_config", "true")
        self.append_debconf((self.configspace + "/manage_config_with_debconf"), "true")
        self.append_debconf((self.configspace + "/backup_existing_nondebconf_config"), "true")

        self.append_debconf("localeconf/lang", self.lang)

        # only write advanced lang stuff if use_lang is NOT set
        for lc_var in [ "lc_collate",
                        "lc_ctype",
                        "lc_messages",
                        "lc_monetary",
                        "lc_numeric",
                        "lc_time" ]:
            # If the "Use system language" checkbox is set, if the user has
            # made an empty combo box, or if the user has set the LC_* variable
            # to be the same as LANG, the LC_* variable is not overriding LANG.
            #if use_lang or self.__dict__[lc_var] == "" or self.__dict__[lc_var] == self.lang:
            if self.use_lang:
                self.debug("setting localeconf/%s to self.lang value: %s" % (lc_var, self.lang))
                self.append_debconf("localeconf/" + lc_var, self.lang)
            elif self.__dict__[lc_var] == "":
                self.debug("self.%s blank; setting localeconf/%s to self.lang value: %s" % (lc_var, lc_var, self.lang))
                self.append_debconf("localeconf/" + lc_var, self.lang)
            elif self.__dict__[lc_var] == self.lang:
                self.debug("self.%s set by user to self.lang value; setting localeconf/%s to self.lang value: %s" % (lc_var, lc_var, self.lang))
                self.append_debconf("localeconf/" + lc_var, self.lang)
            else:
                # LC_* variable is overriding LANG
                self.debug("setting localeconf/%s to %s (overriding self.lang)" % (lc_var, self.__dict__[lc_var]))
                self.append_debconf("localeconf/" + lc_var, self.__dict__[lc_var])
                self.debug("adding %s to self.select_lc_vars" % (lc_var.upper(),))
                self.select_lc_vars.append(lc_var.upper())

        self.debug("self.select_lc_vars = \"%s\"" % (self.select_lc_vars,))

        if self.select_lc_vars:
            # covert the list to a comma-delimited string
            delimiter = ", "
            value_string = delimiter.join(self.select_lc_vars)
            self.debug("setting localeconf/select_lc_vars to %s" % (value_string,))
            self.append_debconf("localeconf/select_lc_vars", value_string)

        self.append_debconf((self.configspace
                             + "/config/inputdevice/keyboard/internal"),
                            self.keyboard)

        # only write advanced keyboard combo stuff if keyboard is set to
        # "custom"
        if self.keyboard == "custom":
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/layout"),
                                self.xkb_layout)
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/rules"),
                                self.xkb_rules)
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/model"),
                                self.xkb_model)
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/variant"),
                                self.xkb_variant)
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/options"),
                                self.xkb_options)
        else:
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/layout"),
                                self.keyboard_values[self.keyboard_list.index(self.keyboard)])
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/rules"),
                                "xfree86")
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/model"),
                                "pc104")
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/variant"), "")
            self.append_debconf((self.configspace
                                 + "/config/inputdevice/keyboard/options"), "")

    def report_debconf(self):
        "report debconf information back to configlet engine"
        return self.debconf_data

_attrs = { "name": "localeconf",
           "display_title": "Configure Keyboard and System Language",
           "description": "Select this option to configure the system's localization settings, and to configure the keyboard under the X Window System.",
           "packages": [ "localeconf", "xserver-xfree86" ],
           "shared_questions": ["shared/default-x-server",
                                "shared/xfree86v3"],
           "priority": 40
}
configlet.register_configlet(Kblang, _attrs)

# vim:ai:et:sts=4:sw=4:tw=0:
