/* lockfile-progs.c

   Copyright 1998 Rob Browning <rlb@defaultvalue.org>

   This code is covered under the terms of the Gnu Public License.
   See the accompanying COPYING file for details.

  To do:

   It might be useful at some point to support a --user option to
   mail-lock that can only be used by the superuser (of course, they
   could just use lockfile-create with an appropriate path...

*/

#define _GNU_SOURCE

#include <lockfile.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <getopt.h>
#include <pwd.h>
#include <sys/types.h>

static const char *action = NULL;
static char *target_file = NULL;
static int retry_count = 9; /* This will be a maximum of 3 minutes */
static int touchlock_oneshot = 0;

/* not used yet, so not documented... */
static int lockfile_verbose = 0;

static int result = 0;

static void
usage(const char *command_name, FILE *file) {
  char *usage_str = NULL;

  if(strcmp(command_name, "mail-lock") == 0) {
    usage_str = "usage: mail-lock [ --retry retry-count ]\n";
  } else if(strcmp(command_name, "mail-unlock") == 0) {
    usage_str = "usage: mail-unlock\n";
  } else if(strcmp(command_name, "mail-touchlock") == 0) {
    usage_str = "usage: mail-touchlock [ --oneshot ]\n";
  } else if(strcmp(command_name, "lockfile-create") == 0) {
    usage_str = "usage: lockfile-create [ --retry retry-count ] file\n";
  } else if(strcmp(command_name, "lockfile-remove") == 0) {
    usage_str = "usage: lockfile-remove file\n";
  } else if(strcmp(command_name, "lockfile-touch") == 0) {
    usage_str = "usage: lockfile-touch [ --oneshot ] file\n";
  } else {
    fprintf(stderr, "lockfile: big problem unknown command name: %s\n",
            command_name);
    exit(1);
  }
  fprintf(file, usage_str);
}

static void
parse_arguments(const int argc, char *argv[]) {

  int opt_result;
  struct option opt_specs[] = {
    { "retry", required_argument, NULL, 'r' },
    { "oneshot", no_argument, NULL, 'o' },
    //{ "verbose", no_argument, NULL, 'v' },
    { NULL, 0, NULL, 0 }
  };
  
  char *cmd_name = rindex(argv[0], '/');
  int mail_cmd_p = 0;
  
  if(cmd_name != NULL) {
    /* Skip the '/' */
    cmd_name++;
  } else {
    cmd_name = argv[0];
  }

  while((opt_result = getopt_long(argc, argv, "", opt_specs, NULL)) != -1) {
    switch(opt_result) {
      case 'o':
        touchlock_oneshot = 1;
        break;
      case 'v':
        lockfile_verbose = 1;
        break;
      case 'r':
        {
          char *rest_of_string;
          long tmp_value = strtol(optarg, &rest_of_string, 10);
          
          if((tmp_value == 0) && (rest_of_string == optarg)) {
            /* Bad value */
            fprintf(stderr, "%s: bad retry-count value\n", cmd_name);
            usage(cmd_name, stderr);
            exit(1);
          } else {
            retry_count = tmp_value;
          }
        }
        break;
      default:
        fprintf(stderr, "%s: getopt returned impossible value 0%o.\n",
                cmd_name, opt_result);
        exit(1);
        break;
    }
  }
  
  if(strcmp(cmd_name, "mail-lock") == 0) {
    action = "lock";
    mail_cmd_p = 1;
  } 
  else if(strcmp(cmd_name, "mail-unlock") == 0) {
    action = "unlock";
    mail_cmd_p = 1;
  }
  else if(strcmp(cmd_name, "mail-touchlock") == 0) {
    action = "touch";
    mail_cmd_p = 1;
  }
  else if(strcmp(cmd_name, "lockfile-create") == 0) {
    action = "lock";
  } 
  else if(strcmp(cmd_name, "lockfile-remove") == 0) {
    action = "unlock";
  }
  else if(strcmp(cmd_name, "lockfile-touch") == 0) {
    action = "touch";
  } else {
    usage(cmd_name, stderr);
    exit(1);
  }

  if(mail_cmd_p) {
    if(optind == argc) {
      uid_t user_id = geteuid();
      struct passwd *user_info = getpwuid(user_id);

      if(user_info == NULL) {
        fprintf(stderr, "%s: fatal error, can't find info for user id %ud\n",
                cmd_name, user_id);
        exit(1);
      }
      
      if(asprintf(&target_file, "/var/spool/mail/%s",
                  user_info->pw_name) == -1) {
        fprintf(stderr, "asprintf failed: line %d\n", __LINE__);
        exit(1);
      }
    } else {
      usage(cmd_name, stderr);
      exit(1);
    }
  } else {
    if((argc - optind) != 1) {
      usage(cmd_name, stderr);
      exit(1);
    }
    target_file = argv[optind];
  }
}

static void
handle_touchdeath(int sig) {
  exit(result);
}

int
main(int argc, char *argv[]) {
  char * lockfilename = NULL;

  parse_arguments(argc, argv);

  if(asprintf(&lockfilename, "%s.lock", target_file) == -1) {
    fprintf(stderr, "asprintf failed: line %d\n", __LINE__);
    exit(1);
  }
  
  if(strcmp(action, "unlock") == 0) {
    result = lockfile_remove(lockfilename);
  } else if(strcmp(action, "lock") == 0) {
    if(lockfile_create(lockfilename, retry_count, 0) == L_SUCCESS) {
      result = 0;
    } else {
      fprintf(stderr, "lockfile creation failed\n");
      result = 1;
    }
  } else if(strcmp(action, "touch") == 0) {
    signal(SIGTERM, handle_touchdeath);

    if(touchlock_oneshot) {
      result = lockfile_touch(lockfilename);
    } else {
      while(1 && (result == 0)) {
        result = lockfile_touch(lockfilename);
        sleep(60);
      }
    }
  }

  if(lockfilename) free(lockfilename);
  return(result);
}
