#!/usr/bin/perl -w

#$Header: /mnt/u1/cvs/logtrend/logtrend-doc/doc/LogTrend-SimpleAgent-0.82.2/lib/LogTrend/Agent/AgentDescriptionToDB/PostgreSQLDBLayer.pm,v 1.1 2002/04/10 01:18:46 jdive Exp $
#
#******************************************
#                                         *
# LogTrend 1.0.0.0 - Atrid Systemes       *
#                                         *
# Author : David Mascle d.mascle@atrid.fr *
#                                         *
#******************************************
#
#$Log: PostgreSQLDBLayer.pm,v $
#Revision 1.1  2002/04/10 01:18:46  jdive
#fixed description, uncompressed simpleagent
#
#Revision 1.10  2001/11/19 10:41:02  lsimonneau
#Critical bugfixes.
#
#Revision 1.9  2001/11/16 10:32:28  lsimonneau
#Major feature enhancement : Implementation of authentification with RSA signatu
#re.
#
#IMPORTANT : Agents are incompatible with older version.
#
#Revision 1.8  2001/10/05 11:55:08  slhullier
#in the other case -> otherwise
#
#Revision 1.7  2001/06/11 10:03:04  slhullier
#
#(Source,Number) uniq
#
#Revision 1.6  2001/06/07 14:08:54  slhullier
#
#Passage de  unshift @INC,'..';  aux packages Logtrend::....
#
#Revision 1.5  2001/03/09 16:16:23  slhullier
#
#Les seuils ne sont plus dans linuxagentdescription.xml
#mais relus dans Configuration.xml
#Gestion de la non reponse du serveur
#
#Revision 1.4  2001/03/02 09:30:50  slhullier
#
#Suppression de la table SOURCES_AGENTS
#
#Revision 1.3  2001/03/01 16:29:49  slhullier
#
#Premieres modifs sur le serveur
#
#Revision 1.2  2001/02/20 17:01:27  slhullier
#
#Mise en commentaire (+tab etc)
#
#Revision 1.1.1.1  2001/02/06 09:48:28  fdubuy
#
#First CVS version : all work done by D.Mascle
#
#

package LogTrend::Agent::AgentDescriptionToDB::PostgreSQLDBLayer;

use strict;
use DBI;
use Error qw(:try);
use LogTrend::Agent::AgentDescriptionToDB::ErrorDeclaration;

use LogTrend::Agent::AgentDescriptionToDB::DBAbstractionLayer;
use vars qw( @ISA );
@ISA = ("LogTrend::Agent::AgentDescriptionToDB::DBAbstractionLayer");

#**********************************************************************************
#                                                                                 *
# PostgreSQLDBLayer is a class inherited from DBAbstractionLayer                  *
#                                                                                 *
# it implements its superclass methods and adds several private utilities methods *
#                                                                                 *
#**********************************************************************************

#**************************************************************************
#
# status : public
#
# function : constructor
#
# parameters :
#
# databasename : the name of the database
# host : the name or the IP of the host of the database server
# port : the port on which the database server is listening for connections
# username : the name of a database user
# password : the password of a database user
#
# returns : a reference to an instance of the class
#
#**************************************************************************

sub new
  {
    my ($classname, $databasename, $host, $port, $username, $password) = @_;
    my $self = $classname->SUPER::new($databasename, $host, $port, $username, $password);
    bless($self, $classname);
    return $self;
  }

sub commit {
    my $self = shift;

    $self->{DBHANDLE}->commit;
}

sub rollback {
    my $self = shift;

    $self->{DBHANDLE}->rollback;
}


#********************************************************************************
#
# status : public
#
# function : starts the connection to the LogTrend database on a PostgeSQL server
#
# parameters : none
#
# returns : 1 if succesfull and 0 otherwise
#
#********************************************************************************

sub connect {
    my $self = shift;
    my $ConnectionParameters = "dbi:Pg(AutoCommit=>0):dbname=$self->{DATABASENAME}";

    $ConnectionParameters = $ConnectionParameters.";host=".$self->{HOST};
    $ConnectionParameters = $ConnectionParameters.";port=".$self->{PORT};

    my $db_handle = DBI->connect($ConnectionParameters, $self->{USERNAME}, $self->{PASSWORD},
                                 {PrintError => 0} );
    if (! $db_handle)  {
        throw Error::DB_Connection($DBI::errstr);
    }

    $self->{DBHANDLE} = $db_handle;
}

#****************************************************************
#
# status : public
#
# function : achieves the diconnection from the PostgreSQL server
#
# parameters : none
#
# returns : 1 if successfull and 0 otherwise
#
#****************************************************************

sub disconnect {
    my $self = shift;
    my $result = 0;

    if (defined $self->{DBHANDLE}) {
        if ($self->{DBHANDLE}->disconnect()) {
            undef $self->{DBHANDLE};
            $result = 1;
        }
    }
    
    return $result;
}

#***************************************************************************
#
# status : public
#
# function : adds an agent to the LogTrend database
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
# agentinformation : the agent's informations
#
#***************************************************************************

sub AddAgent
  {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate, $agentinformation) = @_;
    my $result = 1;

    if($self->CheckAgentExistence($agentsource, $agentnumber, $agentversion, $activationdate)) {
        throw Error::DB_AgentDescription("The agent number $agentnumber on source $agentsource is already declared");
    }

    $self->AddAgentAndSourceAgentRelation($agentsource, $agentnumber, $agentversion, $activationdate, $agentinformation);
}

#***********************************************************************
#
# status : private
#
# function : adds an agent and its relation with a source in the database
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
# agentinformation : the agent's informations
#
# returns : 1 if the adding is successfull and 0 otherwise
#
#************************************************************************

sub AddAgentAndSourceAgentRelation
  {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate, $agentinformation) = @_;
    my $result = 0;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

INSERT INTO agents(number, version, id_source, activationdate, information) 

VALUES(?, ?, ?, ?, ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    $st_handle->execute($agentnumber, $agentversion, $agentsource, $activationdate, $agentinformation)
        or throw Error::DB_Request($st_handle->errstr);

    return $result;
}

#*********************************************************
#
# status : private
#
# function : checks if an agent exists
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
#
# returns : 1 if the source exists and 0 otherwise
#
#*********************************************************

sub CheckAgentExistence {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT COUNT(*) 

FROM agents AS a 

WHERE (a.id_source = ?) AND 
      (a.number = ?)

")  or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    $st_handle->execute($agentsource, $agentnumber)
        or throw Error::DB_Request($st_handle->errstr);
    

    @row = $st_handle->fetchrow_array();

    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }

    return $row[0];
}

#************************************************************
#
# status : public
#
# function : adds a description associated to an agent's data
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
# number : a data's number
# type : a data's type
# unit : a data's unit
# description : a data's description
#
# returns : 1 if succesfull and 0 otherwise
#
#************************************************************

sub AddDataDescription {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate, $number, $type, $unit, $description) = @_;
    my $st_handle;
    my @row;

    my $agentid = $self->GetAgentID($agentsource, $agentnumber, $agentversion, $activationdate);

    if($self->CheckDataDescriptionExistence($agentid, $number)) {
        throw Error::DB_AgentDescription("The data number $number on agent with id $agentid already exists");
    }
    
    $st_handle = $self->{DBHANDLE}->prepare("

INSERT INTO agents_data_descriptions(id_agent, number, type, unit, description) 

VALUES(?, ?, ?, ?, ?)

")  or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    my $typeid = $self->GetTypeID($type);
    my $unitid = $self->GetUnitID($unit);

    $st_handle->execute($agentid, $number, $typeid, $unitid, $description)
        or throw Error::DB_Request($st_handle->errstr);
}

#********************************************************************
#
# status : private
#
# function : returns the id of an agent
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
#
# returns : the id of an agent if successfull and 0 otherwise
#
#********************************************************************

sub GetAgentID
  {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT a.id_agent 

FROM agents AS a 

WHERE (a.id_source = ?) AND (a.number = ?) AND (a.version = ?) AND (a.activationdate = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);


    $st_handle->execute($agentsource, $agentnumber, $agentversion, $activationdate)
        or throw Error::DB_Request($st_handle->errstr);

    @row = $st_handle->fetchrow_array();

    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }
    
    if(! @row) {
        throw Error::DB_AgentDescription("Can't find agent id of agent number $agentnumber on source $agentsource" );
    }

    return $row[0];
}

#*****************************************************************
#
# status : private
#
# function : returns the id of a type
#
# parameters : a type
#
# returns : the id of a type if succesfull and 0 otherwise
#
#*****************************************************************

sub GetTypeID {
    my ($self, $type) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT id_data_type 

FROM data_types 

WHERE (type = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    $st_handle->execute($type)
        or throw Error::DB_Request($self->{DBHANDLE}->errstr);
 
    @row = $st_handle->fetchrow_array();

    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }
    
    if(! @row) {
        throw Error::DB_AgentDescription("Unknown type $type" );
    }
    
    return $row[0];
}

#*****************************************************************
#
# status : private
#
# function : returns the id of a unit
#
# parameters : a unit
#
# returns : the id of a unit if succesfull and 0 otherwise
#
#*****************************************************************

sub GetUnitID {
    my ($self, $unit) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("
SELECT id_data_unit 

FROM data_units 

WHERE (unit = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    $st_handle->execute($unit)
        or throw Error::DB_Request($st_handle->errstr);
 
    @row = $st_handle->fetchrow_array();

    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }
    
    if(! @row) {
        throw Error::DB_AgentDescription("Unknown unit $unit" );
    }
    
    return $row[0];
}


#*********************************************************************
#
# status : private
#
# function : checks the existence of a data's description
#
# parameters : an agent's id and a data description's number
#
# returns : 1 if the data's description exists and 0 otherwise
#
#*********************************************************************

sub CheckDataDescriptionExistence {
    my ($self, $agentid, $number) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT COUNT(*) 

FROM agents_data_descriptions 

WHERE (id_agent = ?) AND (number = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);


    $st_handle->execute($agentid, $number)
        or throw Error::DB_Request($st_handle->errstr);
    
    @row = $st_handle->fetchrow_array();

    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }

    return $row[0];
}

#*************************************************************
#
# status : public
#
# function : adds a description associated to an agent's alarm
#
# parameters :
#
# agentsource : the agent's source
# agentnumber : the agent's number
# agentversion : the agent's version
# activationdate : the agent's activation date
# number : an alarm's number
# level : an alarm's level
# message : an alarm's message
#
# returns : 1 if succesfull and 0 otherwise
#
#************************************************************

sub AddAlarmDescription
  {
    my ($self, $agentsource, $agentnumber, $agentversion, $activationdate, $number, $level, $message) = @_;
    my $st_handle;
    my @row;

    my $agentid = $self->GetAgentID($agentsource, $agentnumber, $agentversion, $activationdate);
    
    if($self->CheckAlarmDescriptionExistence($agentid, $number)) {
        throw Error::DB_AgentDescription("The alarm number $number on agent with id $agentid already exists");
    }

    my $levelid = $self->GetLevelID($level);

    $st_handle = $self->{DBHANDLE}->prepare("

INSERT INTO agents_alarms_descriptions(id_agent, alarm_number, id_alarm_level, message) 

VALUES(?, ?, ?, ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    
    $st_handle->execute($agentid, $number, $levelid, $message)
        or throw Error::DB_Request($st_handle->errstr);

}

#*********************************************************************
#
# status : private
#
# function : checks the existence of an alarm's description
#
# parameters : an agent's id and an alarm description's number
#
# returns : 1 if the data's description exists and 0 otherwise
#
#*********************************************************************

sub CheckAlarmDescriptionExistence {
    my ($self, $agentid, $number) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT COUNT(*) 

FROM agents_alarms_descriptions

WHERE (id_agent = ?) AND (alarm_number = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    
    $st_handle->execute($agentid, $number)
        or throw Error::DB_Request($st_handle->errstr);

    @row = $st_handle->fetchrow_array();
    
    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }

    return $row[0];
}

#***************************************************************************
#
# status : private
#
# function : returns the id of an alarm's level
#
# parameters : an alarm's level
#
# returns : the id of an alarm's level if succesfull and 0 otherwise
#
#***************************************************************************

sub GetLevelID
  {
    my ($self, $level) = @_;
    my $st_handle;
    my @row;

    $st_handle = $self->{DBHANDLE}->prepare("

SELECT id_alarm_level 

FROM alarms_levels 

WHERE (level = ?)

") or throw Error::DB_Request($self->{DBHANDLE}->errstr);

    $st_handle->execute($level)
        or throw Error::DB_Request($st_handle->errstr);
     
    @row = $st_handle->fetchrow_array();
    
    if(defined $st_handle->err) {
        throw Error::DB_Request($st_handle->errstr);
    }

    if(! @row) {
        throw Error::DB_AgentDescription("Unknown alarm level $level" );
    }
    
    return $row[0];
}

1;
