/* Copyright (C) 2000-2003 Markus Lausser (sgop@users.sf.net)
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "mtypes.h"

#include "lopster.h"
#include "utils.h"
#include "global.h"

static mtype_t MType[MEDIA_SIZE] = {
  { NULL, NULL, NULL, 0 },
  { NULL, NULL, NULL, 1 },
  { NULL, NULL, NULL, 2 },
  { NULL, NULL, NULL, 3 },
  { NULL, NULL, NULL, 4 },
  { NULL, NULL, NULL, 5 },
  { NULL, NULL, NULL, 6 },
  { NULL, NULL, NULL, 7 }
};

const char* int2media(int no, int mode) {
  switch (no) {
  case MEDIA_MP3:         return "mp3";
  case MEDIA_FOLDER:      return "folder";
  case MEDIA_AUDIO:       return "audio";
  case MEDIA_VIDEO:       return "video";
  case MEDIA_APPLICATION: return "application";
  case MEDIA_IMAGE:       return "image";
  case MEDIA_TEXT:        return "text";
  default: if (mode == 0) return "default";
  else if (mode == 1)     return "any";
  else                    return "unknown";
  };
}

int media2int(const char* media) {
  if (!media) return MEDIA_NONE;
  if (!strcmp(media, "application")) return MEDIA_APPLICATION;
  else if (!strcmp(media, "mp3"))    return MEDIA_MP3;
  else if (!strcmp(media, "folder")) return MEDIA_FOLDER;
  else if (!strcmp(media, "audio"))  return MEDIA_AUDIO;
  else if (!strcmp(media, "video"))  return MEDIA_VIDEO;
  else if (!strcmp(media, "image"))  return MEDIA_IMAGE;
  else if (!strcmp(media, "text"))   return MEDIA_TEXT;
  else return MEDIA_NONE;
}

static app_t* app_new(const char* command, const char* label) {
  app_t* app = g_malloc(sizeof(*app));
  
  g_assert(command);
  app->app = g_strdup(command);
  app->label = label?g_strdup(label):NULL;
  return app;
}

static suffix_t* suffix_new(const char* ext) {
  suffix_t* suf = g_malloc(sizeof(*suf));
  char* temp;

  g_assert(ext);
  temp = g_strdup(ext);
  suf->ext = NULL;
  make_list_from_string(&(suf->ext), temp, " ");
  g_free(temp);
  suf->app = NULL;
  return suf;
}

void mtype_create_media(int media, char* suffixes) {
  mtype_t* mtype;

  switch (media) {
  case MEDIA_MP3:
    mtype = &MType[MEDIA_MP3];
    mtype->app = g_list_append(mtype->app, app_new("xmms %f", "Open"));
    mtype->app = g_list_append(mtype->app, app_new("xmms -e %f", "Enqueue"));
    break;
  case MEDIA_VIDEO:
    mtype = &MType[MEDIA_VIDEO];
    mtype->app = g_list_append(mtype->app, app_new("mplayer %f", "MPlayer"));
    break;
  case MEDIA_IMAGE:
    mtype = &MType[MEDIA_IMAGE];
    mtype->app = g_list_append(mtype->app, app_new("ee %f", "Electric eyes"));
    break;
  case MEDIA_FOLDER:
    mtype = &MType[MEDIA_FOLDER];
    mtype->app = g_list_append(mtype->app, app_new("gmc %f", "Browse"));
    mtype->app = g_list_append(mtype->app, app_new("xmms %f", "Xmms"));
    return;  // doesnt allow suffixes
  case MEDIA_AUDIO:
  case MEDIA_APPLICATION:
  case MEDIA_TEXT:
    mtype = &MType[media];
    break;
  case MEDIA_NONE:
  default:
    return;
  }
  if (suffixes) {
    char* suf = arg(suffixes, 0);
    while (suf) {
      mtype->suffix = g_list_append(mtype->suffix, suffix_new(suf));
      suf = arg(NULL, 0);
    }
  }
}

/*
static void mtype_create_default() {
  mtype_t* mtype;

  //////////////////////////////////////////////////////////////
  mtype = &MType[MEDIA_MP3];
  
  mtype->app = g_list_append(mtype->app, app_new("xmms %f", "Open"));
  mtype->app = g_list_append(mtype->app, app_new("xmms -e %f", "Enqueue"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("mp3"));
#ifdef HAVE_OGG
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("ogg"));
#endif
  
  //////////////////////////////////////////////////////////////
  mtype = &MType[MEDIA_VIDEO];
  
  mtype->app = g_list_append(mtype->app, app_new("mplayer %f", "MPlayer"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("mpg mpeg"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("avi"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("mov"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("wmf"));
  
  //////////////////////////////////////////////////////////////
  mtype = &MType[MEDIA_IMAGE];
  
  mtype->app = g_list_append(mtype->app, app_new("ee %f", "Electric eyes"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("jpg jpeg"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("bmp"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("gif"));
  mtype->suffix = 
    g_list_append(mtype->suffix, suffix_new("png"));
  
  //////////////////////////////////////////////////////////////
  mtype = &MType[MEDIA_FOLDER];
  
  mtype->app = g_list_append(mtype->app, app_new("gmc %f", "Browse"));
  mtype->app = g_list_append(mtype->app, app_new("xmms %f", "Xmms"));
}
*/

void mtype_load() {
  char line[1024];
  char* fname;
  char* temp;
  char* name;
  char* token;
  int cnt = 0;
  int type;
  mtype_t* mtype;
  suffix_t* suffix;
  FILE* fd;
  app_t* app;

  fname = g_strdup_printf("%s%cmedia.list", 
			  global.options.config_dir, DIR_SEP);
  if ((fd = fopen(fname, "r")) == NULL) {
    g_free(fname);
    //    mtype_create_default();
    //    mtype_save();
    return;
  }

  mtype = NULL;
  suffix = NULL;
  while (mfgets(line, sizeof(line), fd)) {
    cnt++;
    token = arg(line, 0);
    if (!token || *token == '#') continue;
    if (!strcmp(token, "MEDIA")) {
      if (mtype) {
	g_warning("Ignoring line %d in [%s]", cnt, fname);
	continue;
      }
      name = arg(NULL, 0);
      temp = arg(NULL, 0);
      if (!temp || *temp != '{' || temp[1] != 0) {
	g_warning("Malformed line %d in [%s]", cnt, fname);
	continue;
      }
      type = media2int(name);
      if (type < 0 || type >= MEDIA_SIZE) {
	g_warning("Invalid media type in [%s] line %d", fname, cnt);
	continue;
      }
      mtype = &MType[type];
    } else if (!strcmp(token, "OPEN")) {
      name = arg(NULL, 0);
      if (!name) {
	g_warning("Malformed line %d in [%s]", cnt, fname);
	continue;
      }
      temp = arg(NULL, 1);
      if (!temp) {
	g_warning("Malformed line %d in [%s]", cnt, fname);
	continue;
      }
      app = g_malloc(sizeof(app_t));
      app->app = g_strdup(temp);
      app->label = (*name!=0)?g_strdup(name):NULL;
      if (suffix) {
	suffix->app = g_list_append(suffix->app, app);
      } else {
	mtype->app = g_list_append(mtype->app, app);
      }
    } else if (!mtype) {
      g_warning("Ignoring line %d in [%s]", cnt, fname);
      continue;
    } else if (!strcmp(token, "DOWNLOAD")) {
      if (mtype->download) {
	g_warning("Duplicate download folder in [%s] line %d", fname, cnt);
	continue;
      }
      name = arg(NULL, 1);
      if (!name) {
	g_warning("Malformed line %d in [%s]", cnt, fname);
	continue;
      }
      mtype->download = g_strdup(name);
    } else if (!strcmp(token, "}")) {
      if (suffix) {  // we are closing a suffix
	if (!suffix->ext) {
	  g_warning("closing suffix in line %d: suffix is empty ", cnt);
	}
	suffix = NULL;
      } else {       // we are closing a mtype
	/*
	if (mtype->type != MEDIA_FOLDER && 
	    mtype->type != MEDIA_NONE && !mtype->suffix) {
	  g_warning("Type %s should contain suffixes",
		    int2media(mtype->type, -1));
	}
	*/
	mtype = NULL;
      }
    } else if (mtype->type == MEDIA_NONE || mtype->type == MEDIA_FOLDER) {
      g_warning("Ignoring line %d [%s]", cnt, fname);
      continue;
    } else if (!strcmp(token, "SUFFIX")) {
      if (suffix) {
	g_warning("Ignoring line %d in [%s]", cnt, fname);
	continue;
      }
      name = arg(NULL, 0);
      temp = arg(NULL, 0);
      if (!temp || *temp != '{' || temp[1] != 0) {
	g_warning("Malformed line %d in [%s]", cnt, fname);
	continue;
      }
      suffix = g_malloc(sizeof(suffix_t));
      suffix->ext = NULL;
      suffix->app = NULL;
      temp = arg(name, 0);
      while (temp) {
	suffix->ext = g_list_append(suffix->ext, g_strdup(temp));
	temp = arg(NULL, 0);
      }	
      mtype->suffix = g_list_append(mtype->suffix, suffix);
    }
  }
  if (suffix) {
    g_warning("suffix was not closed");
  }
  if (mtype) {
    g_warning("mediatype [%s] was not closed",
	      int2media(mtype->type, -1));
  }
  g_free(fname);
  fclose(fd);
}

void mtype_save() {
  int i1;
  GList* dlist;
  GList* dlist2;
  char* fname;
  char* name;
  mtype_t* mtype;
  suffix_t* suffix;
  app_t* app;
  FILE* fd;

  fname = g_strdup_printf("%s%cmedia.list", 
			  global.options.config_dir, DIR_SEP);

  if ((fd = fopen(fname, "w")) == NULL) {
    g_warning("could not save file [%s]", fname);
    g_free(fname);
    return;
  }

  for (i1 = 0; i1 < MEDIA_SIZE; i1++) {
    mtype = &MType[i1];
    fprintf(fd, "MEDIA %s {\n", int2media(i1, -1));
    if (mtype->download) {
      fprintf(fd, "  DOWNLOAD %s\n", mtype->download);
    }
    for (dlist2 = mtype->app; dlist2; dlist2 = dlist2->next) {
      app = dlist2->data;
      fprintf(fd, "  OPEN \"%s\" %s\n", app->label?app->label:"", app->app);
    }
    for (dlist = mtype->suffix; dlist; dlist = dlist->next) {
      suffix = dlist->data;
      fprintf(fd, "  SUFFIX \"");
      for (dlist2 = suffix->ext; dlist2; dlist2 = dlist2->next) {
	name = dlist2->data;
	if (dlist2->next)
	  fprintf(fd, "%s ", name);
	else
	  fprintf(fd, "%s", name);
      }
      fprintf(fd, "\" {\n");
      for (dlist2 = suffix->app; dlist2; dlist2 = dlist2->next) {
	app = dlist2->data;
	fprintf(fd, "    OPEN \"%s\" %s\n", app->label?app->label:"", app->app);
      }
      fprintf(fd, "  }\n");
    }
    fprintf(fd, "}\n");
  }
  g_free(fname);
  fclose(fd);
}

static mtype_t* mtype_find_from_suffix(char *name) {
  GList *dlist;
  GList *dlist2;
  suffix_t* suffix;
  mtype_t* mtype;
  int i1;
  char* suf;
  char* ext;

  if (!name) return NULL;

  suf = strrchr(name, '.');
  if (suf) suf++;
  else suf = name;

  for (i1 = 1; i1 < MEDIA_SIZE; i1++) {
    mtype = &MType[i1];
    for (dlist = mtype->suffix; dlist; dlist = dlist->next) {
      suffix = dlist->data;
      for (dlist2 = suffix->ext; dlist2; dlist2 = dlist2->next) {
	ext = dlist2->data;
	if (!strcasecmp(suf, ext)) return mtype;
      }
    }
  }
  return &MType[0];
}

int mtype_contains_suffix(int type, char *name) {
  GList *dlist;
  GList *dlist2;
  suffix_t* suffix;
  mtype_t* mtype;
  char* suf;
  char* ext;

  if (type <= 0 || type >= MEDIA_SIZE) return 0;

  suf = strrchr(name, '.');
  if (suf) suf++;
  else suf = name;

  mtype = &MType[type];
  for (dlist = mtype->suffix; dlist; dlist = dlist->next) {
    suffix = dlist->data;
    for (dlist2 = suffix->ext; dlist2; dlist2 = dlist2->next) {
      ext = dlist2->data;
      if (!strcasecmp(suf, ext)) return 1;
    }
  }
  return 0;
}

int mtype_get(char* filename) {
  mtype_t* mtype;

  mtype = mtype_find_from_suffix(filename);
  if (!mtype) return MEDIA_NONE;
  else return mtype->type;
}

GList* mtype_get_apps(int type, char* name) {
  GList *dlist;
  GList *dlist2;
  suffix_t* suffix;
  mtype_t* mtype;
  char* suf;
  GList* result = NULL;
  GList* result2 = NULL;
  char* ext;

  if (name) {
    suf = strrchr(name, '.');
    if (suf) suf++;
    else suf = name;
  } else suf = NULL;

  if (type < 0 || type >= MEDIA_SIZE)
    mtype = mtype_find_from_suffix(name);
  else mtype = &MType[type];

  if (!mtype) return NULL;

  result = g_list_copy(mtype->app);
  if (!suf) return result;

  for (dlist = mtype->suffix; dlist; dlist = dlist->next) {
    suffix = dlist->data;
    for (dlist2 = suffix->ext; dlist2; dlist2 = dlist2->next) {
      ext = dlist2->data;
      if (!strcasecmp(suf, ext)) {
	result2 = g_list_copy(suffix->app);
	result = g_list_concat(result, result2);
	return result;
      }
    }
  }
  return result;
}

int mtype_get_type(mtype_t* mtype) {
  return mtype->type;
}

void mtype_set_download(int type, char* download) {
  if (type < 0 || type >= MEDIA_SIZE) return;

  if (MType[type].download) {
    g_free(MType[type].download);
    MType[type].download = NULL;
  }
  if (download && *download)
    MType[type].download = g_strdup(download);
}

// only used for old rc files, we make special assumptions here!
void mtype_add_suffix(int type, char* suf, char* app, char* label) {
  suffix_t* suffix;

  if (type < 0 || type >= MEDIA_SIZE) return;
  if (!suf || *suf == 0) return;

  suffix = g_malloc0(sizeof(suffix_t));
  MType[type].suffix = 
    g_list_append(MType[type].suffix, suffix);
  suffix->ext = g_list_append(suffix->ext, g_strdup(suf));
  if (app && *app) {
    app_t* appl = g_malloc(sizeof(app_t));
    appl->app = g_strdup(app);
    if (label && *label)
      appl->label = g_strdup(label);
    else
      appl->label = NULL;
    suffix->app = g_list_append(suffix->app, appl);
  }
}

const char* mtype_get_download_from_name(char* filename) {
  mtype_t* mtype;

  mtype = mtype_find_from_suffix(filename);
  
  if (!mtype) return NULL;
  else return mtype->download;
}

const char* mtype_get_download(int type) {
  mtype_t* mtype;

  if (type < 0 || type >= MEDIA_SIZE) return NULL;

  mtype = &MType[type];

  if (!mtype) return NULL;
  else return mtype->download;
}

mtype_t* mtype_find(int index) {
  g_assert(index >= 0 && index < MEDIA_SIZE);
  return &MType[index];
}

static void mtype_clear(mtype_t* mtype) {
  GList* dlist;
  GList* dlist2;

  g_assert(mtype);

  g_free(mtype->download);
  mtype->download = NULL;
	 
  for (dlist = mtype->app; dlist; dlist = dlist->next) {
    app_t* app = dlist->data;
    g_free(app->app);
    g_free(app->label);
    g_free(app);
  }
  g_list_free(mtype->app);
  mtype->app = NULL;

  for (dlist = mtype->suffix; dlist; dlist = dlist->next) {
    suffix_t* suffix = dlist->data;
    for (dlist2 = suffix->ext; dlist2; dlist2 = dlist2->next) {
      char* ext = dlist2->data;
      g_free(ext);
    }
    g_list_free(suffix->ext);
    for (dlist2 = suffix->app; dlist2; dlist2 = dlist2->next) {
      app_t* app = dlist2->data;
      g_free(app->app);
      g_free(app->label);
      g_free(app);
    }
    g_list_free(suffix->app);
    g_free(suffix);
  }
  g_list_free(mtype->suffix);
  mtype->suffix = NULL;
}

void mtype_clear_all() {
  int i1;
  for (i1 = 0; i1 < MEDIA_SIZE; i1++) mtype_clear(&MType[i1]);
}
