#
#  LMusicDellManager.py
#
# This work is released under the GNU GPL, version 2 or later.
#
#
# Implements a queue of operations applied to the currently mounted Dell DJ device
#

from qt import *
from kdecore import *
from utils import *
import time

#
# Event posted from the varous threads to the main Dell manager object
#
class LMusicDellEvent(QCustomEvent):
	def __init__(self,status = None):
		QCustomEvent.__init__(self,QEvent.User+42)
		self._playerStatus = status
	
	def status(self):
		return self._playerStatus

#
# base class for all of the various Dell management thread types
#
class LMusicDellThread(QThread):
	def __init__(self):
		QThread.__init__(self)
		self.lastTime = 0

	#
	# emit a status packet to the UI thread
	#
	def emitEvent(self,status):
		event = LMusicDellEvent(status)
		QApplication.postEvent(LMusicDellManager.singleton(),event)

	#
	# throttle events to 5/sec max
	#
	def maybeEmitEvent(self,status):
		now = long(time.time()*5)
		if now!=self.lastTime:
			self.emitEvent(status)
			self.lastTime = now
	
	#
	# if we were given the name of the playlist, convert it to the actual playlist
	#
	def ensurePlaylist(self,playlist):
		if type(playlist)==unicode or type(playlist)==str:
			from DellLibrary import DellLibrary
			playlist = DellLibrary.singleton().playlistWithName(playlist)
		return playlist

#
# set the owner name
#
class LMusicDellSetOwnerNameThread(LMusicDellThread):
	def __init__(self,name):
		LMusicDellThread.__init__(self)
		from DellLibrary import DellLibrary
		self.library = DellLibrary.singleton()
		self.name = name
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Setting Owner Name','Name':self.name})
		self.library.setOwnerName(self.name)
		self.emitEvent({'Source':'DDJ','Status':'Set Owner Name','Name':self.name})

#
# add tracks to playlist
#
class LMusicDellAddTrackIDsToPlaylistThread(LMusicDellThread):
	def __init__(self,playlist,trackIDs):
		LMusicDellThread.__init__(self)
		self.playlist = self.ensurePlaylist(playlist)
		self.trackIDs = trackIDs
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Adding TrackIDs','Playlist':self.playlist,'TrackIDs':self.trackIDs})
		if self.playlist: self.playlist.addDellTrackIDs(self.trackIDs)
		self.emitEvent({'Source':'DDJ','Status':'Added TrackIDs','Playlist':self.playlist,'TrackIDs':self.trackIDs})

#
# remove a track from a playlist - if the playlist is the master list
# then the track is removed from all playlists and erased from the device
#
class LMusicDellRemoveTrackIDsFromPlaylistThread(LMusicDellThread):
	def __init__(self,playlist,trackIDs):
		LMusicDellThread.__init__(self)
		self.playlist = self.ensurePlaylist(playlist)
		self.trackIDs = trackIDs
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Removing TrackIDs','Playlist':self.playlist,'TrackIDs':self.trackIDs})
		if self.playlist: self.playlist.removeDellTrackIDs(self.trackIDs)
		self.emitEvent({'Source':'DDJ','Status':'Removed TrackIDs','Playlist':self.playlist,'TrackIDs':self.trackIDs})

#
# add a new playlist to the device
#
class LMusicDellAddNewPlaylistThread(LMusicDellThread):
	def __init__(self,name):
		LMusicDellThread.__init__(self)
		from DellLibrary import DellLibrary
		self.library = DellLibrary.singleton()
		self.name = name
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Adding Playlist','Name':self.name})
		playlist = self.library.playlistWithName(self.name)
		if playlist == None: self.library.addNewPlaylist(self.name)
		self.emitEvent({'Source':'DDJ','Status':'Added Playlist','Name':self.name})

#
# remove a playlist from the device
#
class LMusicDellRemovePlaylistThread(LMusicDellThread):
	def __init__(self,playlist):
		LMusicDellThread.__init__(self)
		self.playlist = self.ensurePlaylist(playlist)
		self.library = self.playlist.library()
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Removing Playlist','Playlist':self.playlist})
		if self.playlist: self.library.removePlaylist(self.playlist)
		self.emitEvent({'Source':'DDJ','Status':'Removed Playlist','Playlist':self.playlist})

#
# change the name of a playlist
#
class LMusicDellRenamePlaylistThread(LMusicDellThread):
	def __init__(self,playlist,name):
		LMusicDellThread.__init__(self)
		self.playlist = self.ensurePlaylist(playlist)
		self.name = name
	
	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Renaming Playlist','Playlist':self.playlist,'Name':self.name})
		if self.playlist: self.playlist.setName(self.name)
		self.emitEvent({'Source':'DDJ','Status':'Renamed Playlist','Playlist':self.playlist,'Name':self.name})

def uploadCallback(sent,total,item):
	item.callback(sent,total)

#
# upload a track to the device
#
class LMusicDellUploadTrackToPlaylistThread(LMusicDellThread):
	def __init__(self,playlist,track):
		LMusicDellThread.__init__(self)
		self.playlist = self.ensurePlaylist(playlist)
		self.track = track # note this is a mainLibrary track
	
	def callback(self,sent,total):
		self.maybeEmitEvent({'Source':'DDJ','Status':'Uploading','Playlist':self.playlist,'Track':self.track,'currentTime':sent,'totalTime':total})

	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Uploading Track','Playlist':self.playlist,'Track':self.track})
		if self.playlist: self.playlist.uploadTrack(self.track,uploadCallback,self)
		self.emitEvent({'Source':'DDJ','Status':'Uploaded Track','Playlist':self.playlist,'Track':self.track})

def downloadCallback(sent,total,item):
	item.callback(sent,total)

#
# download a track from the device
#
class LMusicDellDownloadTrackToPlaylistThread(LMusicDellThread):
	def __init__(self,playlist,track):
		LMusicDellThread.__init__(self)
		self.playlist = playlist # note this is a mainLibrary playlist
		self.track = track
	
	def callback(self,sent,total):
		self.maybeEmitEvent({'Source':'DDJ','Status':'Downloading','Playlist':self.playlist,'Track':self.track,'currentTime':sent,'totalTime':total})

	def run(self):
		self.emitEvent({'Source':'DDJ','Status':'Downloading Track','Playlist':self.playlist,'Track':self.track})
		filePath = self.track.download(downloadCallback,self)
		self.emitEvent({'Source':'DDJ','Status':'Downloaded Track','Playlist':self.playlist,'Track':self.track,'filePath':filePath})

#
# refresh the local library based on the current content of the device (unused - we do this synch
#
##class LMusicDellRefreshThread(LMusicDellThread):
##	def __init__(self):
##		LMusicDellThread.__init__(self)
##		from DellLibrary import DellLibrary
##		self.library = DellLibrary.singleton()
##	
##	def run(self):
##		self.emitEvent({'Source':'DDJ','Status':'Refreshing'})
##		self.library.readMusic()
##		self.emitEvent({'Source':'DDJ','Status':'Refreshed'})

def gRefreshCallback(status,index,total,object):
	object.refreshCallback(status,index,total)

class LMusicDellManager(QObject):
	def __init__(self):
		QObject.__init__(self)
		self.actions = []
		self.process = None
		self.altProcess = None # keeps reference to avoid premature GC of process
		from LApplication import LApplication
		self.app = LApplication.singleton()
	
	def refreshCallback(self,status,index,total):
		self.progress.setLabelText(status)
		self.progress.setTotalSteps(total)
		self.progress.setProgress(index)
		self.app.processEvents()

	def doRefresh(self):
		#print "refreshing"
		from DellLibrary import DellLibrary
		self.progress = QProgressDialog(None,"progress",True)
		self.progress.setMinimumDuration(0)
		self.progress.setCaption(i18n("Refreshing"))
		DellLibrary.singleton().readMusic(gRefreshCallback,self) # do this synchronous to avoid threadwars
		self.progress = None
	
	def processNextItem(self):
		if self.process==None or not self.process.running():
			if len(self.actions)>0:
				action = self.actions[0]
				self.actions = self.actions[1:]
				self.process = None
				type = action['Action']
				if type=='Refresh':
					#self.process = LMusicDellRefreshThread()
					self.doRefresh()
					self.processNextItem()
				elif type=='AddNewHostPlaylist':
					from Library import Library
					lib = Library.mainLibrary()
					name = action['Name']
					if lib.playlistWithName(name)==None:
						lib.addNewPlaylist(name) # also synchronous to avoid thread problems
					self.processNextItem()
				elif type=='Sync1':
					self.sync1() # synchronously sets up async actions
					self.processNextItem()
				elif type=='Sync2':
					self.sync2() # synchronously sets up async actions
					self.processNextItem()
				else:
					if type=='SetOwnerName':
						self.process = LMusicDellSetOwnerNameThread(action['Name'])
					elif type=='AddTrackIDsToPlaylist':
						self.process = LMusicDellAddTrackIDsToPlaylistThread(action['Playlist'],action['TrackIDs'])
					elif type=='RemoveTrackIDsFromPlaylist':
						self.process = LMusicDellRemoveTrackIDsFromPlaylistThread(action['Playlist'],action['TrackIDs'])
					elif type=='AddNewPlaylist':
						self.process = LMusicDellAddNewPlaylistThread(action['Name'])
					elif type=='RemovePlaylist':
						self.process = LMusicDellRemovePlaylistThread(action['Playlist'])
					elif type=='RenamePlaylist':
						self.process = LMusicDellRenamePlaylistThread(action['Playlist'],action['Name'])
					elif type=='UploadTrackToPlaylist':
						self.process = LMusicDellUploadTrackToPlaylistThread(action['Playlist'],action['Track'])
					elif type=='DownloadTrackToPlaylist':
						self.process = LMusicDellDownloadTrackToPlaylistThread(action['Playlist'],action['Track'])
					else:
						print "LMusicDellManager: unknown action",action
					if self.process:
						#print "LMusicDellManager starting thread",self.process
						self.process.start()
			else:
				#print "LMusicDellManager no actions"
				pass
		else:
			#print "LMusicDellManager - busy"
			pass
		
	def customEvent(self,e):
		status = e.status()
		if status:
			#print status
			self.emit(PYSIGNAL('status'),(status,None))
			s = status['Status']
			if s in ['Removed TrackIDs','Added TrackIDs','Added Playlist','Removed Playlist','Renamed Playlist','Uploaded Track','Downloaded Track','Refreshed']:
				self.altProcess = self.process # keep reference to keep GC happy
				self.process = None
				if len(self.actions)==0 and s != 'Refreshed': # if action queue empty, append a Refresh unless we just did one
					self.actions.append({'Action':'Refresh'})
				self.processNextItem()
	
	def setOwnerName(self,name):
		self.actions.append({'Action':'SetOwnerName','Name':unikode(name)})
		self.processNextItem()
	
	def addTrackIDsToPlaylist(self,playlist,trackIDs):
		self.actions.append({'Action':'AddTrackIDsToPlaylist','Playlist':playlist,'TrackIDs':trackIDs})
		self.processNextItem()
	
	def removeTrackIDsFromPlaylist(self,playlist,trackIDs):
		self.actions.append({'Action':'RemoveTrackIDsFromPlaylist','Playlist':playlist,'TrackIDs':trackIDs})
		self.processNextItem()

	def addNewPlaylist(self,name):
		self.actions.append({'Action':'AddNewPlaylist','Name':unikode(name)})
		self.processNextItem()

	def addNewPlaylists(self,names):
		for name in names:
			self.actions.append({'Action':'AddNewPlaylist','Name':unikode(name)})
		self.processNextItem()
	
	def removePlaylist(self,playlist):
		self.actions.append({'Action':'RemovePlaylist','Playlist':playlist})
		self.processNextItem()

	def removePlaylists(self,playlists):
		for playlist in playlists:
			self.actions.append({'Action':'RemovePlaylist','Playlist':playlist})
		self.processNextItem()

	def renamePlaylist(self,playlist,name):
		self.actions.append({'Action':'RenamePlaylist','Playlist':playlist,'Name':unikode(name)})
		self.processNextItem()
	
	def uploadTrackToPlaylist(self,playlist,track):
		self.actions.append({'Action':'UploadTrackToPlaylist','Playlist':playlist,'Track':track})
		self.processNextItem()

	def uploadTracksToPlaylist(self,playlist,tracks):
		for track in tracks:
			self.actions.append({'Action':'UploadTrackToPlaylist','Playlist':playlist,'Track':track})
		self.processNextItem()

	def downloadTrackToPlaylist(self,playlist,track):
		self.actions.append({'Action':'DownloadTrackToPlaylist','Playlist':playlist,'Track':track})
		self.processNextItem()

	def downloadTracksToPlaylist(self,playlist,tracks):
		print "downloading",tracks,"to",playlist
		for track in tracks:
			self.actions.append({'Action':'DownloadTrackToPlaylist','Playlist':playlist,'Track':track})
		self.processNextItem()

	def addNewHostPlaylist(self,name):
		self.actions.append({'Action':'AddNewHostPlaylist','Name':unikode(name)})
		self.processNextItem()

	def addNewHostPlaylists(self,names):
		for name in names:
			self.actions.append({'Action':'AddNewHostPlaylist','Name':unikode(name)})
		self.processNextItem()
	
	#
	# sync is split up into two parts in order for the player to stabilize its playlists -
	# it's tough to figure out how to add tracks to the playlist until it actually exists.
	#
	def sync(self):
		self.actions.append({'Action':'Sync1'})
		#self.refresh()
		#self.actions.append({'Action':'Sync2'})
		#self.refresh()
		self.processNextItem()

	def sync1(self):
		#print "LMusicDellManager: sync phase 1"
		from DellLibrary import DellLibrary
		from Library import Library
		from LSettings import LSettings
		dell = DellLibrary.singleton()
		dellMain = dell.mainPlaylist
		library = Library.mainLibrary()
		settings = LSettings.settings()
		progress = QProgressDialog(None,"progress",True)
		progress.setMinimumDuration(0)
		progress.setCaption(i18n("Syncing"))
		progress.setTotalSteps(3)
		progress.setProgress(1)
		musicMode = settings.get("Player Music Sync","union")
		playlistMode = settings.get("Player Playlist Sync","union")
		progress.forceShow()
		self.app.processEvents()
		(tracksAdd,tracksRemove,playlistsAdd,playlistsRemove) = dell.getSyncInfoWith(library)
		#
		# add or remove playlists per settings
		#
		progress.setLabelText(i18n("Syncing playlists"))
		progress.setProgress(2)
		self.app.processEvents()
		if playlistMode=='sync': self.removePlaylists(playlistsRemove.keys())
		elif playlistMode=='union': self.addNewHostPlaylists(playlistsRemove.keys())
		#print "creating Dell playlists",playlistsAdd.keys()
		self.addNewPlaylists(playlistsAdd.keys())
		#
		# copy or remove tracks per settings
		#
		progress.setLabelText(i18n("Syncing tracks"))
		progress.setProgress(3)
		self.app.processEvents()
		if musicMode=='sync': self.removeTracks(tracksRemove.values())
		elif musicMode=='union': self.downloadTracksToPlaylist(None,tracksRemove.values())
		self.uploadTracksToPlaylist(dellMain,tracksAdd.values())
		self.refresh()
		self.actions.append({'Action':'Sync2'})
	
		#
		# now make sure all the tracks are in the proper playlists
		#
	def sync2(self):
		#print "LMusicDellManager: sync phase 2"
		from DellLibrary import DellLibrary
		from Library import Library
		from LSettings import LSettings
		dell = DellLibrary.singleton()
		dellMain = dell.mainPlaylist
		library = Library.mainLibrary()
		settings = LSettings.settings()
		musicMode = settings.get("Player Music Sync","union")
		playlistMode = settings.get("Player Playlist Sync","union")
		progress = QProgressDialog(None,"progress",True)
		progress.setMinimumDuration(0)
		progress.setCaption(i18n("Fixing playlists"))
		hostHashes = library.hashList()
		hostPlaylists = library.getPlaylists()
		progress.setTotalSteps(len(hostPlaylists))
		progress.forceShow()
		self.app.processEvents()
		index = 0
		for hostPlaylist in hostPlaylists:
			index = index+1
			progress.setProgress(index)
			progress.setLabelText(hostPlaylist.name)
			self.app.processEvents()
			if not hostPlaylist.master and not hostPlaylist.trash:
				dellPlaylist = dell.playlistWithName(hostPlaylist.name)
				if dellPlaylist:
					(tracksAdd,tracksRemove) = dellPlaylist.getSyncInfoWith(hostPlaylist)
					if musicMode=='sync': self.removeTracksFromPlaylist(dellPlaylist,tracksRemove.values())
					if musicMode=='union' or musicMode=='add':
						self.uploadTracksToPlaylist(dellPlaylist,tracksAdd.values())
					if musicMode=='union':
						for trackKey in tracksRemove.keys():
							track = hostHashes.get(trackKey,None)
							if track:
								hostPlaylist.addTrack(track)
		self.refresh()

	def refresh(self):
		self.actions.append({'Action':'Refresh'})
		self.processNextItem()
	
	def cancel(self):
		self.actions = []

	def static_singleton():
		global _dellManagerSingleton
		if _dellManagerSingleton==None:
			_dellManagerSingleton = LMusicDellManager()
		return _dellManagerSingleton
	singleton = staticmethod(static_singleton)

_dellManagerSingleton = None
