/*
 * Copyright (C) 2020 Linux Studio Plugins Project <https://lsp-plug.in/>
 *           (C) 2020 Vladimir Sadovnikov <sadko4u@gmail.com>
 *
 * This file is part of lsp-tk-lib
 * Created on: 17 авг. 2020 г.
 *
 * lsp-tk-lib is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * lsp-tk-lib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with lsp-tk-lib. If not, see <https://www.gnu.org/licenses/>.
 */

#include <lsp-plug.in/test-fw/mtest.h>
#include <lsp-plug.in/tk/tk.h>
#include <private/mtest/tk/common.h>

MTEST_BEGIN("tk.widgets.compound", combogroup)
    typedef struct handler_t
    {
        test_type_t    *test;
        char           *label;
    } handler_t;

    static status_t slot_close(tk::Widget *sender, void *ptr, void *data)
    {
        sender->display()->quit_main();
        return STATUS_OK;
    }

    static status_t slot_key_up(tk::Widget *sender, void *ptr, void *data)
    {
        tk::Window *wnd = tk::widget_cast<tk::Window>(sender);
        ws::event_t *ev = static_cast<ws::event_t *>(data);
        test_type_t *_this = static_cast<test_type_t *>(ptr);

        if ((wnd != NULL) && (ev != NULL) && (ev->nType == ws::UIE_KEY_UP))
        {
            _this->printf("Key up: %c (0x%x)\n", (char)ev->nCode, int(ev->nCode));

            if ((ev->nCode == '+') || (ev->nCode == ws::WSK_KEYPAD_ADD))
                wnd->style()->schema()->scaling()->add(0.25f);
            else if ((ev->nCode == '-') || (ev->nCode == ws::WSK_KEYPAD_SUBTRACT))
                wnd->style()->schema()->scaling()->sub(0.25f);
        }
        return STATUS_OK;
    }

    static status_t slot_mouse_in(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_IN: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_out(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_OUT: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_move(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_MOVE: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_down(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_DOWN: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_up(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_UP: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_click(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_CLICK: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_dbl_click(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_DBL_CLICK: %s\n", h->label);

        return STATUS_OK;
    }

    static status_t slot_mouse_tri_click(tk::Widget *sender, void *ptr, void *data)
    {
        handler_t *h = static_cast<handler_t *>(ptr);
        h->test->printf("MOUSE_TRI_CLICK: %s\n", h->label);

        return STATUS_OK;
    }

    static void destroy_handlers(lltl::parray<handler_t> &vh)
    {
        for (size_t i=0, n=vh.size(); i<n; ++i)
        {
            handler_t *h = vh.uget(i);
            ::free(h->label);
            delete h;
        }
    }

    status_t init_widget(tk::Widget *w, lltl::parray<handler_t> &vh, const char *label)
    {
        status_t res = w->init();
        if (res != STATUS_OK)
            return res;

        handler_t *h = new handler_t;
        if ((h == NULL) || (!vh.add(h)))
            return STATUS_NO_MEM;
        h->test     = this;
        h->label    = ::strdup(label);

        tk::handler_id_t hid;
        hid = w->slots()->bind(tk::SLOT_MOUSE_IN, slot_mouse_in, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_DOWN, slot_mouse_down, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_MOVE, slot_mouse_move, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_UP, slot_mouse_up, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_CLICK, slot_mouse_click, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_DBL_CLICK, slot_mouse_dbl_click, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_TRI_CLICK, slot_mouse_tri_click, h);
        if (hid >= 0) hid = w->slots()->bind(tk::SLOT_MOUSE_OUT, slot_mouse_out, h);

        if (hid < 0)
            res = -hid;

        return res;
    }

    tk::ListBoxItem *add_item(lltl::parray<handler_t> &vh, lltl::parray<tk::Widget> &widgets, tk::ComboGroup *cg, size_t &wid, size_t &col, const char *text)
    {
        tk::ListBoxItem *li;
        LSPString id;
        id.fmt_ascii("cgroupitem-%d", int(wid++));

        MTEST_ASSERT(li = new tk::ListBoxItem(cg->display()));
        MTEST_ASSERT(init_widget(li, vh, id.get_ascii()) == STATUS_OK);
        MTEST_ASSERT(widgets.push(li));
        MTEST_ASSERT(cg->items()->add(li) == STATUS_OK);
        li->text_color()->set_rgb24(next_color(col));
        li->text()->set_raw(text);

        return li;
    }

    MTEST_MAIN
    {
        lltl::parray<handler_t> vh;

        tk::Display *dpy = new tk::Display();
        MTEST_ASSERT(dpy != NULL);

        MTEST_ASSERT(dpy->init(0, NULL) == STATUS_OK);

        lltl::parray<tk::Widget> widgets;
        tk::Widget *w = NULL;
        tk::Window *wnd = new tk::Window(dpy);
        tk::Void *wv = NULL;
        tk::Grid *grid = NULL;
        tk::ComboGroup *cg = NULL;

        // Initialize window
        MTEST_ASSERT(init_widget(wnd, vh, "window") == STATUS_OK);
        MTEST_ASSERT(wnd->title()->set_raw("Test combogroup") == STATUS_OK);
        MTEST_ASSERT(wnd->role()->set_raw("combogroup_test") == STATUS_OK);
        wnd->bg_color()->set_rgb(0, 0.75, 1.0);
        wnd->actions()->set_actions(ws::WA_MOVE | ws::WA_RESIZE | ws::WA_CLOSE);
        wnd->border_style()->set(ws::BS_DIALOG);
        wnd->constraints()->set(160, 100, 640, 400);
        wnd->size()->set(320, 200);
        wnd->slot(tk::SLOT_CLOSE)->bind(slot_close, this);
        wnd->slot(tk::SLOT_KEY_UP)->bind(slot_key_up, this);
        wnd->pointer()->set(ws::MP_TABLE_CELL);
        wnd->layout()->set(-0.5, 0.5, 0.5, 0.5);
        MTEST_ASSERT(widgets.push(wnd));

        // Create grid
        MTEST_ASSERT(grid = new tk::Grid(dpy));
        MTEST_ASSERT(init_widget(grid, vh, "grid") == STATUS_OK);
        MTEST_ASSERT(widgets.push(grid));
        MTEST_ASSERT(wnd->add(grid) == STATUS_OK);
        grid->padding()->set_all(4);
        grid->hspacing()->set(8);
        grid->vspacing()->set(4);
        grid->bg_color()->set_rgb(1.0, 0.75, 1.0);
        grid->rows()->set(1);
        grid->columns()->set(2);

        LSPString id;
        size_t gid = 0;
        size_t wid = 0;
        size_t col = 0;
        {
            // Create empty group
            MTEST_ASSERT(id.fmt_ascii("combogroup-%d", int(gid++)));
            MTEST_ASSERT(cg = new tk::ComboGroup(dpy));
            MTEST_ASSERT(init_widget(cg, vh, id.get_ascii()) == STATUS_OK);
            MTEST_ASSERT(widgets.push(cg));
            MTEST_ASSERT(grid->add(cg) == STATUS_OK);

            tk::ListBoxItem *li = add_item(vh, widgets, cg, wid, col, "Widget 1");
            add_item(vh, widgets, cg, wid, col, "Widget 2");
            add_item(vh, widgets, cg, wid, col, "Widget 3");
            add_item(vh, widgets, cg, wid, col, "Widget 4");
            cg->selected()->set(li);

            for (size_t i=0; i<4; ++i)
            {
                MTEST_ASSERT(id.fmt_ascii("void-%d", int(wid++)));

                MTEST_ASSERT(wv = new tk::Void(dpy));
                MTEST_ASSERT(init_widget(wv, vh, id.get_ascii()) == STATUS_OK);
                MTEST_ASSERT(widgets.push(wv));
                MTEST_ASSERT(cg->add(wv) == STATUS_OK);
                wv->constraints()->set_min(32*(i+1), 32*(i+1));
                wv->bg_color()->set_rgb24(next_color(col));
            }
        }

        // Add custom combo group
        MTEST_ASSERT(id.fmt_ascii("combogroup-%d", int(gid++)));
        MTEST_ASSERT(cg = new tk::ComboGroup(dpy));
        MTEST_ASSERT(init_widget(cg, vh, id.get_ascii()) == STATUS_OK);
        MTEST_ASSERT(widgets.push(cg));
        MTEST_ASSERT(grid->add(cg) == STATUS_OK);
        cg->layout()->set_scale(0.0f);
        cg->layout()->set_align(0.5f, 0.5f);
        cg->border_radius()->set(0);
        cg->border_size()->set(0);
        cg->bg_color()->set_rgb24(0x1b1c22);
        cg->color()->set_rgb24(0x26282f);
        cg->text_color()->set_rgb24(0xa8aed3);
        cg->heading()->set_scale(1.0f);
        cg->text_radius()->set(0);
        cg->text_padding()->set_left(5);
        cg->spin_spacing()->set(5);

        tk::ListBoxItem *li = add_item(vh, widgets, cg, wid, col, "Widget 1");
        add_item(vh, widgets, cg, wid, col, "Widget 2");
        add_item(vh, widgets, cg, wid, col, "Widget 3");
        add_item(vh, widgets, cg, wid, col, "Widget 4");
        cg->selected()->set(li);

        for (size_t i=0; i<4; ++i)
        {
            MTEST_ASSERT(id.fmt_ascii("void-%d", int(wid++)));

            MTEST_ASSERT(wv = new tk::Void(dpy));
            MTEST_ASSERT(init_widget(wv, vh, id.get_ascii()) == STATUS_OK);
            MTEST_ASSERT(widgets.push(wv));
            MTEST_ASSERT(cg->add(wv) == STATUS_OK);
            wv->constraints()->set_min(32*(i+1), 32*(i+1));
            wv->bg_color()->set_rgb24(next_color(col));
        }

        // Show window
        wnd->visibility()->set(true);

        MTEST_ASSERT(dpy->main() == STATUS_OK);

        while ((w = widgets.pop()) != NULL)
        {
            w->destroy();
            delete w;
        }

        dpy->destroy();
        delete dpy;
        destroy_handlers(vh);
    }

MTEST_END







