'''
Defines the abstract base class for all L{AEMonitor} subclasses that will
monitor output from LSR to L{AEOutput} devices.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Base

class OutputMonitor(Base.AEMonitor):
  '''
  Abstract base class for L{OutputMonitor}s that will buffer output. Defines 
  methods that must be overriden to do the buffering of output data.
  ''' 
  def showTalk(self, dev):
    '''
    Abstract method. Buffers the talk command sent to the output device.
    
    @param dev: Device to which the talk was sent
    @type dev: L{AEInput.AEInput}
    @raise IOError: When the monitor is no longer accepting data to buffer 
    @raise NotImplementedError: When this method is not overidden in a subclass
    '''
    raise NotImplementedError
  
  def showStop(self, dev):
    '''
    Abstract method. Buffers the stop command sent to the output device.
    
    @param dev: Device to which the stop was sent
    @type dev: L{AEInput.AEInput}
    @raise IOError: When the monitor is no longer accepting data to buffer 
    @raise NotImplementedError: When this method is not overidden in a subclass
    '''
    raise NotImplementedError
  
  def showStyle(self, dev, cmd, value):
    '''
    Abstract method. Buffers the given style and its new value sent to the 
    output device.
    
    @param dev: Device to which the command was sent
    @type dev: L{AEInput.AEInput}
    @param cmd: Command sent to the device
    @type cmd: integer
    @param value: Value for the command sent to the device
    @type value: object
    @raise IOError: When the monitor is no longer accepting data to buffer 
    @raise NotImplementedError: When this method is not overidden in a subclass
    '''
    raise NotImplementedError
  
  def showString(self, dev, string):
    '''
    Abstract method. Buffers the given string sent to the output device.
    
    @param dev: Device to which the character was sent
    @type dev: L{AEInput.AEInput}
    @param string: String sent to the device
    @type string: string
    @raise IOError: When the monitor is no longer accepting data to buffer 
    @raise NotImplementedError: When this method is not overidden in a subclass
    '''
    raise NotImplementedError