'''
Defines L{Tools} for doing output.

@author: Peter Parente
@author: Pete Brunet
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import LSRConstants
import Base, View, Word
from i18n import _
from Error import *

class Output(Base.TaskTools):
  '''
  Provides methods for doing output on any L{AEOutput} device.
  '''
  # Constants used for getting text attributes
  # attribute and value must observe i8n, need to be able to translate both
  FONT_SIZE = 'size'
  FONT_COLOR = 'fg-color'
  FONT_WEIGHT = 'weight'
  FONT_STYLE = 'style'
  FONT_UNDERLINE = 'underline'
  FONT_STYLES = {'italic':_('italic'), 'bold':_('bold'), 'single':_('single')}

  FONT_ATTRS = {FONT_SIZE:_('size'), FONT_COLOR:_('color'), 
                FONT_WEIGHT:_('weight'), FONT_STYLE:_('style'), 
                FONT_UNDERLINE:('underline')}
  FONT_NON_NUMERIC = [FONT_COLOR, FONT_STYLE, FONT_UNDERLINE] 
  
  def msg(self, text, sem=None, layer=None, dev=None):
    '''
    Outputs the text while disregarding settings.

    @param text: Text to output
    @type text: string
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendString(dev, text, sem, layer, True)

  def out(self, text, sem=None, layer=None, dev=None):
    '''
    Outputs the text while disregarding settings and not sending a flush.

    @param text: Text to output
    @type text: string
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendString(dev, text, sem, layer, False)  
    
  def stop(self, dev=None, sem=None, layer=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. When Stopping is True, this has no effect.
    
    @param dev: The device to send the stop to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    @param sem: The semantic information to stop; defaults to None for the
      default semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    if layer is None:
      layer = self.layer
    if self.tier_state.Stopping:
      self.device_man.sendStop(dev, sem, layer)
      self.tier_state.MayStop = True
  
  def stopNow(self, dev=None, sem=None, layer=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. Unlike L{stop}, this ignores the Stopping setting.
    
    @todo: LW: discard cached index markers
    @param dev: The device to send the stop to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    @param sem: The semantic information to stop; defaults to None for the
      default semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    if layer is None:
      layer = self.layer
    self.device_man.sendStop(dev, sem, layer)
    self.tier_state.MayStop = True
    
  def mayStop(self, dev=None, sem=None, layer=None):
    '''
    Stops the device if L{tier_state}.MayStop is True. When False, resets that 
    flag.

    @param dev: The device to send the stop to; defaults to None for the
      default output device
    @type dev: L{AEOutput.Base}
    @param sem: The semantic information to stop; defaults to None for the
      default semantic
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: Was the stop processed (True) or inhibited (False)?
    @rtype: boolean
    '''
    if layer is None:
      layer = self.layer
    if self.tier_state.MayStop:
      self.stop(dev, sem, layer)
      return True
    else:
      self.tier_state.MayStop = True
      return False
      
  def inhibitMayStop(self):
    '''
    Prevents the next call to L{mayStop} from stopping the device provided to
    that call. Sets the L{tier_state}.MayStop flag to False.    
    '''
    self.tier_state.MayStop = False
    
  def stopAll(self, dev=None):
    '''
    Tells the referenced output device to interrupt current output and clear 
    buffered data. Unlike L{stopNow}, this ignores the layer on which the 
    method was called and any semantic information. Everything on the device
    is stopped.
    
    @todo: LW: discard cached index markers
    @param dev: The device to send the stop to; defaults to None for the 
      default output device
    @type dev: L{AEOutput.Base}
    '''
    self.device_man.sendStop(dev, None, None)
  
  def say(self, text, por=None, sem=None, layer=None, dev=None):
    '''
    Outputs the text while respecting settings.

    @todo: PP: Send index marker before sending each word.
    @param text: Text to output
    @type text: string
    @param por: The point of regard of the start of text, or None when the text 
      is not from the screen
    @type por: L{POR}
    @param sem: The semantic stream on which to send output; defaults to None 
      for the default semantic.
    @type sem: integer
    @param layer: Layer on which the event occurred, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @param dev: The device to send the text to; defaults to None for the default 
      output device
    @type dev: L{AEOutput.Base}
    '''
    if layer is None:
      layer = self.layer
    # first, parse into words
    words = Word.buildWordsFromString(text, self.tier_state, por)
    # handle blank strings
    if len(words) == 1 and self.tier_state.Blank and words[0].isAllBlank():
      self.device_man.sendString(dev, _('blank'), sem, layer, True)
    else:
      # send each word, sending talk command at end
      for w in words:
        # send index markers
        self.device_man.sendString(dev, unicode(w), sem, layer, False)
      self.device_man.sendTalk(dev, sem, layer)
      
  def sayTextAttrs(self, por=None, text=None, template='%s', dev=None):
    '''
    Say font attributes (size, style, color, etc) of a given por.
    
    Attributes of note:
    (http://developer.gnome.org/doc/API/2.0/atk/AtkText.html#AtkTextAttribute)
    ATK_TEXT_ATTR_EDITABLE Either "true" or "false" indicating whether text is
    editable or not
    ATK_TEXT_ATTR_UNDERLINE "none", "single", "double" or "low"
    ATK_TEXT_ATTR_STRIKETHROUGH "true" or "false" whether text is strikethrough
    ATK_TEXT_ATTR_SIZE 	The size of the characters.
    ATK_TEXT_ATTR_SCALE The scale of the characters. The value is a string 
    representation of a double
    ATK_TEXT_ATTR_WEIGHT The numeric weight of the characters.  
    ATK_TEXT_ATTR_LANGUAGE The language used
    ATK_TEXT_ATTR_BG_COLOR The background color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_FG_COLOR The foreground color. The value is an RGB value of 
    the format "u,u,u"
    ATK_TEXT_ATTR_DIRECTION The direction of the text, if set. Values are 
    "none", "ltr" or "rtl"
    ATK_TEXT_ATTR_JUSTIFICATION The justification of the text, if set. Values
    are "left", "right", "center" or "fill"
    ATK_TEXT_ATTR_STYLE The slant style of the text, if set. Values are 
    "normal", "oblique" or "italic"
    
    @param por: L{POR} pointing at an accessible whose value will be read or 
      None when the L{task_por} should be used. The value is read only if the 
      text parameter is None.
    @type por: L{POR}
    @param text: Name text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    '''    
    if text is None:
      attrs = self.getAccAllTextAttrs(por)
      if attrs is None:
        # do nothing if no attributes available
        return
      text = []
      for n, v in attrs:
        # non-numeric output (such as color) needs own check for i8n
        if n in self.FONT_NON_NUMERIC:
          if n == 'fg-color':
            v = self.getColorString(v)
          elif v in self.FONT_STYLES:
            v = self.FONT_STYLES[v]
        # else: leave v as it is (a numeric value)
        if n in self.FONT_ATTRS:
          text.append('%s %s' % (self.FONT_ATTRS[n], v))
        else:
          text.append('%s %s' % (n, v))
      text =', '.join(text)
      print text
    self.say(template % text, por, LSRConstants.SEM_TEXT_ATTR, self.layer, dev)
    
  #############################################################################
  # The following set/get methods are autogenerated by EmitGetSet.py                           
  # These are autogenerated in full (rather than with dynamic code generation or
  # with other Python approaces, such as decorators) to allow epydoc to generate
  # complete documentation for Perk writers
  #############################################################################
  def setContinuous(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Continuous for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Continuous.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Continuous is None:
      raise InvalidStyleError
    else:
      s.Continuous = val
  
  def getContinuous(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Continuous for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Continuous.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Continuous is None:
      raise InvalidStyleError
    else:
      return s.Continuous
  
  def setStoppable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Stoppable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Stoppable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Stoppable is None:
      raise InvalidStyleError
    else:
      s.Stoppable = val
  
  def getStoppable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Stoppable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Stoppable.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Stoppable is None:
      raise InvalidStyleError
    else:
      return s.Stoppable
  
  def setSpeakable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Speakable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Speakable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Speakable is None:
      raise InvalidStyleError
    else:
      s.Speakable = val
  
  def getSpeakable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Speakable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Speakable.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Speakable is None:
      raise InvalidStyleError
    else:
      return s.Speakable
  
  def setSoundable(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Soundable for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Soundable.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Soundable is None:
      raise InvalidStyleError
    else:
      s.Soundable = val
  
  def getSoundable(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Soundable for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Soundable.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Soundable is None:
      raise InvalidStyleError
    else:
      return s.Soundable
  
  def setMute(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Mute for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Mute.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Mute is None:
      raise InvalidStyleError
    else:
      s.Mute = val
  
  def getMute(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Mute for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Mute.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Mute is None:
      raise InvalidStyleError
    else:
      return s.Mute
  
  def setPosition(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Position for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Position.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Position is None:
      raise InvalidStyleError
    else:
      s.Position = val
  
  def getPosition(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Position for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Position.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Position is None:
      raise InvalidStyleError
    else:
      return s.Position
  
  def setLanguage(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Language for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Language.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Language is None:
      raise InvalidStyleError
    else:
      s.Language = val
  
  def getLanguage(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Language for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Language.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Language is None:
      raise InvalidStyleError
    else:
      return s.Language
  
  def getChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Channel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Channel.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Channel is None:
      raise InvalidStyleError
    else:
      return s.Channel
  
  def setChannel(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Channel for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Channel.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Channel is None:
      raise InvalidStyleError
    else:
      s.Channel = min(max(val, s.MinChannel), s.MaxChannel)
  
  def getMinChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinChannel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinChannel.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinChannel is None:
      raise InvalidStyleError
    else:
      return s.MinChannel
  
  def getMaxChannel(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxChannel for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxChannel.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxChannel is None:
      raise InvalidStyleError
    else:
      return s.MaxChannel
  
  def getVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Voice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Voice.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Voice is None:
      raise InvalidStyleError
    else:
      return s.Voice
  
  def setVoice(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Voice for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Voice.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Voice is None:
      raise InvalidStyleError
    else:
      s.Voice = min(max(val, s.MinVoice), s.MaxVoice)
  
  def getMinVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinVoice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinVoice.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinVoice is None:
      raise InvalidStyleError
    else:
      return s.MinVoice
  
  def getMaxVoice(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxVoice for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxVoice.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxVoice is None:
      raise InvalidStyleError
    else:
      return s.MaxVoice
  
  def getVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Volume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Volume.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Volume is None:
      raise InvalidStyleError
    else:
      return s.Volume
  
  def setVolume(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Volume for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Volume.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Volume is None:
      raise InvalidStyleError
    else:
      s.Volume = min(max(val, s.MinVolume), s.MaxVolume)
  
  def getMinVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinVolume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinVolume.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinVolume is None:
      raise InvalidStyleError
    else:
      return s.MinVolume
  
  def getMaxVolume(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxVolume for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxVolume.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxVolume is None:
      raise InvalidStyleError
    else:
      return s.MaxVolume
  
  def getRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Rate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Rate.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Rate is None:
      raise InvalidStyleError
    else:
      return s.Rate
  
  def setRate(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Rate for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Rate.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Rate is None:
      raise InvalidStyleError
    else:
      s.Rate = min(max(val, s.MinRate), s.MaxRate)
  
  def getMinRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinRate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinRate.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinRate is None:
      raise InvalidStyleError
    else:
      return s.MinRate
  
  def getMaxRate(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxRate for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxRate.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxRate is None:
      raise InvalidStyleError
    else:
      return s.MaxRate
  
  def getPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Pitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Pitch.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Pitch is None:
      raise InvalidStyleError
    else:
      return s.Pitch
  
  def setPitch(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Pitch for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Pitch.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Pitch is None:
      raise InvalidStyleError
    else:
      s.Pitch = min(max(val, s.MinPitch), s.MaxPitch)
  
  def getMinPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinPitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinPitch.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinPitch is None:
      raise InvalidStyleError
    else:
      return s.MinPitch
  
  def getMaxPitch(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxPitch for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxPitch.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxPitch is None:
      raise InvalidStyleError
    else:
      return s.MaxPitch
  
  def getGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Gender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Gender.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Gender is None:
      raise InvalidStyleError
    else:
      return s.Gender
  
  def setGender(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Gender for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Gender.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Gender is None:
      raise InvalidStyleError
    else:
      s.Gender = min(max(val, s.MinGender), s.MaxGender)
  
  def getMinGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinGender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinGender.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinGender is None:
      raise InvalidStyleError
    else:
      return s.MinGender
  
  def getMaxGender(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxGender for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxGender.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxGender is None:
      raise InvalidStyleError
    else:
      return s.MaxGender
  
  def getAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Aspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Aspiration.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Aspiration is None:
      raise InvalidStyleError
    else:
      return s.Aspiration
  
  def setAspiration(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Aspiration for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Aspiration.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Aspiration is None:
      raise InvalidStyleError
    else:
      s.Aspiration = min(max(val, s.MinAspiration), s.MaxAspiration)
  
  def getMinAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinAspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinAspiration.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinAspiration is None:
      raise InvalidStyleError
    else:
      return s.MinAspiration
  
  def getMaxAspiration(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxAspiration for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxAspiration.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxAspiration is None:
      raise InvalidStyleError
    else:
      return s.MaxAspiration
  
  def getFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Frication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Frication.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Frication is None:
      raise InvalidStyleError
    else:
      return s.Frication
  
  def setFrication(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Frication for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Frication.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Frication is None:
      raise InvalidStyleError
    else:
      s.Frication = min(max(val, s.MinFrication), s.MaxFrication)
  
  def getMinFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinFrication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinFrication.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinFrication is None:
      raise InvalidStyleError
    else:
      return s.MinFrication
  
  def getMaxFrication(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxFrication for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxFrication.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxFrication is None:
      raise InvalidStyleError
    else:
      return s.MaxFrication
  
  def getIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Intonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Intonation.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Intonation is None:
      raise InvalidStyleError
    else:
      return s.Intonation
  
  def setIntonation(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Intonation for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Intonation.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Intonation is None:
      raise InvalidStyleError
    else:
      s.Intonation = min(max(val, s.MinIntonation), s.MaxIntonation)
  
  def getMinIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinIntonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinIntonation.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinIntonation is None:
      raise InvalidStyleError
    else:
      return s.MinIntonation
  
  def getMaxIntonation(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxIntonation for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxIntonation.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxIntonation is None:
      raise InvalidStyleError
    else:
      return s.MaxIntonation
  
  def getHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of HeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of HeadSize.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.HeadSize is None:
      raise InvalidStyleError
    else:
      return s.HeadSize
  
  def setHeadSize(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of HeadSize for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new HeadSize.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.HeadSize is None:
      raise InvalidStyleError
    else:
      s.HeadSize = min(max(val, s.MinHeadSize), s.MaxHeadSize)
  
  def getMinHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinHeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinHeadSize.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinHeadSize is None:
      raise InvalidStyleError
    else:
      return s.MinHeadSize
  
  def getMaxHeadSize(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxHeadSize for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxHeadSize.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxHeadSize is None:
      raise InvalidStyleError
    else:
      return s.MaxHeadSize
  
  def getInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of Instrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of Instrument.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Instrument is None:
      raise InvalidStyleError
    else:
      return s.Instrument
  
  def setInstrument(self, val, dev=None, sem=None, layer=None):
    '''
    Sets the value of Instrument for the given (dev,sem,layer) 3-tuple.
  
    The device will try to honor the request to change to the new value, but
    can ignore it if it does not support the given parameter and/or value.
  
    @param val: The integer value of the new Instrument.
    @type val: integer
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to change the value.
    @type sem: integer 
    @param layer: Layer on which to change the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.Instrument is None:
      raise InvalidStyleError
    else:
      s.Instrument = min(max(val, s.MinInstrument), s.MaxInstrument)
  
  def getMinInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MinInstrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MinInstrument.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MinInstrument is None:
      raise InvalidStyleError
    else:
      return s.MinInstrument
  
  def getMaxInstrument(self, dev=None, sem=None, layer=None):
    '''
    Gets the value of MaxInstrument for the given (dev,sem,layer) 3-tuple.
  
    @param dev: The device on which to get the value; defaults to None for the 
      default output device.
    @type dev: L{AEOutput.Base}
    @param sem: The semantic on which to get the value.
    @type sem: integer 
    @param layer: Layer on which to get the value, one of L{Task.FOCUS_LAYER},
      L{Task.TIER_LAYER}, or L{Task.BACKGROUND_LAYER}
    @type layer: integer
    @return: the current value of MaxInstrument.
    @rtype: integer
    '''
    s = self.device_man.getStyle(dev, sem, layer)
    if s.MaxInstrument is None:
      raise InvalidStyleError
    else:
      return s.MaxInstrument
  
  ##############################################################################
  # The following say___() methods are autogenerated by EmitSay.py                           
  # These are autogenerated in full (rather than with dynamic code generation or
  # with other Python approaches, such as decorators) to allow epydoc to 
  # generate complete documentation for Perk writers
  ##############################################################################
  
  def sayHardline(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the text from the L{POR} to the next hard line.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Hardline text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      end_por = self.getEndofHardline(por)
      text = self.getAccTextBetween(por, end_por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_HARDLINE, self.layer, dev)
    return True
  
  def sayWord(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the word at the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Word text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getWordText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_WORD, self.layer, dev)
    return True
  
  def sayChar(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the character at the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Char text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getCharText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_CHAR, self.layer, dev)
    return True
  
  def sayTitle(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the title of the accessible at the L{POR}, typically its name.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Title text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_TITLE, self.layer, dev)
    return True
  
  def sayLabel(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the label at the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Label text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccLabel(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_LABEL, self.layer, dev)
    return True
  
  def sayItem(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the item at the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Item text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getItemText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ITEM, self.layer, dev)
    return True
  
  def sayWindow(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the window title containing the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Window text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getWindowTitle(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_WINDOW, self.layer, dev)
    return True
  
  def sayApp(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the name of the app containing the L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: App text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAppName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_APP, self.layer, dev)
    return True
  
  def sayFont(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the name of the font at current L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Font text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      # @todo: DBC: this will not work for all applications, improve this
      text = self.getAccTextAttr('family-name', por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_FONT, self.layer, dev)
    return True
  
  def sayColor(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the color of the text at given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Color text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccTextAttr('fg-color', por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_COLOR, self.layer, dev)
    return True
  
  def sayRole(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the role of the accessible at the given L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Role text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccRoleName(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ROLE, self.layer, dev)
    return True
  
  def sayLevel(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the level number at current L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Level text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getAccLevel(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_LEVEL, self.layer, dev)
    return True
  
  def sayState(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the states at current L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: State text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getStateText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_STATE, self.layer, dev)
    return True
  
  def sayStatus(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs the status message at the current L{POR}.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Status text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
      text = self.getItemText(por)
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_STATUS, self.layer, dev)
    return True
  
  def sayError(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs an LSR system error message.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Error text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
  
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_ERROR, self.layer, dev)
    return True
  
  def sayInfo(self, por=None, text=None, template=u'%s', dev=None):
    '''
    Outputs an informational LSR system message.
  
    @param por: L{POR} pointing at an accessible whose value will be read or
      None when the L{task_por} should be used. The value is read only if the
      text parameter is None.
    @type por: L{POR}
    @param text: Info text fetched from the L{POR}
    @type text: string
    @param template: Template for text to be spoken. Defaults to %s indicating 
      only the semantic value should be spoken.
    @type template: string
    @param dev: The device to change; defaults to None for the default output
      device.
    @type dev: L{AEOutput.Base}
    @return: Was anything output?
    @rtype: boolean
    '''
    if text is None:
      por = por or self.task_por
  
    if text is None:
      return False
    self.say(template % text, por, LSRConstants.SEM_INFO, self.layer, dev)
    return True
  
