/* vim:sw=4:sts=4
 * This is part of Lua-Gtk2, the Lua binding for the Gtk2 library.
 * Functions here give access to functions, structs and enums of Gtk2.
 *
 * Copyright (C) 2005, 2007 Wolfgang Oertl
 *
 * Exported functions:
 *   luagtk_dl_init
 *   luagtk_make_func_name
 *   luagtk_g_type_from_name
 *   find_attribute
 *   find_enum
 *   find_func
 *   find_struct
 */

#include "luagtk.h"
#include "luagtk_hash.h"
#include <string.h>	    // strcmp
#include <stdlib.h>	    // bsearch
#include <lauxlib.h>	    // luaL_checkstring


// --- Windows settings ---

#ifdef LUAGTK_win32
 #define DLLOOKUP(dl_handle, name) (FARPROC) GetProcAddress(dl_handle, name)
 #ifdef RUNTIME_LINKING
  #define DLLOAD(name) LoadLibrary(name)
 #else
  #define DLLOAD(name) GetModuleHandle(name)
 #endif
 #define DLCLOSE(name)
 #define RESOLVE_LIBRARIES

static const char dll_list[] =
    "libgtk-win32-2.0-0.dll\000"
    "libgdk-win32-2.0-0.dll\000"
    "libpango-1.0-0.dll\000"
    "libglib-2.0-0.dll\000"
    "libgobject-2.0-0.dll\000"
    "libgdk_pixbuf-2.0-0.dll\000"
 #ifdef HAVE_LIBGTKHTML
    "libgtkhtml-2.dll\000"
 #endif
;

#endif

// --- Linux ---

#ifdef LUAGTK_linux
 #define DLLOOKUP(dl_handle, name) dlsym(dl_handle, name)
 #define DLLOAD(name) dlopen(name, RTLD_LAZY)
 #define DLCLOSE(name) dlclose(name)

 #ifdef RUNTIME_LINKING
  #define RESOLVE_LIBRARIES
static const char dll_list[] =
    "/usr/lib/libgtk-x11-2.0.so\000"
    "/usr/lib/libatk-1.0.so\000"
  #ifdef HAVE_LIBGTKHTML
    "/usr/lib/libgtkhtml-2.so\000"
  #endif
;

 #endif
#endif

/* dll_list etc. needed when the Gtk libraries are to be dynamically loaded.
 * On Windows, the list is needed even without runtime linking to get the
 * module handles.
 */



/**
 * Given a class name in the form "GtkVBox" and a method name like
 * "pack_start", construct a function name like "gtk_vbox_pack_start".
 *
 * All letters are converted to lowercase, but before initially uppercase
 * letters, an underscore is inserted, unless a single letter would then
 * be surrounded by underscores (as is the case for GtkVBox, which turns
 * into gtk_vbox, and not gtk_v_box).
 *
 * Returns 0 on success, else 1.  The only possible error is that the
 * buffer is not big enough.
 */
int luagtk_make_func_name(char *buf, int buf_size, const char *class_name,
    const char *attr_name)
{
    const char *s = class_name;
    char *out = buf;

    // each loop adds one or two characters to the output; a final 0 byte
    // is also required, therefore +2.
    while (*s) {
	if (out - buf + 2 >= buf_size)
	    return 1;
	if (*s >= 'A' && *s <= 'Z') {
	    // the s==class_name+1 test allows g_context_xxx functions.
	    if (s == class_name+1 || (out >= buf+2 && out[-2] != '_'))
		*out++ = '_';
	    *out++ = *s + ('a' - 'A');
	} else {
	    *out++ = *s;
	}
	s++;
    }

    if (attr_name) {
	if (out - buf + 1 + strlen(attr_name) + 1 >= buf_size)
	    return 1;
	*out++ = '_';
	strcpy(out, attr_name);
    } else
	*out = 0;

    return 0;
}


/**
 * Determine the type number for the given class.
 *
 * After using the class for the first time, g_type_from_name returns the ID;
 * otherwise is is required to call the get_type function for this class.
 *
 * @param s  Name of the type
 * @return  0 if not found, else the GType (an integer)
 */
int luagtk_g_type_from_name(const char *s)
{
    int type_nr = g_type_from_name(s);

    // Type already initialized - return it.  This is the usual case.
    if (type_nr)
	return type_nr;

    // cairo_xxx are types that are not in the GType system.  Don't try
    // to get a type_nr for them.
    if (!strncmp(s, "cairo_", 6))
	return 0;

    // Determine the name of the corresponding get_type function.
    char func_name[60];
    struct func_info fi;
    if (luagtk_make_func_name(func_name, sizeof(func_name), s, "get_type"))
	return 0;

    // If s is not a valid type, then the _get_type function doesn't exist
    if (!find_func(func_name, &fi))
	return 0;

    // The function exists - call it.  It returns the type number.
    GType (*func)();
    func = fi.func;
    type_nr = func();

    // Force the class to be initialized - makes g_signals_list_ids work,
    // for example.  Might not always be necessary.  Only for types that
    // are classed - could also use g_type_query.
    gpointer cls = g_type_class_peek(type_nr);
    if (cls) {
	gpointer cls = g_type_class_ref(type_nr);
	g_type_class_unref(cls);
    }

    return type_nr;
}

/**
 * Writes the type's name into the given buffer.  Its length should be
 * LUAGTK_NAMELEN bytes long.
 *
 * @param L  Lua state
 * @param type_idx  Type ID
 */
void luagtk_type_name(lua_State *L, int type_idx, char *buf)
{
    // type_idx must be 1 .. type_count
    if (type_idx <= 0 || type_idx >= type_count)
	luaL_error(L, "%s type_idx out of range: %d", msgprefix, type_idx);

    const struct type_info *ti = type_list + type_idx;

    // fundamental_id must be 1 .. ffi_type_count-1
    const struct ffi_type_map_t *tm = ffi_type_map + ti->fundamental_id;

    if (ti->is_const) {
	strcpy(buf, "const ");
	buf += 6;
    }

    // use the type's name, not the fundamental type's name.
    strcpy(buf, TYPE_NAME(ti));
    buf += strlen(buf);

    // Add the pointers.  Arrays are currently not supported.
    int i;
    for (i=0; i<tm->indirections; i++)
	*buf ++ = '*';

    *buf = 0;
}



/**
 * Search for the string in the ENUM table.  The result might be an enum
 * (typed integer), flags (typed integer, can be ORed together), a regular
 * integer, or a string.
 *
 * @param key The name of the ENUM to look for
 * @param keylen Length of key.  If -1, key must be zero terminated.
 * @param result (output) value of the ENUM
 * @param type_idx (output) number of the ENUM entry in the type list
 * @return 0=error, 1=ENUM found, 2=int found (in *result),
 *    3=string found (on Lua stack)
 */
int find_enum(lua_State *L, const char *key, int keylen, int *result,
    int *type_idx)
{
    int datalen, val;

    if (keylen < 0)
	keylen = strlen(key);
    const unsigned char *res = hash_search(&hash_info_enums, key, keylen,
	&datalen);
    if (!res)
	return 0;

    *type_idx = 0;
    const unsigned char *res_end = res + datalen;

    /* get the flag byte */
    unsigned char c = *res++;

    /* low 5 bits of first byte are either for type_idx or value */
    val = c & 0x1f;

    /* High bit set if type_idx included; fetch another byte for a total of 13
     * bits. */
    if (c & 0x80) {
	*type_idx = (val << 8) + *res++;
	val = 0;
    }

    /* If it is a string, the rest is just that.  Note that a string can have
     * a type_idx, even though that currently doesn't happen - all ENUMs are
     * numeric. */
    if (c & 0x40) {
	lua_pushlstring(L, (char*) res, datalen - 1);
	return 3;
    }
    
    /* collect all bytes for the number */
    while (res < res_end)
	val = (val << 8) + *res++;

    /* bit 5 is the negative sign */
    if (c & 0x20)
	val = -val;

    *result = val;

    /* if type_idx is set, ENUM or FLAG, else regular integer */
    return *type_idx ? 1 : 2;
}


/**
 * Look for an attribute of the given class.  The attributes are ordered
 * by their offset within the structure, but not strictly - unions have
 * their attributes at the same offset.
 */
const struct struct_elem *find_attribute(const struct type_info *si,
    const char *attr_name)
{
    const struct struct_elem *e, *e_end;

    /* Search up to the start of the next entry. */
    e = elem_list + si->st.elem_start;
    e_end = e + si->st.elem_count;

    for (; e < e_end; e++)
	if (!strcmp(attr_name, STRUCT_ELEM_NAME(e)))
	    return e;

    return NULL;
}

typedef struct {
    unsigned char *base;
    unsigned int alloc;
    unsigned int used;
} array_t;

array_t sorted_structs = { 0, 0, 0 };

typedef struct {
    unsigned short type_idx;
} array_elem;

/**
 * Compare two types.  First by name, if equal by indirection count.
 */
static int struct_sorter(const array_elem *a, const array_elem *b)
{
    const struct type_info *ti1 = type_list + a->type_idx;
    const struct type_info *ti2 = type_list + b->type_idx;

    const char *name1 = TYPE_NAME(ti1);
    const char *name2 = TYPE_NAME(ti2);

    int rc = strcmp(name1, name2);
    if (rc)
	return rc;

    const struct ffi_type_map_t *ffi1 = ffi_type_map + ti1->fundamental_id;
    const struct ffi_type_map_t *ffi2 = ffi_type_map + ti2->fundamental_id;

    rc = ffi1->indirections - ffi2->indirections;
    if (rc)
	return rc;

    rc = ti1->is_const - ti2->is_const;
    if (rc)
	return rc;

    // XXX Same for arrays.  Not handled properly yet!
    /// printf("Same type? %s %d, %d\n", name1, a->type_idx, b->type_idx);
    return 0;
}

static inline void array_append(array_t *a, unsigned char *p, int len)
{
    if (a->used + len >= a->alloc) {
	a->alloc = a->alloc ? a->alloc * 2 : 1024;
	a->base = (unsigned char*) realloc(a->base, a->alloc);
    }
    memcpy(a->base + a->used, p, len);
    a->used += len;
}


/**
 * Create a sorted type index.
 *
 * The list of types is generated at build time being sorted by usage
 * frequency, so that lower type_idx are more frequent, because they can be
 * stored with just one byte if lower than 0x80.  To quickly look up a type
 * name, a list sorted by name is required.  This is simply a list of type_idx
 * (2 bytes each), sorted by name.  At startup this has to be created once.
 *
 * Note that the type name does NOT include the "*" for pointers, so that
 * types with the same name appear.  They differ in their "indirections"
 * count, though, and are sorted by that.
 */
static void sort_structures()
{
    const struct type_info *ti = type_list,
	*ti_end = ti + type_count;
    const struct ffi_type_map_t *ffi;

    ti ++;
    while (ti < ti_end) {
	ffi = ffi_type_map + ti->fundamental_id;
	int val = ti - type_list;
	array_append(&sorted_structs, (unsigned char*) &val, 2);
	ti ++;
    }
    qsort(sorted_structs.base, sorted_structs.used / sizeof(array_elem),
	sizeof(array_elem), (int(*)(const void*,const void*))&struct_sorter);

#if 0
    printf("initialized sorted structure list, used %d/%d bytes\n",
	sorted_structs.used, sorted_structs.alloc);
#endif

#if 0
    array_elem *p = (array_elem*) sorted_structs.base,
	*p_end = p + sorted_structs.used / sizeof(array_elem);
    int duplicates = 0;
    int name_ofs_prev = -1;

    while (p < p_end) {
	ti = type_list + p->type_idx;
	ffi = ffi_type_map + ti->fundamental_id;
	printf("%s %s%d\n", TYPE_NAME(ti), ti->is_const ? "const " : "",
	    ffi->indirections);
	if (name_ofs_prev == ti->name_ofs)
	    duplicates ++;
	name_ofs_prev = ti->name_ofs;
	p ++;
    }
    printf("Number of duplicate strings: %d\n", duplicates);
#endif
}

/* comparison for binary search */
static int struct_compare(const void *key, const array_elem *e)
{
    const struct type_info *ti = type_list + e->type_idx;
    const char *name = TYPE_NAME(ti);
    return strcmp(key, name);
}


/**
 * Structures are primarily looked up by their number, but in order to resolve
 * the hierarchy a lookup by name is also required.  A hash table would gain
 * little, as the binary search is quite fast, too.  Would be easier on
 * CPU caches, though.
 */
const struct type_info *find_struct(const char *class_name, int indirections)
{
    if (!sorted_structs.base)
	sort_structures();

    array_elem *p;

    // type_list[0] is a dummy entry; don't bother to search in it.
    // struct_count disregards this first entry, also the last sentinel.
    p = (array_elem*) bsearch(class_name, sorted_structs.base,
	sorted_structs.used / sizeof(array_elem), sizeof(array_elem),
	(int(*)(const void*,const void*))struct_compare);

    if (!p)
	return NULL;

    const struct type_info *ti = type_list + p->type_idx;
    int name_ofs = ti->name_ofs, n;
    const struct ffi_type_map_t *ffi;

    for (;;) {
	ffi = ffi_type_map + ti->fundamental_id;
	n = ffi->indirections - indirections;
	if (!n)
	    return ti;
//	printf("Indirection count mismatch: %s, %d req=%d\n", class_name,
//	    ffi->indirections, indirections);
	p += n < 0 ? 1 : -1;
	ti = type_list + p->type_idx;
//	printf("  moved, now at %s %d\n", TYPE_NAME(ti), 99);
	if (ti->name_ofs != name_ofs) {
	    //printf("oops, now at different name\n");
	    break;
	}
    }

    printf("not found: %s indirections=%d\n", class_name, indirections);
    return NULL;
}


/* globals */
static int dl_count = 0;

#ifdef LUAGTK_linux
void *dl_handle[MAX_DLL];
#endif

#ifdef LUAGTK_win32
HINSTANCE dl_handle[MAX_DLL];
#endif


/*-
 * Find the symbol in the shared library.  It most likely is a function, but
 * might also be a global variable.
 */
static void *_find_symbol(const char *name)
{
#if !defined(LUAGTK_win32) && !defined(RUNTIME_LINKING)
    /* gtk.so is already linked with the shared library - dlopen not used. */
    return DLLOOKUP(NULL, name);
#else
    /* use the list of available handles */
    int i;
    void *p = NULL;

    for (i=0; i<dl_count; i++)
	if ((p = DLLOOKUP(dl_handle[i], name)))
	    break;
    return p;
#endif
}

#ifdef RUNTIME_LINKING

/**
 * Functions that can't be found during dynamic loading of the libraries
 * are replaced by this.  Until they are called, we can continue.
 * A pity that it's not known which function isn't available.  This could
 * be solved by using closures, but then it probably doesn't help much.
 */
static void unavailable_function()
{
    printf("%s ERROR - an unavailable function was called.\n", msgprefix);
    exit(1);
}

#endif

/**
 * Load the dynamic libraries.  Returns 0 on error.
 *
 * On Linux with automatic linking, nothing has to be done; the dynamic linker
 * already has loaded libgtk+2.0 and its dependencies.
 */
int luagtk_dl_init()
{
#ifdef RESOLVE_LIBRARIES
    const char *dlname;

    /* load dynamic libraries */
    for (dlname=dll_list, dl_count=0; *dlname; dlname+=strlen(dlname)+1) {
	if (dl_count >= MAX_DLL) {
	    printf("%s Max. number of libraries exceeded!\n", msgprefix);
	    return 0;
	}
	if (!(dl_handle[dl_count] = DLLOAD(dlname))) {
	    printf("%s Can't load dynamic library %s\n", msgprefix, dlname);
	    // library loading can fail; no problem.  Not all libraries are
	    // always required!
	    continue;
	    // return 0;
	}
	dl_count ++;
    }
#endif

#ifdef RUNTIME_LINKING
    /* If this library isn't linked with the Gtk libraries, then all
     * function calls are indirect through the dl_link table.  Fill in all
     * these pointers.
     *
     * The list of names is the string dl_names, where the names are separated
     * by \0; at the end there's \0\0.
     */
    linkfuncptr *dl = dl_link;
    const char *s;
    int err = 0;

    for (s=dl_names; *s; s += strlen(s) + 1, dl++) {
	if (G_UNLIKELY(!(*dl = _find_symbol(s)))) {
	    printf("%s symbol %s not found in dynamic library.\n",
		msgprefix, s);
	    *dl = unavailable_function;
	    err ++;
	}
    }
    // if (err) return 0;
#endif

    return 1;
}


/**
 * Look for the function in the dynamic library.  If it is not found, this is
 * not an error, because many tries for different widget types may be
 * necessary to find one method.
 * Returns 0 if the function hasn't been found, 1 otherwise.
 */
int find_func(const char *func_name, struct func_info *fi)
{
    int keylen = strlen(func_name), datalen;

    /* Find the function in the hash table.  Note that when using
     * minimal hash tables (see documentation), a function may be found
     * even if it doesn't exist.  This is NOT recommended.
     */
    if (!(fi->args_info = hash_search(&hash_info_funcs, func_name, keylen,
	&datalen)))
	return 0;

    fi->func = _find_symbol(func_name);
    if (fi->func) {
	fi->name = func_name;
	fi->args_len = datalen;
	return 1;
    }

    // symbol not found.
    return 0;
}


