#!/usr/bin/env python
# -*- coding: utf-8 -*-
# -*- Mode: Python -*-
# vi:si:ai:et:sw=4:sts=4:ts=4
#
#
# Copyright Nicolas Bertrand (nico@inattendu.org), 2009
#
# This file is part of Luciole.
#
#    Luciole is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    Luciole is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Luciole.  If not, see <http://www.gnu.org/licenses/>.
#
#
import pygtk
pygtk.require('2.0')

import sys

import gobject

import pygst
pygst.require('0.10')
import gst
import gst.interfaces
import gtk

import luciole_constants as MCONST
import luciole_tools as MT
import os.path


class PhotoSaveBin(gst.Bin) :
    """ Bin Pad to save in jpeg format image from stream. Stream alyays encoded but the result will be saved to file only when a capture is done """
    def __init__(self) :
        """ Pilpeline desc : jpegenc + fakseink """

        self.__gobject_init__()
        ImageBinElems=[]
        
        MyJpegenc = gst.element_factory_make("jpegenc","MyJpegenc")                 # jpeg encoding
        ImageBinElems.append(MyJpegenc)   
        
        photosink = gst.element_factory_make("fakesink","PhotoSink")
        ImageBinElems.append(photosink)   
        
        for elem in ImageBinElems : self.add(elem)
        
        gst.element_link_many(MyJpegenc,photosink)
        
        self.add_pad(gst.GhostPad('sink', MyJpegenc.get_pad('sink')))


class InputImageBin(gst.Bin):
    """ Load image to mix with stream"""
    def get_image2Mix(self): return self.__image2Mix
    def set_image2Mix(self, value): 
        if ( os.path.isfile(value) ):
            self.__image2Mix = value
            image2Mix = property(get_image2Mix, set_image2Mix, None, " Path to Image To Mix ")
  
    def get_framerate(self): return self.__framerate
    def set_framerate(self, value): 
        if ( os.path.isfile(value) ):
            self.__framerate = value
    framerate = property(get_framerate, set_framerate, None, " Framerate for displaying input image ")

    def get_alphaValueImage(self): return self.__alphaValueImage
    def set_alphaValueImage(self, value): 
        self.__alphaValueImage = value
        self.MyAlpha.set_property("alpha",self.__alphaValueImage)    
    alphaValueImage = property(get_alphaValueImage, set_alphaValueImage, None, " Framerate for displaying input image ")

    def __init__(self,image,alphaValue,framerate=25) :
        self.__gobject_init__()
        self.__image2Mix = image
        self.__framerate =  framerate
        self.__alphaValueImage = alphaValue
        ImageBinElems=[]
    
        # load input image
        self.InputImage= gst.element_factory_make('multifilesrc')
        self.InputImage.set_property('location',self.__image2Mix)
        ImageBinElems.append(self.InputImage)
        
        # filter to set frame rate
        self.ImageFilter = gst.element_factory_make("capsfilter")
        self._filter_caps_string = "image/jpeg, framerate=(fraction)%s/1 ,width=%s,height=%s "

        caps_string = self._filter_caps_string%(self.__framerate,MCONST.VIDEO_PAL_RES[0],MCONST.VIDEO_PAL_RES[1])
        caps = gst.Caps(caps_string)
        self.ImageFilter.set_property("caps", caps)
        ImageBinElems.append(self.ImageFilter)
     
        # dec jpeg as stream
        MyJpecdec = gst.element_factory_make('jpegdec')
        ImageBinElems.append(MyJpecdec)
    
        # scale image
        Myvideoscale = gst.element_factory_make('videoscale') 
        ImageBinElems.append(Myvideoscale)
        
        # needed again for setting aspect ratio
        InputImageFilter = gst.element_factory_make('capsfilter',"InputImageFilter")    # set filter on scale
        caps2 = gst.Caps("video/x-raw-yuv, width=%s,height=%s"%(MCONST.VIDEO_PAL_RES[0],MCONST.VIDEO_PAL_RES[1]))
        InputImageFilter.set_property("caps", caps2)
        ImageBinElems.append(InputImageFilter)
    
        self.MyAlpha = gst.element_factory_make("alpha")                             # set alpha of image
        self.MyAlpha.set_property("alpha",self.__alphaValueImage )    
        ImageBinElems.append(self.MyAlpha)
        
        for elem in ImageBinElems : self.add(elem)
        
        gst.element_link_many(self.InputImage,self.ImageFilter, MyJpecdec, Myvideoscale,InputImageFilter, self.MyAlpha)
        
        self.add_pad(gst.GhostPad('src', self.MyAlpha.get_pad('src'))) 



class MixStreamAndImage(gst.Bin) :
    """ Mix the input stream with an image """
    
    def get_image2Mix(self): return self.__image2Mix
    def set_image2Mix(self, value): 
        if ( os.path.isfile(value) ):
            self.__image2Mix = value
            self.ImageBin.image2Mix = self.__image2Mix
    image2Mix = property(get_image2Mix, set_image2Mix, None, " Path to Image To Mix ")

    def get_alphaValueImage(self): return self.__alphaValueImage
    def set_alphaValueImage(self, value):
        self.__alphaValueImage = value
        self.ImageBin.alphaValueImage = self.__alphaValueImage
    alphaValueImage = property(get_alphaValueImage, set_alphaValueImage, None, " alpha value to change. ")

    def __init__(self,alphaValueStream, image, alphaValueImage) :
        self.__gobject_init__()
        self.__image2Mix = image
        self.__alphaValueImage=alphaValueImage
        ImageBinElems=[]
        
        queueB2 = gst.element_factory_make("queue","queueB2")                   # queue of branch 2
        ImageBinElems.append(queueB2)
        
        Myvideorate = gst.element_factory_make('videorate')                    #  rate transformation
        ImageBinElems.append(Myvideorate)
        
        Myvideoscale = gst.element_factory_make('videoscale')                   # scale image
        ImageBinElems.append(Myvideoscale)
        
        VideoFilter = gst.element_factory_make("capsfilter")
        caps = gst.Caps("video/x-raw-yuv, framerate=(fraction)10/1, width=720,height=576") # scale image with caps
        VideoFilter.set_property("caps", caps)
        ImageBinElems.append(VideoFilter)
    
        MyVideoBox = gst.element_factory_make('videobox', "MyVideoBox")         # videoBox where input video stream is displayed
        MyVideoBox.set_property('border-alpha', 0)
        MyVideoBox.set_property('alpha', alphaValueStream)
          
        ImageBinElems.append(MyVideoBox)
        
        # Video mixer : mix input in videobox with image
        MyVideomixer = gst.element_factory_make('videomixer', "MyVideomixer")   # video mixer
        ImageBinElems.append(MyVideomixer)
      
        # input image to mix
        # create Bin for input Image
        self.ImageBin = InputImageBin(self.__image2Mix,alphaValueImage);
        ImageBinElems.append(self.ImageBin)
        
        for elem in ImageBinElems : self.add(elem)
        
        #input stream
        gst.element_link_many(queueB2,Myvideorate,Myvideoscale,VideoFilter,MyVideoBox,MyVideomixer)
        # mix with image
        self.ImageBin.link(MyVideomixer)
    
        # this module has a sink pad and a src pad
        self.add_pad(gst.GhostPad('sink', queueB2.get_pad('sink'))) 
        self.add_pad(gst.GhostPad('src', MyVideomixer.get_pad('src')))  




class ScaleBin(gst.Bin) :
    """ This Bin allow to fit a display in fixed size with preserving aspect ratio """
    
    def __init__(self,width=640,height=480) :
        """
        Description of pipeline : 
                           _________________________________________________
                          |                                               | 
        -->--ghostPad --> | videoscale -->  capsfilter  --> videobox -->  | ghostPad -->--
                          |_______________________________________________|_ 
        """

        self.__gobject_init__()
        ImageBinElems=[]
       
        # compute the image size if neeeded    
        image_size_d = self.calc_video_size(float(width),float(height),MCONST.VIDEO_PAL_RES[0],MCONST.VIDEO_PAL_RES[1])
        
        Myvideoscale = gst.element_factory_make('videoscale')                   # scale image
        ImageBinElems.append(Myvideoscale)
        
        MyVideoFilter = gst.element_factory_make("capsfilter")
        caps = gst.Caps("video/x-raw-yuv,width=%s,height=%s"%(image_size_d['width'],image_size_d['height'])) # scale image with caps
        MyVideoFilter.set_property("caps", caps)
        ImageBinElems.append(MyVideoFilter)
    
        MyVideoBox = gst.element_factory_make('videobox', "MyVideoBox")         # videoBox where input video stream is displayed
        MyVideoBox.set_property('fill',0 )        # fill with black
        # borders are negative values for videobox . postive value used dor crop
        MyVideoBox.set_property('top', -1*image_size_d['top'] )
        MyVideoBox.set_property('bottom', -1*image_size_d['bottom'])
        MyVideoBox.set_property('left', -1*image_size_d['left'])
        MyVideoBox.set_property('right', -1*image_size_d['right'])
        ImageBinElems.append(MyVideoBox)

        # add bins
        for elem in ImageBinElems : self.add(elem)
        
        # link bins
        gst.element_link_many(Myvideoscale,MyVideoFilter,MyVideoBox )
        # this module has a sink pad and a src pad
        self.add_pad(gst.GhostPad('sink', Myvideoscale.get_pad('sink'))) 
        self.add_pad(gst.GhostPad('src', MyVideoBox.get_pad('src'))) 
 
    def calc_video_size(w_src, h_src, w_dst, h_dst):
        """ copmpute the scale for the image : The new image size if needed to scale + border to fill in target destination""" 
        w_scale = (w_src*1.0) / w_dst
        h_scale = (h_src*1.0) / h_dst
        if w_scale >  h_scale :
            scale_f = w_scale
        else  :
            scale_f = h_scale
        if scale_f > 1.0 :
            # Scale to fill in defined zone
            (w_final , h_final) =  (int(w_src/scale_f) , int(h_src/scale_f))
        else :
            # input source lower than target don't scale it
            (w_final , h_final) =  (int(w_src) , int(h_src))

        (border_left, border_right) = ScaleBin.calc_border_size(w_final,w_dst)
        (border_top, border_bottom) = ScaleBin.calc_border_size(h_final,h_dst)
        size_dict = dict()
        size_dict['width'] =  int(w_final)
        size_dict['height'] =  int(h_final)
        size_dict['bottom'] =  int(border_bottom)
        size_dict['top'] =  int(border_top)
        size_dict['left'] =  int(border_left)
        size_dict['right'] =  int(border_right)
        return size_dict 
    calc_video_size = staticmethod(calc_video_size) 

    def calc_border_size(src,tgt) :
        """ This function computes the size of the borders. Images is centered so borders size are equals at one pixel"""
        (border_x, border_y) = (0,0)

        if src < tgt :
            border_size = int( (tgt - src)/2)
            if border_size*2 + src < tgt :
                # to esnsure that border + src = target size
                # means that border size or src are odd; so the first is filled with the missing pixel 
                border_x = int(tgt - src -border_size)
                border_y = border_size
            else :
                (border_x,border_y) = ( border_size, border_size)
    
        return (border_x, border_y)
    calc_border_size = staticmethod(calc_border_size) 


class WebcamInputBin(gst.Bin) :
    """ gst.Bin for Web cam input"""
    def __init__(self,width=640,height=480,source_input='v4l2src',device='/dev/video0') :
        """build Dv cam input bin
        Description of pipeline : 
               __________________________________________________________
              |                                                          | 
        -->-- | v4lsrc -->  capsfilter --> ffmpegcolorspace --> ScaleBin | ghostPad -->--
              |__________________________________________________________|_ 
         
        """

        self.__gobject_init__()
        ImageBinElems=[]
          
        MyVideoSrc = gst.element_factory_make(source_input)
        MyVideoSrc.set_property("device", device)
        ImageBinElems.append(MyVideoSrc)
        
        MyVideoSrcFilter = gst.element_factory_make("capsfilter", "MyVideoSrc")

        caps = gst.Caps("video/x-raw-yuv, width=%s,height=%s"%(width,height) )
        MyVideoSrcFilter.set_property("caps", caps)
        ImageBinElems.append(MyVideoSrcFilter)
        
        MyyuvInput = gst.element_factory_make('ffmpegcolorspace')
        ImageBinElems.append(MyyuvInput)
        
        # scale Webcam Display to DV SCALE 
        MyScaleBin = ScaleBin(width,height)
        ImageBinElems.append(MyScaleBin)
                
        for elem in ImageBinElems : self.add(elem)
        
        gst.element_link_many(MyVideoSrc,MyVideoSrcFilter,MyyuvInput,MyScaleBin)

        #this bin has only a src ghostPad
        self.add_pad(gst.GhostPad('src', MyScaleBin.get_pad('src'))) 
        
     
      
class DvInputBin(gst.Bin) :
    """ gst.Bin for DV cam input"""
    def __init__(self) :
        """build Dv cam input bin
        Description of pipeline : 
               ___________________________________________________________________
              |                                                                   |
        -->-- | dv1394src -->--  dvdemux -->-- dvdec -->-- ffmpegcolorspace -->-- | ghostPad -->--
              |___________________________________________________________________| 
         
         """
        self.__gobject_init__()
        ImageBinElems=[]
        MyVideoSrc = gst.element_factory_make('dv1394src')
        ImageBinElems.append(MyVideoSrc)
        
        #set demuxer ( separation image/sound)
        MyDemux = gst.element_factory_make("dvdemux", "demuxer") 
        ImageBinElems.append(MyDemux)
        MyDemux.connect("pad-added", self.MyDemux_callback)
    
        self.MyVideodecoder = gst.element_factory_make("dvdec", "video-decoder")
        ImageBinElems.append(self.MyVideodecoder)
    
        # ffmpeg needed for pipe ; without the next elements dont understand the stream flow
        MyffmpegInput = gst.element_factory_make('ffmpegcolorspace')
        ImageBinElems.append(MyffmpegInput)
        
        for elem in ImageBinElems : self.add(elem)
        
        gst.element_link_many(MyVideoSrc,MyDemux)
        gst.element_link_many(self.MyVideodecoder,MyffmpegInput)
        
        #this bin has only a src ghostPad
        self.add_pad(gst.GhostPad('src', MyffmpegInput.get_pad('src'))) 

    
    def MyDemux_callback(self, demuxer, pad):
        """ Call back function to create the video pad of dvdemux."""  
        if pad.get_property("template").name_template == "video":
            dec_pad = self.MyVideodecoder.get_pad("sink")
            pad.link(dec_pad)
    
          
class FakeInputBin(gst.Bin) :
    """ gst.Bin for fake input"""
    def __init__(self) :
        """build Fake input bin.
        Description of pipeline : 
        _______________________________________
        |                                      |
        | videotestsrc -->--  capsfilter -->-- | ghostPad -->--
        |______________________________________| 
         
        """
        self.__gobject_init__()
        ImageBinElems=[]
        
        MyVideoSrc = gst.element_factory_make('videotestsrc')
        ImageBinElems.append(MyVideoSrc)
        
        MyVideoSrcFilter = gst.element_factory_make("capsfilter", "MyVideoSrc")
        caps = gst.Caps("video/x-raw-yuv, framerate=(fraction)10/1, width=720,height=576")
        MyVideoSrcFilter.set_property("caps", caps)
        ImageBinElems.append(MyVideoSrcFilter)
        
        for elem in ImageBinElems : self.add(elem)
        
        gst.element_link_many(MyVideoSrc,MyVideoSrcFilter)
        
        #this bin has only a src ghostPad
        self.add_pad(gst.GhostPad('src', MyVideoSrcFilter.get_pad('src'))) 
        


class VideoWidget(object):
    """ class usage to be understood """    
    def __init__(self,DrawingArea):
        """ init videwWidget : update Drawingarea widget with gstreamer propoerties """
        self.VideoArea = DrawingArea
        self.VideoArea.imagesink = None
        self.VideoArea.unset_flags(gtk.DOUBLE_BUFFERED)

    def do_expose_event(self, event):
        """ nbd@grape : with gstreamer is this function useful . Get from player.py example in pygst """
        if self.VideoArea.imagesink:
            self.VideoArea.imagesink.expose()
            return False
        else:
            return True
    
    def set_sink(self, sink):
        assert self.VideoArea.window.xid
        self.VideoArea.imagesink = sink
        self.VideoArea.imagesink.set_xwindow_id(self.VideoArea.window.xid)

class LucioleGstreamer(object) :
    """ Main class for hardware video acquisition with gstreamer"""      
                
    #
    # Class constants declaration
    __HardType=MCONST.HardType
    (NOMIX,MIX) = range(2)
    __MixType=(NOMIX,MIX)
    __ToMixImageName="ToMix.jpeg"
  
 
    #
    # Properties declaration
    #
    def get_inputType(self): return self.__inputType
    def set_inputType(self, value): 
        if value in self.__HardType :
            self.__inputType = value
        else :
            print " Invalide Type",value," ",self.__HardType
    def del_inputType(self): del self.__inputType
    inputType = property(get_inputType, set_inputType, del_inputType, " Hard input type ")
  
    def get_mix(self): return self.__mix
    def set_mix(self, value): 
        if value in self.__MixType :
            self.__mix = value
    def del_mix(self): del self.__mix
    mix = property(get_mix, set_mix, del_mix, " Mix type ")

    def get_CaptureImagePath(self): return self.__CaptureImagePath
    def set_CaptureImagePath(self, value): self.__CaptureImagePath = value
    def del_CaptureImagePath(self): del self.__CaptureImagePath
    CaptureImagePath = property(get_CaptureImagePath, set_CaptureImagePath, del_CaptureImagePath, " Path to save file ")

    def get_alphaStream(self): return self.__alphaStream
    def set_alphaStream(self, value): 
        if ( (value >= 0.0) and  (value <= 1.0) ):
            self.__alphaStream = value
    def del_alphaStream(self): del self.__alphaStream
    alphaStream = property(get_alphaStream, set_alphaStream, del_alphaStream, " Stream alpha value ")
  
    def get_alphaImage(self): return self.__alphaImage
    def set_alphaImage(self, value): 
        if ( (value >= 0.0) and  (value <= 1.0)) :
            self.__alphaImage = value
            if self.MixBin : self.MixBin.alphaValueImage = self.__alphaImage
    def del_alphaImage(self): del self.__alphaImage
    alphaImage = property(get_alphaImage, set_alphaImage, del_alphaImage, " Image alpha value ")
  
    def get_image2Mix(self): return self.__image2Mix
    def set_image2Mix(self, value): 
        if ( os.path.isfile(value) ):
            MT.copyf(value,self.__image2Mix)    
            if self.MixBin : self.MixBin.image2Mix = self.__image2Mix
    def del_image2Mix(self): del self.__image2Mix
    image2Mix = property(get_image2Mix, set_image2Mix, del_image2Mix, " Path to Image To Mix ")

    def get_webcam_data(self): return self.__webcam_data
    def set_webcam_data(self, value): self.__webcam_data = value 
    def del_webcam_data(self): del self.__webcam_data
    webcam_data = property(get_webcam_data, set_webcam_data, del_webcam_data, " webcam parameters for gstreamer : source_input,device,height, width ")
  
    def __init__(self,videowidget,baseDirPath="/dev/tmp", on_error = None, cb_capture_done = None) :
        """ Initialisation of class LucioleGstreamer 
            Input parameters :
            - videowidget : the widget into display acquisition
            - baseDirPath : a base path to store the image to mix
            - cb_on_error : callback to inidicate error
            - cb_capture_done : callback to indicate that capture is done.
        """
      
        self.__inputType = MCONST.FAKE
        self.__alphaStream = 1.0
        self.__alphaImage =0.5
        self.__mix  = self.NOMIX
        self.__image2Mix = os.path.join(baseDirPath,self.__ToMixImageName)
        #init videowidget
        #self.DispWidget = videowidget
        self.videowidget = VideoWidget(videowidget)  
        self._on_error = on_error
        self._cb_capture_done = cb_capture_done

        self.playing = False
        self.MixBin=None

        # init webcam data - set default values
        #  standard vga
        self.__webcam_data={}
        self.__webcam_data['width'] = 640
        self.__webcam_data['height'] = 480
        self.__webcam_data['device'] = "/dev/video0"
        self.__webcam_data['source_input'] = "v4l2src"

    def reset_pipe(self): 
        """ Gstreamer pipe configuration :
        Build the elements according acquisition input, 
        availablity of mixer, and set output to videop display and file.        
        """
        ElementList = []
        self.pipe = gst.Pipeline()
        if (self.__inputType == MCONST.WEBCAM) :
            InputBin = WebcamInputBin(
                        width = self.__webcam_data['width'],
                        height = self.__webcam_data['height'],
                        source_input = self.__webcam_data['source_input'],
                        device = self.__webcam_data['device']
                        )
        elif (self.__inputType == MCONST.DVCAM) :
            InputBin = DvInputBin()
        else :
            InputBin = FakeInputBin()
        ElementList.append(InputBin)
   
        
        
        


        #create tee ( One for file sink, the other for video sink)
        MyTee = gst.element_factory_make("tee", "MyTee")
        ElementList.append(MyTee)
    
        # both "branch" of the tee are queued
        
        #    
        # File queue
        #    
        queueFile = gst.element_factory_make("queue","queueFile")              
        ElementList.append(queueFile)
        #fileSink = SaveCapturedImageBin(self.__CaptureImagePath)
        fileSink = PhotoSaveBin()
        ElementList.append(fileSink)
        
        
        #
        # Display queue
        #
        queueDisplay = gst.element_factory_make("queue","queueDisplay")        
        ElementList.append(queueDisplay)
        if (self.__mix == self.MIX) :
            self.MixBin =MixStreamAndImage(self.__alphaStream, self.__image2Mix, self.__alphaImage)
            ElementList.append(self.MixBin)

        Myffmpeg = gst.element_factory_make('ffmpegcolorspace')          # ffmpegcolorspace for display
        ElementList.append(Myffmpeg)
    
        MyImageSink = gst.element_factory_make('xvimagesink')
        ElementList.append(MyImageSink)
          
        #
        # Add elements to pipeline
        #
        for elem in ElementList : self.pipe.add(elem)
        #
        # link pipeline elements
        #
        #link input
        gst.element_link_many(InputBin,MyTee)
        # link  tee File branch 
        gst.element_link_many(MyTee,queueFile,fileSink)
        #link tee display Branch
        gst.element_link_many(MyTee,queueDisplay)
        if (self.__mix == self.MIX) :
            gst.element_link_many(queueDisplay,self.MixBin,Myffmpeg,MyImageSink)    
        else :
            gst.element_link_many(queueDisplay,Myffmpeg,MyImageSink)
    
        self.on_eos = False            # usage tbd 
        #self.videowidget = VideoWidget(self.DispWidget)
    
        bus = self.pipe.get_bus()
        bus.enable_sync_message_emission()
        bus.add_signal_watch()
        bus.connect('sync-message::element', self.on_sync_message)
        bus.connect('message', self.on_message)

    def on_sync_message(self, bus, message):
        """ Gstreamer sync Message callback. """    
        if message.structure is None:
            return
        if message.structure.get_name() == 'prepare-xwindow-id':
            self.videowidget.set_sink(message.src)
            message.src.set_property('force-aspect-ratio', True)
          
    def on_message(self, bus, message):
        """ Gstreamer message callback"""
        t = message.type
        if t == gst.MESSAGE_ERROR:
            err, debug = message.parse_error()
            error_message = "%s : %s" %(err.copy(),err.message)
            
            self.pipe.set_state(gst.STATE_NULL)
            self.playing = False        # usage tbd
            if self._on_error != None :
                self._on_error(err.code,error_message)
                self.playing = False        # usage tbd
        elif t == gst.MESSAGE_EOS:
            if self.on_eos:
                self.on_eos()
            self.playing = False     # usage tbd
        elif t == gst.MESSAGE_STATE_CHANGED :
            pass
    

    def pause(self):
        gst.info("pausing pipe")
        self.pipe.set_state(gst.STATE_PAUSED)
        state = self.pipe.get_state(timeout=1000) # wait the state to GST_STATE_CHANGE_NO_PREROLL with a timeout od 1 micro second
        self.playing = False

    def play(self):
        gst.info("playing pipe")
        self.pipe.set_state(gst.STATE_PLAYING)
        self.playing = True
    
    def stop(self):
        self.pipe.set_state(gst.STATE_NULL)
        gst.info("stopped pipe")
        self.playing = False
    
    def get_state(self, timeout=1):
        return self.pipe.get_state(timeout=timeout)

    def is_playing(self):
        return self.playing
    
    def capture(self) :
        """ capture is requested : 
            - get jpegnec src pad (output pad of jpege enc)
            - and add a buffer probe with callback when frame is encoded
        """
        # only one src pad with jpegenc
        pad = self.pipe.get_by_name("MyJpegenc").src_pads().next()
        # add the probe
        self.grabprobe = pad.add_buffer_probe(self._cb_process_frame)
    
        
    def _cb_process_frame(self, pad, buffer):
        """ Callbak to inidicate that buffer is available from probe """
        # remove the prove
        pad.remove_buffer_probe(self.grabprobe)
        
        # save to file 
        file = open(self.__CaptureImagePath,'w')
        file.write(buffer)
        file.close()
        
        # callback to indicate that job is done
        # the execution here is not in the same thread as gui , so call capture_done with idle add
        if self._cb_capture_done != None : gobject.idle_add(self._cb_capture_done) 

        return True						
